/*=============================================================
 * remove.c -- Remove child from family or spouse from family.
 * Copyright(c) 1992 by Thomas T. Wetmore; all rights reserved.
 *   Version 2.3.4 - 24 Jun 93 - controlled
 *   Version 2.3.5 - 01 Sep 93 - modified
 *=============================================================
 */
#include <stdio.h>
#include "standard.h"
#include "table.h"
#include "gedcom.h"

extern STRING idcrmv, ntchld, ntprnt, idsrmv, idsrmf, normls, cfcrmv;
extern STRING okcrmv, ntsinf, cfsrmv, oksrmv;

/*==========================================
 * remove_child -- Remove child from family.
 *========================================*/
BOOLEAN remove_child (child, nolast)
NODE child;	/* child */
BOOLEAN nolast;	/* don't remove last in family? */
{
	NODE famc, husb, wife, chil, rest, last, node;
	STRING key;

/* Identify child and check for FAMC node */
	if (!child) child = ask_for_indi(idcrmv, FALSE, FALSE);
	if (!child) return FALSE;
	if (!FAMC(child)) {
		message(ntchld);
		return FALSE;
	}
	ASSERT(famc = indi_to_famc(child));
	split_fam(famc, &husb, &wife, &chil, &rest);
	join_fam(famc, husb, wife, chil, rest);
	if (nolast && num_fam_xrefs(husb, wife, chil) < 2) {
		message(normls);
		return FALSE;
	}
	if (!ask_yes_or_no(cfcrmv)) return TRUE;

/* Remove FAMC line from child */
	last = NULL;
	node = nchild(child);
	while (node && strcmp("FAMC", ntag(node))) {
		last = node;
		node = nsibling(node);
	}
	ASSERT(node);
	nsibling(last) = nsibling(node);
	stdfree(node);

/* Remove CHIL line from family */
	split_fam(famc, &husb, &wife, &chil, &rest);
	last = NULL;
	node = chil;
	key = nxref(child);
	while (node && !eqstr(nval(node), key)) {
		last = node;
		node = nsibling(node);
	}
	ASSERT(node);
	if (!last)
		chil = nsibling(node);
	else
		nsibling(last) = nsibling(node);
	stdfree(node);

/* Update database with changed records */
	join_fam(famc, husb, wife, chil, rest);
	if (!husb && !wife && !chil)
		delete_fam(famc);
	else
		fam_to_dbase(famc);
	indi_to_dbase(child);
	message(okcrmv);
	return TRUE;
}
/*============================================
 * remove_spouse -- Remove spouse from family.
 *==========================================*/
BOOLEAN remove_spouse (indi, fam, nolast)
NODE indi, fam;
BOOLEAN nolast;	/* don't remove last member of family? */
{
	INT i, sex;
	NODE husb, wife, chil, rest, node, last;
	KEY key = NULL;

/* Identify spouse to remove */
	if (!indi) indi = ask_for_indi(idsrmv, FALSE, FALSE);
	if (!indi) return FALSE;
	if (!FAMS(indi)) {
		message(ntprnt);
		return FALSE;
	}
	sex = SEX(indi);
	ASSERT(sex == SEX_MALE || sex == SEX_FEMALE);

/* Identify family to remove spouse from */
	if (!fam) fam = choose_family(indi, "e", idsrmf);
	if (!fam) return FALSE;
	key = nxref(fam);

/* Make sure spouse is in family and can be removed */
	split_fam(fam, &husb, &wife, &chil, &rest);
	join_fam(fam, husb, wife, chil, rest);
	node = (sex == SEX_MALE) ? husb : wife;
	if (!eqstr(nxref(indi), nval(node))) {
		message(ntsinf);
		return FALSE;
	}
	if (nolast && num_fam_xrefs(husb, wife, chil) < 2) {
		message(normls);
		return FALSE;
	}
	if (!ask_yes_or_no(cfsrmv)) return FALSE;

/* Remove FAMS line from spouse and update database */
	last = NULL;
	node = nchild(indi);
	while (node && !eqstr("FAMS", ntag(node))) {
		last = node;
		node = nsibling(node);
	}
	ASSERT(last);
	while (node && eqstr("FAMS", ntag(node)) && !eqstr(key, nval(node))) {
		last = node;
		node = nsibling(node);
	}
	ASSERT(node && eqstr("FAMS", ntag(node)));
	nsibling(last) = nsibling(node);
	stdfree(node);
	indi_to_dbase(indi);

/* Modify family tree and update database */
	split_fam(fam, &husb, &wife, &chil, &rest);
	if (sex == SEX_MALE) {
		stdfree(husb);
		husb = NULL;
	} else {
		stdfree(wife);
		wife = NULL;
	}
	join_fam(fam, husb, wife, chil, rest);
	if (!husb && !wife && !chil)
		delete_fam(fam);
	else
		fam_to_dbase(fam);
	message(oksrmv);
	return TRUE;
}
/*========================================================
 * num_fam_xrefs -- Find number of person links in family.
 *======================================================*/
INT num_fam_xrefs (husb, wife, chil)
NODE husb, wife, chil;
{
	INT num = 0;
	if (husb) num++;
	if (wife) num++;
	return num + node_list_length(chil);
}
