/*-------------------------------------------------------------------------
        panel.c - part of PANELS implementation for ncurses/Linux
        zmbenhal@netcom.com; author: wht@n4hgf.Mt-Park.GA.US
--------------------------------------------------------------------------*/

#include <curses.h>
#include "panel.h"

#define getbegx(w)      (w)->_begx
#define getbegy(w)      (w)->_begy
#define getmaxx(w)      (w)->_maxx
#define getmaxy(w)      (w)->_maxy

PANEL *__bottom_panel = (PANEL *)0;
PANEL *__top_panel = (PANEL *)0;
PANEL __stdscr_pseudo_panel = { (WINDOW *)0 };

#define Wnoutrefresh(pan) wnoutrefresh((pan)->win)
#define Touchpan(pan) touchwin((pan)->win)
#define Touchline(pan,start,count) touchline((pan)->win,start,count)

/*+-------------------------------------------------------------------------
        __panels_overlapped(pan1,pan2) - check panel overlapped
--------------------------------------------------------------------------*/
static int
__panels_overlapped(pan1,pan2)
register PANEL *pan1;
register PANEL *pan2;
{
        if(!pan1 || !pan2) return 0;
        if((pan1->wstarty >= pan2->wstarty) && (pan1->wstarty < pan2->wendy) &&
                (pan1->wstartx >= pan2->wstartx) && (pan1->wstartx < pan2->wendx))
                return(1);
        if((pan1->wstarty >= pan1->wstarty) && (pan2->wstarty < pan1->wendy) &&
                (pan1->wstartx >= pan1->wstartx) && (pan2->wstartx < pan1->wendx))
                return 1;
        return 0;
}

/*+-------------------------------------------------------------------------
        __free_obscure(pan)
--------------------------------------------------------------------------*/
static void
__free_obscure(pan)
PANEL *pan;
{
PANELOBS *tobs = pan->obscure;          /* "this" one */
PANELOBS *nobs;                         /* "next" one */

        while(tobs) {
                nobs = tobs->above;
                free((char *)tobs);
                tobs = nobs;
        }
        pan->obscure = (PANELOBS *)0;
}

/*+-------------------------------------------------------------------------
        __override(pan,show)
--------------------------------------------------------------------------*/
static void
__override(pan,show)
PANEL *pan;
int show;
{
register y;
register PANEL *pan2;
PANELOBS *tobs = pan->obscure;                          /* "this" one */

        if(show == 1)
                Touchpan(pan);
        else if(!show) {
                Touchpan(pan);
/*
                Touchline(&__stdscr_pseudo_panel,pan->wendy,getmaxy(pan->win));
*/
                Touchpan(&__stdscr_pseudo_panel);
        } else if(show == -1) {
                while(tobs && (tobs->pan != pan))
                        tobs = tobs->above;
        }

        while(tobs) {
                if((pan2 = tobs->pan) != pan) {
                        for(y = pan->wstarty; y < pan->wendy; y++) {
                                if( (y >= pan2->wstarty) && (y < pan2->wendy) &&
                                        ((is_linetouched(pan->win,y - pan->wstarty) == 1) ||
                                        (is_linetouched(stdscr,y) == 1)))
                                {
                                        Touchline(pan2,y - pan2->wstarty,1);
                                }
                        }
                }
                tobs = tobs->above;
        }
}

/*+-------------------------------------------------------------------------
        __calculate_obscure()
--------------------------------------------------------------------------*/
static void
__calculate_obscure()
{
PANEL *pan;
register PANEL *pan2;
register PANELOBS *tobs;                /* this one */
PANELOBS *lobs = (PANELOBS *)0;         /* last one */

        pan = __bottom_panel;
        while(pan) {
                if(pan->obscure)
                        __free_obscure(pan);
                lobs = (PANELOBS *)0;           /* last one */
                pan2 = __bottom_panel;
                while(pan2) {
                        if(__panels_overlapped(pan,pan2)) {
                                if(!(tobs = (PANELOBS *)malloc(sizeof(PANELOBS))))
                                        return;
                                tobs->pan = pan2;
                                tobs->above = (PANELOBS *)0;
                                if(lobs)
                                        lobs->above = tobs;
                                else
                                        pan->obscure = tobs;
                                lobs  = tobs;
                        }
                        pan2 = pan2->above;
                }
                __override(pan,1);
                pan = pan->above;
        }

}

/*+-------------------------------------------------------------------------
        __panel_is_linked(pan) - check to see if panel is in the stack
--------------------------------------------------------------------------*/
static int
__panel_is_linked(pan)
PANEL *pan;
{
register PANEL *pan2 = __bottom_panel;

        while(pan2) {
                if(pan2 == pan)
                        return(1);
                pan2 = pan2->above;
        }
        return OK;
}

/*+-------------------------------------------------------------------------
        __panel_link_top(pan) - link panel into stack at top
--------------------------------------------------------------------------*/
static void __panel_link_top (pan)
PANEL *pan;
{

        pan->above = (PANEL *)0;
        pan->below = (PANEL *)0;
        if(__top_panel) {
                __top_panel->above = pan;
                pan->below = __top_panel;
        }
        __top_panel = pan;
        if(!__bottom_panel)
                __bottom_panel = pan;
        __calculate_obscure();

}

/*+-------------------------------------------------------------------------
        __panel_link_bottom(pan) - link panel into stack at bottom
--------------------------------------------------------------------------*/
static void __panel_link_bottom(pan)
PANEL *pan;
{

        pan->above = (PANEL *)0;
        pan->below = (PANEL *)0;
        if(__bottom_panel) {
                __bottom_panel->below = pan;
                pan->above = __bottom_panel;
        }
        __bottom_panel = pan;
        if(!__top_panel)
                __top_panel = pan;
        __calculate_obscure();
}

/*+-------------------------------------------------------------------------
        __panel_unlink(pan) - unlink panel from stack
--------------------------------------------------------------------------*/
static void __panel_unlink (pan)
PANEL *pan;
{
register PANEL *prev;
register PANEL *next;

        __override(pan,0);
        __free_obscure(pan);

        prev = pan->below;
        next = pan->above;

        if(prev) {              /* if non-zero, don't update list head */
                prev->above = next;
                if(next)
                        next->below = prev;
        }
        else if(next)
                next->below = prev;
        if(pan == __bottom_panel)
                __bottom_panel = next;
        if(pan == __top_panel)
                __top_panel = prev;

        __calculate_obscure();

        pan->above = (PANEL *)0;
        pan->below = (PANEL *)0;
}

/*+-------------------------------------------------------------------------
        panel_window(pan) - get window associated with panel
--------------------------------------------------------------------------*/
WINDOW *panel_window(pan)
PANEL *pan;
{
        return(pan->win);
}

/*+-------------------------------------------------------------------------
        update_panels() - wnoutrefresh windows in an orderly fashion
--------------------------------------------------------------------------*/
void
update_panels ()
{
PANEL *pan;

        pan = __bottom_panel;
        while(pan) {
                __override(pan,-1);
                pan = pan->above;
        }

        if(is_wintouched(stdscr))
                Wnoutrefresh(&__stdscr_pseudo_panel);
        
        if(pan = __bottom_panel) {
                while(pan) {
                        if(is_wintouched(pan->win))
                                Wnoutrefresh(pan);
                        pan = pan->above;
                }
        }
}

/*+-------------------------------------------------------------------------
        hide_panel(pan) - remove a panel from stack
--------------------------------------------------------------------------*/
int hide_panel (pan)
register PANEL *pan;
{
        if(!pan) return(ERR);
        if(!__panel_is_linked(pan)) {
                pan->above = (PANEL *)0;
                pan->below = (PANEL *)0;
                return(ERR);
        }
        __panel_unlink(pan);
        return OK;
}

/*+-------------------------------------------------------------------------
        show_panel(pan) - place a panel on top of stack
may already be in stack
--------------------------------------------------------------------------*/
int show_panel (pan)
register PANEL *pan;
{

        if(!pan) return(ERR);
        if(pan == __top_panel) return(OK);
        if(__panel_is_linked(pan))
                (void) hide_panel(pan);
        __panel_link_top(pan);
        return OK;
}

/*+-------------------------------------------------------------------------
        top_panel(pan) - place a panel on top of stack
--------------------------------------------------------------------------*/
int top_panel(pan)
register PANEL *pan;
{
        return show_panel(pan);
}

/*+-------------------------------------------------------------------------
        del_panel(pan) - remove a panel from stack, if in it, and free struct
--------------------------------------------------------------------------*/
int del_panel(pan)
register PANEL *pan;
{
        if(pan) {
                if(__panel_is_linked(pan))
                        (void)hide_panel(pan);
                free((char *)pan);
                return OK;
        }
        return ERR;
}

/*+-------------------------------------------------------------------------
        bottom_panel(pan) - place a panel on bottom of stack
may already be in stack
--------------------------------------------------------------------------*/
int bottom_panel(pan)
register PANEL *pan;
{
        if(!pan) return(ERR);
        if(pan == __bottom_panel) return(OK);
        if(__panel_is_linked(pan))
                (void)hide_panel(pan);
        __panel_link_bottom(pan);
        return OK;
}

/*+-------------------------------------------------------------------------
        new_panel(win) - create a panel and place on top of stack
--------------------------------------------------------------------------*/
PANEL *new_panel(win)
WINDOW *win;
{
PANEL *pan = (PANEL *)malloc(sizeof(PANEL));

        if(!__stdscr_pseudo_panel.win) {
                __stdscr_pseudo_panel.win = stdscr;
                __stdscr_pseudo_panel.wstarty = 0;
                __stdscr_pseudo_panel.wstartx = 0;
                __stdscr_pseudo_panel.wendy = LINES;
                __stdscr_pseudo_panel.wendx = COLS;
                __stdscr_pseudo_panel.user = "stdscr";
                __stdscr_pseudo_panel.obscure = (PANELOBS *)0;
        }
        if (pan) {
                pan->win = win;
                pan->above = (PANEL *)0;
                pan->below = (PANEL *)0;
                pan->wstarty = getbegy(win);
                pan->wstartx = getbegx(win);
                pan->wendy = pan->wstarty + getmaxy(win);
                pan->wendx = pan->wstartx + getmaxx(win);
                pan->user = (char *)0;
                pan->obscure = (PANELOBS *)0;
                (void)show_panel(pan);
        }
        return(pan);
}

/*+-------------------------------------------------------------------------
        panel_above(pan)
--------------------------------------------------------------------------*/
PANEL *panel_above(pan)
PANEL *pan;
{
        return pan ? pan->above : __bottom_panel;
}

/*+-------------------------------------------------------------------------
        panel_below(pan)
--------------------------------------------------------------------------*/
PANEL * panel_below(pan)
PANEL *pan;
{
        return pan ? pan->below : __top_panel;
}

/*+-------------------------------------------------------------------------
        set_panel_userptr(pan,uptr)
--------------------------------------------------------------------------*/
int set_panel_userptr(pan,uptr)
PANEL *pan;
char *uptr;
{
        if(!pan) return ERR;
        pan->user = uptr;
        return OK;
}

/*+-------------------------------------------------------------------------
        panel_userptr(pan)
--------------------------------------------------------------------------*/
char *panel_userptr(pan)
PANEL *pan;
{
        return pan ? pan->user : (char *) 0;
}

/*+-------------------------------------------------------------------------
        move_panel(pan,starty,startx)
--------------------------------------------------------------------------*/
int move_panel (pan, starty, startx)
PANEL *pan;
int starty, startx;
{
WINDOW *win;

        if(!pan) return(ERR);
        if(__panel_is_linked(pan))
                __override(pan,0);
        win = pan->win;
        if(mvwin(win,starty,startx))
                return(ERR);
        pan->wstarty = getbegy(win);
        pan->wstartx = getbegx(win);
        pan->wendy = pan->wstarty + getmaxy(win);
        pan->wendx = pan->wstartx + getmaxx(win);
        if(__panel_is_linked(pan))
                __calculate_obscure();
        return(OK);
}

/*+-------------------------------------------------------------------------
        replace_panel(pan,win)
--------------------------------------------------------------------------*/
int replace_panel(pan,win)
PANEL *pan;
WINDOW *win;
{
        if(!pan) return(ERR);
        if(__panel_is_linked(pan))
                __override(pan,0);
        pan->win = win;
        if(__panel_is_linked(pan))
                __calculate_obscure();
        return(OK);
}

/*+-------------------------------------------------------------------------
        panel_hidden(pan)
--------------------------------------------------------------------------*/
int panel_hidden(pan)
PANEL *pan;
{
        if(!pan) return ERR;
#if 0
        return(__panel_is_linked(pan) ? ERR : OK);
#endif
        return !__panel_is_linked(pan);
}
