/*===============================================================
 * more.c -- More builtin functions of report interpreter.
 * Copyright(c) 1992 by Thomas T. Wetmore IV; all rights reserved.
 *   Version 2.3.4 - 24 Jun 93 - controlled
 *   Version 2.3.5 - 26 Sep 93 - modified
 *================================================================
 */
#include <stdio.h>
#include "standard.h"
#include "table.h"
#include "gedcom.h"
#include "cache.h"
#include "interp.h"
#include "indiseq.h"

extern STRING notone, ifone;
extern NODE format_and_choose_indi();

/*==============================================================
 * _extractnames -- Extract name parts from person or NAME node.
 *   usage: extractnames(NODE, LIST, VARB, VARB) -> VOID
 *============================================================*/
WORD _extractnames (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP nexp = (INTERP) ielist(node);
	INTERP lexp = inext(nexp);
	INTERP lvar = inext(lexp);
	INTERP svar = inext(lvar);
	NODE lin = (NODE) evaluate(nexp, stab, eflg);
	LIST list;
	STRING str;
	INT len, sind;
	if (*eflg || !lin) return NULL;
	list = (LIST) evaluate(lexp, stab, eflg);
	if (*eflg || !list) return NULL;
	*eflg = TRUE;
	if (!iistype(lvar, IIDENT)) return NULL;
	if (!iistype(svar, IIDENT)) return NULL;
	*eflg = FALSE;
	if (strcmp("NAME", ntag(lin)) && !(lin = NAME(lin))) return NULL;
	str = nval(lin);
	if (!str || *str == 0) return NULL;
	name_to_list(str, list, &len, &sind);
	assign_iden(stab, iident(lvar), len);
	assign_iden(stab, iident(svar), sind);
	return NULL;
}
/*===============================================================
 * _extractplaces -- Extract place parts from event or PLAC NODE.
 *   usage: extractplaces(NODE, LIST, VARB) -> VOID
 *=============================================================*/
WORD _extractplaces (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP nexp = (INTERP) ielist(node);
	INTERP lexp = inext(nexp);
	INTERP lvar = inext(lexp);
	NODE lin = (NODE) evaluate(nexp, stab, eflg);
	LIST list;
	STRING str;
	INT len;
	if (*eflg || !lin) return NULL;
	list = (LIST) evaluate(lexp, stab, eflg);
	if (*eflg || !list) return NULL;
	*eflg = TRUE;
	if (!iistype(lvar, IIDENT)) return NULL;
	assign_iden(stab, iident(lvar), 0);
	*eflg = FALSE;
	if (strcmp("PLAC", ntag(lin)) && !(lin = PLAC(lin))) return NULL;
	str = nval(lin);
	if (!str || *str == 0) return NULL;
	place_to_list(str, list, &len);
	assign_iden(stab, iident(lvar), len);
	return NULL;
}
/*===================================
 * _database -- Return database name.
 *   usage: database() -> STRING
 *=================================*/
WORD _database (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	extern STRING btreepath;
	*eflg = FALSE;
	return btreepath;
}
/*============================================
 * _index -- Find nth occurrence of substring.
 *   usage: index(STRING, STRING, INT) -> INT
 *==========================================*/
WORD _index (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	STRING sub, str = (STRING) evaluate(arg, stab, eflg);
	INT num, index();
	if (*eflg) return NULL;
	arg = inext(arg);
	sub = (STRING) evaluate(arg, stab, eflg);
	if (*eflg) return NULL;
	arg = inext(arg);
	num = (INT) evaluate(arg, stab, eflg);
	if (*eflg) return NULL;
	return (WORD) index(str, sub, num);
}
/*===============================================
 * _substring -- Find substring of string.
 *   usage: substring(STRING, INT, INT) -> STRING
 *=============================================*/
WORD _substring (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	STRING substring(), str = (STRING) evaluate(arg, stab, eflg);
	INT lo, hi;
	if (*eflg) return NULL;
	arg = inext(arg);
	lo = (INT) evaluate(arg, stab, eflg);
	if (*eflg) return NULL;
	arg = inext(arg);
	hi = (INT) evaluate(arg, stab, eflg);
	if (*eflg) return NULL;
	return (WORD) substring(str, lo, hi);
}
/*=======================================================
 * index -- Find nth occurrence of sub in str (uses KMP).
 *=====================================================*/
static char pi[MAXLINELEN];
static INT index (str, sub, num)
STRING str, sub;
INT num;
{
        INT n = strlen(str), m = strlen(sub), q = 0, i, found = 0;
        compute_pi(sub);
        for (i = 1; i <= n; i++) {
                while (q > 0 && sub[q] != str[i-1])
                        q = pi[q];
                if (sub[q] == str[i-1]) q++;
                if (q == m) {
                        if (++found == num) return i - m + 1;
                        q = pi[q];
                }
        }
        return 0;
}
/*=========================================
 * compute_pi -- Support routine for index.
 *=======================================*/
static compute_pi (sub)
STRING sub;
{
        INT m = strlen(sub), k = 0, q;
        pi[1] = 0;
        for (q = 2; q <= m; q++) {
                while (k > 0 && sub[k] != sub[q-1])
                        k = pi[k];
                if (sub[k] == sub[q-1]) k++;
                pi[q] = k;
        }
}
/*===============================
 * substring -- Return substring.
 *=============================*/
STRING substring (s, i, j)
STRING s;
INT i, j;
{
	static char scratch[MAXLINELEN+1];
	STRING strncpy();
	if (!s || *s == 0 || i <= 0 || i > j || j > strlen(s)) return NULL;
	strncpy(scratch, &s[i-1], j-i+1);
	scratch[j-i+1] = 0;
	return (STRING) scratch;
}
/*================================================
 * chooseindi -- Have user choose person from set.
 *   usage: chooseindi(SET) -> INDI
 *==============================================*/
WORD _chooseindi (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INDISEQ seq = (INDISEQ) evaluate(ielist(node), stab, eflg);
	NODE indi;
	if (!seq || *eflg || length_indiseq(seq) < 1) return NULL;
	indi = format_and_choose_indi(seq, FALSE, FALSE, TRUE, ifone, notone);
	if (!indi) return NULL;
	return (WORD) indi_to_cacheel(indi);
}
/*==================================================
 * choosesubset -- Have user choose subset from set.
 *   usage: choosesubset(SET) -> SET
 *================================================*/
WORD _choosesubset (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	STRING msg;
	INDISEQ new, seq = (INDISEQ) evaluate(ielist(node), stab, eflg);
	if (!seq || *eflg || length_indiseq(seq) < 1) return NULL;
	new = copy_indiseq(seq);
	format_indiseq(new, FALSE, FALSE);
	msg = (length_indiseq(new) > 1) ? notone : ifone;
	new = (INDISEQ) choose_list_from_indiseq(msg, new);
	return (WORD) new;
}
/*===========================================================
 * choosechild -- Have user choose child of person or family.
 *   usage: choosechild(INDI|FAM) -> INDI
 *=========================================================*/
WORD _choosechild (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	CACHEEL cel = (CACHEEL) evaluate(ielist(node), stab, eflg);
	STRING key;
	NODE indi, fam;
	INDISEQ seq;
	if (*eflg || !cel) return NULL;
	key = ckey(cel);
	if (*key == 'I') {
		indi = key_to_indi(key);
		seq = indi_to_children(indi);
		if (!seq || length_indiseq(seq) < 1) return NULL;
		indi = format_and_choose_indi(seq, FALSE, FALSE, TRUE,
		    ifone, notone);
		remove_indiseq(seq, FALSE);
		if (!indi) return NULL;
		return (WORD) indi_to_cacheel(indi);
	} else if (*key == 'F') {
		fam = key_to_fam(key);
		seq = fam_to_children(fam);
		if (!seq || length_indiseq(seq) < 1) return NULL;
		indi = format_and_choose_indi(seq, FALSE, FALSE, TRUE,
		    ifone, notone);
		remove_indiseq(seq, FALSE);
		if (!indi) return NULL;
		return (WORD) indi_to_cacheel(indi);
	}
	return NULL;
}
/*===================================================
 * choosespouse -- Have user choose spouse of person.
 *   usage: choosespouse(INDI) -> INDI
 *=================================================*/
WORD _choosespouse (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	INDISEQ seq;
	if (*eflg) return NULL;
	seq = indi_to_spouses(indi);
	if (!seq || length_indiseq(seq) < 1) return NULL;
	indi = format_and_choose_indi(seq, FALSE, TRUE, TRUE, ifone, notone);
	remove_indiseq(seq, FALSE);
	if (!indi) return NULL;
	return (WORD) indi_to_cacheel(indi);
}
/*================================================
 * choosefam -- Have user choose family of person.
 *   usage: choosefam (INDI) -> FAM
 *==============================================*/
WORD _choosefam (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE fam, indi = eval_indi(ielist(node), stab, eflg, NULL);
	INDISEQ seq;
	if (*eflg) return NULL;
	seq = indi_to_families(indi);
	if (!seq || length_indiseq(seq) < 1) return NULL;
	fam = format_and_choose_indi(seq, TRUE, TRUE, TRUE, ifone, notone);
	remove_indiseq(seq, FALSE);
	if (!fam) return NULL;
	return (WORD) fam_to_cacheel(fam);
}
/*=====================================================
 * menuchoose -- Have user choose from list of options.
 *   usage: menuchoose (LIST [,STRING]) -> INT
 *===================================================*/
WORD _menuchoose (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INT i, len;
	STRING msg, *strngs;
	STRING ttl = "Please choose from the following list.";
	INTERP arg = (INTERP) ielist(node);
	LIST list = (LIST) evaluate(arg, stab, eflg);
	if (!list || *eflg || length_list(list) < 1) return (WORD) 0;
	msg = NULL;
	arg = (INTERP) inext(arg);
	if (arg) msg = evaluate(arg, stab, eflg);
	if (*eflg) return (WORD) 0;
	if (msg && *msg) ttl = msg;
	len = length_list(list);
	strngs = (STRING *) stdalloc(len*sizeof(STRING));
	i = 0;
	FORLIST(list, el)
		strngs[i++] = (STRING) el;
	ENDLIST
	i = choose_from_list(ttl, len, strngs);
	stdfree(strngs);
	return (WORD) (i + 1);
}
/*=================================
 * system -- Run a shell command.
 *   usage: system (STRING) -> VOID
 *===============================*/
WORD _system (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	STRING cmd = (STRING) evaluate(ielist(node), stab, eflg);
	if (*eflg || !cmd || *cmd == 0) return NULL;
	endwin();
	system("clear");
	system(cmd);
	return NULL;
}
/*==============================================
 * firstindi -- Return first person in database.
 *   usage: firstindi() -> INDI
 *============================================*/
WORD _firstindi (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi;
	static char key[10];
	STRING record;
	INT len, i = 0;
	*eflg = FALSE;
	while (TRUE) {
		sprintf(key, "I%d", ++i);
		if (!(record = retrieve_record(key, &len)))
			return NULL;
		if (!(indi = string_to_node(record))) {
			stdfree(record);
			continue;
		}
		stdfree(record);
		free_nodes(indi);/*yes*/
		return (WORD) indi_to_cacheel(indi);
	}
}
/*============================================
 * nextindi -- Return next person in database.
 *   usage: nextindi(INDI) -> INDI
 *==========================================*/
WORD _nextindi (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	static char key[10];
	STRING record;
	INT len, i;
	if (*eflg) return NULL;
	strcpy(key, indi_to_key(indi));
	i = atoi(&key[1]);
	while (TRUE) {
		sprintf(key, "I%d", ++i);
		if (!(record = retrieve_record(key, &len)))
			return NULL;
		if (!(indi = string_to_node(record))) {
			stdfree(record);
			continue;
		}
		stdfree(record);
		free_nodes(indi);/*yes*/
		return (WORD) indi_to_cacheel(indi);
	}
}
/*================================================
 * previndi -- Return previous person in database.
 *   usage: previndi(INDI) -> INDI
 *==============================================*/
WORD _previndi (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	static char key[10];
	STRING record;
	INT len, i;
	if (*eflg) return NULL;
	strcpy(key, indi_to_key(indi));
	i = atoi(&key[1]);
	while (TRUE) {
		sprintf(key, "I%d", --i);
		if (!(record = retrieve_record(key, &len)))
			return NULL;
		if (!(indi = string_to_node(record))) {
			stdfree(record);
			continue;
		}
		stdfree(record);
		free_nodes(indi);/*yes*/
		return (WORD) indi_to_cacheel(indi);
	}
}
/*============================================
 * lastindi -- Return last person in database.
 *   usage: lastindi() -> INDI
 *==========================================*/
WORD _lastindi (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
}
/*=============================================
 * firstfam -- Return first family in database.
 *   usage: firstfam() -> FAM
 *===========================================*/
WORD _firstfam (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE fam;
	static char key[10];
	STRING record;
	INT len, i = 0;
	*eflg = FALSE;
	while (TRUE) {
		sprintf(key, "F%d", ++i);
		if (!(record = retrieve_record(key, &len)))
			return NULL;
		if (!(fam = string_to_node(record))) {
			stdfree(record);
			continue;
		}
		stdfree(record);
		free_nodes(fam);/*yes*/
		return (WORD) fam_to_cacheel(fam);
	}
}
/*===========================================
 * nextfam -- Return next family in database.
 *   usage: nextfam(FAM) -> FAM
 *=========================================*/
WORD _nextfam (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE fam = eval_fam(ielist(node), stab, eflg, NULL);
	static char key[10];
	STRING record;
	INT len, i;
	if (*eflg) return NULL;
	strcpy(key, fam_to_key(fam));
	i = atoi(&key[1]);
	while (TRUE) {
		sprintf(key, "F%d", ++i);
		if (!(record = retrieve_record(key, &len)))
			return NULL;
		if (!(fam = string_to_node(record))) {
			stdfree(record);
			continue;
		}
		stdfree(record);
		free_nodes(fam);/*yes*/
		return (WORD) fam_to_cacheel(fam);
	}
}
/*===============================================
 * prevfam -- Return previous family in database.
 *   usage: prevfam(FAM) -> FAM
 *=============================================*/
WORD _prevfam (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE fam = eval_fam(ielist(node), stab, eflg, NULL);
	static char key[10];
	STRING record;
	INT len, i;
	if (*eflg) return NULL;
	strcpy(key, fam_to_key(fam));
	i = atoi(&key[1]);
	while (TRUE) {
		sprintf(key, "F%d", --i);
		if (!(record = retrieve_record(key, &len)))
			return NULL;
		if (!(fam = string_to_node(record))) {
			stdfree(record);
			continue;
		}
		stdfree(record);
		free_nodes(fam);/*yes*/
		return (WORD) fam_to_cacheel(fam);
	}
}
/*============================================
 * lastfam -- Return last family in database.
 *   usage: lastfam() -> FAM
 *==========================================*/
WORD _lastfam (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
}
