/*====================================================================
 * interp.c -- This file contains the higher level interpret functions
 *   of the report interpreter.
 * Copyright(c) 1992 by Thomas T. Wetmore IV; all rights reserved.
 *   Version 2.3.4 - 24 Jun 93 - controlled
 *   Version 2.3.5 - 16 Aug 93 - modified
 *=====================================================================
 */
#include <stdio.h>
#include "standard.h"
#include "table.h"
#include "gedcom.h"
#include "interp.h"
#include "cache.h"
#include "indiseq.h"

extern STRING llprograms;

FILE *ifp = NULL, *ofp = NULL;
TABLE proctab, globtab;
INT rplineno;
INT semerrors = 0;
STRING ierror = "Error in line %d: ";
static STRING noreport = "No report was generated.";

/*======================================
 * initinterp -- Initialize interpreter.
 *====================================*/
initinterp ()
{
	initrassa();
	initset();
	semerrors = 0;
}
/*====================================
 * finishinterp -- Finish interpreter.
 *==================================*/
finishinterp ()
{
	finishrassa();
}
/*==============================================
 * interp_report -- Main routine of interpreter.
 *============================================*/
interp_report ()
{
	FILE *ask_for_file();
	INTERP proc;
	TABLE stab;

/* Get names of report program */
	ifp = ask_for_file("r", "What is the name of the report program?",
	    NULL, llprograms);
	if (!ifp)  {
		message(noreport);
		return;
	}
	ofp = NULL;

/* Create procedure table and parse report program */
	initinterp();
	proctab = create_table();
	globtab = create_table();
	rplineno = 1;
	if (yyparse()) return;
	if (semerrors) {
		wprintf("The report program contains semantic errors.\n");
		return;
	}

/* Interpret main routine; create and pass down empty symbol table */
	if (!(proc = (INTERP) valueof(proctab, "main"))) {
		message("The report program needs a main procedure.");
		remove_table(proctab, NULL);
		remove_table(globtab, NULL);
		return;
	}
	stab = create_table();
	if (interpret((INTERP) ibody(proc), stab))
		message("The report was successfully generated.");
	else {
		message("The report was not generated because of errors.");
	}
	remove_table(proctab, NULL);
	remove_table(globtab, NULL);
	remove_table(stab, NULL);
	finishinterp();
	fclose(ifp);
	fclose(ofp);
}
/*=================================================
 * interpret -- Interpret a list of template nodes.
 *===============================================*/
#define rpline (node->i_line)
BOOLEAN interpret (node, stab)
INTERP node;
TABLE stab;
{
	STRING str;
	BOOLEAN eflg;
	while (node) {
		switch (itype(node)) {
		case ILITERAL:
			poutput(iliteral(node));
			break;
		case IIDENT:
			str = evaluate_iden(node, stab, &eflg);
			if (eflg) {
				wprintf(ierror, rpline);
				wprintf("ident: %s\n", iident(node));
				return FALSE;
			}
			poutput(str);
			break;
		case IFUNC:
			str = evaluate_func(node, stab, &eflg);
			if (eflg) {
				wprintf(ierror, rpline);
				wprintf("func: %s\n", iname(node));
				return FALSE;
			}
			if (str) poutput(str);
			break;
		case IPROC:
			FATAL();
		case ICHILDREN:
			if (!interp_children(node, stab)) {
				wprintf(ierror, rpline);
				wprintf("children\n");
				return FALSE;
			}
			break;
		case ISPOUSES:
			if (!interp_spouses(node, stab)) {
				wprintf(ierror, rpline);
				wprintf("spouses\n");
				return FALSE;
			}
			break;
		case IFAMILIES:
			if (!interp_families(node, stab)) {
				wprintf(ierror, rpline);
				wprintf("families\n");
				return FALSE;
			}
			break;
		case IINDICES:
			if (!interp_indisetloop(node, stab)) {
				wprintf(ierror, rpline);
				wprintf("indisetloop\n");
				return FALSE;
			}
			break;
		case IINDI:
			if (!interp_forindi(node, stab)) {
				wprintf(ierror, rpline);
				wprintf("forindi\n");
				return FALSE;
			}
			break;
		case IFAM:
			if (!interp_forfam(node, stab)) {
				wprintf(ierror, rpline);
				wprintf("forfam\n");
				return FALSE;
			}
			break;
		case ILIST:
			if (!interp_forlist(node, stab)) {
				wprintf(ierror, rpline);
				wprintf("forlist\n");
				return FALSE;
			}
			break;
		case INOTES:
			if (!interp_fornotes(node, stab)) {
				wprintf(ierror, rpline);
				wprintf("fornotes\n");
				return FALSE;
			}
			break;
		case INODES:
			if (!interp_fornodes(node, stab)) {
				wprintf(ierror, rpline);
				wprintf("fornodes\n");
				return FALSE;
			}
			break;
		case ITRAV:
			if (!interp_traverse(node, stab)) {
				wprintf(ierror, rpline);
				wprintf("traverse\n");
				return FALSE;
			}
			break;
		case IIF:
			if (!interp_if(node, stab)) {
				wprintf(ierror, rpline);
				wprintf("if\n");
				return FALSE;
			}
			break;
		case IWHILE:
			if (!interp_while(node, stab)) {
				wprintf(ierror, rpline);
				wprintf("while\n");
				return FALSE;
			}
			break;
		case ICALL:
			if (!interp_call(node, stab)) {
				wprintf(ierror, rpline);
				wprintf("call %s\n", iname(node));
				return FALSE;
			}
			break;
		default:
			wprintf("HUH, HUH, HUH, HUNH!\n");
			return FALSE;
		}
		node = inext(node);
	}
	return TRUE;
}
/*==============================================
 * interp_children -- Interpret a children loop.
 *============================================*/
BOOLEAN interp_children (node, stab)
INTERP node; TABLE stab;
{
	BOOLEAN eflg;
	INT nchil;
	CACHEEL fcel, cel;
	NODE fam = (NODE) eval_fam(ifamily(node), stab, &eflg, &fcel);
	if (eflg || (fam && strcmp(ntag(fam), "FAM"))) return FALSE;
	if (!fam) return TRUE;	/* this is not an error */
	lock_cache(fcel);
	FORCHILDREN(fam, chil, nchil)
		insert_table(stab, ichild(node),
		    (WORD) (cel = indi_to_cacheel(chil)));
		insert_table(stab, inum(node), (WORD) nchil);
		lock_cache(cel);
		if (!interpret((INTERP) ibody(node), stab))
			return FALSE;
		unlock_cache(cel);
	ENDCHILDREN
	unlock_cache(fcel);
	return TRUE;
}
/*============================================
 * interp_spouses -- Interpret a spouses loop.
 *==========================================*/
BOOLEAN interp_spouses (node, stab)
INTERP node; TABLE stab;
{
	BOOLEAN eflg;
	INT nspouses;
	CACHEEL icel, scel, fcel;
	NODE indi = (NODE) eval_indi(iprinc(node), stab, &eflg, &icel);
	if (eflg || !indi || strcmp(ntag(indi), "INDI")) return FALSE;
	lock_cache(icel);
	FORSPOUSES(indi, spouse, fam, nspouses)
		insert_table(stab, ispouse(node),
		    (WORD) (scel = indi_to_cacheel(spouse)));
		insert_table(stab, ifamvar(node),
		    (WORD) (fcel = fam_to_cacheel(fam)));
		lock_cache(scel);
		lock_cache(fcel);
		insert_table(stab, inum(node), (WORD) nspouses);
		if (!interpret((INTERP) ibody(node), stab)) return FALSE;
		unlock_cache(scel);
		unlock_cache(fcel);
	ENDSPOUSES
	unlock_cache(icel);
	return TRUE;
}
/*==============================================
 * interp_families -- Interpret a families loop.
 *============================================*/
BOOLEAN interp_families (node, stab)
INTERP node; TABLE stab;
{
	BOOLEAN eflg;
	INT nfamilies;
	CACHEEL icel, fcel, scel;
	NODE indi = (NODE) eval_indi(iprinc(node), stab, &eflg, &icel);
	if (eflg || !indi || strcmp(ntag(indi), "INDI")) return FALSE;
	lock_cache(icel);
	FORFAMILIES(indi, fam, spouse, nfamilies)
		insert_table(stab, ifamvar(node),
		    (WORD) (fcel = fam_to_cacheel(fam)));
		insert_table(stab, ispouse(node),
		    (WORD) (scel = indi_to_cacheel(spouse)));
		insert_table(stab, inum(node), (WORD) nfamilies);
		lock_cache(fcel);
		if (scel) lock_cache(scel);
		if (!interpret((INTERP) ibody(node), stab)) return FALSE;
		unlock_cache(fcel);
		if (scel) unlock_cache(scel);
	ENDFAMILIES
	unlock_cache(icel);
	return TRUE;
}
/*==============================================
 * interp_fornotes -- Interpret a fornotes loop.
 *============================================*/
BOOLEAN interp_fornotes (node, stab)
INTERP node; TABLE stab;
{
	BOOLEAN eflg;
	NODE root = (NODE) evaluate(inode(node), stab, &eflg);
	if (eflg || !root) return FALSE;
	FORTAGVALUES(root, "NOTE", sub, vstring)
		insert_table(stab, istrng(node), (WORD) vstring);
		if (!interpret((INTERP) ibody(node), stab)) return FALSE;
	ENDTAGVALUES
	return TRUE;
}
/*==============================================
 * interp_fornodes -- Interpret a fornodes loop.
 *============================================*/
BOOLEAN interp_fornodes (node, stab)
INTERP node; TABLE stab;
{
	BOOLEAN eflg;
	NODE root, sub;
	root = (NODE) evaluate(inode(node), stab, &eflg);
	if (eflg || !root) return FALSE;
	sub = nchild(root);
	while (sub) {
		insert_table(stab, isubnode(node), (WORD) sub);
		if (!interpret((INTERP) ibody(node), stab)) return FALSE;
		sub = nsibling(sub);
	}
	return TRUE;
}
/*============================================
 * interp_forindi -- Interpret a forindi loop.
 *==========================================*/
BOOLEAN interp_forindi (node, stab)
INTERP node; TABLE stab;
{
	NODE indi;
	static char key[10];
	STRING record;
	INT len, i = 0;
	while (TRUE) {
		sprintf(key, "I%d", ++i);
		if (!(record = retrieve_record(key, &len))) break;
		if (!(indi = string_to_node(record))) continue;
		insert_table(stab, iindivar(node),
		    (WORD) indi_to_cacheel(indi));
		insert_table(stab, inum(node), (WORD) i);
		if (!interpret((INTERP) ibody(node), stab)) return FALSE;
		free_nodes(indi);
		stdfree(record);
	}
	return TRUE;
}
/*==========================================
 * interp_forfam -- Interpret a forfam loop.
 *========================================*/
BOOLEAN interp_forfam (node, stab)
INTERP node; TABLE stab;
{
	NODE fam;
	static char key[10];
	STRING record;
	INT len, i = 0;
	while (TRUE) {
		sprintf(key, "F%d", ++i);
		if (!(record = retrieve_record(key, &len))) break;
		if (!(fam = string_to_node(record))) continue;
		insert_table(stab, iindivar(node),
		    (WORD) fam_to_cacheel(fam));
		insert_table(stab, inum(node), (WORD) i);
		if (!interpret((INTERP) ibody(node), stab)) return FALSE;
		free_nodes(fam);
		stdfree(record);
	}
	return TRUE;
}
/*=================================================
 * interp_indisetloop -- Interpret an indiset loop.
 *===============================================*/
BOOLEAN interp_indisetloop (node, stab)
INTERP node; TABLE stab;
{
	BOOLEAN eflg;
	INDISEQ seq = (INDISEQ) evaluate(iindex(node), stab, &eflg);
	if (eflg || !seq) return FALSE;
	FORINDISEQ(seq, el, ncount)
		insert_table(stab, iindivar(node),
		    (WORD) key_to_indi_cacheel(skey(el)));
		insert_table(stab, ivalvar(node), (WORD) sval(el));
		insert_table(stab, inum(node), (WORD) (ncount + 1));
		if (!interpret((INTERP) ibody(node), stab)) return FALSE;
	ENDINDISEQ
	return TRUE;
}
/*=========================================
 * interp_forlist -- Interpret a list loop.
 *=======================================*/
BOOLEAN interp_forlist (node, stab)
INTERP node; TABLE stab;
{
	BOOLEAN eflg;
	INT ncount = 1;
	LIST lst = (LIST) evaluate(ilist(node), stab, &eflg);
	if (eflg || !lst) return FALSE;
	FORLIST(lst, el)
		insert_table(stab, ielement(node), (WORD) el);
		insert_table(stab, inum(node), (WORD) ncount++);
		if (!interpret((INTERP) ibody(node), stab)) return FALSE;
	ENDLIST
	return TRUE;
}
/*================================================
 * interp_if -- Interpret an if control structure.
 *==============================================*/
BOOLEAN interp_if (node, stab)
INTERP node; TABLE stab;
{
	BOOLEAN eflg;
	WORD value = evaluate_cond(icond(node), stab, &eflg);
	if (eflg) return FALSE;
	if (value) return interpret((INTERP) ithen(node), stab);
	if (ielse(node)) return interpret((INTERP) ielse(node), stab);
	return TRUE;
}
/*=====================================================
 * interp_while -- Interpret a while control structure.
 *===================================================*/
BOOLEAN interp_while (node, stab)
TABLE stab;  INTERP node;
{
	BOOLEAN eflg;
	WORD value;
	while (TRUE) {
		value = evaluate_cond(icond(node), stab, &eflg);
		if (eflg) return FALSE;
		if (value == (WORD) FALSE) return TRUE;
		if (!interpret((INTERP) ibody(node), stab)) return FALSE;
	}
}
/*===================================================
 * interp_call -- Interpret a call control structure.
 *=================================================*/
BOOLEAN interp_call (node, stab)
TABLE stab;  INTERP node;
{
	TABLE newtab;
	INTERP arg, parm, proc = (INTERP) valueof(proctab, iname(node));
	BOOLEAN rc;
	if (!proc) {
		wprintf("``%s'': undefined procedure\n", iname(node));
		return FALSE;
	}
	newtab = create_table();
	arg = (INTERP) iargs(node);
	parm = (INTERP) iparams(proc);
	while (arg && parm) {
		BOOLEAN eflg;
		WORD value = evaluate(arg, stab, &eflg);
		if (eflg) return FALSE;
		insert_table(newtab, iident(parm), (WORD) value);
		arg = inext(arg);
		parm = inext(parm);
	}
	if (arg || parm) {
		wprintf("``%s'': mismatched args and params\n", iname(node));
		remove_table(newtab, NULL);
		return FALSE;
	}
	rc = interpret((INTERP) ibody(proc), newtab);
	remove_table(newtab, NULL);
	return rc;
}
