#ifndef LINT
static char * sccsdef = "@(#)usrreq.c	1.1	(Alex Crain) 6/20/89";
#endif

/*
 *  usrreq.c - Unix domain functions.
 *
 *  Written by Alex Crain.
 *
 *  This file is based in the Berkeley file uipc_socket.c,
 *  but is *not* guarenteed to be in any way compatable. It is
 *  close enough to the Berkeley code that the following applies...
 *
 *  Copyright (c) 1982, 1986, 1988 Regents of the University of California.
 *  All rights reserved.
 * 
 *  Redistribution and use in source and binary forms are permitted
 *  provided that this notice is preserved and that due credit is given
 *  to the University of California at Berkeley. The name of the University
 *  may not be used to endorse or promote products derived from this
 *  software without specific prior written permission. This software
 *  is provided "as is" without express or implied warranty.
 *
 */

#include <sys/types.h>
#include <uipc/conf.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/user.h>
#include <sys/inode.h>
#include <sys/proc.h>
#include <sys/stat.h>
#include <sys/var.h>
#include <sys/tune.h>
#include <sys/errno.h>
#include <uipc/mbuf.h>
#include <uipc/socket.h>
#include <uipc/socketvar.h>
#include <uipc/protosw.h>
#include <uipc/domain.h>
#include <uipc/unpcb.h>
#include <uipc/un.h>
#include <uipc/fproto.h>

struct	sockaddr sun_noname = { AF_UNIX };
ino_t	unp_ino;

int
uipc_usrreq (so, req, m, nam, rights)
  struct socket * so;
  int req;
  struct mbuf * m, * nam, * rights;
{
   struct unpcb * unp = sotounpcb (so);
   struct socket * so2;
   int error = 0;

   if (req == PRU_CONTROL)
       return EOPNOTSUPP;

   if (req != PRU_SEND && rights && rights->m_len)
    {
       error = EOPNOTSUPP;
       goto release;
    }

   if (unp == (struct unpcb *) 0 && req != PRU_ATTACH)
    {
       error = EINVAL;
       goto release;
    }

   switch (req)
    {
    case PRU_ATTACH:
       if (unp)
	{
	   error = EISCONN;
	   break;
	}
       error = unp_attach (so);
       break;

    case PRU_DETACH:
       unp_detach (unp);
       break;

    case PRU_BIND:
       error = unp_bind (unp, nam);
       break;

    case PRU_LISTEN:
       if (unp->unp_inode == 0)
	   error = EINVAL;
       break;

    case PRU_CONNECT:
       error = unp_connect (so, nam);
       break;

    case PRU_CONNECT2:
       error = unp_connect2 (so, (struct socket *) nam);
       break;

    case PRU_DISCONNECT:
       unp_disconnect (unp);
       break;

    case PRU_ACCEPT:
       if (unp->unp_conn && unp->unp_conn->unp_addr)
	{
	   nam->m_len = unp->unp_conn->unp_addr->m_len;
	   bcopy (mtod (unp->unp_conn->unp_addr, caddr_t),
		  mtod (nam, caddr_t), (unsigned) nam->m_len);
	}
       else
	{
	   nam->m_len = sizeof (sun_noname);
	   * (mtod (nam, struct sockaddr *)) = sun_noname;
	}
       break;

    case PRU_SHUTDOWN:
       socantsendmore (so);
       unp_usrclosed (unp);
       break;

    case PRU_RCVD:
       switch (so->so_type)
	{
	case SOCK_DGRAM:
	   panic ("uipc 1");
	   /* NOTREACHED */;

	case SOCK_STREAM:
#define rcv (&so->so_rcv)
#define snd (&so2->so_snd)
	   if (unp->unp_conn == 0)
	       break;
	   /*
	    *  Adjust backpressure on sender 
	    *  and wakeup any waiting to write.
	    */
	   so2 = unp->unp_conn->unp_socket;
	   snd->sb_mbmax += unp->unp_mbcnt - rcv->sb_mbcnt;
	   unp->unp_mbcnt = rcv->sb_mbcnt;
	   snd->sb_hiwat += unp->unp_cc - rcv->sb_cc;
	   unp->unp_cc = rcv->sb_cc;
	   sowwakeup (so2);
#undef snd
#undef rcv
	   break;

	default:
	   panic ("uipc 2");
	}
       break;

    case PRU_SEND:
       if (rights && (error = unp_internalize (rights)))
	   break;

       switch (so->so_type)
	{
	case SOCK_DGRAM:
	   {
	      struct sockaddr * from;

	      if (nam)
	       {
		  if (unp->unp_conn)
		   {
		      error = EISCONN;
		      break;
		   }
		  if (error = unp_connect (so, nam))
		      break;
	       }
	      else
	       {
		  if (unp->unp_conn == 0)
		   {
		      error = ENOTCONN;
		      break;
		   }
	       }
	      so2 = unp->unp_conn->unp_socket;
	      if (unp->unp_addr)
	          from = mtod (unp->unp_addr, struct sockaddr *);
	      else
	          from = &sun_noname;
	      if (sbspace (&so2->so_rcv) > 0 &&
		  sbappendaddr (&so2->so_rcv, from, m, rights))
	       {
		  sorwakeup (so2);
		  m = 0;
	       }
	      else
	          error = ENOBUFS;
	      if (nam)
	          unp_disconnect (unp);
	      break;
	   }
		  
	case SOCK_STREAM:
#define rcv (&so2->so_rcv)
#define snd (&so->so_snd)
	   if (so->so_state & SS_CANTSENDMORE)
	    {
	       error = EPIPE;
	       break;
	    }
	   if (unp->unp_conn == 0)
	       panic ("uipc 3");
	   so2 = unp->unp_conn->unp_socket;
	   /*
	    * Send to paired receive port, and then reduce
	    * senders hiwater makrs to maintain backpressure.
	    * Wake up readers.
	    */
	   if (rights)
	       (void) sbappendrights (rcv, m, rights);
	   else
	       sbappend (rcv, m);
	   snd->sb_mbmax -= rcv->sb_mbcnt - unp->unp_conn->unp_mbcnt;
	   unp->unp_conn->unp_mbcnt = rcv->sb_mbcnt;
	   snd->sb_hiwat -= rcv->sb_cc - unp->unp_conn->unp_cc;
	   unp->unp_conn->unp_cc = rcv->sb_cc;
	   sorwakeup (so2);
	   m = 0;
	   break;
#undef snd
#undef rcv

	default:
	   panic ("uipc 4");
	}
       break;

    case PRU_SENSE:
/*       ((struct stat *) m)->st_blksize = so->so_snd.sb_hiwat; */
       if (so->so_type == SOCK_STREAM && unp->unp_conn != 0)
	{
	   so2 = unp->unp_conn->unp_socket;
/*	   ((struct stat *) m)->st_blksize += so2->so_rcv.sb_cc; */
	}
       ((struct stat *) m)->st_dev = NODEV;
       if (unp->unp_ino == 0)
	   unp->unp_ino = unp_ino++;
       ((struct stat *) m)->st_ino = unp->unp_ino;
       return 0;

    case PRU_ABORT:
       unp_drop(unp, ECONNABORTED);
       break;

    case PRU_RCVOOB:
       return EOPNOTSUPP;
       
    case PRU_SENDOOB:
       error = EOPNOTSUPP;
       break;

    case PRU_SOCKADDR:
       break;

    case PRU_PEERADDR:
       if (unp->unp_conn && unp->unp_conn->unp_addr)
	{
	   nam->m_len = unp->unp_conn->unp_addr->m_len;
	   bcopy (mtod (unp->unp_conn->unp_addr, caddr_t),
		  mtod (nam, caddr_t), (unsigned) nam->m_len);
	}
       break;

    case PRU_SLOWTIMO:
       break;

    default:
       panic ("prusrreq");
    }

 release:
   if (m)
       m_freem (m);
   return error;
}

#define UNPST_SENDSPACE 4096
#define UNPST_RECVSPACE 4096
#define UNPDG_SENDSPACE 2048	/* max datagram size */
#define UNPDG_RECVSPACE 2048

int unp_rights;

int
unp_attach (so)
  struct socket * so;
{
   struct mbuf * m;
   struct unpcb * unp;
   int error = 0;

   switch (so->so_type)
    {
    case SOCK_DGRAM:
       error = soreserve (so, UNPDG_SENDSPACE, UNPDG_RECVSPACE);
       break;

    case SOCK_STREAM:
       error = soreserve (so, UNPST_SENDSPACE, UNPST_RECVSPACE);
       break;
    }
   if (error)
       return error;

   if ((m = m_getclr (M_DONTWAIT, MT_PCB)) == NULL)
       return ENOBUFS;

   unp = mtod(m, struct unpcb *);
   so->so_pcb = (caddr_t) unp;
   unp->unp_socket = so;

   return 0;
}

void
unp_detach (unp)
  struct unpcb * unp;
{
   if (unp->unp_inode)
    {
       unp->unp_inode->i_socket = 0;
       iput (unp->unp_inode);
       unp->unp_inode = 0;
    }
   if (unp->unp_conn)
       unp_disconnect (unp);
   while (unp->unp_refs)
       unp_drop (unp->unp_refs, ECONNRESET);
   soisdisconnected (unp->unp_socket);
   unp->unp_socket->so_pcb = 0;
   m_freem (unp->unp_addr);
   (void) m_free (dtom (unp));
   if (unp_rights)
       unp_gc ();
}

int
unp_bind (unp, nam)
  struct unpcb * unp;
  struct mbuf * nam;
{
   struct sockaddr_un * soun = mtod (nam, struct sockaddr_un *);
   struct inode * ip;
   int error;

   if (unp->unp_inode != NULL || nam->m_len == MLEN)
       return EINVAL;

   *(mtod (nam, caddr_t) + nam->m_len) = '\0';
   u.u_dirp = soun->sun_path;

   if (ip = namei (schar, 1))
    {
       iput (ip);
       return EADDRINUSE;
    }

   if (error = u.u_error)
    {
       u.u_error = 0;
       return error;
    }

   if ((ip = maknode (IFREG | 0777)) == NULL)
    {
       error = u.u_error;
       u.u_error = 0;
       return error;
    }
   
   ip->i_uid = u.u_uid;
   ip->i_gid = u.u_gid;
   ip->i_socket = unp->unp_socket;
   unp->unp_inode = ip;
   unp->unp_addr = m_copy (nam, 0, (int) M_COPYALL);
   prele (ip);
   return 0;
}

int
unp_connect (so, nam)
  struct socket * so;
  struct mbuf * nam;
{
   struct sockaddr_un * soun = mtod (nam, struct sockaddr_un *);
   struct inode * ip;
   int error;
   struct socket * so2;
   caddr_t dirp = u.u_dirp;
   caddr_t base = u.u_base;
   unsigned count = u.u_count;

   if ((nam->m_len + (nam->m_off - MMINOFF)) == MLEN)
    {
       error = EMSGSIZE;
       goto bad0;
    }
   * (mtod (nam, caddr_t) + nam->m_len) = '\0';
   u.u_dirp = soun->sun_path;
   if ((ip = namei (schar, 0)) == 0)
    {
       error = u.u_error;
       u.u_error = 0;
       goto bad0;
    }
   if (access (ip, IWRITE))
    {
       error = u.u_error;
       u.u_error = 0;
       goto bad;
    }
   if ((so2 = ip->i_socket) == 0)
    {
       error = ECONNREFUSED;
       goto bad;
    }
   if (so2->so_type != so->so_type)
    {
       error = EPROTOTYPE;
       goto bad;
    }
   if (so->so_proto->pr_flags & PR_CONNREQUIRED &&
       ((so2->so_options & SO_ACCEPTCONN) == 0 ||
	(so2 = sonewconn (so2)) == 0))
    {
       error = ECHILD; /* ECONNREFUSED;*/
       goto bad;
    }
   error = unp_connect2 (so, so2);
 bad:
   iput (ip);
 bad0:
   u.u_base = base;
   u.u_count = count;
   u.u_dirp = dirp;
   return error;
}

int
unp_connect2 (so1, so2)
  struct socket * so1, * so2;
{
   struct unpcb * unp1, * unp2;

   if (so2->so_type != so1->so_type)
       return EPROTOTYPE;
   unp1 = sotounpcb (so1);
   unp2 = sotounpcb (so2);
   unp1->unp_conn = unp2;
   switch (so1->so_type)
    {
    case SOCK_DGRAM:
       unp1->unp_nextref = unp2->unp_refs;
       unp2->unp_refs = unp1;
       soisconnected (so1);
       break;

    case SOCK_STREAM:
       unp2->unp_conn = unp1;
       soisconnected (so1);
       soisconnected (so2);
       break;
       
    default:
       panic ("unp_connect 2");
    }
   return 0;
}

void
unp_disconnect (unp)
  struct unpcb * unp;
{
   struct unpcb * unp2 = unp->unp_conn;

   if (unp2 == 0)
       return;
   unp->unp_conn = 0;
   switch (unp->unp_socket->so_type)
    {
    case SOCK_DGRAM:
       if (unp2->unp_refs == unp)
	   unp2->unp_refs = unp->unp_nextref;
       else
	{
	   unp2 = unp2->unp_refs;
	   for (;;)
	    {
	       if (unp2 == 0)
		   panic ("unp_disconnect");
	       if (unp2->unp_nextref == unp)
		   break;
	       unp2 = unp2->unp_nextref;
	    }
	   unp2->unp_nextref = unp->unp_nextref;
	}
       unp->unp_nextref = 0;
       unp->unp_socket->so_state &= ~SS_ISCONNECTED;
       break;

    case SOCK_STREAM:
       soisdisconnected (unp->unp_socket);
       unp2->unp_conn = 0;
       soisdisconnected (unp2->unp_socket);
       break;
    }
}

/* ARGSUSED */
void
unp_usrclosed (unp)
  struct unpcb * unp;
{
   /* do not very much */
   ;
}

void
unp_drop (unp, errno)
  struct unpcb * unp;
  int errno;
{
   struct socket * so = unp->unp_socket;

   so->so_error = errno;
   unp_disconnect (unp);
   if (so->so_head)
    {
       so->so_pcb = (caddr_t) 0;
       m_freem (unp->unp_addr);
       (void) m_free (dtom (unp));
       sofree (so);
    }
}

ushort f_msgcount[NFILEMAX];
#define fptoi(FP) (((FP)-file)/sizeof (struct file))

/* ARGSUSED */
int
unp_externalize (rights)
  struct mbuf * rights;
{
   int newfds = rights->m_len / sizeof (int);   
   int i, f;
   struct file ** rp = mtod (rights, struct file **);
   struct file * fp;

   if (newfds > ufavail ())
    {
       for (i = 0; i < newfds; i++)
	{
	   fp = *rp;
	   unp_discard (fp);
	   * rp++ = 0;
	}
       return EMSGSIZE;
    }
   for (i = 0, f = 0; i < newfds; i++)
    {
       f = ufalloc (f);
       if (f < 0)
	   panic ("unp_externalize");
       fp = * rp;
       u.u_ofile[f] = fp;
       f_msgcount[fptoi (fp)]--;
       unp_rights --;
       * (int *) rp++ = f;
    }
   return 0;
}

/* ARGSUSED */
int
unp_internalize (rights)
  struct mbuf * rights;
{
   struct file ** rp, * fp;
   int oldfds = rights->m_len / sizeof (int);
   int i;

   rp = mtod (rights, struct file **);

   for (i = 0; i < oldfds; i++)
       if (getf (* (int *) rp++) == 0)
	   return EBADF;

   rp = mtod (rights, struct file **);

   for (i = 0; i < oldfds; i++)
    {
       fp = getf (* (int *) rp);
       * rp++ = fp;
       fp->f_count++;
       f_msgcount[fptoi (fp)]++;
       unp_rights++;
    }
   return 0;
}

int unp_defer, unp_gcing;
extern struct domain unixdomain;

void
unp_gc ()
{
   struct file * fp;
   struct socket * so;
   int i;

   if (unp_gcing)
       return;
   unp_gcing = 1;
 restart:
   unp_defer = 0;
   for (fp = file; fp < (struct file *) v.ve_file; fp++)
       fp->f_flag &= ~(FMARK | FDEFER);
   do
    {
       for (fp = file; fp < (struct file *) v.ve_file; fp++)
	{
	   if (fp->f_count == 0)
	       continue;
	   if (fp->f_flag & FDEFER)
	    {
	       fp->f_flag &= ~FDEFER;
	       unp_defer--;
	    }
	   else
	    {
	       if (fp->f_flag & FMARK)
		   continue;
	       if (fp->f_count == f_msgcount[fptoi (fp)])
		   continue;
	       fp->f_flag |= FMARK;
	    }
	   if (fp->f_inode)
	       continue;
	   so = filesock (fp);
	   if (so->so_proto->pr_domain != &unixdomain ||
	       so->so_proto->pr_flags & PR_RIGHTS)
	       continue;
	   if (so->so_rcv.sb_flags & SB_LOCK)
	    {
	       sbwait (&so->so_rcv);
	       goto restart;
	    }
	   unp_scan (so->so_rcv.sb_mb, (int (*)()) unp_mark);
	}
    } while (unp_defer);
   for (fp = file, i = 0; fp < (struct file *) v.ve_file; fp++, i++)
    {
       if (fp->f_count == 0)
	   continue;
       if (fp->f_count == f_msgcount[i] &&
	   (fp->f_flag & FMARK) == 0)
	   while (f_msgcount[i])
	       unp_discard (fp);
    }
   unp_gcing = 0;
}

void
unp_dispose (m)
  struct mbuf * m;
{
   void unp_discard ();

   if (m)
       unp_scan (m, (int (*)()) unp_discard);
}

/* ARGSUSED */
void
unp_scan (m0, op)
  struct mbuf * m0;
  int (* op)();
{
   struct mbuf * m;
   struct file ** rp;
   int i;
   int qfds;

   while (m0)
    {
       for (m = m0; m; m = m->m_next)
	   if (m->m_type == MT_RIGHTS && m->m_len)
	    {
	       qfds = m->m_len / sizeof (struct file **);
	       rp = mtod (m, struct file **);
	       for (i = 0; i < qfds; i++)
		   (* op) (* rp++);
	       break;
	    }
       m0 = m0->m_act;
    }
}

void
unp_mark (fp)
  struct file * fp;
{
   if (fp->f_flag & FMARK)
       return;
   unp_defer++;
   fp->f_flag |= (FMARK | FDEFER);
}

void
unp_discard (fp)
  struct file * fp;
{
   f_msgcount[fptoi (fp)]--;
   unp_rights--;
   closef (fp);
}

