%{

/* 
 * ifile processor for unix-pc disassembler
 *
 * Written by Alex Crain - January 1988
 *
 * Copyright 1988 Alex Crain
 */

/*
 * Usage: parseifile [-d] [-f ifile] [-s table size]
 *
 * Where:
 *	-d	Turn on yydebug. Used only for program debugging.
 *	-f	specify an ifile to parse. default is /lib/shlib.ifile
 *	-s	table size of result symbol table. Default is 2400, and
 *		does not need to be increased unless your ifile has
 *		more then 2400 entries (on entry per line).
 *	-n	symbol table name. Default is symtab[].
 *
 * The output is a c structure in the form:
 * struct symbol { 
 *	int address;
 *	char * name;
 * } symtab[] = {
 *	{sorted table entries}
 * };
 *
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "parseifile.h"

char * malloc ();
long strtol ();
int GetC (), NextC (), yylex ();
void fatal (), gen_file (), UnGetC (), sort_symbols (), exit (), free ();

#define yyerror(MSG) fatal (MSG)

struct symbol {
   long address;
   char * name;
};

static char * ifile_name = DEFAULT_IFILE;
static char * table_name = DEFAULT_SYMTAB;
static FILE * ifile;
static struct symbol * symtab;
static int table_size = 0, max_table_size = DEFAULT_TABLE_SIZE;

extern int yydebug;
#define YYDEBUG 1

void
main (argc, argv)
int argc;
char ** argv;
{
   int c;
   extern char * optarg;

   yydebug = 0;

   while ((c = getopt (argc, argv, "df:n:s:")) != EOF)
       switch (c)
	{
	case 'd':
	   yydebug = 1;
	   break;
	case 'f':
	   ifile_name = optarg;
	   break;
	case 'n':
	   table_name = optarg;
	   break;
	case 's':
	   max_table_size = strtol (optarg, (char **) 0, 0);
	   break;
	case '?':
	   fatal ("Usage: parseifile [-d] [-f ifile] [-s table-size].");
	}
   
   if ((ifile = fopen (ifile_name, "r")) == NULL)
       fatal ("Cannot open ifile.");

   symtab = (struct symbol *) malloc (sizeof(struct symbol) * max_table_size);
   if (symtab == NULL)
       fatal ("Out of memory.");

   (void) yyparse ();
   sort_symbols ();
   gen_file ();
   exit (0);
   /*NOTREACHED*/
}

%}

%union {
	long	integer;
	char *	string;
}

%token FLAG
%token '{'
%token '}'
%token '='
%token '+'
%token ';'
%token <string> SOMETHING

%type <integer> expression

%%

file
	: definitions
	;

definitions
	: definition
	| definitions definition
	;

definition
	: directive
	| assignment
	| FLAG
	;

directive
	: SOMETHING block
		{ if ($1) (void) free ($1); }
	;

assignment
	: SOMETHING '=' expression ';'
		{ add_symbol ($1, $3); }
	;

expression 
	: SOMETHING
		{ if (isdigit ($1[0]))
		      $$ = strtol ($1, (char **)0, 0);
		  else
		      $$ = lookup_symbol ($1);
		  (void) free ($1);
		}
	| expression '+' SOMETHING
		{ if (isdigit ($3[0]))
		      $$ = $1 + strtol ($3, (char **)0, 0);
		  else
		      $$ = $1 + lookup_symbol ($3);
		  (void) free ($3);
		}
	;      

block
	: '{' stuff_list '}'
	| '{' '}'
	;

stuff_list
	: stuff
	| stuff_list stuff
	;

stuff
	: SOMETHING
		{ if ($1) (void) free ($1); }
	| block
	| '+'
	| '='
	| ';'
	;

%%

int lookahead = 0;
int yylineno = 0;

int
GetC ()
{
   int c;

   if (lookahead)
    {
       c = lookahead;
       lookahead = 0;
    }
   else
    {
       c = getc (ifile);
       if (c == '\n')
	   yylineno++;
    }
   return c;
}

int
NextC ()
{
   if (! lookahead)
       lookahead = getc (ifile);
   return lookahead;
}

void 
UnGetC (c)
  int c;
{
   lookahead = c;
}

int
yylex ()
{
   int c;
   static char buffer[128];

 start:
   while (((c = GetC ()) != EOF) && isspace (c));
   if (c == '/' && NextC () == '*')
    {
       do {
	  c=GetC (); 
       } while (c != EOF && (c != '*' || NextC () != '/'));
       c = GetC ();
       goto start;
    }
	  
   if (isalnum (c) || c == '_')
    {
       int i = 0;

       while ((isalnum (c) || c == '_') && i < 128)
	{
	   buffer[i++] = c;
	   c = GetC ();
	}
       UnGetC (c);
       buffer[i] = '\0';
       yylval.string = malloc (sizeof (buffer) + 1);
       if (yylval.string)
	   (void) strcpy (yylval.string, buffer);
       else
	   fatal ("Out of memory.");
       if (yydebug) 
	   (void) printf ("(%s)", buffer);
       return SOMETHING;
    }

   if (c == '-')
    {
       while ((c = GetC ()) != EOF && c != '\n');
       return FLAG;
    }

   switch (c)
    {
    case '{':
    case '}':
    case '+':
    case '=':
    case ';':
       return c;
    case EOF:
       return EOF;
    default:
       yylval.string = (char *) 0;
       return SOMETHING;
    }
}

void
add_symbol (symbol,address)
  char * symbol;
  long address;
{
   symtab[table_size].name = symbol;
   symtab[table_size].address = address;
   table_size++;
}

void
gen_file ()
{
   int i;

   (void) puts (HEADER);
   (void) printf ("#define %s_TABLE_SIZE %d\n", table_name, table_size);
   (void) printf ("struct symbol %s[] = {", table_name);
   for (i=0; i<table_size; i++)
       (void) printf ("{0x%x,\"%s\"},\n", symtab[i].address, symtab[i].name);
   (void) puts ("};");
}

void
fatal (msg)
  char * msg;
{
   (void) fputs ("pareifile: ", stderr);
   if (yylineno)
       (void) fprintf (stderr, "[line %d]: ", yylineno);
   (void) fputs (msg, stderr);
   (void) putc('\n', stderr);
   (void) exit (1);

}

long
lookup_symbol(s)
  char * s;
{
   int i = 0;

   while (i < table_size)
    {
       if (strcmp (symtab[i].name, s))
	   i++;
       else
	   return symtab[i].address;
    }
   fatal ("unreferenced symbol.");
   /*NOTREACHED*/
}
	   
	   
void 
sort_symbols ()
{
   int i, j;
   for (i = table_size - 1; i; i--)
       for (j = 0; j < table_size - 1; j++)
	   if (symtab[j].address > symtab[j+1].address)
	    {
	       union { long l; char * s; } temp;
	       
	       temp.l = symtab[j].address;
	       symtab[j].address = symtab[j+1].address;
	       symtab[j+1].address = temp.l;
	       temp.s = symtab[j].name;
	       symtab[j].name = symtab[j+1].name;
	       symtab[j+1].name = temp.s;
	    }
}
