/* lpd - the printer daemon				Author: Ralf Wenk */

/* This program handles jobs found in the lpd spooldirectory. It uses
 * a file in the spooldirectory named 'Lock' to test if a daemon is
 * already running. While this is not a semaphore it is possible that
 * a job is not processed immediately.
 *
 * Ralf Wenk	last update:	Tue Jun  9 16:17:02 1992
 *
 * $Id: lpd.c,v 1.1 92/06/09 17:01:25 ralf Exp $
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include <limits.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <stdio.h>


#define TRUE	1
#define FALSE	0
#define EOS    '\0'
#define TTIME	3			/* test every n seconds */
#define SPDIR	"/usr/spool/lpd"	/* spool directory */
#define LPDLOCK	"Lock"			/* shows that a lpd is running */
#define FMODE	(S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)	/* file creation mode */
#define IJOBNAME "A"			/* impossible jobname */
#define TRIESMAX (600 / TTIME)		/* wait 10 minutes, then cry */
#define BUSYTEXT "Printer returns busy status since 10 minutes. I'll terminate.\n"


FILE *pfp = NULL;			/* pipefilepointer for mail */
DIR *spdir;				/* spool directory */
char jfname[NAME_MAX+1];		/* name of current job */
int jobok;				/* job can be processed */


/* No header defines this */
extern FILE *popen();
extern int pclose();


#ifndef HAVESETPGRP
/*
 * QAD simulation
 */
int setpgrp ()
{
  signal( SIGINT, SIG_IGN );
}	/* setpgrp */
#endif


/*
 * prepare for termination
 */
void cleanup ()
{
  signal( SIGUSR1, SIG_IGN );
  unlink( LPDLOCK );
  closedir( spdir );
}	/* cleanup */


/*
 * unusual termination caused by output problems or signals
 */
void terminate ()
{
  cleanup();
  exit( 0 );
}	/* terminate */


/*
 * a job was canceled, is it the current one ?
 */
void jobcancel ()
{
  int fd;

  signal( SIGUSR1, SIG_IGN );
  if (( fd = open( jfname, O_RDONLY )) >= 0 )
    close( fd );
  else
    if ( errno != ENFILE && errno != EMFILE )
      jobok = FALSE;
  signal( SIGUSR1, jobcancel );
}	/* jobcancel */


/*
 * invoke the mailer
 */
void sendmail ( to, what )
char *to, *what;
{
  char cmd[PATH_MAX];

  if ( pfp == NULL )
  {
    sprintf( cmd,"mail -s 'Problems with your lp job %s' %s", jfname, to );
    pfp = popen( cmd,"w");
  }
  fprintf( pfp, what );
}	/* sendmail */


/*
 * Print a file
 */
void print ( from, to, user )
int from, to;
char *user;
{
  char buf[BUFSIZ];
  char *bptr;
  int in, out;
  int try;
  int myerrno;

  while ( jobok && ( in = read( from, buf, BUFSIZ )) != 0 )
  {
    if ( in > 0 )
    {
      try = 0;
      bptr = buf;
      while ( jobok && ( out = write( to, bptr, in )) != in && try < TRIESMAX )
      {
        if ( out > 0 )			/* if only a part was written */
        {
          in -= out;
          bptr += out;
        }
        try++;
        sleep( TTIME );
      }
      if ( try == TRIESMAX )
      {
        myerrno = errno;
        if ( myerrno == EIO )
          sendmail( user, BUSYTEXT );
        else
          sendmail( user, strerror( myerrno ));
        terminate();
      }
    }
  }
}	/* print */


/*
 * process a job
 */
dojob ( fp )
FILE *fp;
{
  char buff[BUFSIZ];
  char txt[PATH_MAX];
  char *source, *dest, *user;
  int copy, copies;
  int sfd;				/* source filedescriptor */
  int dfd;				/* destination filedescriptor */

  while ( jobok && fgets( buff, BUFSIZ, fp ) != NULL )
  {
    copies = atoi( strtok( buff," "));
    source = strtok( (char *)NULL," ");
    dest = strtok( (char *)NULL," ");
    user = strtok( (char *)NULL,"\n");
    while (( sfd = open( source, O_RDONLY )) < 0 && errno == EINTR );
    if ( sfd < 0 )
    {
      sprintf( txt,"Can't read %s (%s)\n", source, strerror( errno ));
      sendmail( user, txt );
    }
    else
    {
      while (( dfd = open( dest, O_WRONLY )) < 0 && errno == EINTR );
      if ( dfd < 0 )
      {
        sprintf( txt,"Can't write %s (%s)\n", dest, strerror( errno ));
        sendmail( user, txt );
      }
      else
      {
        for ( copy = 0; copy < copies; copy++ )
        {
          print( sfd, dfd, user );
          while ( lseek( sfd, (off_t)0, SEEK_SET ) != 0 && errno == EINTR );
        }
        while ( close( dfd ) < 0 );
      }
      while ( close( sfd ) < 0 );
    }
    if ( strncmp( SPDIR, source, strlen( SPDIR )) == 0 )
      while ( jobok && unlink( source ) < 0 );
  }
  fclose( fp );
  while ( jobok && unlink( jfname ) < 0 );
  if ( pfp != NULL )			/* terminate mail input */
  {
    sendmail( user,"\nYour lpd\n");
    pclose( pfp );
    pfp = NULL;
  }
}	/* dojob */


/*
 * search in spooldirectory for the next job
 */
FILE *nextjob ()
{
  FILE *fp;
  struct dirent *entry;

  do
  {
    jobok = TRUE;
    strcpy( jfname, IJOBNAME );
    rewinddir( spdir );
    while (( entry = readdir( spdir )) != NULL )
    {
      if ( isdigit( entry->d_name[0] ) && strcmp( jfname, entry->d_name ) > 0 )
        strcpy( jfname, entry->d_name );
    }
    fp = fopen( jfname,"r");
  } while ( !jobok );			/* in case jobcancel hits */
  return( fp );
}	/* nextjob */


/*
 * test and terminate if a daemon already exists
 * else create the lock and process ervery interesting file found
 */
int main( argc, argv )
int argc;
char *argv[];
{
  FILE *fp;
  struct stat stbuff;
  char pid[6];
  int fd;

  jobok = TRUE;
  close( STDIN_FILENO );
  close( STDOUT_FILENO );
  close( STDERR_FILENO );
  setpgrp();
  if ( chdir( SPDIR ) == 0 )
  {
    stat(".", &stbuff );		/* who ist the ownwer of SPDIR ? */
    setuid( stbuff.st_uid );		/* change to */
    spdir = opendir(".");
    signal( SIGUSR1, jobcancel );
    signal( SIGTERM, terminate );
    if ( access( LPDLOCK, 0 ) && ( fd = creat( LPDLOCK, FMODE )) >= 0 )
    {
      sprintf( pid,"%d", getpid());
      write( fd, pid, strlen( pid ));
      close( fd );
      while (( fp = nextjob()) != NULL )
      {
        jobok = TRUE;
        dojob( fp );
      }
      cleanup();
    }
  }
  exit( 0 );
}	/* main */

