/* lpr - printer front end				Author: Ralf Wenk */

/* This program links or copies files to the line printer daemon
 * directory and then calls the daemon to spool them.
 *
 * Ralf Wenk	last update:	Tue Jun  9 16:17:35 1992
 *
 * $Id: lp.c,v 1.1 92/06/09 17:01:18 ralf Exp $
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <pwd.h>
#include <time.h>
#include <fcntl.h>
#include <limits.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>


#define TRUE  1
#define FALSE 0
#define EOS   '\0'
#define DEF_DEV	"/dev/lp"		/* default device */
#define PRINTER	"PRINTER"		/* environment variables for output */
#define LPDEST	"LPDEST"		/* device or destination */
#define STDIN	"-"			/* "standard input" filename */
#define SPDIR	"/usr/spool/lpd"	/* spool direcotry */
#define FMODE	(S_IRUSR | S_IWUSR)	/* file creation mode */
#define MMODE	(S_IRWXG | S_IRWXO)	/* umask mode */
#define TMAX	10000000L		/* seconds after primary job # can */
					/* be reused */
#define NOBODY	"-"

char fname[PATH_MAX+1];

char *lpd[]    = { "/usr/lib/lpd",	/* possible places of lpd */
		   "/etc/lpd",
		   "/usr/bin/lpd",
		   NULL };


/*
 * make a unique filename for data
 * the 1st one will hold the job information
 */
char *mkfname ()
{
  static unsigned int seqn = 0;

  sprintf( fname,"%s/D.%04x.%04x", SPDIR, getpid(), seqn );
  seqn++;
  return( fname );
}	/* mkfname */


/*
 * queue the job in by renaming the 1st datafile to a jobfile
 */
int job ()
{
  char jobname[PATH_MAX+1];

  sprintf( fname,"%s/D.%04x.%04x", SPDIR, getpid(), 0 );
  sprintf( jobname,"%s/%ld.%04x", SPDIR, time((time_t *)NULL) % TMAX, getpid());
  return( rename( fname, jobname ));
}	/* job */


/*
 * get the name of the real user
 */
char *getuser ()
{
  struct passwd *pwdptr;
  char *cptr;

  if (( pwdptr = getpwuid( getuid())) != NULL )
    cptr = pwdptr->pw_name;
  else
    cptr = NOBODY;
  return( cptr );
}	/* getuser */


/*
 * copy some bytes between two files
 */
void copy ( from, to )
int from, to;
{
  char buf[BUFSIZ];
  unsigned elements;

  while (( elements = read( from, buf, BUFSIZ )) > 0 )
    write( to, buf, elements );
}	/* copy */


/*
 * handle the stdin case
 */
int dostdin ( jfp, copies, dest, user )
FILE *jfp;
int copies;
char *dest, *user;
{
  char *cptr;
  int fd;
  int result;

  result = 0;
  cptr = mkfname();
  if (( fd = creat( cptr, FMODE )) < 0 )
    result = 1;
  else
  {
    copy( STDIN_FILENO, fd );
    close( fd );
    fprintf( jfp,"%u %s %s %s\n", copies, cptr, dest, user );
  }
  return( result );
}	/* dostdin */


/*
 * handle all non stdin cases
 */
int doother ( jfp, copies, source, dest, user )
FILE *jfp;
int copies;
char *source, *dest, *user;
{
  int pfd[2];
  char *cptr;
  int fd;
  int status;
  int result;

  result = 0;
  if ( pipe( pfd ) != 0 )
    result = 1;
  else
    switch ( fork())
    {
      case -1 :				/* failed */
        break;
      case 0 :				/* child */
        fclose( jfp );
        close( pfd[0] );
        setuid( getuid());		/* set effective uid to real uid */
        if (( fd = open( source, O_RDONLY )) < 0 )
          result = 1;
        else
          copy( fd, pfd[1] );
        exit( result );			/* files closed by exit() */
        break;
      default :				/* parent */
        cptr = mkfname();
        close( pfd[1] );
        if (( fd = creat( cptr, FMODE )) < 0 )
          result = 1;
        else
        {
          copy( pfd[0], fd );
          wait( &status );		/* clean up child */
          close( pfd[0] );
          close( fd );
          if ( WIFEXITED( status ) && WEXITSTATUS( status ) == 0 )
            fprintf( jfp,"%u %s %s %s\n", copies, cptr, dest, user );
          else
          {
            result = 1;
            unlink( cptr );		/* remove file if child failed */
          }
        }
        break;
    }
  return( result );
}	/* doother */


/*
 * insert given file(s) into /usr/spool/lpd and start the daemon
 */
int main( argc, argv )
int argc;
char *argv[];
{
  FILE *jfp;				/* job filedescriptor */
  char *user;				/* name of user invoking this cmd */
  char *cptr;
  char *dest;				/* output destination */
  int copies;				/* number of copies */
  int copyit;				/* files not needed after exit */
  int result;				/* exit status of this command */
  char option;
  extern char *optarg;
  extern int optind;

  result = 0;
  copyit = FALSE;
  user = getuser();
  dest = DEF_DEV;
  umask( MMODE );
  if (( cptr = getenv( PRINTER )) != NULL )
    dest = cptr;
  if (( cptr = getenv( LPDEST )) != NULL )
    dest = cptr;
  copies = 1;
  while (( option = getopt( argc, argv,"cd:n:")) != EOF )
  {
    switch ( option )
    {
      case 'c':
        copyit = TRUE;
        break;
      case 'd':
        dest = optarg;
        break;
      case 'n':
        if ( atoi( optarg ) > 0 )
          copies = atoi( optarg );
        break;
      default:
        fprintf( stderr,"Usage: %s [-c][-d dest][-n copies][file ...]\n",
                 argv[0] );
        exit( 1 );
        break;
    }
  }
  if (( jfp = fopen( mkfname(),"w")) == NULL )
    result = 1;
  else
  {
    if ( optind == argc )
      result = dostdin( jfp, copies, dest, user );
    else
      while ( optind < argc )
      {
        if ( strcmp( argv[optind], STDIN ) == 0 )
          result = dostdin( jfp, copies, dest, user );
        else if ( copyit )
          result = doother( jfp, copies, argv[optind], dest, user );
        else if (( cptr = getcwd( fname, PATH_MAX )) == NULL )
          result = doother( jfp, copies, argv[optind], dest, user );
        else
        {
          if ( argv[optind][0] == '/' )
            *cptr = EOS;
          fprintf( jfp,"%u %s/%s %s %s\n", copies, cptr, argv[optind], dest,
                   user );
        }
        optind++;
      }
    fclose( jfp );
    if ( job() != 0 )
      result = 1;
  }
  if ( fork() == 0 )
  {
    int index;
    index = 0;
    while ( lpd[index] )
      execl( lpd[index++],"lpd", NULL );
  }
  exit( result );
}	/* main */

