/* backup <device> <files> copies files to backup device. by H. Salomons */

#include "backup.h"

#define NO_DEBUG_ALL

#ifdef DEBUG_ALL
#define SHOW_RESULT
#define DEBUG_PROFILE
#define DEBUG_ARGUMENTS
#define DEBUG_EXCLUDE
#define DEBUG_RESET
#define DEBUG_BACKTMP
#define DEBUG_MALLOC
#endif

static int  namcount;
static char *namlist[MAX_ARGS];
static char *sincefile;

int         main(argc, argv)
    int         argc;
    char      **argv;
{
    int         result = 0;
    int         depth = 0;
    char        verbose = 0;
    char       *cp;
    char       *cwd;
    char        workdir[MAX_NAME];
    int         argcount;
    char       *arglist[MAX_ARGS];
    char      **argp;
    extern char *getenv();
    extern char *getcwd();

    progname = *argv;
    /*
     * the program does 'chdir', so it needs to know where it came from to
     * backup the local files.
     */
    cwd = getcwd(workdir, MAX_NAME);

    block_count = 0L;
    block_max = (long) DOUBLE_SIZE;
    oseqnr = 1;
    since = 0L;
    fnout = NULLF;
    dirback = BACKDIR;
    listname = BACKLST;
    diskname = BACKTAR;
    sincefile = (char *) 0;
    numskip = 0;
    file_number = 0;

    if (isatty(1))
	/* don't scroll on terminal */
	*eoln = '\r';
    else
	/* do scroll in pipe or file */
	*eoln = '\n';

    reset_pointers();

    argcount = 0;
    namcount = 0;

    /* get profile(HOME) */
    if ((cp = getenv(HOME)) != (char *) 0)
	getprofile(arglist, &argcount, MAX_ARGS, cp, PROFILE);

    /* get profile(.) */
    getprofile(arglist, &argcount, MAX_ARGS, (char *) 0, PROFILE);

#ifdef DEBUG_PROFILE
    /* debug argument list */
    {
	int         i;
	for (i = 0; i < argcount; i++)
	    fprintf(stdout, "%d: %lx '%s'\n", i, arglist[i], arglist[i]);
    }
#endif
    while (--argc)
	arglist[argcount++] = *++argv;

#ifdef DEBUG_PROFILE
    {
	int         i;

	for (i = 0; i < argcount; i++)
	    fprintf(stdout, "%d: %lx '%s'\n", i, arglist[i], arglist[i]);
    }
#endif

    /* scan the arguments */
    for (argp = arglist; argcount > 0; argcount--)
    {
	cp = *argp++;
#ifdef DEBUG_ARGUMENTS
	fprintf(stderr, "scan argument(%s)\n", cp);
#endif
	if (*cp == '-')
	{
	    switch (*++cp)
	    {
	    case 's':
		/*
		 * appoint a file whose st_mtime is used for an incremental
		 * backup '-s <file>'.
		 */
		if (--argcount > 0)
		{
		    since = getftime(sincefile = *argp++);
#ifdef DEBUG_ARGUMENTS
		    fprintf(stderr, "since (%s)\n", since);
#endif
		}
		else
		    usage(1);
		break;

	    case 'n':
		/*
		 * the sequence number of the backup-list and subsequently
		 * the backup floppy
		 */
		if (--argcount > 0)
		{
		    oseqnr = atoi(*argp++);
#ifdef DEBUG_ARGUMENTS
		    fprintf(stderr, "number (%d)\n", oseqnr);
#endif
		}
		else
		    usage(1);
		break;

	    case 'l':
		/*
		 * the list-name to use between doback and mtar
		 */
		if (--argcount > 0)
		{
		    listname = *argp++;
#ifdef DEBUG_ARGUMENTS
		    fprintf(stderr, "list (%s)\n", listname);
#endif
		}
		else
		    usage(1);
		break;

	    case 't':
		/* the temporary file dir. */
		if (--argcount > 0)
		{
		    dirback = *argp++;
#ifdef DEBUG_ARGUMENTS
		    fprintf(stderr, "dirback (%s)\n", dirback);
#endif
		}
		else
		    usage(1);
		break;

	    case 'd':
		/* the disk-name to use for ? */
		if (--argcount > 0)
		{
		    diskname = *argp++;
#ifdef DEBUG_ARGUMENTS
		    fprintf(stderr, "diskname (%s)\n", diskname);
#endif
		}
		else
		    usage(1);
		break;

	    case 'x':
		/* suffices to skip */
		if (--argcount > 0)
		{
		    skip[numskip] = *argp++;
#ifdef DEBUG_ARGUMENTS
		    fprintf(stderr, "skip (%s)\n", skip[numskip]);
#endif
		    if (numskip < MAX_SKIP)
			numskip++;
		}
		else
		    usage(1);

		break;

	    case 'c':
		do_compress = 0;
		break;

	    case 'v':
		verbose = 1;
		break;

	    case 'b':
		/* the disk-size in blocks of 512 bytes (tar blocks) */
		if (--argcount > 0)
		{
		    block_max = atoi(*argp++);
		    if (block_max < 50)
		    {
			usage(2);
		    }
		}
		else
		    usage(1);
		break;

	    default:
		usage(0);
	    }
	}
	else
	    /* this is a directory name or a file name */
	    namlist[namcount++] = cp;
    }

    if (namcount == 0)
	usage(3);

    if (verbose)
	usage(4);

    for (argp = namlist; namcount > 0; namcount--)
    {
	cp = *argp++;
	/*
	 * then scan the directory tree, depth first, directories first
	 */
	if (list_files(cp, depth, LIST_DIR) < 0)
	    continue;

	/* at last the files in the top-directory */
	(void) list_files(cp, depth, LIST_FILE);

	/* backup this directory */
	do_backup(start);

	/* clean up bookkeeping */
	reset_pointers();

	/* get back to the current working directory */
	chdir(cwd);
    }
    close_curnt();
    return (result);
}

int         usage(nr)
    int         nr;
{
    extern char *ctime();
    char       *cp;
    int         i;

    /* get the time string */
    cp = ctime(&since);

    /* strip linefeed from ctime string */
    *(cp + 24) = '\0';

    /* skip day-of-the week string */
    cp += 4;

    switch (nr)
    {
    case 1:
	fprintf(stderr, "parameter missing\n\n");
	break;

    case 2:
	fprintf(stderr, "backup device/file too small\n\n");
	break;

    case 3:
	fprintf(stderr, "no directories or files to back up\n");
	break;

    case 4:
	fprintf(stderr,
	"%s: backup files and directories, current options:\n", progname);
	fprintf(stderr, "-c suppress compression (compression %s)\n",
		do_compress ? "on" : "off");
	if (sincefile != (char *) 0)
	    fprintf(stderr,
	"-s : incremental backup since %s\n   (modify time of file %s)\n",
		    cp, sincefile);
	else
	    fprintf(stderr, "-s : full backup\n");
	fprintf(stderr, "-t %s : temporary directory\n", dirback);
	fprintf(stderr, "-n %d : disk number\n", oseqnr);
	fprintf(stderr, "-l %s : list name\n", listname);
	fprintf(stderr, "-d %s : backup device or file\n", diskname);
	fprintf(stderr, "-b %ld : volume size of backup device\n", block_max);
	fprintf(stderr, "-v : verbose (on)\n");
	if (numskip == 0)
	    fprintf(stderr, "-x : no files excluded\n");
	else
	{
	    fprintf(stderr, "-x : files with extensions excluded: ");
	    for (i = 0; i < numskip; i++)
		fprintf(stderr, ".%s ", skip[i]);
	    fprintf(stderr, "\n");
	}
	fprintf(stderr, "list of files or directories:\n");
	for (i = 0; i < namcount; i++)
	    fprintf(stderr, "   %s\n", namlist[i]);
	fprintf(stderr, "\n");
	return;
    default:
	break;
    }

    fprintf(stderr,
	    "usage: %s [-c] [-s s] [-t t] [-n n] [-l l] [-d d] [-b b] [-x x] ...\n",
	    progname);
    exit(1);
}

int         list_files(name, depth, mode)
    char       *name;
    int         depth;
    int         mode;
{
    struct stat statbuf;
    char        pname[MAX_NAME];

#ifdef DEBUG_ARGUMENTS
    fprintf(stderr, "list_files(%s, %d, %d, %s)\n",
	    name, depth, mode, dirback);
#endif
    /* save the current file or path name */
    strcpy(pname, name);
    pathname[depth] = pname;

#ifdef DEBUG_ARGUMENTS
    fprintf(stderr, "pathname[%d]='%s'\n", depth, pathname[depth]);
#endif

    /* if this file exists */
    if (stat(name, &statbuf) == 0)
    {
	/* select the type info */
	switch (statbuf.st_mode & S_IFMT)
	{
	    /* don't backup block and character devices */
	case S_IFBLK:
	case S_IFCHR:
	    break;

	case S_IFDIR:
	    /* if the current name is a directory */
	    if (mode == LIST_DIR)
	    {
		/* add path to the list (a null-operation) */
		add_path(depth, &statbuf);

		/* depth first, start with directories */
		dlist(pname, depth, LIST_DIR);

		/* next the regular files */
		dlist(pname, depth, LIST_FILE);
	    }
	    break;

	case S_IFREG:

	    /* record the filename, try to compress etc. */
	    if (mode == LIST_FILE)
		add_file(depth, &statbuf);
	    break;

	default:
	    /* this should not happen... */
	    break;
	}
    }
    else
    {
	fprintf(stderr, "%s: can't access %s\n", progname, name);
	return (-1);
    }
    return (0);
}

/* this routine walks through a directory */

int         dlist(name, depth, mode)
    char       *name;
    int         depth;
    int         mode;
{
    DIR        *dp;
    struct dirent *dir;

#ifdef SHOW_RESULT
    fprintf(stderr, "opendir(%s)\n", name);
#endif

    /* open the directory */
    if ((dp = opendir(name)) == (DIR *) 0)
	fprintf(stderr, "can't opendir(%s)\n", name);
    else
    {
	/* do a change directory */
	if (chdir(name) < 0)
	    fprintf(stderr, "can't chdir to %s\n", name);
	else
	{
	    /* skip the entries '.' and '..' */
	    if ((dir = readdir(dp)) == (NULLD))
		fprintf(stderr, "read error on '.'\n");
	    else if ((dir = readdir(dp)) == NULLD)
		fprintf(stderr, "read error on '..'\n");
	    else
	    {
		/* scan through the direcory entries */
		while ((dir = readdir(dp)) != NULLD)
		{
#ifdef SHOW_RESULT
		    fprintf(stderr, "name: %s\n", dir->d_name);
#endif
		    /*
		     * if not a deleted file (else inode == 0)
		     */
		    if (dir->d_ino)
			/*
			 * add directory recursively, this may fail if the
			 * tree is nested too deeply, the directory remains
			 * open!
			 */
			list_files(dir->d_name, depth + 1, mode);
		}
	    }
#ifdef SHOW_RESULT
	    fprintf(stderr, "call do_backup(%lx)\n", start);
#endif
	    /* backup this directory */
	    do_backup(start);

	    /* clean up bookkeeping */
	    reset_pointers();

	    /* return to were you came from */
	    chdir("..");
	}
	/* close the current directory */
	closedir(dp);
    }
    return (0);
}


/* sort the current directory, largest file first */

int         do_backup(myptr)
    fi_ptr      myptr;
{
    int         i;
    int         done;
    fi_ptr     *list;
    fi_ptr      work;
    int         mycmp();

    if (myptr)
    {
	/*
	 * the global counter 'list_count' holds the number of files that
	 * need to be backed-up. 'list' is the array of pointers needed for
	 * sorting.
	 */
	list = MALLOC(fi_ptr, list_count);
	work = start;
	for (i = 0; i < list_count; i++)
	{
	    list[i] = work;
	    work = work->fi_next;
	}

	/* quick sort the list */
	qsort(list, list_count, sizeof(fi_ptr), mycmp);

	/*
	 * try to fit all the files in the current tar archive all files
	 * that did not fit in the first archive will be put in following
	 * archive(s)
	 */
	do
	{
	    done = TRUE;
	    for (i = 0; i < list_count; i++)
	    {
		/* not yet backed up */
		if (list[i] != NULL_FI)
		{
		    /* try to back up */
		    if (backup_one(list[i], file_number++) == 1)
			done = FALSE;
		    else
			/* backup worked */
			list[i] = NULL_FI;
		}
	    }
	    /*
	     * if done is still false, the current list of files won't fit
	     * in the current archive 'oname'.
	     */
	    if (done == FALSE)
	    {
		/* close the current file */
		close_curnt();

		open_next();
		block_count = 0L;
	    }
	} while (!done);
	FREE(list);
    }
}

#ifdef ASCENDING
#define QSORT_GREATER (1)
#define QSORT_SAME (0)
#define QSORT_SMALLER (-1)
#else
#define QSORT_GREATER (-1)
#define QSORT_SAME (0)
#define QSORT_SMALLER (1)
#endif

int         mycmp(first, second)
    fi_ptr     *first;
    fi_ptr     *second;
{
    long        val1 = (*first)->fi_size;
    long        val2 = (*second)->fi_size;

    if (val1 == val2)
	return (QSORT_SAME);
    if (val1 > val2)
	return (QSORT_GREATER);
    return (QSORT_SMALLER);
}

#ifdef MY_ALLOC
extern unsigned long allocated();
#endif

int         reset_pointers()
{
#ifdef DEBUG_RESET
    fprintf(stderr, "call free_info:\n");
#endif
    free_info(start);
    fptr = &start;
    start = (fi_ptr) 0;
    list_count = 0;
#ifdef DEBUG_RESET
#ifdef MY_ALLOC
    fprintf(stdout, "reset pointers, allocated %ld\n", allocated());
#endif
#endif
}

void        add_path(depth, statbuf)
    int         depth;
    struct stat *statbuf;
{
    int         i;
    char        path_buf[MAX_NAME];
    fileinfo    work;

    path_buf[0] = '\0';
    for (i = 0; i <= depth; i++)
    {
	strcat(path_buf, pathname[i]);
	if (i != depth)
	    strcat(path_buf, "/");
    }
    work.fi_name = path_buf;
    work.fi_size = statbuf->st_size;
    work.fi_bsiz = 0L;
    work.fi_back = (char *) 0;
    work.fi_next = (fi_ptr) 0;

    while (backup_one(&work, -1) != 0)
    {
	/* close the current file */
	close_curnt();

	open_next();
	block_count = 0L;
    }
}


int         exclude(name)
    char       *name;
{
    register char *ext;
    register int i;

#ifdef DEBUG_EXCLUDE
    fprintf(stderr, "exclude %s\n", name);
#endif
    if ((ext = strrchr(name, '.')) != (char *) 0)
    {
	ext++;
#ifdef DEBUG_EXCLUDE
	fprintf(stderr, "extension %s\n", ext);
#endif
	for (i = 0; i < numskip; i++)
	{
	    if (strcmp(ext, skip[i]) == 0)
	    {
#ifdef DEBUG_EXCLUDE
		fprintf(stderr, "excluded\n");
#endif
		return (TRUE);
	    }
	}
    }
    return (FALSE);
}

void        add_file(depth, statbuf)
    int         depth;
    struct stat *statbuf;
{
    int         i;
    char        myname[MAX_NAME];
    fi_ptr      myptr;
    char       *cp;
    char       *ctime();

    if (exclude(pathname[depth]))
	return;

    if ((statbuf->st_ctime) > since)
    {
	/* construct the path */
	*myname = '\0';
	for (i = 0; i < depth; i++)
	{
	    strcat(myname, pathname[i]);
	    strcat(myname, "/");
	}
	/* add the name */
	strcat(myname, pathname[depth]);

	/* skip the backup temporary directory */
#ifdef DEBUG_BACKTMP
	fprintf(stdout, "%s == %s\n", myname, dirback);
#endif
	if (strncmp(myname, dirback, strlen(dirback)) == 0)
	{
	    fprintf(stderr, "file %s (in %s) skipped\n", myname, dirback);
	    return;
	}

	myptr = MALLOC(fileinfo, 1);
	if (myptr == (fi_ptr) 0)
	{
	    fprintf(stderr, "add_file: can't malloc\n");
	    exit(1);
	}
	*fptr = myptr;
	fptr = &(myptr->fi_next);
	myptr->fi_next = (fi_ptr) 0;
	list_count++;

	if ((cp = strsave(myname)) == (char *) 0)
	{
	    fprintf(stderr, "can't malloc name\n");
	    exit(1);
	}
	myptr->fi_name = cp;
	myptr->fi_size = statbuf->st_size;
	myptr->fi_bsiz = 0L;
	myptr->fi_back = (char *) 0;
    }
}
