/* backtar.c - tape archiver after: Michiel Huisjes */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#define NO_DEBUG

typedef char BOOL;

#define TRUE	1
#define FALSE	0
#define HEADER_SIZE	512
#define NAME_SIZE	100
#define LIST_SIZE   512
#define LINE_SIZE   256

#define BLOCK_BOUNDARY	 20

typedef union
{
    char        hdr_block[HEADER_SIZE];
    struct m
    {
	char        m_name[NAME_SIZE];
	char        m_mode[8];
	char        m_uid[8];
	char        m_gid[8];
	char        m_size[12];
	char        m_time[12];
	char        m_checksum[8];
	char        m_linked;
	char        m_link[NAME_SIZE];
    }           member;
}           HEADER;

static HEADER header;

#define INT_TYPE	(sizeof(header.member.m_uid))
#define LONG_TYPE	(sizeof(header.member.m_size))

#define NIL_HEADER	((HEADER *) 0)
#define NIL_PTR		((char *) 0)
#define BLOCK_SIZE	((unsigned) 512)
#define INTER_COUNT 18

extern char *progname;
static int  isfloppy;
static int  tar_fd;
static char io_buffer[BLOCK_SIZE];
static char path[NAME_SIZE];
static char pathname[NAME_SIZE];
static int  interval = INTER_COUNT;	/* this program knows too much */

static int  total_blocks;
extern char *malloc();
extern void exit();
extern void free();
extern void sync();
char       *strsave();
static int  make_header();
static int  clear_header();
static int  copy();
int         adjust_boundary();
static int  mwrite();

static char *path_name(file)
    register char *file;
{
    sprintf(pathname, "%s%s", path, file);
    return pathname;
}

static int  checksum()
{
    register char *ptr = header.member.m_checksum;
    register int ac = 0;

    while (ptr < &header.member.m_checksum[INT_TYPE])
	*ptr++ = ' ';

    ptr = header.hdr_block;
    while (ptr < &header.hdr_block[BLOCK_SIZE])
	ac += *ptr++;

    return ac;
}

int         add_tar(tf_handle, orig, back)
    int         tf_handle;
    char       *orig;
    char       *back;
{
    char       *cp;
    char       *cq;
    struct stat st;
    register int fd;

    tar_fd = tf_handle;
#ifdef DEBUG
    fprintf(stderr, "fd %d, orig %s, back %s\n",
	    tf_handle, CHARP(orig), CHARP(back));
#endif
    if (back != NIL_PTR)
    {
	cp = back;
	cq = orig;
    }
    else
    {
	cp = orig;
	cq = NIL_PTR;
    }
    if (stat(cp, &st) < 0)
    {
	fprintf(stderr, "can't find %s\n", cp);
	return;
    }
    if (st.st_mode & S_IFREG)
    {
	if ((fd = open(cp, 0)) < 0)
	{
	    fprintf(stderr, "Can't open %s\n", cp);
	    return;
	}
	make_header(path_name(cp), cq, &st);
	mwrite(tar_fd, &header, (unsigned) sizeof(header));
	copy(fd, tar_fd, st.st_size);
	close(fd);
    }
    else if ((st.st_mode & S_IFMT) == S_IFDIR)
    {
	make_header(path_name(cp), cq, &st);
	mwrite(tar_fd, &header, (unsigned) sizeof(header));
    }
    else
    {
	fprintf(stderr, "%s: unknown file type %ld. Not added.\n",
		cp, (long) st.st_mode);
	return;
    }
#ifdef DEBUG
    fprintf(stderr, "done orig %s\n", CHARP(orig));
#endif
}

static int  make_header(file, back, st)
    char       *file;
    char       *back;
    register struct stat *st;
{
    register char *ptr = header.member.m_name;
    struct stat st_orig;
    struct stat *sp;
    long        size;

    clear_header();
    sp = st;
    size = st->st_size;

    while (*ptr++ = *file++);
    if (back != NIL_PTR)
    {
	sp = &st_orig;
	header.member.m_linked = 'Z';
	ptr = header.member.m_link;
	while (*ptr++ = *back++);
	ptr = header.member.m_link;
	if (stat(ptr, sp) != 0)
	{
	    fprintf(stderr, "lost original file '%s'\n", ptr);
	    exit(1);
	}
    }
    else
	header.member.m_linked = ' ';

    if (st->st_mode & S_IFDIR)
    {
	*(ptr - 1) = '/';
	size = 0L;
    }
    sprintf(header.member.m_mode, "%05o ", sp->st_mode & 07777);
    sprintf(header.member.m_uid, "%05o ", sp->st_uid);
    sprintf(header.member.m_gid, "%05o ", sp->st_gid);
    sprintf(header.member.m_size, "%lo ", size);
    sprintf(header.member.m_time, "%lo ", sp->st_mtime);
    sprintf(header.member.m_checksum, "%05o", checksum());
}

static int  clear_header()
{
    register char *ptr = header.hdr_block;

    while (ptr < &header.hdr_block[BLOCK_SIZE])
	*ptr++ = '\0';
}

int         adjust_boundary()
{
    clear_header();
    mwrite(tar_fd, &header, (unsigned) sizeof(header));
}

static int  mwrite(fd, address, bytes)
    int         fd;
    unsigned    bytes;
    char       *address;
{
    if (write(fd, address, bytes) != bytes)
    {
	fprintf(stderr, "%s: write error\n", progname);
	exit(1);
    }
    total_blocks++;
    if (isfloppy)
	if ((total_blocks % interval) == 0)
	    sync();
}

static int  copy(from, to, bytes)
    int         from, to;
    register long bytes;
{
    unsigned  rest;
    int       blocks;

    blocks = (int) ((bytes + (long) BLOCK_SIZE - 1) / (long) BLOCK_SIZE);
    while (blocks--)
    {
	(void) read(from, io_buffer, BLOCK_SIZE);
	rest = (bytes > (long) BLOCK_SIZE) ? BLOCK_SIZE : (int) bytes;
	mwrite(to, io_buffer, (to == tar_fd) ? BLOCK_SIZE : rest);
	bytes -= (long) rest;
    }
}
