/* Yalex.c		Lexical analyzer */
/* 25-Jul-87 IBM */
/* 25-Mar-88 VAXVMS */
/* 07-Jul-89 ZTC */

/* Copyright 1987,1988,1989 David A. Clunie. All rights reserved.
   PO Box 811, Parkville 3052 AUSTRALIA.
   This program may be freely distributed for non-commercial use. */

/*	Defines:	yalex()		nxtchar()	unchar()
			where()		flinenum()

	Statics:	clrbuf()	addbuf()	insbuf()
			iscomment()	isstring()	iswhite()
			skipwhite()	skipwst()	match()
			acopy()		ucopy()		dcopy()
			gettag()	puttag()	nxtcopy()

	Uses:		chksym()	errmsg()	settok()
*/

#include <stdio.h>

#include <ctype.h>

#ifdef TTYHEAD
#include TTYHEAD
#endif

#define	isodigit(x)	(x >= '0' && x <= '7')
#define	isidstart(x)	(isalpha(x) || x == '_')
#define	isidbody(x)	(isalnum(x) || x == '_')

#define PHASE1

#include "yadefs.h"

#include "yalex.h"			/* Renamed ytab.h from parser */

YYSTYPE yylval;				/* NB. Must be ancestor of yaparse.c */

static int linenum = 0;			/* Count of input line (starts at 1) */
static char lexbuf[BUFLNG];
static char *bufp,*bufend;
static char *tagname;			/* Union tag in actions */

int
yylex()
{
    int match();
    void clrbuf(),addbuf(),insbuf(),skipwhite(),nxtchar();
    void dcopy(),ucopy(),acopy();
    void flinenum();
    SYMBOL *chksym();
    void settok();
    void unchar();

    register int i;
    int t;
    int delim;

    if (linenum == 0) nxtchar();	/* First time through */

    clrbuf();
    skipwhite();

    if (isdigit(ch)) {
	do {
    	    addbuf(ch);
    	    nxtchar();
	} while (isdigit(ch));
	yylval.u_symbol=chksym(lexbuf);
	return NUMBER;
    }

    if (isidstart(ch)) {			/* Identifier */
	do {
    	    addbuf(ch);
    	    nxtchar();
	} while (isidbody(ch));
	skipwhite();				/* Lookahead for colon */
	yylval.u_symbol=chksym(lexbuf);
	if (ch == ':') {
    	    return C_IDENTIFIER;
	}
	else {
    	    return IDENTIFIER;
	}
    }

    switch (ch) {
	case '\"':
	case '\'':
			delim=ch;
			nxtchar();			/* Character Const */
			switch (ch) {
			    case '\"':
			    case '\'':
					if (ch == delim) {
					    i=0;	/* '' -> null */
					}
					else {		/* ordinary char */
					    i=ch;
					    addbuf(ch);
					    nxtchar();
					}
					break;
			    case '\\':	addbuf(ch);
					nxtchar();	/* escape character */
					if (isodigit(ch)) {
					    i=0;
					    while (isodigit(ch)) {
						i=(i<<3) + (ch-'0');
						addbuf(ch);
						nxtchar();
					    }
					    i=i & 0377;
					}
					else {
					    switch (ch) {
						case 'n':  i='\n'; break;
						case 't':  i='\t'; break;
						case 'b':  i='\b'; break;
						case 'r':  i='\r'; break;
						case 'f':  i='\f'; break;
						case '\\': i='\\'; break;
						case '\'': i='\''; break;
						case '\"': i='\"'; break;
						default:   i=ch;
					    }
					    addbuf(ch);
					    nxtchar();
					}
					break;
			    default:	i=ch;		/* ordinary char */
					addbuf(ch);
					nxtchar();
			}
			t=LITERAL;
			while (ch != delim && ch != EOF) {
			    t=IDENTIFIER;	/* if > 1 character */
			    addbuf(ch);
			    nxtchar();
			}
			if (ch != EOF) {	/* trailing quote */
			    nxtchar();
			}
			if (t == IDENTIFIER) {
			    skipwhite();
			    if (ch == ':') {
				t=C_IDENTIFIER;
			    }
			}
			else {			/* LITERAL - reinsert quotes */
			    addbuf('\'');
			    insbuf('\'');
			}
			yylval.u_symbol=chksym(lexbuf);
			if (t == LITERAL) {
			    settok(yylval.u_symbol,i);
			    /* token number is char value */
			}
			return t;

	case '=':	nxtchar();
			if (ch != '{') {
			    unchar();
			    ch='=';
			}
			/* fall thru */
	case '{':	delim=(ch == '=') ? ';' : '}';
			if (section == RULES) {
			    fprintf(faction,"\ncase %d: {\n",nextrule);
			    flinenum(faction);
			    nxtchar();
			    acopy(faction,delim);
			    if (delim == ';') fputs(";",faction);
			    fputs("\n} break;\n",faction);
    			    return ACTBODY;
    			}
			else {				/* assume definition */
			    ucopy();
			    return UNIONBODY;
			}

	case '\\':
	case '%':	nxtchar();
	    		switch (ch) {
			    case '\\':
    			    case '%':	nxtchar();
					if (section == RULES) {
					    return 0;	/* End marker */
					}
					else {
					    return MARK;
					}
   	 		    case '{':	flinenum(fytabc);
					nxtchar();
					dcopy(fytabc);
    					return DEFBODY;
    			    case '<':	nxtchar(); return YLEFT;
    			    case '>':	nxtchar(); return YRIGHT;
    			    case '2':	nxtchar(); return YNONASSOC;
    			    case '0':	nxtchar(); return YTOKEN;
    			    case '=':	nxtchar(); return YPREC;
    			    case 'l':	if (match("left")) {
    					    return YLEFT;
    					}
    					break;
    			    case 'r':	if (match("right")) {
    					    return YRIGHT;
    					}
    					break;
    			    case 'n':	if (match("nonassoc")) {
    					    return YNONASSOC;
    					}
    					break;
    			    case 'b':	if (match("binary")) {
    					    return YNONASSOC;
    					}
    					break;
    			    case 't':	if (match("token")) {
    					    return YTOKEN;
    					}
    					else if (match("erm")) {
    					    return YTOKEN;
    					}
    					else if (match("ype")) {
    					    return YTYPE;
    					}
    					break;
    			    case 'p':	if (match("prec")) {
    					    return YPREC;
    					}
    					break;
    			    case 's':	if (match("start")) {
    					    return YSTART;
    					}
    					break;
    			    case 'u':	if (match("union")) {
    					    return YUNION;
    					}
    					break;
    			}
			errmsg("Invalid %command",FATAL);
			return BADCHAR;

	case	':':	nxtchar(); return COLON;
	case	';':	nxtchar(); return SEMICOLON;
	case	',':	nxtchar(); return COMMA;
	case	'<':	nxtchar(); return LANGLE;
	case	'>':	nxtchar(); return RANGLE;
	case	'|':	nxtchar(); return BAR;
    }
    if (ch == EOF) {
	return 0;
    }
    else {
	nxtchar();
	return BADCHAR;
    }
}

static void
clrbuf()
{
    bufp=lexbuf;				/* first char in lexbuf */
    *bufp='\0';
    bufend=lexbuf+BUFLNG-1;			/* last char in lexbuf */
}

static void
addbuf(i)
register int i;
{
    void errmsg();

    if (bufp < bufend) {			/* ensures space for \0 */
	*bufp=i;
	*++bufp='\0';
    }
    else
	errmsg("Lex buffer ovf",ABORT);
}

static void
insbuf(i)
int i;
{
    void errmsg();
    register char *p;

    if (bufp < bufend) {
	for (p= ++bufp; p > lexbuf; p--) {
	    *p= *(p-1);		/* "= *" and "= ++" avoid old assignment ops */
	}
	*p=i;
    }
    else
	errmsg("Lex buffer ovf",ABORT);
}

static int				/* Does current ch start a string ? */
isstring(f1,f2)				/* If so copy to f1,f2 & get next ch */
FILE *f1,*f2;
{
    void nxtcopy();
    int quote;

    if (ch == '\'' || ch == '\"') {
	quote=ch;
	do {
	    nxtcopy(f1,f2);
	    if (ch == '\\') {			/* Avoid escaped quotes */
		nxtcopy(f1,f2);			/* Ignore escape character */
		nxtcopy(f1,f2);			/* and its successor */
	    }
	} while (ch != quote && ch != EOF);
	if (ch == quote) nxtcopy(f1,f2);
	return 1;
    }
    else {
	return 0;
    }
}

static int				/* Does current ch start a comment ? */
iscomment(f1,f2)			/* If so copy to f1,f2 & get next ch */
FILE *f1,*f2;
{
    void unchar(),nxtcopy(),nxtchar();

    if (ch == '/') {
	nxtchar();
	if (ch == '*') {
	    if (f1) putc('/',f1);
	    if (f2) putc('/',f2);
	    do {
		nxtcopy(f1,f2);
		if (ch == '*') {
		    nxtcopy(f1,f2);
		    if (ch == '/') {
			nxtcopy(f1,f2);
			return 1;	/* Leaves next char past / in ch */
		    }
		}
	    } while (ch != EOF);
	    return 1;			/* Comment terminated by EOF !! */
	}
	else {
	    unchar();
	    ch='/';
	}
    }
    return 0;
}

static int				/* Is current ch whitespace ? */
iswhite(f1,f2)				/* If so copy to f1,f2 & get next ch */
FILE *f1;
FILE *f2;
{
    void nxtcopy();

    if (isspace(ch)) {
	nxtcopy(f1,f2);
	return 1;
    }
    else {
	return 0;
    }
}

static void				/* Skip whitespace & comments */
skipwhite()
{
    int iswhite(),iscomment();

    while (iswhite(NULL,NULL) || iscomment(NULL,NULL)) {
    }
}

static void				/* Skip whitespace,comments, strings */
skipwst(f1,f2)				/* (copying skipped chars to f1 & f2)*/
FILE *f1,*f2;
{
    int iswhite(),iscomment(),isstring();

    while (iswhite(f1,f2) || iscomment(f1,f2) || isstring(f1,f2)) {
    }
}

static int
match(s)
register char *s;
{
    void nxtchar();

    while (ch == *s) {
	++s;
	nxtchar();
    }
    return (*s == 0);			/* Matched until end of string */
}

static void				/* Output current character */
nxtcopy(f1,f2)				/* and then get the next one */
FILE *f1;
FILE *f2;
{
    void nxtchar();

    if (f1) putc(ch,f1);
    if (f2) putc(ch,f2);
    nxtchar();
}

void
nxtchar()
{
    if (linenum == 0 || ch == '\n') {	/* Won't look at nonexistent */
					/* ch first time through */
	++linenum;
	if (isatty(fileno(finput))) {
	    fputs(PROMPT,stderr);
	}
    }
    ch=getc(finput);
    if (iseofchar(ch)) {	/* defined in yasystem.h */
	ch=EOF;
    }
}

void
where(f)				/* Display input location */
FILE *f;
{
    fprintf(f,"[At line %d]",linenum);
}

void
unchar()
{
    ungetc(ch,finput);
}

void
flinenum(f)				/* Write input linenum to output */
FILE *f;
{
    if (o_linenum)
	fprintf(f,"\n#line %d\n",linenum);  /* NB. MUST be at start of line */
}

static void
acopy(f,end)			/* Copy until '}' or ';', replacing "$nn" */
FILE *f;			/* Doesn't detect either in comments */
int end;			/* either '}' or ';' (NOT copied) */
{
    void puttag(),gettag(),skipwst(),nxtchar();
    int i,depth;

    depth=0;
    while (ch != EOF) {
	skipwst(f,NULL);
	switch (ch) {
		case ';':	nxtchar();
				if (end == ';') {
				    return;
				}
				putc(';',f);
				break;
		case '{':	++depth;
				putc('{',f);
				nxtchar();
				break;
		case '}':	nxtchar();
				if (depth-- == 0 && end == '}') {
				    return;
				}
				putc('}',f);
				break;
		case '$':	nxtchar();
				gettag();
				if (ch == '$') {
				    fputs("yyval",f);
				    puttag(f,1,0);
				    nxtchar();
				}
				else {
				    if (isdigit(ch) || ch == '-') {
					clrbuf();
					if (ch == '-') {
					    addbuf(ch);
					    nxtchar();
					    if (!isdigit(ch)) {
						errmsg("$- but no number",
							FATAL);
					    }
					}
					while (isdigit(ch)) {
					    addbuf(ch);
					    nxtchar();
					}
					i=atoi(lexbuf);
					if (i > g_offset) {
					    errmsg("$nn is too far right",
						FATAL);
					}
					fprintf(f,"yypvt[%d]",i-g_offset);
					puttag(f,0,i);
				    }
				    else {
					errmsg("$ not followed by $ or number",
						FATAL);
				    }
				}
				break;
		default:	putc(ch,f);
				nxtchar();
				break;
	}
    }
}

static void
ucopy()				/* Copy until "}" except in comment */
{
    void skipwst(),nxtchar();

    fputs("\ntypedef union {\n",fytabc);
    flinenum(fytabc);
    if (o_define) {
	fputs("\ntypedef union {\n",fytabh);
	flinenum(fytabh);
    }
    else {
	fytabh=0;		/* For skipwst's benefit */
    }
    nxtchar();
    for (;;) {
	skipwst(fytabc,fytabh);
	if (ch == '}' || ch == EOF) {
	    break;
	}
	putc(ch,fytabc);
	if (o_define) putc(ch,fytabh);
	nxtchar();
    }
    nxtchar();			/* absorb trailing } */

    fputs("} YYSTYPE;\n",fytabc);
    if (o_define) fputs("} YYSTYPE;\n",fytabh);
}

static void
dcopy(f)			/* Copy until "%}" except in comment */
FILE *f;
{
    void skipwst(),nxtchar();

    for (;;) {
	skipwst(f,NULL);
	if (ch == EOF) {
	    break;
	}
	if (ch == '%' || ch == '\\') {
	    int delim=ch;
	    nxtchar();
	    if (ch == '}') {
		nxtchar();
		break;
	    }
	    else {
		unchar();
		ch=delim;
	    }
	}
	putc(ch,f);
	nxtchar();
    }
}

void
gettag()			/* Read union tag xxx from $<xxx>nn */
{
    if (ch == '<') {
 	clrbuf();
	nxtchar();
	while (ch != '>') {
	    addbuf(ch);
	    nxtchar();
	}
	nxtchar();		/* Absorb trailing > */
	tagname=xalloc(strlen(lexbuf)+1);
	strcpy(tagname,lexbuf);
    }
    else {
	tagname=NULL;
    }
}

void
puttag(f,lflag,nn)		/* Put optional union tag into output stream */
FILE *f;
int lflag;			/* 0 = right symbol ($nn), 1 = left ($$) */
int nn;				/* $nn if right symbol */
{
    char *tag;
    LIST *list;

    if (g_typeon) {
	trace(("puttag: type checking is on\n"));
	if (lflag) {
	    trace(("puttag: lflag\n"));
	    if (tagname) {		/* Explicitly specified type */
		tag=tagname;		/* used for embedded actions */
	    }
	    else {			/* Type of left non-terminal */
		tag=g_cid->type;	/* used for trailing actions */
		if (tag == 0) {
		    errmsg("$$ has no type",FATAL);
		}
	    }
	    /* NB. Don't know whether or embedded or trailing action yet
	       because we can't lookahead that far, but we need to emit
	       a tag now. Check the appropriateness of the selection
	       after the action.
	    */
	}
	else {
	    if (nn >= 1) {	/* Find symbol in current rule */
		trace(("puttag: checking rule %04x for nn %d\n",g_rule,nn));
		for (list=g_rule->list; nn > 1 && list; --nn,list=list->next);
		if (list) {
		    tag=list->sym->type;
		    if (tag) {
			if (tagname) {
			    if (strcmp(tag,tagname) == 0) {
				errmsg("Explicit type unnecessary",WARNING);
			    }
			    else {
				errmsg("Explicit type different from symbol",
				    FATAL);
			    }
			}
		    }
		    else {
			if (tagname) {
			    tag=tagname;
			}
			else {
			    errmsg("Right-hand symbol has no type",FATAL);
			}
		    }
		}
		else {
		    /* right context reference illegal - checked already */
		    tag=0;
		}
	    }
	    else {
		if (tagname) {
		    tag=tagname;
		}
		else {
		    errmsg("Left context reference requires explicit type",
			FATAL);
		    tag=0;
		}
	    }
	}

	if (tag) {
	    trace(("puttag: tag is <%s>\n",tag));
	    fprintf(f,".%s",tag);
	}
    }
    else {			/* Type checking not turned on */
	if (tagname) {		/* But explicit type was specified */
	    errmsg("Type checking is not activated",FATAL);
	}
    }
}

