/* IBM PC console driver (display code).
 *
 * The console driver is split into two parts.  This file contains the display
 * driver code.  The keyboard driver code is contained in the file keyboard.c.
 *
 * The console is really a collection of consoles.  One of which is currently
 * being displayed.  This console is referred to as the foreground console.
 * The remaining consoles are only kept up to date in local memory.  The
 * function keys are used to select the current foreground console.  A user
 * program can write to a console even if it is in the background.
 *
 * The video controller chip (6845) has a register that points to the address
 * in memory of the first line of the screen.  By increasing the value of this
 * register the screen can be moved up a line.  This means that the screen can
 * be made to scroll very fast since it is not necessary to move the entire
 * screen about in memory.  Eventually the last line of the screen will exceed
 * the video RAM.  Depending on the controller the memory may wrap around to
 * the start of video RAM, or it may be necessary to copy the screen back up to
 * the start of video RAM.  Scrolling the virtual console is performed in a
 * similar fashion - the data is contained in a cyclic buffer, and a pointer
 * to the first character is maintained.  The current video RAM contents
 * reflect the foreground virtual console memory.
 *
 * The IBM display is memory mapped, so outputing characters such as line
 * feed, backspace and bell are tricky.  Each character on the screen is
 * described by a pair of adjacent bytes.  The first byte contains character
 * attribute information, such as the display colors.  The second byte contains
 * the code of the character to be displayed.  Accessing the display memory is
 * expensive since it typically is not dual ported, this means that the CPU has
 * to perform wait states if it tries to access it while the video controller
 * is.  A block copy operation to the video memory might run at one tenth of
 * the speed of a block copy to normal memory.  On the CGA card the CPU is not
 * forced to wait, instead the video controller is locked out while the CPU
 * accesses memory.  This however causes snow to appear on the screen (small
 * blank scan lines).  Trying to prevent snow greatly reduces display speed.
 *
 * To make the console driver fast a number of optimizations have been
 * performed.  Printable characters are processed by a special purpose loop
 * instead of calling the general purpose character processing routine.  A
 * short word that can be or'ed with a character to give the value to be placed
 * in the video RAM is maintained.  Characters are copied directly to the
 * virtual console memory, and the screen image is updated from this memory
 * using a block copy operation only when necessary.  An array containing the
 * background color is maintained so that the screen can quickly be blanked out
 * using memcpy.
 *
 * The behavior of the console driver upon reaching the last column of the
 * screen could be improved.  With line wrap turned on it is not possible to
 * place a character in the lower right hand corner of the screen.  Attempting
 * to do so causes the screen to scroll up a line.  With line wrap turned off
 * the escape codes for positioning the cursor on a different row do not work
 * after a character has been placed in the last column.  And assumptions such
 * as the number of characters on the screen up to and including the cursor
 * being given by dp->cursor + 1 are violated.  This is assumed in the command
 * ESC [1J.  The command ESC [1K is known to fail under similar circumstances.
 * Other commands may likewise fail.
 */

#include "kernel.h"
#include <sgtty.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include "protect.h"
#include "tty.h"

/* Constants used by the console driver. */
#define C_VID_SIZE    0x2000	/* color, 8k word of video RAM */
#define M_VID_SIZE    0x0800	/* monochrome, 2k word of video RAM */
#define CGA_RETRACE      160	/* CGA number of chars. to display at once */
#define BLANK_BUF_SIZE	  40	/* size of buffer containing blank chars. */
#define DEF_FLUSH_COST	  12	/* default flush cost */
#define SCR_COLS          80	/* # characters on a line */
#define SCR_ROWS          25	/* # lines on the screen */
#define SCR_CHARS (SCR_COLS * SCR_ROWS)	/* # characters on the screen */
#define BLACK		   0	/* black = color 0 (on both mono. and color) */
#define WHITE		   7	/* white = color 7 (on both mono. and color) */
#define GO_FORWARD         0	/* scroll forward */
#define GO_BACKWARD        1	/* scroll backward */
#define MAX_ESC_PARMS     10	/* number of escape sequence params allowed */
#define MAX_ESC_STR       20	/* length of escape control string allowed */

/* Constants relating to the controller chips. */
#define M_6845         0x3B0	/* port base for 6845 mono */
#define C_6845         0x3D0	/* port base for 6845 color */
#define EGA            0x3C0	/* port base for EGA card */
#define INDEX              4	/* 6845's index port */
#define DATA               5	/* 6845's data port */
#define STATUS		  10	/* 6845's status port */
#define CUR_SIZE          10	/* 6845's cursor size register */
#define VID_ORG           12	/* 6845's origin register */
#define CURSOR            14	/* 6845's cursor register */

/* Beeper. */
#define BEEP_FREQ     0x0533	/* value to put into timer to set beep freq */
#define B_TIME		   3	/* length of CTRL-G beep is ticks */

/* Macro to map a console number to a display_s structure, and vice-versa. */
#define screen_addr(num)	(&screen_lines[num])
#define screen_num(dp)		((struct display_s *) (dp) \
					 - (struct display_s *) screen_lines)

/* Global variables used by console driver, and assembly language code. */
PUBLIC int vid_mask;		/* 0x3fff for color (16K) or 0x0fff for mono */
PUBLIC int vid_port;		/* I/O port for accessing 6845 */
PUBLIC int blank_color;		/* obsolete (referenced by klib88.x) */

/* Private variables used by the console driver. */
PRIVATE int vid_size;		/* 0x2000 for color (16K) or 0x0800 for mono */
PRIVATE unsigned vid_base;	/* base of video ram (0xB000 or 0xB800) */
PRIVATE int vid_org;		/* position of first character of video ram */
PRIVATE int new_org;		/* new value to be loaded into vid_org */
PRIVATE int buf_first, buf_last;
	/* Characters between buf_first, and buf_last have yet to be copied
	 * from the virtual console to the video RAM.  dp->cursor always lies
	 * between buf_first and buf_last.
	 */
PRIVATE int flush_cost;
	/* Fixed overhead of flushing characters from the virtual screen to
	 * the video RAM relative to the per character overhead.
	 *
	 *	 time(flush n chars.) + time(flush m chars.)
	 *		 = time(flush n + m + flush_cost chars.)
	 *
	 * Typically the flush cost will be between 8 and 16 depending upon
	 * the speed of the graphics card and the compiler used to compile
	 * Minix.
	 *
	 * A faster graphics card will result in a smaller per character
	 * overhead, and hence a larger flush cost.
	 *
	 * Compiling Minix with a better compiler than the standard K&R C ACK
	 * compiler will reduce the flush cost since the fixed overhead will
	 * be reduced, but the assembly language vid_copy routine will remain
	 * unchanged.
	 */

/* Map from ANSI colors to the attributes used by the PC. */
PRIVATE int ansi_colors[8] = {0, 4, 2, 6, 1, 5, 3, 7};

/* Rendition structure, specifies character rendering options. */
struct rendition_struct {
  int bold;			/* increased intensity flag */
  int underlined;		/* underlined flag */
  int blinking;			/* slowly blinking flag */
  int negative;			/* inverse video flag */
  int fg_color;			/* foreground color */
  int bg_color;			/* background color */
};

/* Display screen structure, 1 per virtual console. */
struct display_s {
  int display;			/* 1 if screen is the foreground console */

  /* General parameters and state. */
  int column;			/* current column number (0-origin) */
  int row;			/* current row (0 at top of screen) */
  int linewrap;			/* 1 if wrap long lines */
  int softscroll;		/* 1 = software scrolling, 0 = hardware */
  int snow;			/* 1 = screen snows, try and prevent it */
  int cursor_visible;		/* 1 = cursor is visible */
  struct rendition_struct rendition;	/* current rendition */
  struct rendition_struct default_rendition;	/* default rendition */

  /* The next parameter is derived from the current rendition. */
  int attr;			/* current attribute byte << 8 */

  /* Escape state machine and parameter values. */
  char esc_state;		/* 0=normal, 1=ESC, 2=ESC [, 3=ESC P,
				 * 4=ESC P s ESC
				 */
  char esc_intro;		/* distinguishing character following ESC */
  int esc_parmv[MAX_ESC_PARMS];	/* list of escape parameters */
  int *esc_parmp;		/* pointer to current escape parameter */
  char esc_strv[MAX_ESC_STR];	/* escape control string parameter */
  char *esc_strp;		/* pointer to escape string character */

  /* Hardware parameters. */
  int org;			/* position of first char. in cyclic buffer */
  int cursor;			/* offset of cursor from org */
  short obuf[SCR_CHARS];	/* virtual screen buffer */
  short blanks[BLANK_BUF_SIZE];	/* buffer containing blank chars. */
};

/* Console driver display screen structures. */
PRIVATE struct display_s screen_lines[NR_CONSOLES];

FORWARD void beep();
FORWARD void do_escape();
FORWARD void flush();
FORWARD void long_vid_copy();
FORWARD void move_to();
FORWARD void out_char();
FORWARD void parse_escape();
FORWARD void scr_blank();
FORWARD void scr_copy();
FORWARD void scroll_screen();
FORWARD void select_attr();
FORWARD void set_6845();
FORWARD void set_cursor();
FORWARD void set_flush_cost();
FORWARD void set_rendition();
FORWARD void set_softscroll();
FORWARD void stop_beep();
FORWARD long time_flush();

/*===========================================================================*
 *				console					     *
 *===========================================================================*/
PUBLIC void console(num, tbuf, size, mode)
int num, size, mode;		/* mode is the value of tp->tty_mode. */
register char *tbuf;
{
/* Copy the data from the buffer onto the screen.  The values to be displayed
 * are buffered locally in the virtual screen buffer before being copied to the
 * screen.
 */

  struct display_s *dp;		/* display pointer */
  int remaining, domax, attr, done, first;
  short *start, *limit;
  register short *dest;

  dp = screen_addr(num);

  new_org = vid_org;
	/* Already equal, leave above line in the code however for clarity. */
  buf_first = buf_last = dp->cursor;
  remaining = size;

  /* Output each byte to the virtual console.  If we are not at the end of the
   * line and the character is a normal printing character it will be copied
   * directly to the virtual screen instead of being processed using out_char.
   * This is the inner loop of the output routine, so attempt to make it fast.
   */
  while (remaining > 0) {
	domax = SCR_COLS - 1 - dp->column;
	if (domax > remaining) domax = remaining;
	if (*tbuf < ' ' || dp->esc_state > 0 || domax < 1) {
		/* Complex - call out_char. */
		out_char(dp, *tbuf++, mode);
		remaining--;
	} else {
		/* Easy - do it quickly. */
		attr = dp->attr;
		first = dp->org + dp->cursor;
		if (first >= SCR_CHARS) first -= SCR_CHARS;
		start = &dp->obuf[first];
		limit = start + domax;
		dest = start;
		while (dest < limit && *tbuf >= ' ') {
			*dest++ = attr | (*tbuf & BYTE);
			tbuf++;
		}
		done = dest - start;
		dp->column += done;
		dp->cursor += done;
		if (buf_last < dp->cursor) buf_last = dp->cursor;
		remaining -= done;
	}
  }

  flush(dp);			/* transfer anything buffered to the screen */

  /* Update hardware parameters. */
  if (dp->display) {
	if (vid_org != new_org) {
		vid_org = new_org;
		set_6845(VID_ORG, vid_org);
	}
	set_cursor(dp);
  }
}


/*===========================================================================*
 *				out_char				     *
 *===========================================================================*/
PRIVATE void out_char(dp, c, mode)
register struct display_s *dp;
char c;				/* character to be output */
int mode;
{
/* Output a character on the console.  This is currently only called for the
 * complex cases, although it is quite general and could be used to handle all
 * cases.
 */

  int first;

  /* Check for escape sequences first. */
  if (dp->esc_state > 0) {
	parse_escape(dp, c);
	return;
  }

  switch (c) {
	case 000:		/* null is typically used for padding */
		return;		/* better not do anything */

	case 007:		/* ring the bell */
		beep();
		return;

	case 013:		/* CTRL-K */
		move_to(dp, dp->column, dp->row - 1);
		return;

	case 014:		/* CTRL-L */
		move_to(dp, dp->column + 1, dp->row);
		return;

	case 016:		/* CTRL-N */
		move_to(dp, dp->column + 1, dp->row);
		return;

	case '\b':		/* backspace */
		move_to(dp, dp->column - 1, dp->row);
		return;

	case '\n':		/* line feed */
		if (dp->row == SCR_ROWS - 1)
			scroll_screen(dp, GO_FORWARD);
		else
			dp->row++;
		move_to(dp, (mode & CRMOD ? 0 : dp->column), dp->row);
		return;

	case '\r':		/* carriage return */
		move_to(dp, 0, dp->row);
		return;

	case '\t':		/* tab */
		do {
			if (dp->column >= SCR_COLS - 1) {
				out_char(dp, ' ', mode);
			} else {
				first = dp->org + dp->cursor;
				if (first >= SCR_CHARS) first -= SCR_CHARS;
				dp->obuf[first] = dp->attr | ' ';
				dp->column++;
				dp->cursor++;
				if (buf_last < dp->cursor)
					buf_last = dp->cursor;
			}
		} while (dp->column & TAB_MASK);
		return;

	case 033:		/* ESC - start of an escape sequence */
		dp->esc_state = 1;	/* mark ESC as seen */
		return;

	default:		/* normal printable character */
		/* See if a long line should be wrapped or truncated. */
		if (dp->column >= SCR_COLS) {
			if (dp->linewrap)
				out_char(dp, '\n', CRMOD);	/* wrap */
			else
				return;		/* truncate */
		}

		/* Put character in the virtual console buffer. */
		first = dp->org + dp->cursor;
		if (first >= SCR_CHARS) first -= SCR_CHARS;
		dp->obuf[first] = dp->attr | (c & BYTE);
		dp->column++;
		dp->cursor++;
		if (buf_last < dp->cursor) buf_last = dp->cursor;

		/* See if cursor wraps onto the next line. */
		if (dp->linewrap && dp->column >= SCR_COLS) {
			out_char(dp, '\n', CRMOD);
		}

		return;
  }
}


/*===========================================================================*
 *				parse_escape				     *
 *===========================================================================*/
PRIVATE void parse_escape(dp, c)
register struct display_s *dp;
char c;				/* next character of escape sequence */
{
/* Handle escape sequences.  This procedure is called once for each character
 * of an escape sequence.  Hence it uses static variables to keep track of the
 * state seen so far.
 *
 * The following ISO/ANSI escape sequences are currently supported (n is a
 * numeric string, s is a character string).
 *
 *  Name  Encoding                       Synopsis
 *
 *   RI   ESC M      Moves up 1 line, scrolls backwards if on first line.
 *                       Note, ISO standard does not guarantee scrolling.
 *   DCS  ESC Ps ESC \ Device control s.
 *                       linewrap.on=turn line wrap on,
 *                       linewrap.off=turn line wrap off,
 *                       snow.on=turn snow on,
 *                       snow.off=turn snow off,
 *                       softscroll.on=turn soft scrolling on,
 *                       softscroll.off=turn soft scrolling off,
 *			 cursor.on=make the cursor visible,
 *			 cursor.off=make the cursor invisible,
 *			 rendition.set=save current rendition as the default,
 *			 rendition.reset=clear default rendition,
 *                       keyboard.pc=select standard PC keyboard,
 *                       keyboard.olivetti=select Olivetti keyboard,
 *                       keyboard.dutch=select Dutch keyboard,
 *                       keyboard.extended=select extended keyboard.
 *   RIS  ESC c      Reset display.
 *   CUU  ESC [nA    Moves up n lines (default 1).
 *   CUD  ESC [nB    Moves down n lines (default 1).
 *   CUF  ESC [nC    Moves right n spaces (default 1).
 *   CUB  ESC [nD    Moves left n spaces (default 1).
 *   CUP  ESC [n1;n2H  Moves cursor to (n1,n2) (default 1, 1).
 *   ED   ESC [nJ    Clears screen (default 0).
 *                       0=from cursor to bottom of screen,
 *                       1=from top of screen to cursor,
 *                       2=entire screen.
 *   EL   ESC [nK    Clears line (default 0).
 *                       0=from cursor to end of line,
 *                       1=from start of line to cursor,
 *                       2=entire line.
 *   IL   ESC [nL    Inserts n lines ar cursor (default 1).
 *   DL   ESC [nM    Deletes n lines at cursor (default 1).
 *   DCH  ESC [nP    Deletes n chars at cursor (default 1).
 *   ICH  ESC [n@    Inserts n chars at cursor (default 1).
 *   SGR  ESC [n1;n2...m    Enables renditions n1, n2, ... (default 0).
 *                       0=default, 1=bold, 4=underlined, 5=blinking,
 *                       7=negative image, 22=normal intensity, 24=not
 *                       underlined, 25=steady, 27=positive image,
 *                       30-37=foreground color, 39=default foreground color
 *                       40-47=background color, 49=default background color.
 */
int i;

  switch (dp->esc_state) {
	case 1:			/* ESC seen */
		dp->esc_intro = '\0';
		dp->esc_parmp = dp->esc_parmv;
		/* Maximum of 2 unspecified parameters (occurs in ESC [H). */
		dp->esc_parmv[0] = dp->esc_parmv[1] = 0;
		dp->esc_strp = dp->esc_strv;
		switch (c) {
		  case '[':	/* Control Sequence Introducer */
			dp->esc_intro = c;
			dp->esc_state = 2;
			break;
		  case 'P':	/* Device Control String */
			dp->esc_state = 3;
			break;
		  case 'M':	/* Reverse Index */
		  case 'c':	/* Reset Display */
			do_escape(dp, c);
			break;
		  default:	/* invalid, ignore */
			dp->esc_state = 0;
			break;
		}
		break;

	case 2:		/* ESC [ seen */
		if (c >= '0' && c <= '9') {
			*dp->esc_parmp = *dp->esc_parmp * 10 + (c - '0');
			break;
		} else if (c == ';') {
			if (dp->esc_parmp + 1 < dp->esc_parmv + MAX_ESC_PARMS)
				*(++dp->esc_parmp) = 0;
			break;
		} else {
			do_escape(dp, c);
		}
		break;

	case 3:		/* ESC P seen */
		switch (c) {
		  case 033:
			*dp->esc_strp = '\0';
			dp->esc_state = 4;
			break;
		  default:
			if (dp->esc_strp < dp->esc_strv + MAX_ESC_STR - 1)
				*(dp->esc_strp++) = c;
			else
				dp->esc_state = 0; /* too long, ignore */
			break;
		}
		break;

	case 4:		/* ESC P s ESC seen */
		switch (c) {
		  case '\\':
			do_escape(dp, 'P');
			break;
		  default:	/* illegal state */
			dp->esc_state = 0;
			break;
		}
		break;

	default:	/* illegal state */
		dp->esc_state = 0;
		break;
  }
}


/*===========================================================================*
 *				do_escape				     *
 *===========================================================================*/
PRIVATE void do_escape(dp, c)
register struct display_s *dp;
char c;				/* next character in escape sequence */
{
/* Perform an escape sequence, now that the sequence has been recognised. */

  int n, m, value, src, dst, count;
  int *parm;

  /* Handle a sequence beginning with just ESC. */
  if (dp->esc_intro == '\0') {
	switch (c) {
		case 'M':		/* Reverse Index */
			if (dp->row == 0)
				scroll_screen(dp, GO_BACKWARD);
			else
				move_to(dp, dp->column, dp->row - 1);
			break;

		case 'P':		/* Device Control String */
			if (strcmp(dp->esc_strv, "linewrap.on") == 0)
				dp->linewrap = TRUE;
			else if (strcmp(dp->esc_strv, "linewrap.off") == 0)
				dp->linewrap = FALSE;
			else if (strcmp(dp->esc_strv, "snow.on") == 0)
				dp->snow = FALSE;
			else if (strcmp(dp->esc_strv, "snow.off") == 0)
				/* Must only turn snow off for a CGA display,
				 * system could hang if snow was turned off
				 * and the display was not a CGA display.
				 */
				dp->snow = color & !ega;
			else if (strcmp(dp->esc_strv, "softscroll.on") == 0)
				set_softscroll(dp, TRUE);
			else if (strcmp(dp->esc_strv, "softscroll.off") == 0)
				set_softscroll(dp, FALSE);
			else if (strcmp(dp->esc_strv, "cursor.on") == 0)
				dp->cursor_visible = TRUE;
			else if (strcmp(dp->esc_strv, "cursor.off") == 0)
				dp->cursor_visible = FALSE;
			else if (strcmp(dp->esc_strv, "rendition.set") == 0)
				dp->default_rendition = dp->rendition;
			else if (strcmp(dp->esc_strv,
				        "rendition.reset") == 0) {
				dp->default_rendition.bold = FALSE;
				dp->default_rendition.underlined = FALSE;
				dp->default_rendition.blinking = FALSE;
				dp->default_rendition.negative = FALSE;
				dp->default_rendition.fg_color = WHITE;
				dp->default_rendition.bg_color = BLACK;
			} else if (strcmp(dp->esc_strv, "keyboard.pc") == 0)
				load_keyboard(screen_num(dp), IBM_PC);
			else if (strcmp(dp->esc_strv,
					"keyboard.olivetti") == 0)
				load_keyboard(screen_num(dp), OLIVETTI);
			else if (strcmp(dp->esc_strv, "keyboard.dutch") == 0)
				load_keyboard(screen_num(dp), DUTCH_EXT);
			else if (strcmp(dp->esc_strv,
					"keyboard.extended") == 0)
				load_keyboard(screen_num(dp), US_EXT);
			/* else invalid, ignore */
			break;

		case 'c':		/* Reset Display */
			dp->linewrap = LINEWRAP;
			dp->softscroll = ps;
			load_keyboard(screen_num(dp), BOOT_KBD);
			dp->rendition.bold = FALSE;
			dp->rendition.underlined = FALSE;
			dp->rendition.blinking = FALSE;
			dp->rendition.negative = FALSE;
			dp->rendition.fg_color
				= dp->default_rendition.fg_color = WHITE;
			dp->rendition.bg_color
				= dp->default_rendition.bg_color = BLACK;
			select_attr(dp);
			/* clear screen and home cursor */
			scr_blank(dp, 0, SCR_CHARS);
			move_to(dp, 0, 0);
			break;

		default:		/* invalid, ignore */
			break;
	}
  } else {
	/* Handle a sequence beginning with ESC [ and parameters. */
	if (dp->esc_intro == '[') {
		value = dp->esc_parmv[0];
		switch (c) {
		    case 'A':		/* ESC [nA moves up n lines */
			n = (value == 0 ? 1 : value);
			move_to(dp, dp->column, dp->row - n);
			break;

		    case 'B':		/* ESC [nB moves down n lines */
			n = (value == 0 ? 1 : value);
			move_to(dp, dp->column, dp->row + n);
			break;

		    case 'C':		/* ESC [nC moves right n spaces */
			n = (value == 0 ? 1 : value);
			move_to(dp, dp->column + n, dp->row);
			break;

		    case 'D':		/* ESC [nD moves left n spaces */
			n = (value == 0 ? 1 : value);
			move_to(dp, dp->column - n, dp->row);
			break;

		    case 'H':		/* ESC [m;nH moves cursor to (m,n) */
			m = (dp->esc_parmv[0] == 0 ? 0 : dp->esc_parmv[0] - 1);
			n = (dp->esc_parmv[1] == 0 ? 0 : dp->esc_parmv[1] - 1);
			move_to(dp, n, m);
			break;

		    case 'J':		/* ESC [nJ clears the screen */
			switch (value) {
			    case 0:	/* from cursor */
				scr_blank(dp, dp->cursor,
					  SCR_CHARS - dp->cursor);
				break;
			    case 1:	/* to cursor */
				scr_blank(dp, 0, dp->cursor + 1);
				break;
			    case 2:	/* entire screen */
				scr_blank(dp, 0, SCR_CHARS);
				break;
			}
			break;

		    case 'K':		/* ESC [nK clears the current line */
			switch (value) {
			    case 0:	/* from cursor */
				scr_blank(dp, dp->cursor,
					  SCR_COLS - dp->column);
				break;
			    case 1:	/* to cursor */
				scr_blank(dp, dp->cursor - dp->column,
					  dp->column + 1);
				break;
			    case 2:	/* entire line */
				scr_blank(dp, dp->cursor - dp->column,
					  SCR_COLS);
				break;
			}
			break;

		    case 'L':		/* ESC [nL inserts n lines at cursor */
			n = value;
			if (n < 1) n = 1;
			if (n > (SCR_ROWS - dp->row)) n = SCR_ROWS - dp->row;

			src = dp->cursor - dp->column;
			dst = src + n * SCR_COLS;
			count = SCR_CHARS - dst;
			scr_copy(dp, src, dst, count);
			scr_blank(dp, src, n * SCR_COLS);
			break;

		    case 'M':		/* ESC [nM deletes n lines at cursor */
			n = value;
			if (n < 1) n = 1;
			if (n > (SCR_ROWS - dp->row)) n = SCR_ROWS - dp->row;

			dst = dp->cursor - dp->column;
			src = dst + n * SCR_COLS;
			count = SCR_CHARS - src;
			scr_copy(dp, src, dst, count);
			scr_blank(dp, dst + count, n * SCR_COLS);
			break;

		    case 'P':		/* ESC [nP deletes n chars at cursor */
			n = value;
			if (n < 1) n = 1;
			if (n > (SCR_COLS - dp->column))
				n = SCR_COLS - dp->column;

			dst = dp->cursor;
			src = dst + n;
			count = SCR_COLS - dp->column - n;
			scr_copy(dp, src, dst, count);
			scr_blank(dp, dst + count, n);
			break;

		    case '@':		/* ESC [n@ inserts n chars at cursor */
			n = value;
			if (n < 1) n = 1;
			if (n > (SCR_COLS - dp->column))
				n = SCR_COLS - dp->column;

			src = dp->cursor;
			dst = src + n;
			count = SCR_COLS - dp->column - n;
			scr_copy(dp, src, dst, count);
			scr_blank(dp, src, n);
			break;

		    case 'm':		/* ESC [n1;n2...m enables renditions
					 * n1, n2, ... */
			for (parm = dp->esc_parmv; parm <= dp->esc_parmp;
			     parm++) {
				set_rendition(dp, *parm);
			}
			select_attr(dp);
			break;

		    default:
			break;
		}	/* closes switch(c) */
	}	/* closes if (dp->esc_intro == '[') */
  }
  dp->esc_state = 0;
}


/*===========================================================================*
 *				set_rendition				     *
 *===========================================================================*/
PRIVATE void set_rendition(dp, value)
register struct display_s *dp;
int value;			/* aspect of rendition to set */
{
/* Set a particular graphics rendition aspect. */

  struct rendition_struct *dpr;

  dpr = &dp->rendition;
  switch (value) {
	case 0:		/* default rendition */
		dp->rendition = dp->default_rendition;
		break;

	case 1:		/* bold */
		dpr->bold = TRUE;
		break;

	case 4:		/* underlined */
		dpr->underlined = TRUE;
		break;

	case 5:		/* blinking */
		dpr->blinking = TRUE;
		break;

	case 7:		/* negative image */
		dpr->negative = TRUE;
		break;

	case 22:	/* not bold */
		dpr->bold = FALSE;
		break;

	case 24:	/* not underlined */
		dpr->underlined = FALSE;
		break;

	case 25:	/* not blinking */
		dpr->blinking = FALSE;
		break;

	case 27:	/* positive image */
		dpr->negative = FALSE;
		break;


	case 39:	/* default foreground color */
		dpr->fg_color = dp->default_rendition.fg_color;
		break;

	case 49:	/* default background color */
		dpr->bg_color = dp->default_rendition.bg_color;
		break;

	default:
		if (value >= 30 && value <= 37) {
			dpr->fg_color = value - 30;
		} else if (value >= 40 && value <= 47) {
			dpr->bg_color = value - 40;
		} /* else invalid, ignore */
		break;
  }
}


/*===========================================================================*
 *				select_attr				     *
 *===========================================================================*/
PRIVATE void select_attr(dp)
register struct display_s *dp;
{
/* Set dp->attr based on dp->rendition, and the display hardware. */

  struct rendition_struct *dpr;
  int blank_color, fg, bg;
  register short *dest;
  short *limit;

  dpr = &dp->rendition;

  /* Set dp->attr, and dp->rendition. */
  fg = color ? ansi_colors[dpr->fg_color] : WHITE;
  bg = color ? ansi_colors[dpr->bg_color] : BLACK;
  if (dpr->negative)
      dp->attr = (fg << 12) | (bg << 8);
  else
      dp->attr = (bg << 12) | (fg << 8);
  /* Bold inverse video often looks bad, because what is normally the
   * background color is made bold, rather than what is normally the
   * foreground color.  Don't do bold inverse video.
   */
  if (dpr->bold & !dpr->negative) dp->attr |= 0x0800;	/* intense */
  blank_color = dp->attr;
  if (dpr->underlined)
	if (color) {
		dp->attr |= 0x0800;	/* intense */
		blank_color = dp->attr;
		/* blank_color is set here so that the cursor will blink in
		 * the intense color.
		 */
	} else {
		dp->attr &= 0xf900;	/* underline */
		/* blank_color is not set here so that new lines are not
		 * underlined.
		 */
	}
  if (dpr->blinking) dp->attr |= 0x8000;	/* blink */

  /* Update blanks buffer if necessary. */
  if (dp->blanks[0] != blank_color) {
	limit = &dp->blanks[BLANK_BUF_SIZE];
	for (dest = &dp->blanks[0]; dest < limit; dest++) {
		*dest = blank_color;
	}
  }
}


/*===========================================================================*
 *				move_to					     *
 *===========================================================================*/
PRIVATE void move_to(dp, x, y)
register struct display_s *dp;
int x;				/* column (0 <= x <= 79) */
int y;				/* row (0 <= y <= 24, 0 at top) */
{
/* Move the cursor to (x, y). */

  if (x < 0 || x >= SCR_COLS || y < 0 || y >= SCR_ROWS) return;
  dp->column = x;		/* set x co-ordinate */
  dp->row = y;			/* set y co-ordinate */
  dp->cursor = y * SCR_COLS + x;

  /* See if it cheaper to copy the buffered characters on the the screen or to
   * extend the buffered region to include the cursor.
   */
  if (dp->cursor < buf_first - flush_cost
      || dp->cursor > buf_last + flush_cost) {
		flush(dp);
  } else {
	if (dp->cursor < buf_first) buf_first = dp->cursor;
	if (dp->cursor > buf_last) buf_last = dp->cursor;
  }
}


/*===========================================================================*
 *				scr_blank				     *
 *===========================================================================*/
PRIVATE void scr_blank(dp, offset, count)
register struct display_s *dp;
int offset, count;
{
/* Blank out the screen starting at character offset and continuing for count
 * characters.
 */

  int remaining, first, ct;
  char *source, *dest;

  /* Blank virtual console. */
  first = dp->org + offset;
  remaining = count;
  while (remaining > 0) {
	if (first >= SCR_CHARS) first -= SCR_CHARS;
	ct = MIN(remaining, SCR_CHARS - first);
	ct = MIN(ct, BLANK_BUF_SIZE);
	source = (char *) &dp->blanks[0];
	dest = (char *) &dp->obuf[first];
	memcpy(dest, source, 2 * ct);
	remaining -= ct;
	first += ct;
  }

  /* See if it is cheaper to copy the blank characters to the screen or to
   * extend the buffered region to include the cursor.
   */
  if (offset + count < buf_first - flush_cost
      || offset > buf_last + flush_cost) {
	flush(dp);
  }
  if (offset < buf_first) buf_first = offset;
  if (offset + count > buf_last) buf_last = offset + count;
}


/*===========================================================================*
 *				scr_copy				     *
 *===========================================================================*/
PRIVATE void scr_copy(dp, from, to, count)
register struct display_s *dp;
int from, to, count;
{
/* Copy characters on the screen from character src to character dst for count
 * characters.
 */

  int src, dst, remaining, ct;
  char *source, *dest;

  src = dp->org + from;
  dst = dp->org + to;
  remaining = count;
  if (dst < src) {
	/* Copy "forwards" - first character first. */
	while (remaining > 0) {
		/* memmove will be performed more than one if characters wrap
		 * around in the cyclic buffer.
		 */
		if (src >= SCR_CHARS) src -= SCR_CHARS;
		if (dst >= SCR_CHARS) dst -= SCR_CHARS;
		ct = MIN(SCR_CHARS - src, SCR_CHARS - dst);
		ct = MIN(ct, remaining);
		source = (char *) &dp->obuf[src];
		dest = (char *) &dp->obuf[dst];
		memmove(dest, source, 2 * ct);
		remaining -= ct;
		src += ct;
		dst += ct;
	  }
  } else {
	/* Copy "backwards" - last character first. */
	src += remaining;
	dst += remaining;
	if (src > SCR_CHARS) src -= SCR_CHARS;
	if (dst > SCR_CHARS) dst -= SCR_CHARS;
	while (remaining > 0) {
		/* memmove will be performed more than one if characters wrap
		 * around in the cyclic buffer.
		 */
		if (src <= 0) src += SCR_CHARS;
		if (dst <= 0) dst += SCR_CHARS;
		ct = MIN(src, dst);
		ct = MIN(ct, remaining);
		source = (char *) &dp->obuf[src - ct];
		dest = (char *) &dp->obuf[dst - ct];
		memmove(dest, source, 2 * ct);
		remaining -= ct;
		src -= ct;
		dst -= ct;
	}
  }

  /* See if it is cheaper to copy the new characters to the screen or to extend
   * the buffered region to include the cursor.
   */
  if (to + count < buf_first - flush_cost || to > buf_last + flush_cost) {
	flush(dp);
  }
  if (to < buf_first) buf_first = to;
  if (to + count > buf_last) buf_last = to + count;
}


/*===========================================================================*
 *				scroll_screen				     *
 *===========================================================================*/
PRIVATE void scroll_screen(dp, dir)
register struct display_s *dp;
int dir;			/* GO_FORWARD or GO_BACKWARD */
{
/* Scroll the screen forwards or backwards a line. */

  int offset;

  /* Scroll the virtual screen. */
  if (dir == GO_FORWARD) {
	dp->org += SCR_COLS;
	if (dp->org >= SCR_CHARS) dp->org -= SCR_CHARS;
	buf_first = MAX(buf_first - SCR_COLS, 0);
	buf_last = MAX(buf_last - SCR_COLS, 0);
  } else {
	dp->org -= SCR_COLS;
	if (dp->org < 0) dp->org += SCR_CHARS;
	buf_first = MIN(buf_first + SCR_COLS, SCR_CHARS);
	buf_last = MIN(buf_last + SCR_COLS, SCR_CHARS);
  }

  if (dp->display) {
	/* Scrolling the screen is a real nuisance due to the various
	 * incompatible video cards.  This driver supports hardware scrolling
	 * (MDA, HDA, CGA, and most EGA cards) and software scrolling
	 * (incompatible EGA cards).
	 */

	if (dp->softscroll) {
		/* Software scrolling for non-IBM compatible display cards.
		 * Have to update the entire video RAM from the virtual
		 * console buffer.
		 */
		buf_first = 0;
		buf_last = SCR_CHARS;

	} else if (ega) {
		/* Use video origin, but don't assume the hardware can wrap. */
		if (dir == GO_FORWARD) {
			new_org += SCR_COLS;
			if (new_org + SCR_CHARS >= vid_size) {
				new_org = 0;
				/* Force entire screen to be updated. */
				buf_first = 0;
				buf_last = SCR_CHARS;
			}
		} else {		/* scroll backwards */
			new_org -= SCR_COLS;
			if (new_org < 0) {
				new_org = vid_size - SCR_CHARS;
				/* Force entire screen to be updated. */
				buf_first = 0;
				buf_last = SCR_CHARS;
			}
		}

	} else {
		/* Normal scrolling using the 6845 registers. */
		if (dir == GO_FORWARD) {
			new_org += SCR_COLS;
			if (new_org >= vid_size) new_org -= vid_size;
		} else {
			new_org -= SCR_COLS;
			if (new_org < 0) new_org += vid_size;
		}
	}
  }

  /* See if it is cheaper to copy the buffered characters to the screen or to
   * extend the buffered region to include the cursor.
   */
  if (dp->cursor < buf_first - flush_cost
      || dp->cursor > buf_last + flush_cost) {
		flush(dp);
  } else {
	if (dp->cursor < buf_first) buf_first = dp->cursor;
	if (dp->cursor > buf_last) buf_last = dp->cursor;
  }

  /* Blank the new line at top or bottom. */
  offset = (dir == GO_FORWARD ? SCR_CHARS - SCR_COLS : 0);
  scr_blank(dp, offset, SCR_COLS);
}


/*===========================================================================*
 *				flush					     *
 *===========================================================================*/
PRIVATE int old_org;

PRIVATE void flush(dp)
register struct display_s *dp;
{
/* Have the characters that have been buffered in the virtual console screen
 * image transfered to the physical screen.
 */

  int first, count;

  if (dp->display) {
	snow = dp->snow;

	/* Break up a call to vid_copy for machines that can only write during
	 * vertical retrace.  Writing during the normal scan period would
	 * cause snow to appear on the screen. Vid_copy itself does the wait.
	 * Even if the machine does not snow the copy will involve two cycles
	 * through the loop if the buffered characters wrap around in the
	 * cyclic buffer.
	 */
	while (buf_first != buf_last) {
		first = dp->org + buf_first;
		if (first >= SCR_CHARS) first -= SCR_CHARS;
		count = MIN(buf_last - buf_first, SCR_CHARS - first);
		if (snow) count = MIN(count, CGA_RETRACE);
		vid_copy((char *) (&dp->obuf[first]), vid_base,
			 2 * (new_org + buf_first), count);
		buf_first += count;
	}

	/* Update vid_org as quickly as possible on a monochrome display to
	 * try and minimize flicker caused by characters that have been added
	 * outside of the screen area projecting into the current screen
	 * image.  The additional screen memory that is present means that
	 * this is unlikely to happen on a color display.
	 */
	if (!color && vid_org != new_org) {
		vid_org = new_org;
		set_6845(VID_ORG, vid_org);
	}
  }

  buf_first = buf_last = dp->cursor;
}


/*===========================================================================*
 *				set_softscroll				     *
 *===========================================================================*/
PRIVATE void set_softscroll(dp, soft_on)
register struct display_s *dp;
int soft_on;
{
/* Turn softscrolling on or off. */

  dp->softscroll = soft_on;

  if (dp->display & soft_on) {
	new_org = vid_org = 0;
	set_6845(VID_ORG, vid_org);

	/* Transfer the virtual console image onto the screen. */
	buf_first = 0;
	buf_last = SCR_CHARS;
	flush(dp);
	set_cursor(dp);
  }
}


/*===========================================================================*
 *				set_cursor				     *
 *===========================================================================*/
PRIVATE void set_cursor(dp)
register struct display_s *dp;
{
/* Position the hardware cursor on the physical screen. */

  if (dp->cursor_visible && dp->column < SCR_COLS) {
	set_6845(CURSOR, vid_org + dp->cursor);	/* position cursor */
  } else {
	/* Hide cursor.  It would be best to simply turn the cursor off, but
	 * some video cards appear to be unable to do this.  Instead the cursor
	 * is positioned outside of the screen area.  Scrolling the screen can
	 * cause the cursor to become momentarily visible.
	 */
	set_6845(CURSOR, vid_org + SCR_CHARS);
  }
}


/*===========================================================================*
 *				set_6845				     *
 *===========================================================================*/
PRIVATE int old_high_org;

PRIVATE void set_6845(reg, val)
int reg;			/* which register pair to set */
int val;			/* 16-bit value to set it to */
{
/* Set a register pair inside the 6845.
 *
 * Registers 10-11 control the format of the cursor (how high it is, etc).
 * Registers 12-13 tell the 6845 where in video ram to start (in WORDS)
 * Registers 14-15 tell the 6845 where to put the cursor (in WORDS)
 *
 * Note that registers 12-15 work in words, i.e. 0x0000 is the top left
 * character, but 0x0001 (not 0x0002) is the next character.  This addressing
 * is different from the way the 8088 addresses the video ram, where 0x0002 is
 * the address of the next character.
 */

  int new_high_org;

  /* Try and avoid an annoying screen glitch caused by the 6845 loading an
   * in-between value for the video origin.
   */
  if (reg == VID_ORG) {
	/* Try to not load the register while a horizontal retrace is occuring.
	 * The origin usually changes in steps of SCR_COLUMNS, so it is not
	 * worth checking for an unchanged low byte.
	 */

	new_high_org = val & (BYTE << 8);
	if (new_high_org != old_high_org) {
		old_high_org = new_high_org;
		while (in_byte(vid_port + STATUS) & 0x01)
			;	/* wait till not in horizontal retrace */
	}
  }
  lock();

  out_byte(vid_port + INDEX, reg);	/* set the index register */
  out_byte(vid_port + DATA, (val >> 8) & BYTE);	/* output high byte */
  out_byte(vid_port + INDEX, reg + 1);	/* again */
  out_byte(vid_port + DATA, val & BYTE);	/* output low byte */

  unlock();
}


/*===========================================================================*
 *				beep					     *
 *===========================================================================*/
PRIVATE int beeping = FALSE;

PRIVATE void beep()
{
/* Making a beeping sound on the speaker (output for CRTL-G).
 *
 * This routine works by turning on the bits 0 and 1 in port B of the 8255
 * chip that drive the speaker.
 */

  message mess;

  if (beeping) return;
  out_byte(TIMER_MODE, 0xB6);	/* set up channel 2 (square wave) */
  out_byte(TIMER2, BEEP_FREQ & BYTE);	/* load frequency low-order */
  out_byte(TIMER2, (BEEP_FREQ >> 8) & BYTE);	/* high-order bits */
  lock();			/* guard PORT_B from keyboard intr handler */
  out_byte(PORT_B, in_byte(PORT_B) | 3);	/* turn on beep bits */
  unlock();
  beeping = TRUE;

  mess.m_type = SET_ALARM;
  mess.CLOCK_PROC_NR = TTY;
  mess.DELTA_TICKS = B_TIME;
  mess.FUNC_TO_CALL = (void (*)()) stop_beep;
  sendrec(CLOCK, &mess);
}


/*===========================================================================*
 *				stop_beep				     *
 *===========================================================================*/
PRIVATE void stop_beep()
{
/* Turn off the beeper by turning off bits 0 and 1 in PORT_B. */

  lock();			/* guard PORT_B from keyboard intr handler */
  out_byte(PORT_B, in_byte(PORT_B) & ~3);
  beeping = FALSE;
  unlock();
}


/*===========================================================================*
 *				time_flush				     *
 *===========================================================================*/
PRIVATE long time_flush(dp, chars)
register struct display_s *dp;
int chars;
{
/* Measure the time taken to flush chars characters to the screen.  Setting
 * chars to zero can be used to measure the time consumed by the timing loop.
 * The returned time is in arbitrary time units.
 */

  unsigned initial_count, final_count;
  long elapsed, count;

  lock ();	/* Disable interrupts while performing measurements. */

  /* Since the clock task has yet to be initialized the counter is currently
   * counting down from some unknown value.  If this value was too small it
   * would not be possible to perform any timing measurements.  Thus the
   * counter needs to be initialized prior to starting the tty task.  This has
   * not been done here because it would require changes to clock.c.  Instead
   * the counter is temporarily initialized here, and clock.c initializes it
   * properly later on.
   */
  out_byte(TIMER_MODE, 0x36);
  out_byte(TIMER0, 255);
  out_byte(TIMER0, 255);

  /* We accumulate results until roughly 100ms has passed.  Such a long
   * measurement time is necessary because a graphics card may appear to be
   * faster than it really is if we happen to only measure it while it was
   * performing a vertical retrace.  The counter runs at roughly 1MHz, thus
   * 100ms is equivelent to around 100000 counts.  For convenience measurements
   * that wrap are discarded.
   */
  elapsed = 0;
  count = 0;
  while (elapsed < 100000) {
	if (chars == 0) {
		initial_count = read_counter();
		final_count = read_counter();
	} else {
		buf_first = 0; buf_last = chars;
		initial_count = read_counter();
		flush(dp);
		final_count = read_counter();
	}
	if (final_count <= initial_count) {
		elapsed += initial_count - final_count;
		count++;
	}
  }

  unlock();	/* Re-enable interrupts. */

  /* Returned time is scaled up to make integer calculations accurate. */
  return elapsed * 1000 / count;
}


/*===========================================================================*
 *				set_flush_cost				     *
 *===========================================================================*/
PRIVATE void set_flush_cost(dp)
register struct display_s *dp;
{
/* Set the flush cost based on the speed of the video hardware. */

  long loop_time, time_1, time_100;
  long per_char_time;

  loop_time = time_flush(dp, 0);	/* empty loop time */
  time_1 = time_flush(dp, 1) - loop_time;	/* time to flush 1 char. */
  time_100 = time_flush(dp, 100) - loop_time;	/* time to flush 100 chars. */

  per_char_time = (time_100 - time_1) / (100 - 1);
  if (per_char_time == 0) per_char_time = 1;	/* for safety */
  /* Tests to see if flush should be called only call flush if the number of
   * characters exceeds flush_cost, hence in computing flush_cost the precise
   * value should be rounded down to the nearest integer.
   */
  flush_cost = (time_1 - per_char_time) / per_char_time;
  /* Check that value is sensible. */
  if (flush_cost < 0 || flush_cost > 100) flush_cost = DEF_FLUSH_COST;
}


/*===========================================================================*
 *				scr_init				     *
 *===========================================================================*/
PUBLIC void scr_init(num)
int num;
{
/* Initialize the display driver for the specified device.  This function must
 * be called in order for each of the console devices present.
 */

  register struct display_s *dp;
  int scr;

  dp = screen_addr(num);

  /* Initialize screen parameters. */
  dp->display = (num == fg_console);
  dp->linewrap = LINEWRAP;
  dp->softscroll = ps;		/* soft scrolling by default for PS/2 */
  dp->cursor_visible = TRUE;
  dp->rendition.fg_color = dp->default_rendition.fg_color = WHITE;
  dp->rendition.bg_color = dp->default_rendition.bg_color = BLACK;
  select_attr(dp);

  /* Set up hardware and other shared stuff if this is the first call. */
  if (num == 0) {
	/* Set initial values. */
	fg_console = 0;

	/* Tell the EGA card, if any, to simulate a 16K CGA card. */
	out_byte(EGA + INDEX, 4);	/* register select */
	out_byte(EGA + DATA, 1);/* don't use extra memory */

	/* Initialize hardware descriptor values. */
	if (color) {
		vid_base = protected_mode ? COLOR_SELECTOR
			: physb_to_hclick(COLOR_BASE);
		vid_size = C_VID_SIZE;
		vid_port = C_6845;
	} else {
		vid_base = protected_mode ? MONO_SELECTOR
			: physb_to_hclick(MONO_BASE);
		vid_size = (ega ? C_VID_SIZE : M_VID_SIZE);
		vid_port = M_6845;
	}
	vid_mask = 2 * vid_size - 1;

	/* Hardware setup. */
	set_6845(VID_ORG, vid_org);	/* use page 0 of video ram */
	set_6845(CUR_SIZE, CURSOR_SHAPE);	/* set cursor shape */
	set_cursor(dp);

	flush_cost = DEF_FLUSH_COST;	/* until calculated below */
  }

  /* Initialize screen contents.  Hardware has been set up. */
  scr_blank(dp, 0, SCR_CHARS);		/* clear virtual screen */
  flush(dp);				/* update physical screen */

  /* Set flush cost if this is the first call.  Screen has been cleared. */
  if (num == 0) set_flush_cost(dp);
}


/*===========================================================================*
 *				swap_screen				     *
 *===========================================================================*/
PUBLIC void swap_screen(old_console)
int old_console;		/* previous console */
{
/* Change foreground screen to the value given by global variable fg_console.
 * This routine is invoked by the keyboard driver.
 */

  register struct display_s *dp;

  /* Get rid of old screen. */
  dp = screen_addr(old_console);
  dp->display = FALSE;

  /* Load new screen */
  dp = screen_addr(fg_console);
  dp->display = TRUE;
  new_org = vid_org;

  /* Transfer the virtual console image onto the screen. */
  buf_first = 0;
  buf_last = SCR_CHARS;
  flush(dp);
  set_cursor(dp);
}


/*===========================================================================*
 *				toggle_scroll				     *
 *===========================================================================*/
PUBLIC void toggle_scroll()
{
/* Toggle between hardware and software scroll.  This routine is invoked by
 * the keyboard driver.
 */

  register struct display_s *dp;

  dp = screen_addr(fg_console);
  set_softscroll(dp, 1 - dp->softscroll);
}
