/*
 * Command: 	bi - extract loadable kernel image from the 'Boot Diskette'
 * Syntax:	bi [-v] [device [file]]
 * Flags:	-v print size of the core image
 * Example:	bi /dev/fd0 minix.sys	# extruct core image from drive A
 *  'bi' extracts core image of the Minix kernel from the 'boot diskette',
 * and write it into the specified file.  The default input device is the
 * "/dev/fd0" and the default output file is stdout.  This program is 
 * usefull for using the 'MXboot' system.
 *
 * Written by S.Hirabayashi.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#define DFLDRV	"/dev/fd0"
#define DFLIMG	"stdout"
#define SCTSIZ	512	/* sector size */
#define BUFSIZE SCTSIZ * 32	/* this must be positive integer */

#define OFFSET	504	/* see tools/build.c */
#define IDBYTE	510
#define IPL_ID	0xaa55

int ifd, ofd;
char *dev;		/* input FD device */
char *file;		/* output file */
int vflg;
char *cmd;
char buf[BUFSIZE];

main(argc, argv) char **argv;
{
  char *s;
  int i, j, n;

  cmd = argv[0];
  while (--argc > 0 && (*++argv)[0] == '-')
	for (s = argv[0]+1; *s; s++)
		switch (*s) {
		case 'v':
			vflg++;
			break;
		default:
			usage();
		}

  if (sizeof(buf) % SCTSIZ)
	error("sizeof(buf) % SCTSIZ is not equal 0", (char *)0);

  dev = (argc > 0) ? argv[0] : DFLDRV;
  file = (argc > 1) ? argv[1] : DFLIMG;

  if ((ifd = open(dev, O_RDONLY)) == -1)
	error("can't open %s", dev);

  if (read(ifd, buf, SCTSIZ) != SCTSIZ)
	error("disk read error on %s", dev);

  s = buf + OFFSET;
  if ((unsigned short *)(buf + IDBYTE) == IPL_ID) s -= 2;
  n = *s & 0xff | ((*(s+1) & 0xff) << 8) + 1;
  if (vflg)
	fprintf(stderr, "Total image size: %u Blocks, %ld Bytes\n",
		n, ((unsigned long)n * SCTSIZ));

  if (argc > 1) {
	if ((ofd = open(file, O_WRONLY|O_TRUNC|O_CREAT, 0666)) == -1)
		error("cannot open %s", *argv);
  }
  else
	ofd = 1;

  if (write(ofd, buf, SCTSIZ) == -1)
	error("write error on %s", file);

  while (n) {
	i = (n > (sizeof(buf) / SCTSIZ)) ? sizeof(buf) : n * SCTSIZ;
	if ((j = read(ifd, buf, i)) != i)
		error("disk read error on %s", dev);
	if (write(ofd, buf, j) != j)
		error("write error on %s", file);
	n -= j / SCTSIZ;
  }
  exit(0);
}

usage()
{
  fprintf(stderr, "Usage: %s [-v] [device [file]]\n", cmd);
  exit(1);
}

error(s, t) char *s, *t;
{
  fprintf(stderr, "%s: ", cmd);
  fprintf(stderr, s, t);
  fprintf(stderr, "\n");
  exit(1);
}
