/* 
 * This file contains an experimental disk and tape driver for the
 * Adaptec 154x SCSI Host Adapter family, written by James da Silva.
 *
 * This is version 1.0.  Please send comments, criticisms and bug fixes
 * to me via e-mail at jds@cs.umd.edu, or on my BBS at 1-301-277-9408.
 *
 * I wrote this driver using the technical documentation for the AHA available
 * from the Adaptec BBS at 1-408-945-7727, and from the SCSI standard drafts
 * available on NCR's SCSI BBS at 1-316-636-8700.  I suggest you get both
 * these documents if you want to understand and hack this code.
 *
 * The driver supports the following operations (using message format m2):
 *
 *    m_type      DEVICE    PROC_NR    COUNT    POSITION  ADDRESS
 * ----------------------------------------------------------------
 * |  DISK_READ | device  | proc nr |  bytes  |  offset | buf ptr |
 * |------------+---------+---------+---------+---------+---------|
 * | DISK_WRITE | device  | proc nr |  bytes  |  offset | buf ptr |
 * ----------------------------------------------------------------
 * |SCATTERED_IO| device  | proc nr | requests|         | iov ptr |
 * ----------------------------------------------------------------
 *
 * The supported device numbers are as follows:
 *   #  Name    	Device
 *   0  sd0,rsd0	first disk, entire disk
 *   1  sd1,rsd1	first disk, partition 1
 *   2  sd2,rsd2	first disk, partition 2
 *   3  sd3,rsd3	first disk, partition 3
 *   4  sd4,rsd4	first disk, partition 4
 *   5  sd5,rsd5	second disk, entire disk
 *   6  sd6,rsd6	second disk, partition 1
 *   7  sd7,rsd7	second disk, partition 2
 *   8  sd8,rsd8	second disk, partition 3
 *   9  sd9,rsd9	second disk, partition 4
 *  10+ repeat above to taste
 *
 * 128  nrst0		first tape, no rewind
 * 129  rst0		first tape, rewind
 * 130  nrst1		second tape, no rewind
 * 131  rst1		second tape, rewind
 * 132+ repeat above to taste
 *
 * The file contains one entry point:
 *
 *   scsi_task:		main entry when system is brought up
 *
 */
#include "kernel.h"
#include <minix/callnr.h>
#include <minix/com.h>
#include <minix/partition.h>
#include <fcntl.h>
#include <sys/mtio.h>

#define TAPE_SUPPORT	1	/* 1=include tape support, 0=don't */
#define DEBUG		0	/* 1=print errmsg, 2=dump ccb, 3=trace all */
#define IMMED_BIT	1
#define ARCHIVE_2525S	0

#define aha_diskdev	diskdev_aha
#define aha_disk	disk_aha
#define aha_reset	reset_aha
#define aha_result	result_aha
#define scsi_init	init_scsi
#define scsi_inquiry	inquiry_scsi
#define scsi_tpio	tpio_scsi
#define scsi_tpopen	tpopen_scsi
#define scsi_tpclose	tpclose_scsi
#define scsi_tpioctl	tpioctl_scsi

/* AHA-154x port addresses */
#define AHA_BASEREG	0x330	/* Base port address of AHA registers */
#define AHA_CNTLREG	AHA_BASEREG+0	/* Control Register - write only */
#define AHA_STATREG	AHA_BASEREG+0	/* Status Register - read only */
#define AHA_DATAREG	AHA_BASEREG+1	/* Data Register - read/write */
#define AHA_INTRREG	AHA_BASEREG+2	/* Interrupt Flags - read only */

/* control register bits */
#define AHA_HRST	0x80	/* bit 7 - Hard Reset */
#define AHA_SRST	0x40	/* bit 6 - Soft Reset */
#define AHA_IRST	0x20	/* bit 5 - Interrupt Reset */
#define AHA_SCRST	0x10	/* bit 4 - SCSI Bus Reset */
/*			0x08	/* bit 3 - Reserved (set to 0) */
/*			0x04	/* bit 2 - Reserved (set to 0) */
/*			0x02	/* bit 1 - Reserved (set to 0) */
/*			0x01	/* bit 0 - Reserved (set to 0) */

/* status register bits */
#define AHA_STST	0x80	/* bit 7 - Self Test in Progress */
#define AHA_DIAGF	0x40	/* bit 6 - Internal Diagnostic Failure */
#define AHA_INIT	0x20	/* bit 5 - Mailbox Initialization Required */
#define AHA_IDLE	0x10	/* bit 4 - SCSI Host Adapter Idle */
#define AHA_CDF		0x08	/* bit 3 - Command/Data Out Port Full */
#define AHA_DF		0x04	/* bit 2 - Data In Port Full */
/*			0x02	/* bit 1 - Reserved */
#define AHA_INVDCMD	0x01	/* bit 0 - Invalid Host Adapter Command */

/* interrupt flags register bits */
#define AHA_ANYINT	0x80	/* bit 7 - Any Interrupt */
/*			0x40	/* bit 6 - Reserved */
/*			0x20	/* bit 5 - Reserved */
/*			0x10	/* bit 4 - Reserved */
#define AHA_SCRD	0x08	/* bit 3 - SCSI Reset Detected */
#define AHA_HACC	0x04	/* bit 2 - Host Adapter Command Compete */
#define AHA_MBOE	0x02	/* bit 1 - Mailbox Out Empty */
#define AHA_MBIF	0x01	/* bit 0 - Mailbox In Full */

/* AHA Command Codes */
#define AHACOM_INITBOX	 0x01	/* Mailbox Initialization */
#define AHACOM_STARTSCSI 0x02	/* Start SCSI Command */
#define AHACOM_HAINQUIRY 0x04	/* Host Adapter Inquiry */
#define AHACOM_SETIMEOUT 0x06	/* Set SCSI selection time out value */
#define AHACOM_INSTALLED 0x0A	/* Return Installed Devices */
#define AHACOM_GETCONFIG 0x0B	/* Return Configuration Data */
#define AHACOM_GETSETUP  0x0D	/* Return Setup Data */

/* AHA Mailbox Out Codes */
#define AHA_MBOXFREE	0x00	/* Mailbox is Free */
#define AHA_MBOXSTART	0x01	/* Start Command */
#define AHA_MBOXABORT	0x02	/* Abort Command */
/* AHA Mailbox In Codes */
#define AHA_MBOXOK	0x01	/* Command Completed Successfully */
#define AHA_MBOXERR	0x04	/* Command Completed with Error */


/* Basic types */
typedef unsigned char byte;
typedef struct { byte v[3]; } int24; /* AHA uses 24 bit, big-endian values! */

/* AHA Mailbox structure */
typedef struct {
byte status;		/* Command or Status byte */
int24 ccbptr;		/* pointer to Command Control Block */
} mailbox_t;

/* SCSI Group 0 Command Descriptor Block structure */
typedef struct {
    byte opcode;		/* SCSI Operation Code */
#	define SCSI_UNITRDY  0x00	/* Test Unit Ready */	
#	define SCSI_REWIND   0x01	/* Rewind */	
#	define SCSI_RDLIMITS 0x05	/* Read Block Limits Opcode */
#	define SCSI_READ     0x08	/* Group 0 Read Opcode */
#	define SCSI_WRITE    0x0A	/* Group 0 Write Opcode */
#	define SCSI_WREOF    0x10	/* Write File Marks */	
#	define SCSI_SPACE    0x11	/* Space over filemarks/blocks */
#	define SCSI_INQUIRY  0x12	/* Group 0 Inquiry Opcode */
#	define SCSI_ERASE    0x19	/* Group 0 Erase Opcode */
#	define SCSI_MDSENSE  0x1A	/* Group 0 Mode Sense Opcode */
#	define SCSI_LOAD     0x1B	/* Group 0 LOAD/UNLOAD */
    union {
	struct {	/* disk i/o commands */
            int24 d_lba;	/* LUN and logical block address */
	    byte d_nblocks;	/* transfer size in blocks */
	} d;
	struct {	/* tape i/o commands */
	    byte t_fixed;	/* fixed length? */
	    int24 t_trlength;	/* transfer length */
	} t;
    } u;
#   define lba		u.d.d_lba	
#   define nblocks	u.d.d_nblocks
#   define fixed	u.t.t_fixed
#   define trlength	u.t.t_trlength
    byte control;		/* reserved and link bit fields, set to 0 */
} cdb_t;

/* SCSI Request Sense Information */
typedef struct {
    byte errc;			/* Error Code, Error Class, and Valid bit */
    byte segnum;		/* Segment Number */
    byte key;			/* Sense Key */
#	define sense_key(key)	(key & 0x1F)	/* the key portion */
#	define sense_ili(key)	(key & 0x20)	/* ILI? */
#	define sense_eom(key)	(key & 0x40)	/* EOM? */
#	define sense_eof(key)	(key & 0x80)	/* EOF? */
    byte info[4];		/* sense info, big endian */
    byte len;			/* additional length */
    byte extra[6];		/* additional sense bytes */
} sense_t;

/* SCSI Inquiry Information */
typedef struct {
    byte devtype;		/* Peripheral Device Type */
#	define SCSI_DEVDISK	0	/* Direct-access */
#	define SCSI_DEVTAPE	1  	/* Sequential-access */
#	define SCSI_DEVPRN	2	/* Printer */
#	define SCSI_DEVCPU	3	/* Processor */
#	define SCSI_DEVWORM	4	/* Write-Once Read-Multiple device */
#	define SCSI_DEVCDROM	5	/* Read-Only Direct-access */
#	define SCSI_DEVMAX	5	/* last device type we know about */
    byte devqual;		/* Device-Type Qualifier */
#	define scsi_rmb(d) ((d) & 0x80) /* Removable? */
    byte stdver;		/* version of standard compliance */
#	define scsi_isover(v)   ((v) & 0xC0) >> 6)	/* ISO version */
#	define scsi_ecmaver(v)  ((v) & 0x38) >> 3)	/* ECMA version */
#	define scsi_ansiver(v)  ((v) & 0x07))		/* ANSI version */
    byte reserved;
    byte len;			/* Length of remaining info */
    byte extra[64];		/* some room for vendor unique data */
} inquiry_t;

/* AHA Command Control Block structure */
typedef struct {
    byte opcode;		/* Operation Code */
    byte addrcntl;		/* Address and Direction Control: */
#       define ccb_scid(id)     (((id)<<5)&0xE0) /* SCSI ID field */
#	define CCB_OUTCHECK	0x10		 /* outbound length check */
#	define CCB_INCHECK	0x08		 /* inbound length check */
#	define CCB_NOCHECK	0x00		 /* no length check */
#	define ccb_lun(lun)     ((lun)&0x07)	 /* SCSI LUN field */
    byte cmdlen;		/* SCSI Command Length (6 for Group 0) */
    byte senselen;		/* Request/Disable Sense, Allocation Length */
#	define CCB_SENSEREQ	0x0E		/* Request Sense, 14 bytes */
#	define CCB_SENSEOFF	0x01		/* Disable Request Sense */
    int24 datalen;		/* Data Length:  3 bytes, big endian */
    int24 dataptr;		/* Data Pointer: 3 bytes, big endian */
    int24 linkptr;		/* Link Pointer: 3 bytes, big endian */
    byte linkid;		/* Command Linking Identifier */
    byte hastat;		/* Host Adapter Status */
    byte tarstat;		/* Target Device Status */
    byte reserved[2];		/* Reserved, set to 0 */
    cdb_t cmd;			/* SCSI Command Descriptor Block */
    sense_t sense;		/* SCSI Request Sense Information */
} ccb_t;

/* Miscellaneous parameters */
#define SECTOR_SIZE      512	/* physical sector size in bytes */
#define SCSI_TIMEOUT	 250	/* SCSI selection timeout (ms), 0 = none */
#define AHA_TIMEOUT   200000	/* max loops waiting (calib. for 486@25Mhz) */ 
#define MAX_DISKS          5	/* this can be increased if needed */
#define MAX_TAPES          2	/* this can be increased if needed */

#define SECTORS_PER_MEG  (1024L*1024L/SECTOR_SIZE)
#define NR_DISKDEV      (MAX_DISKS * (1 + NR_PARTITIONS))

/* Variables */

PRIVATE struct {	/* per-partition table */
  long first;			/* first sector */
  long size;			/* size of partition in sectors */
} aha_diskdev[NR_DISKDEV];

PRIVATE struct {	/* per-disk table */
  int target;			/* SCSI Target ID */
  int lun;			/* SCSI Logical Unit Number */
  /* !! should include capacity (and geometry?) info */
} aha_disk[MAX_DISKS];
PRIVATE int nr_disks;		/* Number of disk drives actually detected */

#if TAPE_SUPPORT
PRIVATE struct {	/* per-tape table */
  int target;			/* SCSI Target ID */
  int lun;			/* SCSI Logical Unit Number */
  long blocksize;		/* current block size */
  int in_use;			/* is the tape drive in use? */ 
  int open_mode;		/* open for reading or writing? */ 
  int at_eof;			/* got EOF mark */
} aha_tape[MAX_TAPES];
PRIVATE int nr_tapes;		/* Number of tape drives actually detected */
#endif	/* TAPE_SUPPORT */

/* SCSI device types */
PRIVATE char *scsi_devstr[SCSI_DEVMAX+1] = {
  "DISK", "TAPE", "PRINTER", "CPU", "WORM", "CDROM" 
};

#if DEBUG
/* SCSI sense key types */
#define SCSI_MAXSENSE		(sizeof(scsi_sense)/sizeof(char *) - 1)
PRIVATE char *scsi_sense[] = {
  "NO SENSE INFO", "RECOVERED ERROR", "NOT READY", "MEDIUM ERROR",
  "HARDWARE ERROR", "ILLEGAL REQUEST", "UNIT ATTENTION", "DATA PROTECT",
  "BLANK CHECK", "VENDOR UNIQUE ERROR", "COPY ABORTED", "ABORTED COMMAND",
  "EQUAL", "VOLUME OVERFLOW", "MISCOMPARE"
};
#endif

PRIVATE message sd_mess;	/* message buffer for in and out */
PRIVATE int aha_result;		/* operation result code */
PRIVATE int aha_present;	/* Is the controller there? */ 
PRIVATE byte installed[8];	/* bitmap of installed targets/units */
PRIVATE mailbox_t mailbox[2];	/* out and in boxes */
PRIVATE ccb_t ccb;		/* our command control block */
PRIVATE inquiry_t inqdata;	/* results of Inquiry command */
PRIVATE byte buf[BLOCK_SIZE];   /* Buffer used by the startup routine */


/* Functions */

FORWARD void scsi_task();		/* entry point */
FORWARD void scsi_init();		/* initialize, detect devices */
FORWARD int scsi_inquiry();		/* send inquiry for init */

FORWARD void scsi_ndisk();		/* disk-specific functions */
FORWARD int scsi_dkio();

#if TAPE_SUPPORT
FORWARD void scsi_ntape();		/* tape-specific functions */
FORWARD int scsi_tpio();
FORWARD int scsi_tpopen();
FORWARD int scsi_tpclose();
FORWARD int scsi_tpioctl();
#endif

FORWARD int scsi_command();		/* AHA-specific functions */
FORWARD void aha_command();
FORWARD int aha_reset();
FORWARD void aha_wait();

#define long2aha(l) (*long2ahap(l))	/* kludge for bcc - no struct return */
FORWARD int24 *long2ahap();
#define ptr2aha(p) (*ptr2ahap(p))	/* kludge for bcc - no struct return */
FORWARD int24 *ptr2ahap();

#if DEBUG
FORWARD void errorpr();
#else
#define errorpr(msg,adr,str)
#endif

#if DEBUG > 1 
FORWARD void errordump();
#else
#define errordump() 
#endif


/*===========================================================================*
 *				scsi_task				     *
 *===========================================================================*/
PUBLIC void scsi_task()
{
/* Main program of the Adaptec SCSI disk/tape driver task. */

  int r, caller, proc_nr;

  /* First initialize the controller and find the devices */
  scsi_init();

  /* Here is the main loop of the SCSI task.  It waits for a message, carries
   * it out, and sends a reply.
   */

  while (TRUE) {
	/* get a request to do some work */
	receive(ANY, &sd_mess);
	if (sd_mess.m_source < 0) {	/* !! why not? */
		printf("sc0: got stray message from %d ", sd_mess.m_source);
		continue;
	}
	caller = sd_mess.m_source;
	proc_nr = sd_mess.PROC_NR;

	if(!aha_present) { 
		/* no AHA controller installed */
		r = EIO;
	} 
#if TAPE_SUPPORT
	else if(sd_mess.DEVICE & 0x80) {
		/* handle tape requests */
		switch(sd_mess.m_type) {
		    case DISK_READ:
		    case DISK_WRITE:
			r = scsi_tpio(&sd_mess); break;
		    case SCATTERED_IO:
			r = do_vrdwt(&sd_mess, scsi_tpio); break;
		    case TTY_OPEN:
			r = scsi_tpopen(&sd_mess); break;
		    case TTY_CLOSE:
			r = scsi_tpclose(&sd_mess); break;
		    case TTY_IOCTL:
			r = scsi_tpioctl(&sd_mess); break;
		    default:
			r = EIO; 
			printf("st?: bad msg type %d\n", sd_mess.m_type);
		}
	}
#endif
	else {
		/* handle disk requests */ 

		switch(sd_mess.m_type) {
		    case DISK_READ:
		    case DISK_WRITE:
			r = scsi_dkio(&sd_mess); break;
		    case SCATTERED_IO:
			r = do_vrdwt(&sd_mess, scsi_dkio); 
			break;
		    case TTY_OPEN:
		    case TTY_CLOSE:
			r = OK;
			break;
		    case TTY_IOCTL:
			r = ENOTTY;
			break;		    
		    default:
			r = EIO;
			printf("sd?: bad request type %d\n", sd_mess.m_type);
		}
	}

	/* Finally, prepare and send the reply message. */
	sd_mess.m_type = TASK_REPLY;	
	sd_mess.REP_PROC_NR = proc_nr;

	sd_mess.REP_STATUS = r;	/* # of bytes transferred or error code */
	send(caller, &sd_mess);	/* send reply to caller */
  }
}


/*===========================================================================*
 *				scsi_init				     *
 *===========================================================================*/
PRIVATE void scsi_init()
{
  byte cmd[1], haidata[4], getcdata[4];
  int targ, lun;


  aha_present = aha_reset();
  if(!aha_present) return; /* no controller, forget it */

  /* get information about controller type and configuration */

  cmd[0] = AHACOM_HAINQUIRY;
  aha_command(1, cmd, 4, haidata);
  if(haidata[0] == 0) haidata[0] = 'O';

  cmd[0] = AHACOM_GETCONFIG;
  aha_command(1, cmd, 3, getcdata);

#if DEBUG
  printf("sc0: ID %d Adaptec 154x ID %c OPT %c REV %c.%c\n",
	getcdata[2] & 0x07, haidata[0], haidata[1], haidata[2], haidata[3]);
#endif

  /* find all installed SCSI devices */
    
  cmd[0] = AHACOM_INSTALLED;
  aha_command(1, cmd, 8, installed);

  nr_disks = 0;
#if TAPE_SUPPORT
  nr_tapes = 0;
#endif

  for(targ=0; targ < 8; targ++) for(lun=0; lun < 8; lun++) {
	if(installed[targ] & (1 << lun)) {
		scsi_inquiry(targ, lun);
		switch(inqdata.devtype) {
		case SCSI_DEVDISK: scsi_ndisk(targ, lun); break;
#if TAPE_SUPPORT
		case SCSI_DEVTAPE: scsi_ntape(targ, lun); break;
#endif
		default:
			printf("sc0: ID %d.%d: ", targ, lun);
			if(inqdata.devtype > SCSI_DEVMAX) printf("UNKNOWN ");
			else printf("%-7s ", scsi_devstr[inqdata.devtype]);
			printf("%s (unsupported)\n", inqdata.extra);
		}
	}
  }
}


/*===========================================================================*
 *				scsi_inquiry				     *
 *===========================================================================*/
PRIVATE int scsi_inquiry(targ, lun)
int targ, lun;
{
  int result, i;
  byte *p, *q;
    
  /* set up scsi command */
    
  ccb.cmd.opcode = SCSI_INQUIRY;
  ccb.cmd.lba = long2aha((long)0);
  ccb.cmd.nblocks = sizeof(inquiry_t);
    
  /* send the command */

  scsi_command(targ, lun, CCB_NOCHECK, (long)
	umap(proc_ptr, D, (vir_bytes)&inqdata, 1), (long)sizeof(inquiry_t));
   
  /* Massage extra field into a printable string.  Why is this necessary? */

  for(p = q = inqdata.extra; p < inqdata.extra + inqdata.len; p++)
	if(*p >= ' ' && *p < 127) *q++ = *p;	/* keep only printable chars */
  *q = '\0';
}


/*===========================================================================*
 *				scsi_ndisk				     *
 *===========================================================================*/
PRIVATE void scsi_ndisk(targ, lun)
int targ, lun;
{
  /* Add new disk to tables */
  int p, dev, tenmegs;
  struct part_entry *pt;

  if(nr_disks >= MAX_DISKS) {
	printf("sc0: ID %d.%d TOO MANY DISKS, set MAX_DISKS higher (now %d)\n",
		targ, lun, MAX_DISKS);
	return;
  }

  aha_disk[nr_disks].target = targ;
  aha_disk[nr_disks].lun = lun;

#if DEBUG
  printf("sd%d: ID %d.%d: %-7s %s\n", nr_disks, targ, lun,
	scsi_devstr[inqdata.devtype], inqdata.extra);
#endif

  /* !! should get drive capacity */

  /* read partition table */

  ccb.cmd.opcode = SCSI_READ;  /* set up scsi command */
  ccb.cmd.lba = long2aha((long) 0);
  ccb.cmd.nblocks = 1;
  scsi_command(targ, lun, CCB_INCHECK, 
	umap(proc_ptr, D, buf, SECTOR_SIZE), (long)SECTOR_SIZE);

  if(buf[510] != 0x55 || buf[511] != 0xAA) {
	printf("sd%d: invalid partition table, skipping disk\n", nr_disks);
	return;
  }

  /* set up disk devices based on partitions */

  pt = (struct part_entry *)&buf[PART_TABLE_OFF];		
  dev = nr_disks*5;

  aha_diskdev[dev].first = 0;		/* device 0 = whole disk */
  aha_diskdev[dev++].size = 0;		/* size = 0 means infinite */

  for(p = 0; p < NR_PARTITIONS; p++, dev++) {
	tenmegs = pt[p].size * 10 / SECTORS_PER_MEG;
#if DEBUG
	printf("sd%d: %d.%d meg partition, blocks %6D ... %6D\n", dev,
		tenmegs/10, tenmegs%10,	pt[p].lowsec, pt[p].lowsec+pt[p].size);
#endif
	aha_diskdev[dev].first = pt[p].lowsec;
	aha_diskdev[dev].size = pt[p].size;
  }
  nr_disks ++;
}


/*===========================================================================*
 *				scsi_dkio				     *
 *===========================================================================*/
PRIVATE int scsi_dkio(m_ptr)
message *m_ptr;
{
  phys_bytes address;
  long block;
  int dev, disk, n, iocheck; 

  address = numap(m_ptr->PROC_NR, (vir_bytes) m_ptr->ADDRESS, BLOCK_SIZE);

  dev = m_ptr->DEVICE;
  disk = dev / (1 + NR_PARTITIONS);
  n = m_ptr->COUNT / SECTOR_SIZE;
  block = m_ptr->POSITION / SECTOR_SIZE;

  /* check for bad device number */
  if(dev < 0 || dev >= NR_DISKDEV || disk >= nr_disks) {
	errorpr(m_ptr, address, "bad device number");
	return EIO;
  }

  /* check for misaligned request */
  if(m_ptr->POSITION % SECTOR_SIZE != 0 || m_ptr->COUNT % SECTOR_SIZE != 0) {
	errorpr(m_ptr, address, "misaligned request");
	return EINVAL;
  }

  /* check for partition overrun and adjust request to fit */
  if(aha_diskdev[dev].size && block + n > aha_diskdev[dev].size) {
	n = aha_diskdev[dev].size - block;
	/* start block >= size? */
	if(n <= 0) return 0;
  }

  block += aha_diskdev[dev].first;	/* adjust to physical block number */

  /* set up SCSI I/O command */

  if(m_ptr->m_type == DISK_READ) {
	ccb.cmd.opcode = SCSI_READ;
	iocheck = CCB_INCHECK;
  }
  else if(m_ptr->m_type == DISK_WRITE) {
	ccb.cmd.opcode = SCSI_WRITE;
	iocheck = CCB_OUTCHECK;
  }
  else { 				/* shouldn't even have got here */
	errorpr(m_ptr, address, "bad op code");
	return EINVAL;
  }

  ccb.cmd.lba = long2aha(block);
  ccb.cmd.nblocks = n;
  if(scsi_command(aha_disk[disk].target, aha_disk[disk].lun, iocheck,
		address, m_ptr->COUNT)) {
	errorpr(m_ptr, address, "scsi error");
	return EIO;
  }
#if DEBUG > 2
  errorpr(m_ptr, address, "ok");
#endif
  return n * SECTOR_SIZE;
}


#if TAPE_SUPPORT

/*===========================================================================*
 *				scsi_ntape				     *
 *===========================================================================*/
PRIVATE void scsi_ntape(targ, lun)
int targ, lun;
{
  int i;
  long minblk, maxblk;

  if(nr_tapes >= MAX_TAPES) {
	printf("sc0: ID %d.%d TOO MANY TAPES, set MAX_TAPES higher (now %d)\n",
		targ, lun, MAX_TAPES);
	return;
  }

  aha_tape[nr_tapes].target = targ;
  aha_tape[nr_tapes].lun = lun;
  aha_tape[nr_tapes].in_use = 0;
  aha_tape[nr_tapes].at_eof = 0;

#if DEBUG
  printf("st%d: ID %d.%d: %-7s %s\n", nr_tapes, targ, lun,
	scsi_devstr[inqdata.devtype], inqdata.extra);
#endif

  /* set up scsi modesense command */
    
  ccb.cmd.opcode = SCSI_MDSENSE;
  ccb.cmd.lba = long2aha(0L);
  ccb.cmd.nblocks = 255;
    
  /* send the command */

  scsi_command(targ, lun, CCB_NOCHECK, (long)
	umap(proc_ptr, D, (vir_bytes)buf, 1), (long)256);

#if DEBUG
  printf("st%d: data len %d medium type %d speed/bufmode 0x%X desc len %d\n",
	nr_tapes, buf[0], buf[1], buf[2], buf[3]);
  for(i=4;i<4+buf[3];i+=8) {
	printf("st%d: density %d, nblocks %D, block len %D\n", nr_tapes,
		buf[i], 
		buf[i+1]*65536L+buf[i+2]*256L+buf[i+3],
		buf[i+5]*65536L+buf[i+6]*256L+buf[i+7]);
  }
#endif
  /* read limits */

  /* set up scsi command */
    
  ccb.cmd.opcode = SCSI_RDLIMITS;
  ccb.cmd.lba = long2aha(0L);
  ccb.cmd.nblocks = 0;
    
  /* send the command */

  scsi_command(targ, lun, CCB_NOCHECK, (long)
	umap(proc_ptr, D, (vir_bytes)buf, 1), (long)6);

  minblk = buf[4]*256L + buf[5]; 
  maxblk = buf[1]*65536L + buf[2]*256L + buf[3];
#if ARCHIVE_2525S
  aha_tape[nr_tapes].blocksize = minblk * 512;
#else
  aha_tape[nr_tapes].blocksize = minblk;
#endif
#if DEBUG
  printf("st%d: limits: min block len %D, max block len %D\n", 
	nr_tapes, minblk, maxblk);
#endif

  nr_tapes++;
}


/*===========================================================================*
 *				scsi_tpio				     *
 *===========================================================================*/
PRIVATE int scsi_tpio(m_ptr)
message *m_ptr;
{
  phys_bytes address;
  long block, residue, n;
  int tape, iocheck, rewind, i;


  address = numap(m_ptr->PROC_NR, (vir_bytes) m_ptr->ADDRESS, BLOCK_SIZE);
  tape = (m_ptr->DEVICE & 0x7F) >> 1;
  rewind = m_ptr->DEVICE & 1;

  n = m_ptr->COUNT / aha_tape[tape].blocksize;

  /* check for bad device number */
  if(tape < 0 || tape >= nr_tapes) {
	errorpr(m_ptr, address, "bad device number");
	return EIO;
  }

  /* check for bad count */
  if(m_ptr->COUNT % aha_tape[tape].blocksize != 0) {
	errorpr(m_ptr, address, "oddsized request");
	return EINVAL;
  }

  if(m_ptr->m_type == DISK_READ) {
	if(aha_tape[tape].at_eof) return 0;
	ccb.cmd.opcode = SCSI_READ;
	iocheck = CCB_INCHECK;
  }
  else if(m_ptr->m_type == DISK_WRITE) {
	ccb.cmd.opcode = SCSI_WRITE;
	iocheck = CCB_OUTCHECK;
  }
  else { 		/* shouldn't even have got here */
	errorpr(m_ptr, address, "bad op code");
	return EINVAL;
  }

  ccb.cmd.fixed = 1;
  ccb.cmd.trlength = long2aha(n);
  if(scsi_command(aha_tape[tape].target, aha_tape[tape].lun, iocheck,
		address, m_ptr->COUNT)) {
	if(sense_key(ccb.sense.key) == 0 &&
 	   (sense_eof(ccb.sense.key) || sense_eom(ccb.sense.key) || 
	    sense_ili(ccb.sense.key))) {
		/* not an error, but eof or eom */
		aha_tape[tape].at_eof = 1;
	        for(residue=i=0;i<4;i++)
			residue = (residue << 8) + ccb.sense.info[i];
		return m_ptr->COUNT - (residue * aha_tape[tape].blocksize);
	}	
	errorpr(m_ptr, address, "scsi error");
	return EIO;
  }
#if DEBUG > 2
  errorpr(m_ptr, address, "ok");
#endif
  return m_ptr->COUNT;
}


/*===========================================================================*
 *				scsi_tpopen				     *
 *===========================================================================*/
PRIVATE int scsi_tpopen(m_ptr)
message *m_ptr;
{
  int tape;

  tape = (m_ptr->DEVICE & 0x7F) >> 1;

  /* check for bad device number */
  if(tape < 0 || tape >= nr_tapes) {
	errorpr(m_ptr, 0L, "bad device number");
	return EIO;
  }

  /* make sure tape is not already open */
  if(aha_tape[tape].in_use) {
	errorpr(m_ptr, 0L, "tape already open");
	return EBUSY;
  }

  /* make sure tape unit is online */
  ccb.cmd.opcode = SCSI_UNITRDY;
  ccb.cmd.lba = long2aha(0L);
  ccb.cmd.nblocks = 0;
  if(scsi_command(aha_tape[tape].target,aha_tape[tape].lun,CCB_NOCHECK,0L,0L)){
	errorpr(m_ptr, 0L, "tape not online");
	return EIO;
  }

  aha_tape[tape].in_use = 1;
  aha_tape[tape].at_eof = 0;
  aha_tape[tape].open_mode = m_ptr->COUNT;
#if DEBUG > 2
  errorpr(m_ptr, 0L, "ok");
#endif
  return OK; 
}


/*===========================================================================*
 *				scsi_tpclose				     *
 *===========================================================================*/
PRIVATE int scsi_tpclose(m_ptr)
message *m_ptr;
{
  int tape, rewind;

  tape = (m_ptr->DEVICE & 0x7F) >> 1;
  rewind = m_ptr->DEVICE & 1;

  /* check for bad device number */
  if(tape < 0 || tape >= nr_tapes) {
  	errorpr(m_ptr, 0L, "bad device number");
	return EIO;
  }

  /* make sure tape is open */
  if(!aha_tape[tape].in_use) {
  	errorpr(m_ptr, 0L, "tape not open");
	return EIO;
  }

  aha_tape[tape].in_use = 0;
  aha_tape[tape].at_eof = 0;
 
  /* write filemark if open for writing */
  if(aha_tape[tape].open_mode & W_BIT) {
	ccb.cmd.opcode = SCSI_WREOF;
	ccb.cmd.fixed = 0;
	ccb.cmd.trlength = long2aha(1L);
	if(scsi_command(aha_tape[tape].target, aha_tape[tape].lun,
		CCB_NOCHECK, 0L, 0L)) {
  		errorpr(m_ptr, 0L, "could not write filemark on close");
		return EIO;
  	}
  }

  /* rewind if rewind device */
  if(rewind) {
	ccb.cmd.opcode = SCSI_REWIND;
	ccb.cmd.fixed = 0 | IMMED_BIT;
	ccb.cmd.trlength = long2aha(0L);
	if(scsi_command(aha_tape[tape].target, aha_tape[tape].lun,
		CCB_NOCHECK, 0L, 0L)){
  		errorpr(m_ptr, 0L, "could not rewind on close");
		return EIO;
	}
  }
#if DEBUG > 2
  errorpr(m_ptr, 0L, "ok");
#endif
  return OK;
}

/*===========================================================================*
 *				scsi_tpioctl				     *
 *===========================================================================*/
PRIVATE int scsi_tpioctl(m_ptr)
message *m_ptr;
{
  int tape, size;
  struct mtop op;
  byte *p;

  tape = (m_ptr->DEVICE & 0x7F) >> 1;

  /* check for bad device number */
  if(tape < 0 || tape >= nr_tapes) {
  	errorpr(m_ptr, 0L, "bad device number");
	return EIO;
  }

  if(m_ptr->TTY_REQUEST != MTIOCTOP) {
  	errorpr(m_ptr, (long)m_ptr->TTY_REQUEST, "bad ioctl request number");
	return ENOTTY;
  }

  size = sizeof(struct mtop); 
  phys_copy(numap(m_ptr->PROC_NR, (vir_bytes) m_ptr->ADDRESS, (vir_bytes)size),
		umap(proc_ptr, S, (vir_bytes) &op, (vir_bytes)size),
		(phys_bytes)size);

  switch(op.mt_op) {
    case MTREW:
	ccb.cmd.opcode = SCSI_REWIND;
	ccb.cmd.fixed = 0 | IMMED_BIT;
	ccb.cmd.trlength = long2aha(0L);
	break;
    case MTFSF:
    case MTFSR:
    case MTBSF:
    case MTBSR:
	ccb.cmd.opcode = SCSI_SPACE;
	if(op.mt_op == MTFSR || op.mt_op == MTBSR)
		ccb.cmd.fixed = 0;
	else
		ccb.cmd.fixed = 1;
	if(op.mt_op == MTBSF || op.mt_op == MTBSR)
		ccb.cmd.trlength = long2aha((long)-op.mt_count);
	else 
		ccb.cmd.trlength = long2aha((long)op.mt_count);
	break;
    case MTERASE:
	ccb.cmd.opcode = SCSI_ERASE;
	ccb.cmd.fixed = 2 | IMMED_BIT;
	ccb.cmd.trlength = long2aha(0L);
	break;
    case MTRETEN:
    case MTOFFL:
	ccb.cmd.opcode = SCSI_LOAD;
	ccb.cmd.fixed = 0 | IMMED_BIT;
	ccb.cmd.trlength = long2aha((op.mt_op == MTRETEN) ? 3L: 0L);
	break;
    case MTWEOF:
    default:
  	errorpr(m_ptr, (long)op.mt_op, "unimplemented ioctl command");
	return ENOTTY;
  }

  if(scsi_command(aha_tape[tape].target, aha_tape[tape].lun,
	CCB_NOCHECK, 0L, 0L)) {
  	errorpr(m_ptr, (long)op.mt_op, "error during ioctl command");
	return EIO;
  }
  return OK;
}

#endif	/* TAPE_SUPPORT */


/*===========================================================================*
 *				scsi_command				     *
 *===========================================================================*/
PRIVATE int scsi_command(targ, lun, checkbits, data, len)
int targ, lun, checkbits;
long data, len;
{
  int key;

  /* set up aha command block */
   
  ccb.addrcntl = ccb_scid(targ) | ccb_lun(lun) | checkbits;
  ccb.datalen = long2aha(len);
  ccb.dataptr = long2aha(data);

  /* handshake with controller to send command */

  mailbox[0].status = AHA_MBOXSTART;		/* fill aha's mailbox */
  out_byte(AHA_DATAREG, AHACOM_STARTSCSI);	/* hey, you've got mail! */
  aha_wait();					/* wait for completion */
  mailbox[1].status = AHA_MBOXFREE;  		/* free up inbox */

  /* check results of operation */

  if(ccb.hastat != 0 || ccb.tarstat != 0) {
#if DEBUG
	if(ccb.hastat != 0) {
		/* weird host adapter status */
		printf("\nsc%d.%d: host adaptor error %d\n", targ, lun, ccb.hastat);
 	}
	else {
		/* check sense data */
		key = sense_key(ccb.sense.key);
		if(ccb.tarstat == 2 && key == 0) 
			return 1;	/* attention cond, not really error */ 
		printf("\nsc0: target error status %d, sense key %d (%s)\n",
			ccb.tarstat, ccb.sense.key, 
			key > SCSI_MAXSENSE ? "UNKNOWN" : scsi_sense[key]);
	}
	errordump();
#endif
	return 1;
  }
  return 0;
}


/*===========================================================================*
 *				aha_command				     *
 *===========================================================================*/
PRIVATE void aha_command(outlen, outptr, inlen, inptr)
int outlen, inlen;
char *outptr, *inptr;
{
  int i, stat;

  /* send command bytes */
    
  for(i=0; i < outlen; i++) {
	while(in_byte(AHA_STATREG) & AHA_CDF);		/* !! timeout */
	out_byte(AHA_DATAREG, *outptr++);
  }

  /* receive data bytes */
    
  for(i=0; i < inlen; i++) {
	while(!(in_byte(AHA_STATREG) & AHA_DF));	/* !! timeout */
	*inptr++ = in_byte(AHA_DATAREG);
  }

  aha_wait();		/* wait for hardware completion message */

  /* !! should check status register here for invalid command */
}


/*===========================================================================*
 *				aha_reset				     *
 *===========================================================================*/
PRIVATE int aha_reset()
{
  int stat;
  long retries;
  byte ahacmd[5];

  cim_aha_scsi();
	
  /* reset controller, wait for self test to complete */

  retries = AHA_TIMEOUT;
  out_byte(AHA_CNTLREG, AHA_HRST);
  while(in_byte(AHA_STATREG) & AHA_STST && --retries);
  if(!retries) {
	printf("sc0: AHA154x Controller not responding\n");
	milli_delay(5000);
	return 0;
  }
	
  /* check for self-test failure */
  stat = in_byte(AHA_STATREG);
  if(stat & AHA_DIAGF) {
	printf("sc0: AHA154x Controller failed self-test\n");
	milli_delay(5000);
	return 0;
  }

  /* !! maybe a santity check here: make sure IDLE and INIT are set? */ 

  /* init mailbox and tell controller where it is */
    
  mailbox[0].status = AHA_MBOXFREE;
  mailbox[0].ccbptr = ptr2aha(&ccb);
  mailbox[1].status = AHA_MBOXFREE;
  /* mailbox[1].ccbptr filled by adapter after command execution */

  ahacmd[0] = AHACOM_INITBOX;
  ahacmd[1] = 1;				/* one mailbox set */
  *((int24 *)&ahacmd[2]) = ptr2aha(&mailbox[0]);	/* here it is */
  aha_command(5, ahacmd, 0, 0);

  /* !! maybe sanity check: check status reg for initialization success */

  /* Set SCSI selection timeout */

  ahacmd[0] = AHACOM_SETIMEOUT;
  ahacmd[1] = SCSI_TIMEOUT != 0;		/* timeouts on/off */
  ahacmd[2] = 0;				/* reserved */
  ahacmd[3] = SCSI_TIMEOUT / 256;		/* MSB */
  ahacmd[4] = SCSI_TIMEOUT % 256;		/* LSB */
  aha_command(5, ahacmd, 0, 0);
    
  /* initialize command control block */

  ccb.opcode = 0;				/* always initiator cmd */
  ccb.cmdlen = sizeof(ccb.cmd);			/* always use group 0 */
  ccb.senselen = CCB_SENSEREQ;			/* always want sense info */
  ccb.linkptr = long2aha((long)0);		/* never link commands */
  ccb.linkid = 0;
  ccb.reserved[0] = 0;
  ccb.reserved[1] = 0;
  ccb.cmd.control = 0;				/* SCSI cmd link field */

  return 1;
}


/*===========================================================================*
 *				aha_wait				     *
 *===========================================================================*/
PRIVATE void aha_wait()
{
  static message intr_mess;
  receive(HARDWARE, &intr_mess);
  /* !! check INTRREG for unexpected interrupts, like external SCSI reset */
  out_byte(AHA_CNTLREG, AHA_IRST);	/* clear interrupts */
  cim_aha_scsi();
}


/*===========================================================================*
 *				long2aha				     *
 *===========================================================================*/
PRIVATE int24 *long2ahap(l)
long l;
{
  register byte *p = (byte *) &l;
  static int24 val;	
        
  val.v[0] = p[2]; val.v[1] = p[1]; val.v[2] = p[0];
  return &val;
}


/*===========================================================================*
 *				ptr2aha					     *
 *===========================================================================*/
PRIVATE int24 *ptr2ahap(p)
byte *p;
{
  register phys_bytes addr;

  addr = umap(proc_ptr, D, (vir_bytes) p, BLOCK_SIZE);
  return long2ahap((long)addr);		
}


#if DEBUG
/*===========================================================================*
 *				errorpr					     *
 *===========================================================================*/
PRIVATE void errorpr(m_ptr, address, str)
message *m_ptr;
phys_bytes address;
{
    char *req;

#if TAPE_SUPPORT
  if(m_ptr->DEVICE & 0x80) 
	printf("%srst%d: ",(m_ptr->DEVICE & 1) ? "":"n",
			(m_ptr->DEVICE & 0x7F) >> 1);
  else
#endif
	printf("sd%d: ", m_ptr->DEVICE);
  switch(m_ptr->m_type) {
    case DISK_READ:
    case DISK_WRITE:
  	printf("%s req, caller %d bytes %d offset %X address %X",
		m_ptr->m_type == DISK_READ? "READ" : "WRITE",
		m_ptr->PROC_NR, m_ptr->COUNT, m_ptr->POSITION,
		(long)address);
	break;
#if TAPE_SUPPORT
    case TTY_OPEN:
    case TTY_CLOSE:
	printf("%s req, caller %d",
		m_ptr->m_type == TTY_OPEN? "OPEN" : "CLOSE",
		m_ptr->PROC_NR);
	break;
    case TTY_IOCTL:
	printf("IOCTL req, caller %d mtop %d", m_ptr->PROC_NR, address);
 	break;
#endif
    default:
	printf("bad request type %d, caller %d",
		m_ptr->m_type, m_ptr->PROC_NR);
    }
    printf(": %s\n", str);
}
#endif /* DEBUG */


#if DEBUG > 1
/*===========================================================================*
 *				errordump				     *
 *===========================================================================*/
PRIVATE void errordump()
{
  int i;

  printf("aha ccb dump:");
  for(i=0;i<sizeof(ccb);i++) {
	if(i % 19 == 0) printf("\n");
	printf(" %02X",((byte *)&ccb)[i]);
  }
  printf("\n");
}
#endif /* DEBUG > 1 */
