/* getline.c */

#include "const.h"
#include "type.h"
#include "chars.h"
#include "var.h"

#ifndef PUBLIC

#define FORWARD extern
#define PUBLIC
#define PRIVATE static
#define static

#define FALSE 0
#define TRUE 1

#include "chars.h"

typedef int bool_pt;
typedef unsigned char bool_t;
typedef unsigned count_t;
typedef int char16_t;       /* s.b. unsigned ? */

FORWARD char16_t inchar();
FORWARD void outbyte();

PRIVATE char_pt prompt = '?';

#endif /* PUBLIC */

#define BUFSIZE 256
#define LINESIZE (80 - 1 - 3 + 1)
                            /* 1 to avoid scroll, 3 for prompt, 1 for null */
PRIVATE char buf[BUFSIZE];
#define endbuf (buf + sizeof buf)
PRIVATE int bufindex;
PRIVATE char *buflow = endbuf;
#define endline (line + sizeof line - 1)  /* 1 less for null */
PRIVATE char line[LINESIZE];
PRIVATE char *high;
PRIVATE char hotline[] = { 0, 0 };
PRIVATE char *low;
PRIVATE bool_t inprogress = FALSE;
PRIVATE bool_t insertmode;

FORWARD bool_pt left();
FORWARD void oldline();
FORWARD void show();
FORWARD bool_pt right();

PUBLIC char *getline()
{
	char16_t ch;
	count_t length;

	if ( !inprogress )
	{
		low = line;
		high = endline;
		bufindex = -1;
		insertmode = FALSE;
		inprogress = TRUE;
	}
	show();
	while ( TRUE )
	{
		ch = inchar();
		if ( ch == '\r' || ch == '\n' )
		{
			while ( left() )
				;
			*endline = 0;
			if ( (length = endline - high + 1) <= 1 )
				continue;
			if ( length > 2 )
			{
				if ( length > buflow - buf )
				{
					memcpy( buf + length, buflow = buf, sizeof buf - length );
					buf[sizeof buf - 1] = 0;
				}
				else
					buflow -= length;
				memcpy( buflow, high, length );
			}
			inprogress = FALSE;
			return high;
		}
		switch( ch )
		{
		case BACKSPACE1:
			left();
		case DEL1:
		case DEL2:
			if ( high < endline )
				++high;
			break;
		case DOWN1:
		case DOWN2:
			if ( bufindex >= 0 )
				/* used before this line */
				--bufindex;
			oldline();
			break;
		case END1:
		case END2:
			while ( right() )
				;
			break;
		case ESCAPE:
			hotline[0] = ch;
			return hotline;
		case INS1:
		case INS2:
			insertmode = (TRUE + FALSE) - insertmode;
			break;
		case LEFT1:
		case LEFT2:
			left();
			break;
		case RIGHT1:
		case RIGHT2:
			if ( right() )
				;
			break;
		case START1:
		case START2:
			while ( left() )
				;
			break;
		case UP1:
		case UP2:
			++bufindex;
			oldline();
			break;
		case WORDLEFT1:
		case WORDLEFT2:
			while ( (high == endline || *high != ' ') && left() )
				;
			while ( left() && *high == ' ' )
				;
			break;
		case WORDRIGHT1:
		case WORDRIGHT2:
			while ( high < endline && *high != ' ' )
				right();
			while ( high < endline && *high == ' ' )
				right();
			break;
		default:
			if ( ch >= FIRST_EXTENDED_CHAR )
			{
				hotline[0] = ETOA( ch );
				return hotline;
			}
			if ( ch >= ' ' && ch <= '~' && low < high )
			{
				*low++ = ch;
				if ( !insertmode && high < endline )
					++high;
			}
			break;
		}
		show();
	}
}

PRIVATE bool_pt left()
{
	if ( low <= line )
		return FALSE;
	*--high = *--low;
	return TRUE;
}

PRIVATE void oldline()
{
	/* written for small size but low speed */
	char *bufptr;
	count_t count;

	for ( bufptr = buflow, count = 0; bufptr < endbuf; ++count )
		bufptr += strlen( bufptr ) + 1;
	if ( count != 0 )
	{
		if ( (bufindex %= (int) count) < 0 )
			bufindex += count;
		for ( bufptr = buflow, count = bufindex; count != 0; --count )
			bufptr += strlen( bufptr ) + 1;
		memcpy( line, bufptr, count = strlen( bufptr ) );
		low = line + count;
		high = endline;
	}
}
PRIVATE bool_pt right()
{
	if ( high >= endline )
		return FALSE;
	*low++ = *high++;
	return TRUE;
}

PRIVATE void show()
{
	/* 
		Print line over top of old one.
		Grotty because it has to work for dumb ttys.
	*/
	static int lastlength = 0;
	int length;
	register char *lineptr;

	show1();
	for ( lineptr = high; lineptr < endline; ++lineptr )
		outbyte( *lineptr );
	length = (low - line) + (endline - high);
	for ( lastlength -= length; lastlength > 0; --lastlength )
		outspace();
	lastlength = length;
	show1();
}

PRIVATE void show1()
{
	register char *lineptr;

	outbyte( '\r' );
	outbyte( prompt );
	outstr( "> " );
	for ( lineptr = line; lineptr < low; ++lineptr )
		outbyte( *lineptr );
}
