/*	zexec 1.1 - decompress and execute		Author: Kees J. Bot
 *								11 May 1994
 * When placed before a compressed executable like this:
 *
 *	#!/usr/bin/zexec /usr/bin/zcat
 *
 * Zexec feeds the rest of the file to zcat and executes the result.
 */
#define nil 0
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>

void tell(const char *s)
{
	write(2, s, strlen(s));
}

void report(const char *label)
{
	int err= errno;

	tell("zexec: ");
	tell(label);
	tell(": ");
	tell(strerror(err));
	tell("\n");
}

void fatal(const char *label)
{
	report(label);
	exit(1);
}

int main(int argc, char **argv)
{
	char *decomp, *exe;
	int decfd, exefd, tmpfd;
	char buf[1024];
	char tmp[5 + 14 + 1];
	char *p;
	ssize_t n;
	pid_t pid;
	int status;
	int pfd[2];
	int err;
	struct flock lk;

	if (argc < 3) {
		tell("Usage: zexec decompressor executable [arg ...]\n");
		return 1;
	}
	decomp= argv[1];
	exe= argv[2];

	/* Try to put a write lock on the decompressor or parallel zexec's
	 * may use lots of memory.
	 */
	if ((decfd= open(decomp, O_RDWR)) != -1) {
		(void) fcntl(decfd, F_SETFD, fcntl(decfd, F_GETFD)|FD_CLOEXEC);
		lk.l_type= F_WRLCK;
		lk.l_whence= SEEK_SET;
		lk.l_start= 0;
		lk.l_len= 0;
		(void) fcntl(decfd, F_SETLKW, &lk);
	}

	/* Compressed executable. */
	if ((exefd= open(exe, O_RDONLY)) < 0) fatal(exe);

	if ((n= read(exefd, buf, sizeof(buf))) < 0) fatal(exe);

	/* Try to find the end of the #! line in the first 1k.  Use the
	 * whole file to decompress if it can't be found.
	 */
	if (n >= 3 && buf[0] == '#' && buf[1] == '!'
			&& (p= memchr(buf, '\n', n)) != nil) {
		n= p - buf + 1;
	} else {
		n= 0;
	}

	if (lseek(exefd, (off_t) n, SEEK_SET) == -1) fatal(exe);

	/* Create and open a temporary file. */
	strcpy(tmp, "/tmp/");
	strncpy(tmp + 5, (p= strrchr(exe, '/')) == nil ? exe : p + 1, 14 - 7);
	strcat(tmp + 5, ".XXXXXX");
	if ((tmpfd= mkstemp(tmp)) < 0) fatal(tmp);

	/* The executable will be decompressed in a temp file.  To remove the
	 * temp file eventually we keep a process around that does the unlink
	 * when this pipe closes on exec:
	 */
	if (pipe(pfd) < 0) {
		err= errno;
		(void) unlink(tmp);
		errno= err;
		fatal("pipe()");
	}
	(void) fcntl(pfd[1], F_SETFD, fcntl(pfd[1], F_GETFD) | FD_CLOEXEC);

	/* Ye basic fork and exec with I/O redirection. */
	switch ((pid= fork())) {
	case -1:
		err= errno;
		(void) unlink(tmp);
		errno= err;
		fatal("fork()");
	case 0:
		/* Start the remover in a sub-subprocess, so that it will be
		 * a child of init eventually.
		 */
		switch (fork()) {
		case -1:
			err= errno;
			(void) unlink(tmp);
			errno= err;
			fatal("fork()");
		case 0:
			/* Wait for the pipe to perish. */
			close(exefd);
			close(tmpfd);
			close(pfd[1]);
			setsid();
			(void) read(pfd[0], buf, 1);
			(void) unlink(tmp);
			_exit(0);
		}

		/* Start the decompressor. */
		close(pfd[0]);
		close(pfd[1]);

		dup2(exefd, 0);
		dup2(tmpfd, 1);
		close(exefd);
		close(tmpfd);
		execl(decomp, decomp, (char *) nil);
		fatal(decomp);
	}
	close(pfd[0]);
	close(exefd);
	close(tmpfd);

	if (waitpid(pid, &status, 0) < 0) fatal("wait()");
	if (WEXITSTATUS(status) != 0) exit(1);
	if (WIFSIGNALED(status) != 0) {
		tell("zexec: ");
		tell(decomp);
		tell(" died by a signal\n");
		exit(1);
	}

	/* Execute the decompressed executable. */
	if (chmod(tmp, 0100) < 0) fatal(tmp);
	execv(tmp, argv + 2);
	fatal(tmp);
}
