/*	vdisk_ctrl 1.0 - control virtual disk devices	Author: Kees J. Bot
 *								17 Dec 1995
 */
#define nil 0
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/svrctl.h>
#include <minix/vdisk.h>

void fatal(const char *label)
{
	fprintf(stderr, "vdisk_ctrl: %s: %s\n", label, strerror(errno));
	exit(1);
}

void *allocate(void *mem, size_t len)
{
	mem= realloc(mem, len);
	if (mem == nil) {
		fprintf(stderr, "vdisk_ctrl: Out of memory\n");
		exit(1);
	}
	return mem;
}

int line = 1;

void syntax_err(char *conf_file, char *what)
{
	fprintf(stderr, "vdisk_ctrl: \"%s\", line %d: %s\n",
						conf_file, line, what);
	exit(1);
}

char *getword(FILE *cfp, char *conf_file)
/* Read one word from the configuration file. */
{
	int c;
	char *word;
	size_t i, n;

	/* Skip whitespace. */
	do {
		c= getc(cfp);
		if (c == EOF) {
			if (ferror(cfp)) fatal(conf_file);
			return nil;
		}
		if (c == '#') {
			while ((c= getc(cfp)) != '\n') {
				if (c == EOF) {
					if (ferror(cfp)) fatal(conf_file);
					return nil;
				}
			}
		}
		if (c == '\n') line++;
	} while (c <= ' ');

	word= nil;
	i= 0;
	n= 0;
	do {
		if (i == n) word= allocate(word, (n= n+16) * sizeof(word[0]));
		word[i++]= c;
	} while ((c= getc(cfp)) > ' ');

	if (c == EOF) {
		if (ferror(cfp)) fatal(conf_file);
	} else {
		ungetc(c, cfp);
	}
	word= allocate(word, (i+1) * sizeof(word[0]));
	word[i]= 0;
	return word;
}

char *prepend_dev(char *name)
/* Prepend "/dev/" in front of a relative pathname. */
{
	size_t n;

	if (name[0] == '/') return name;
	n= strlen(name) + 1;
	name= allocate(name, (5 + n) * sizeof(name[0]));
	memmove(name + 5, name, n);
	memcpy(name, "/dev/", 5);
	return name;
}

int sugar(char *word)
/* Return '=', '+', or '*' if word is "=", "+", or "*", otherwise 0. */
{
	return word[0] != 0 && word[1] == 0 && strchr("=+*", word[0]) != nil
		? word[0] : 0;
}

int main(int argc, char **argv)
{
	struct fsdevmap fsdevmap;
	struct vdisk_attach vdatt;
	char *conf_file;
	char *word;
	char *vdev;
	char **devs= nil;
	FILE *cfp;
	int vd;
	size_t i, n;
	struct stat st;

	/* For now we only accept '-c config-file'. */
	if (argc == 2 && strncmp(argv[1], "-c", 2) == 0 && argv[1][2] != 0) {
		conf_file= argv[1] + 2;
	} else
	if (argc == 3 && strcmp(argv[1], "-c") == 0) {
		conf_file= argv[2];
	} else {
		fprintf(stderr, "Usage: vdisk_ctrl -c config-file\n");
		exit(1);
	}

	if (strcmp(conf_file, "-") == 0) {
		cfp= stdin;
	} else {
		if ((cfp= fopen(conf_file, "r")) == nil) fatal(conf_file);
	}

	/* Repeatedly read something of the form vdev = dev + dev + ... */
	word= getword(cfp, conf_file);
	while (word != nil) {
		if (sugar(word) != 0) {
			syntax_err(conf_file, "virtual disk name expected");
		}
		vdev= prepend_dev(word);
		word= getword(cfp, conf_file);
		if (word == nil || sugar(word) != '=') {
			syntax_err(conf_file, "'=' expected");
		}
		free(word);
		word= getword(cfp, conf_file);
		n= 0;
		for (;;) {
			if (word == nil || sugar(word) != 0) {
				syntax_err(conf_file, "device name expected");
			}
			devs= allocate(devs, (n + 1) * sizeof(devs[0]));
			devs[n++]= prepend_dev(word);
			word= getword(cfp, conf_file);
			if (word == nil || sugar(word) != '+') break;
			free(word);
			word= getword(cfp, conf_file);
		}

		/* One vdev = dev + dev + ... list read, so program a virtual
		 * disk.
		 */
		if ((vd= open(vdev, O_RDONLY)) < 0) fatal(vdev);

		for (i = 0; i < n; i++) {
			if (stat(devs[i], &st) < 0) fatal(devs[i]);
			if (!S_ISBLK(st.st_mode) && !S_ISCHR(st.st_mode)) {
				fprintf(stderr,
					"vdisk_ctrl: %s: Not a device\n",
					devs[i]);
				exit(1);
			}
			fsdevmap.dev= st.st_rdev;
			if (svrctl(FSDEVMAP, &fsdevmap) < 0) {
				fprintf(stderr,
		"vdisk_ctrl: can't map %s to <task, minor>: %s\n",
					devs[i], strerror(errno));
				exit(1);
			}

			vdatt.index= i;
			vdatt.task_nr= fsdevmap.task_nr;
			vdatt.minor= fsdevmap.minor;
			if (ioctl(vd, DIOCVDATTACH, &vdatt) < 0) {
				fprintf(stderr,
				"vdisk_ctrl: can't attach %s to %s: %s\n",
					devs[i], vdev, strerror(errno));
				exit(1);
			}
			free(devs[i]);
		}
		close(vd);
		free(vdev);
	}
	exit(0);
}
