/*	tangle 1.0 - Weave trees or untangle them	Author: Kees J. Bot
 *								12 Jun 1996
 */
#define nil 0
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <utime.h>
#include <string.h>
#include <errno.h>
#include <dirent.h>
#include <signal.h>
#include <time.h>
#include <utime.h>

#ifndef S_ISLNK
#define lstat		stat
#endif

/* To make timestamps old, but unequal to each other.  (31532400 = seconds in
 * the epoch year 1970.)
 */
#define OLDTIME(t)	((t) % 31532400)

int exstatus;

void report(const char *label)
{
	fprintf(stderr, "tangle: %s: %s\n", label, strerror(errno));
	exstatus= 1;
}

void fatal(const char *label)
{
	report(label);
	exit(exstatus);
}

void report2(const char *cmd, const char *arg1, const char *arg2)
{
	fprintf(stderr, "tangle: %s %s %s: %s\n",
		cmd, arg1, arg2, strerror(errno));
	exstatus= 1;
}

void *allocate(void *mem, size_t len)
{
	if ((mem= realloc(mem, len)) == nil && len > 0) fatal("memory problem");
	return mem;
}

#define deallocate(mem)	free(mem)

char *copystr(const char *s)
{
	char *c= allocate(nil, (strlen(s) + 1) * sizeof(c[0]));
	strcpy(c, s);
	return c;
}

typedef struct pathname {
	char		*path;	/* The actual pathname. */
	size_t		idx;	/* Index for the terminating null byte. */
	size_t		lim;	/* Actual length of the path array. */
} pathname_t;

void path_init(pathname_t *pp)
/* Initialize a pathname to the null string. */
{
	pp->path= allocate(nil, (pp->lim= 16) * sizeof(pp->path[0]));
	pp->path[pp->idx= 0]= 0;
}

void path_add(pathname_t *pp, const char *name)
/* Add a component to a pathname. */
{
	size_t lim;
	char *p;
	int slash;

	lim= pp->idx + strlen(name) + 2;

	if (lim > pp->lim) {
		pp->lim= lim + lim/2;	/* add an extra 50% growing space. */
		pp->path= allocate(pp->path, pp->lim * sizeof(pp->path[0]));
	}

	p= pp->path + pp->idx;
	slash= (pp->idx > 0);
	if (pp->idx == 1 && p[-1] == '/') p--;

	while (*name != 0) {
		if (*name == '/') {
			slash= 1;
		} else {
			if (slash) { *p++ = '/'; slash= 0; }
			*p++= *name;
		}
		name++;
	}
	if (slash && p == pp->path) *p++= '/';
	*p = 0;
	pp->idx= p - pp->path;
}

void path_trunc(pathname_t *pp, size_t didx)
/* Delete part of a pathname to a remembered length. */
{
	pp->path[pp->idx= didx]= 0;
}

#if kept_for_comments_only

const char *path_name(const pathname_t *pp)
/* Return the actual name as a char array. */
{
	return pp->path;
}

size_t path_length(const pathname_t *pp)
/* The length of the pathname. */
{
	return pp->idx;
}

void path_drop(pathname_t *pp)
/* Release the storage occupied by the pathname. */
{
	free(pp->path);
}
#endif

#define path_name(pp)		((const char *) (pp)->path)
#define path_length(pp)		((pp)->idx)
#define path_drop(pp)		free((void *) (pp)->path)

typedef struct namelist {	/* Obviously a list of names. */
	struct namelist	*next;
	char		*name;
} namelist_t;

void sort(namelist_t **anl)
/* A stable mergesort disguised as line noise.  Must be called like this:
 *	if (L!=nil && L->next!=nil) sort(&L);
 */
{
	/* static */ namelist_t *nl1, **mid;  /* Need not be local */
	namelist_t *nl2;

	nl1= *(mid= &(*anl)->next);
	do {
		if ((nl1= nl1->next) == nil) break;
		mid= &(*mid)->next;
	} while ((nl1= nl1->next) != nil);

	nl2= *mid;
	*mid= nil;

	if ((*anl)->next != nil) sort(anl);
	if (nl2->next != nil) sort(&nl2);

	nl1= *anl;
	for (;;) {
		if (strcmp(nl1->name, nl2->name)<=0) {
			if ((nl1= *(anl= &nl1->next)) == nil) {
				*anl= nl2;
				break;
			}
		} else {
			*anl= nl2;
			nl2= *(anl= &nl2->next);
			*anl= nl1;
			if (nl2 == nil) break;
		}
	}
}

namelist_t *collect(const char *dir)
/* Return a sorted list of directory entries.  Returns null with errno != 0
 * on error.
 */
{
	namelist_t *names, **pn= &names;
	DIR *dp;
	struct dirent *entry;

	if ((dp= opendir(dir)) == nil) return nil;

	while ((entry= readdir(dp)) != nil) {
		if (entry->d_name[0] == '.'
			&& (entry->d_name[1] == 0
				|| (entry->d_name[1] == '.'
					&& entry->d_name[2] == 0))) {
			continue;
		}
		*pn= allocate(nil, sizeof(**pn));
		(*pn)->name= copystr(entry->d_name);
		pn= &(*pn)->next;
	}
	closedir(dp);
	*pn= nil;
	errno= 0;
	if (names != nil && names->next != nil) sort(&names);
	return names;
}

char *pop_name(namelist_t **names)
/* Return one name of a name list. */
{
	static namelist_t *junk;

	if (junk != nil) {
		deallocate(junk->name);
		deallocate(junk);
	}
	if ((junk= *names) == nil) {
		return nil;
	} else {
		*names= junk->next;
		return junk->name;
	}
}

enum howmake { LINK, COPY };

char *makefile(const char *template, enum howmake how, const char *orig)
/* Link, or copy a file to a file in the same directory as 'template'.  Return
 * the name of the new file, or null on error.
 */
{
	char *temp;
	const char *end;
	size_t len, n, m, r;
	static unsigned long cycle;
	int fd0, fd1;
	char buf[64 << (sizeof(int) + sizeof(char *))];

	if (how == COPY) {
		if ((fd0= open(orig, O_RDONLY)) < 0) {
			report(orig);
			return nil;
		}
	}

	if ((end= strrchr(template, '/')) == nil) end= template; else end++;
	len= end - template;
	temp= allocate(nil, (len + 6 + 3*sizeof(long)) * sizeof(temp[0]));

	for (;;) {
		sprintf(temp, "%.*stangle%lu", len, template, cycle);
		if (how == LINK) {
			if (link(orig, temp) == 0) return temp;
		} else {
			fd1= open(temp, O_WRONLY | O_CREAT | O_EXCL, 0600);
			if (fd1 >= 0) break;
		}
		if (errno != EEXIST) {
			if (how == LINK) {
				report2("ln", orig, temp);
			} else {
				report(temp);
				close(fd0);
			}
			deallocate(temp);
			return nil;
		}
		cycle++;	/* try another name */
	}

	/* Finish the copy, we only created the target. */
	while ((n= read(fd0, buf, sizeof(buf))) > 0) {
		m= 0;
		while (m < n) {
			if ((r= write(fd1, buf, n - m)) < 0) {
				report(temp);
				close(fd0);
				close(fd1);
				deallocate(temp);
				return nil;
			}
			m+= r;
		}
	}
	if (n < 0) {
		report(orig);
		close(fd0);
		close(fd1);
		deallocate(temp);
		return nil;
	}
	close(fd0);
	close(fd1);
	return temp;
}

typedef struct file {
	struct file	*next;
	ino_t		ino;
	dev_t		dev;
	time_t		mtime;
	off_t		size;
	char		*name;
} file_t;

#define HASH_LEN	(64 << (sizeof(int) + sizeof(char *)))

file_t *oldhlist[HASH_LEN];	/* List of files already seen */
file_t *newhlist[HASH_LEN];	/* List of files newly processed */

file_t **searchdi(file_t **hlist, struct stat *stp)
{
	file_t *hp, **hpp;

	hpp= &hlist[(stp->st_mtime ^ stp->st_size) % HASH_LEN];

	while ((hp= *hpp) != nil) {
		if (stp->st_ino == hp->ino && stp->st_dev == hp->dev) break;
		hpp= &hp->next;
	}
	return hpp;
}

int basenamecmp(const char *file1, const char *file2)
{
	const char *b1, *b2;

	if ((b1= strrchr(file1, '/')) == nil) b1= file1; else b1++;
	if ((b2= strrchr(file2, '/')) == nil) b2= file2; else b2++;
	return strcmp(b1, b2);
}

file_t **searchmsn(file_t **hlist, struct stat *stp, const char *name)
{
	file_t *hp, **hpp;

	hpp= &hlist[(stp->st_mtime ^ stp->st_size) % HASH_LEN];

	while ((hp= *hpp) != nil) {
		if (stp->st_mtime == hp->mtime
			&& stp->st_size == hp->size
			&& basenamecmp(name, hp->name) == 0)
			break;
		hpp= &hp->next;
	}
	return hpp;
}

void blocksig(int block)
/* Block, or unblock signals interfering with critial actions. */
{
	sigset_t mask;

	sigemptyset(&mask);
	sigaddset(&mask, SIGHUP);
	sigaddset(&mask, SIGINT);
	sigaddset(&mask, SIGQUIT);
	sigaddset(&mask, SIGTERM);
#ifdef SIGTSTP
	sigaddset(&mask, SIGTSTP);
#endif
	sigprocmask(block ? SIG_BLOCK : SIG_UNBLOCK, &mask, nil);
}

int uflag;			/* -u: Untangle instead? */
int zflag;			/* -0: And make file very old? */
int vflag;			/* -v: Talk about it? */

void tangle_tree(pathname_t *path)
{
	struct stat stb;

	if (lstat(path_name(path), &stb) < 0) {
		if (errno != ENOENT) report(path_name(path));
		return;
	}

	if (S_ISDIR(stb.st_mode)) {
		/* The node is a directory, traverse it. */
		namelist_t *names;
		char *name;
		size_t idx;

		if ((names= collect(path_name(path))) == nil && errno != 0) {
			report(path_name(path));
			return;
		}
		idx= path_length(path);
		while ((name= pop_name(&names)) != nil) {
			path_add(path, name);
			tangle_tree(path);
			path_trunc(path, idx);
		}
	} else
	if (S_ISREG(stb.st_mode)) {
		/* The node is a regular file, see if we can link it. */
		file_t *hp, **hpp;
		char *temp;

		/* Already linked? */
		if (*searchdi(oldhlist, &stb) != nil) return;

		/* Same time, size and name? */
		hp= *(hpp= searchmsn(oldhlist, &stb, path_name(path)));

		if (hp == nil) {
			/* Not in the remembered list.  Put in new list. */
			hpp= searchdi(newhlist, &stb);
			if (*hpp != nil) return;
			hp= allocate(nil, sizeof(*hp));
			hp->name= copystr(path_name(path));
			hp->next= nil;
			hp->ino= stb.st_ino;
			hp->dev= stb.st_dev;
			hp->mtime= stb.st_mtime;
			hp->size= stb.st_size;
			hp->next= *hpp;
			*hpp= hp;
			return;
		}

		/* A remembered file with the same time, size and name.  How
		 * can it and the current file not be the same?  Link them
		 * together.
		 */
		blocksig(1);
		temp= makefile(path_name(path), LINK, hp->name);
		if (temp == nil) {
			blocksig(0);
			return;
		}
		if (rename(temp, path_name(path)) < 0) {
			report2("mv", temp, path_name(path));
			(void) unlink(temp);
			blocksig(0);
			deallocate(temp);
			return;
		}
		blocksig(0);
		deallocate(temp);
		if (vflag) {
			printf("ln %s %s\n", hp->name, path_name(path));
			fflush(stdout);
		}
	}
}

void tangle(char *file)
{
	struct stat stb;
	pathname_t path;
	int i;

	if (lstat(file, &stb) < 0) report(file);

	path_init(&path);
	path_add(&path, file);
	tangle_tree(&path);
	path_drop(&path);

	for (i= 0; i < HASH_LEN; i++) {
		file_t **ohpp, **nhpp, *t;

		ohpp= &oldhlist[i];
		nhpp= &newhlist[i];
		while (*nhpp != nil) {
			t= *nhpp;
			*nhpp= t->next;
			t->next= *ohpp;
			*ohpp= t;
		}
	}
}

void untangle_tree(pathname_t *path)
{
	struct stat stb;

	if (lstat(path_name(path), &stb) < 0) {
		if (errno != ENOENT) report(path_name(path));
		return;
	}

	if (S_ISDIR(stb.st_mode)) {
		/* The node is a directory, traverse it. */
		namelist_t *names;
		char *name;
		size_t idx;

		if ((names= collect(path_name(path))) == nil && errno != 0) {
			report(path_name(path));
			return;
		}
		idx= path_length(path);
		while ((name= pop_name(&names)) != nil) {
			path_add(path, name);
			untangle_tree(path);
			path_trunc(path, idx);
		}
	} else
	if (S_ISREG(stb.st_mode)) {
		/* The node is a regular file, see if we must untangle it. */
		file_t *hp, **hpp;
		char *temp;
		struct utimbuf ub;

		hp= *(hpp= searchdi(oldhlist, &stb));

		/* Previously linked to something already untangled? */
		if (hp != nil) {
			blocksig(1);
			temp= makefile(path_name(path), LINK, hp->name);
			if (temp == nil) {
				blocksig(0);
				return;
			}
			if (rename(temp, path_name(path)) < 0) {
				report2("mv", temp, path_name(path));
				(void) unlink(temp);
				blocksig(0);
				deallocate(temp);
				return;
			}
			blocksig(0);
			deallocate(temp);
			if (vflag) {
				printf("ln %s %s\n", hp->name, path_name(path));
				fflush(stdout);
			}
			return;
		}

		/* No need to untangle if linked to nothing else. */
		if (stb.st_nlink <= 1) {
			if (zflag) {
				ub.actime= stb.st_atime;
				ub.modtime= OLDTIME(stb.st_mtime);
				if (utime(path_name(path), &ub) < 0) {
					report(path_name(path));
					return;
				}
			}
			return;
		}

		/* Remember this file for links inside the tree. */
		hp= allocate(nil, sizeof(*hp));
		hp->name= copystr(path_name(path));
		hp->next= nil;
		hp->ino= stb.st_ino;
		hp->dev= stb.st_dev;
		hp->mtime= stb.st_mtime;
		hp->size= stb.st_size;
		hp->next= *hpp;
		*hpp= hp;

		/* Copy, unlink, rename. */
		blocksig(1);
		temp= makefile(path_name(path), COPY, hp->name);
		if (temp == nil) {
			blocksig(0);
			return;
		}
		ub.actime= time(nil);
		ub.modtime= zflag ? OLDTIME(stb.st_mtime) : stb.st_mtime;

		if (utime(temp, &ub) < 0
			|| (chown(temp, stb.st_uid, stb.st_gid) < 0
							&& errno != EPERM)
			|| chmod(temp, stb.st_mode & 07777) < 0
		) {
			report(temp);
			(void) unlink(temp);
			blocksig(0);
			deallocate(temp);
			return;
		}
		if (rename(temp, path_name(path)) < 0) {
			report2("mv", temp, path_name(path));
			(void) unlink(temp);
			blocksig(0);
			deallocate(temp);
			return;
		}
		blocksig(0);
		deallocate(temp);

		if (vflag) {
			printf("rm&cp %s\n", path_name(path));
			fflush(stdout);
		}
	}
}

void untangle(char *file)
{
	struct stat stb;
	pathname_t path;

	if (lstat(file, &stb) < 0) report(file);

	path_init(&path);
	path_add(&path, file);
	untangle_tree(&path);
	path_drop(&path);
}

void usage(void)
{
	fprintf(stderr, "Usage: tangle [-v] file1 file2 ...\n");
	fprintf(stderr, "       tangle -u [-0v] file ...\n");
	exit(1);
}

int main(int argc, char **argv)
{
	int i;

	i= 1;
	while (i < argc && argv[i][0] == '-') {
		char *opt= argv[i++]+1;

		if (opt[0] == '-' && opt[1] == 0) break;

		while (*opt != 0) switch (*opt++) {
		case 'u':	uflag= 1;	break;
		case '0':	zflag= 1;	break;
		case 'v':	vflag= 1;	break;
		default:	usage();
		}
	}

	if (!uflag) {
		if (zflag) usage();
		if ((argc - i) < 2) usage();

		while (i < argc) tangle(argv[i++]);
	} else {
		if (i == argc) usage();

		while (i < argc) untangle(argv[i++]);
	}
	exit(exstatus);
}
