/*	scsiprobe 1.3 - Probe SCSI devices for info	Author: Kees J. Bot
 *								9 Apr 1993
 */
#define nil	0
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/scsi.h>

#define arraysize(a)	(sizeof(a)/sizeof((a)[0]))
#define arraylimit(a)	((a) + arraysize(a))

void fatal(char *label)
{
	fprintf(stderr, "scsiprobe: %s: %s\n", label, strerror(errno));
	exit(1);
}

/* SCSI Sense key bits. */
#define SENSE_KEY	0x0F	/* The key part. */
#define SENSE_ILI	0x20	/* Illegal block size. */
#define SENSE_EOM	0x40	/* End-of-media. */
#define SENSE_EOF	0x80	/* Filemark reached. */

/* From aha_scsi.c: */

static char *devstr[] = {
	"DISK", "TAPE", "PRINTER", "CPU", "WORM", "CD-ROM",
	"SCANNER", "OPTICAL", "JUKEBOX", "COMMUNICATION"
};

char *str_scsi_sense[] = {
	"NO SENSE INFO", "RECOVERED ERROR", "NOT READY", "MEDIUM ERROR",
	"HARDWARE ERROR", "ILLEGAL REQUEST", "UNIT ATTENTION", "DATA PROTECT",
	"BLANK CHECK", "VENDOR UNIQUE ERROR", "COPY ABORTED", "ABORTED COMMAND",
	"EQUAL", "VOLUME OVERFLOW", "MISCOMPARE", "SENSE RESERVED"
};

typedef unsigned char byte;

/* SCSI Request Sense Information */
typedef struct {
    byte errc;			/* Error Code, Error Class, and Valid bit */
    byte segnum;		/* Segment Number */
    byte key;			/* Sense Key */
#	define SENSE_KEY	0x0F	/* the key portion */
#	define SENSE_RES	0x10	/* reserved bit */
#	define SENSE_ILI	0x20	/* illegal block size */
#	define SENSE_EOM	0x40	/* end-of-media */
#	define SENSE_EOF	0x80	/* filemark reached */
    byte info[4];		/* sense info */
    byte len;			/* additional length */
    byte comspec[4];		/* command specific info */
    byte add_code;		/* additional sense code */
    byte add_qual;		/* additional sense code qualifier */
} sense_t;

/* SCSI Inquiry Information */
typedef struct {
    byte devtype;		/* Peripheral Device Type */
#	define SCSI_DEVDISK	0	/* Direct-access */
#	define SCSI_DEVTAPE	1	/* Sequential-access */
#	define SCSI_DEVPRN	2	/* Printer */
#	define SCSI_DEVCPU	3	/* Processor */
#	define SCSI_DEVWORM	4	/* Write-Once Read-Multiple device */
#	define SCSI_DEVCDROM	5	/* Read-Only Direct-access */
#	define SCSI_DEVMAX	5	/* Last device type we know about */
    byte devqual;		/* Device-Type Qualifier */
#	define scsi_rmb(d)	(((d) & 0x80) != 0)	/* Removable? */
    byte stdver;		/* Version of standard compliance */
#	define scsi_isover(v)   (((v) & 0xC0) >> 6)	/* ISO version */
#	define scsi_ecmaver(v)  (((v) & 0x38) >> 3)	/* ECMA version */
#	define scsi_ansiver(v)  ((v) & 0x07)		/* ANSI version */
    byte format;		/* Response data format */
    byte len;			/* length of remaining info */
    byte reserved[2];
    byte flags;
#	define scsi_sync(f)	(((f) & 0x10) != 0)	/* Sync SCSI? */
    char vendor[8];		/* Vendor name */
    char product[16];		/* Product name */
    char revision[4];		/* Revision level */
    char extra[20];		/* Vendor specific */
} inquiry_t;

void show_hex(int indent, byte *page, int len)
{
	int i, j;

	for (i = 0; i < len; i+= 0x10) {
		printf("\n%*s", indent-1, "");
		for (j = i; j < i + 0x10; j++) {
			printf(j < len ? " %02x" : "   ", page[j]);
		}
		printf("   ");
		for (j = i; j < i + 0x10 && j < len; j++) {
			fputc((unsigned) (page[j] - ' ') < (0x7F - ' ')
				? page[j] : ' ', stdout);
		}
	}
	fputc('\n', stdout);
}

void scsi_report(char *device, sense_t *sense)
{
							/* 7.2.14, 7-35 */
	printf("SCSI sense after error on %s: [7.2.14, 7-35]\n", device);
	printf("  Valid = %d | Error Code = 0x%02x\n",
		sense->errc >> 7, sense->errc & 0x7F);
	printf("  Segment number = %d\n", sense->segnum);
	printf("  EOF = %d | EOM = %d | ILI = %d | Sense Key = 0x%x (%s)\n",
		sense->key >> 7,
		(sense->key >> 6) & 1,
		(sense->key >> 5) & 1,
		sense->key & 0x0F,
		str_scsi_sense[sense->key & 0x0F]);
	printf("  Information = %d\n",
		((long) sense->info[0] << 24) | ((long) sense->info[1] << 16)
		| (sense->info[2] << 8) | sense->info[3]);
	printf("  Additional Sense Length = %d\n", sense->len);
	printf("  Command Specific Information = 0x%02x%02x%02x%02x\n",
		sense->comspec[0], sense->comspec[1],
		sense->comspec[2], sense->comspec[3]);
	printf("  Additional Sense Code & Qualifier = 0x%02x 0x%02x\n",
		sense->add_code, sense->add_qual);
}

void scsi_fatal(char *device, sense_t *sense)
{
	scsi_report(device, sense);
	exit(1);
}

void show_type(int devtype, int subtype, int devspec)
{
	char *what;

	switch ((devtype << 8) | subtype) {		/* 8.3.3, 8-45 */
	case 0x0000:	what= "Disk";	break;
	case 0x0001:	what= "Floppy SS";	break;
	case 0x0002:	what= "Floppy DS";	break;
	case 0x0005:	what= "Floppy 8\" SSSD";	break;
	case 0x0006:	what= "Floppy 8\" DSSD";	break;
	case 0x0009:	what= "Floppy 8\" SSDD";	break;
	case 0x000A:	what= "Floppy 8\" DSDD";	break;
	case 0x000D:	what= "Floppy 5.25\" SSSD";	break;
	case 0x0012:	what= "Floppy 5.25\" DSSD";	break;
	case 0x0016:	what= "Floppy 5.25\" DSDD";	break;
	case 0x001A:	what= "Floppy 5.25\" DSHD";	break;
	case 0x001E:	what= "Floppy 3.5\" DSDD";	break;
	case 0x0040:	what= "12 track Tape";	break;
	case 0x0044:	what= "24 track Tape";	break;
	default:
		what= "??";
	}
	printf("  Medium Type = 0x%02x (%s)\n", subtype, what);
	switch (devtype) {
	case 0x00:					/* 8.3.3, 8-46 */
		printf("  WP = %d (Write %s) [8.3.3, 8-46]\n",
			devspec >> 7, devspec & 0x80 ? "protected" : "enabled");
		printf("  DPOFUA = %d (%sCache)\n",
			(devspec >> 4) & 1, devspec & 0x10 ? "" : "No ");
		break;
	case 0x01:					/* 9.3.3, 9-21 */
		printf("  WP = %d (Write %s) [9.3.3, 9-21]\n",
			devspec >> 7, devspec & 0x80 ? "protected" : "enabled");
		switch ((devspec >> 4) & 0x07) {
		case 0x00:	what= "Write-through";	break;
		case 0x01:	what= "Write-back";	break;
		case 0x02:	what= "Write-back, flush others";	break;
		default:	what= "??";
		}
		printf("  Buffered Mode = 0x%x (%s)\n",
			(devspec >> 4) & 0x07, what);
		printf("  Speed = %d%s\n", devspec & 0x0F,
			(devspec & 0x0F) == 0 ? " (default)" : "");
		break;
	default:
		printf("  Device Specific Parameter = %02x\n", devspec);
	}
}

void show_density(int devtype, int density)
{
	char *what;

	switch ((devtype << 8) | density) {
	case 0x0100:
		what= "Default Density";
		break;
	case 0x0101:					/* 9.3.3, 9-22 */
		what= "1/2\" 9 track 800 bpi reel tape (X3.22-1983)";
		break;
	case 0x0102:
		what= "1/2\" 9 track 1600 bpi reel tape (X3.39-1986)";
		break;
	case 0x0103:
		what= "1/2\" 9 track 6250 bpi reel tape (X3.54-1986)";
		break;
	case 0x0104:
	case 0x0105:
		what= "1/4\" 4/9 track 8000 bpi cartridge tape (X3.136-1986)";
		break;
	case 0x0106:
		what= "1/2\" 9 track 3200 bpi reel tape (X3.157-1987)";
		break;
	case 0x0107:
		what= "1/4\" 4 track 6400 bpi cartridge tape (X3.116-1986)";
		break;
	case 0x0108:
		what= "0.15\" 4 track 8000 bpi cassette tape (X3.158-1987)";
		break;
	case 0x0109:
		what= "1/2\" 18 track 37871 bpi cartridge tape (X3B5/87-099)";
		break;
	case 0x010A:
		what= "1/2\" 22 track 6667 bpi cartridge tape (X3B5/86-199)";
		break;
	case 0x010B:
		what= "1/4\" 4 track 1600 bpi cartridge tape (X3.56-1986)";
		break;
	case 0x010C:
		what= "1/2\" 24 track 12690 bpi cartridge tape (HI-TC1)";
		break;
	case 0x010D:
		what= "1/2\" 24 track 25380 bpi cartridge tape (HI-TC2)";
		break;
	case 0x010F:
		what= "1/4\" 15 track 10000 bpi cartridge tape (QIC-120)";
		break;
	case 0x0110:
		what= "1/4\" 18 track 10000 bpi cartridge tape (QIC-150)";
		break;
	case 0x0111:
		what= "1/4\" 26 track 16000 bpi cartridge tape (QIC-320)";
		break;
	case 0x0112:
		what= "1/4\" 30 track 51667 bpi cartridge tape (QIC-1350)";
		break;
	case 0x0113:
		what= "0.15\" helical scan 61000 bpi cassette tape (X3B5/88-185A)";
		break;
	case 0x0114:
		what= "8mm helical scan 54000 bpi cassette tape (X3B5/88-036)";
		break;
	default:
		what= "??";
	}
	printf("    Density = 0x%02x (%s)\n", density, what);
}

void show_blocks(int devtype, byte *block, int len)
{
	printf("  Block descriptors: (%d bytes)\n", len);

	while (len >= 8) {				/* 7.3.3, 7-63 */
		show_density(devtype, block[0]);
		printf("    Number of Blocks = %ld, Block Size = %ld [7.3.3, 7-63]\n",
			((long) block[1] << 16) | (block[2] << 8) | block[3],
			((long) block[5] << 16) | (block[6] << 8) | block[7]);
		block+= 8;
		len-= 8;
	}
}

unsigned n_notches;
byte notch_page[24];
#define isnotched(page)	(((notch_page+16)[7 - (page) / 8] >> ((page) % 8)) & 1)

void show_1page(int devtype, byte *page)
{
	int i;

	switch (page[0] & 0x3F) {
	case 0x0A:					/* 7.3.3.1, 7-66 */
		printf("    Control Mode Page: [7.3.3.1, 7-66]\n");
		printf("      RLEC = %d\n", page[2] & 1);
		printf("      Queue Algorithm Modifier = 0x%x\n",
			page[3] >> 4);
		printf("      QErr = %d | DQue = %d | EECA = %d\n",
			(page[3] >> 1) & 1,
			page[3] & 1,
			page[4] >> 7);
		printf("      RAENP = %d | UAAENP = %d | EAENP = %d\n",
			(page[4] >> 2) & 1,
			(page[4] >> 1) & 1,
			page[4] & 1);
		printf("      Ready AEN Holdoff Period = %u\n",
			(page[6] << 8) | page[7]);
		break;
	case 0x02:					/* 7.3.3.2, 7-68 */
		printf("    Disconnect-Reconnect Page: [7.3.3.2, 7-68]\n");
		printf("      Buffer Full Ratio = %d\n", page[2]);
		printf("      Buffer Empty Ratio = %d\n", page[3]);
		printf("      Bus Inactivity Limit = %u\n",
			(page[4] << 8) | page[5]);
		printf("      Disconnect Time Limit = %u\n",
			(page[6] << 8) | page[7]);
		printf("      Connect Time Limit = %u\n",
			(page[8] << 8) | page[9]);
		printf("      Maximum Burst Size = %u\n",
			(page[10] << 8) | page[11]);
		printf("      DTDC = 0x%x\n", page[12] & 0x03);
		break;
	case 0x08:					/* 8.3.3.1, 8-48 */
		printf("    Caching Page: [8.3.3.1, 8-48]\n");
		printf("      WCE = %d (Write-%s) | MF = %d | RCD = %d\n",
			(page[2] >> 2) & 1,
			page[2] & 0x04 ? "back" : "through",
			(page[2] >> 1) & 1,
			page[2] & 1);
		printf("      Demand Read Retention Priority = 0x%x\n",
			page[3] >> 4);
		printf("      Write Retention Priority = 0x%x\n",
			page[3] & 0x0F);
		printf("      Disable Pre-fetch Transfer Length = %u\n",
			(page[4] << 8) | page[5]);
		printf("      Minimum Pre-fetch = %u\n",
			(page[6] << 8) | page[7]);
		printf("      Maximum Pre-fetch = %u\n",
			(page[8] << 8) | page[9]);
		printf("      Maximum Pre-fetch Ceiling = %u\n",
			(page[10] << 8) | page[11]);
		break;
	case 0x05:					/* 8.3.3.2, 8-50 */
		printf("    Flexible Disk Page: [8.3.3.2, 8-50]\n");
		printf("      Transfer rate = %u kb/s\n",
			(page[2] << 8) | page[3]);
		printf("      Number of Heads = %d\n", page[4]);
		printf("      Sectors per Track = %d\n", page[5]);
		printf("      Data Bytes per Sector = %u\n",
			(page[6] << 8) | page[7]);
		printf("      Number of Cylinders = %u\n",
			(page[8] << 8) | page[9]);
		printf("      Starting Cylinder-Write Precompensation = %u\n",
			(page[10] << 8) | page[11]);
		printf("      Starting Cylinder-Reduced Write Current = %u\n",
			(page[12] << 8) | page[13]);
		printf("      Drive Step Rate = %u\n",
			(page[14] << 8) | page[15]);
		printf("      Drive Step Pulse Width = %d\n", page[16]);
		printf("      Head Settle Delay = %u\n",
			(page[17] << 8) | page[18]);
		printf("      Motor on Delay = %d\n", page[19]);
		printf("      Motor off Delay = %d\n", page[20]);
		printf("      TRDY = %d | SSN = %d | MO = %d\n",
			page[21] >> 7,
			(page[21] >> 6) & 1,
			(page[21] >> 5) & 1);
		printf("      Step Pulses per Cylinder = %d\n",
			page[22] & 0x0F);
		printf("      Write Compensation = %d\n", page[23]);
		printf("      Head Load Delay = %d\n", page[24]);
		printf("      Head Unload Delay = %d\n", page[25]);
		printf("      Pin 34 | Pin 2 | Pin 4 | Pin 1 = 0x%02x%02x",
			page[26], page[27]);
		printf("      Medium Rotation Rate = %u rpm\n",
			(page[28] << 8) | page[29]);
		break;
	case 0x03:					/* 8.3.3.3, 8-52 */
		printf("    Format Device Page: [8.3.3.3, 8-52]\n");
		printf("      Tracks per Zone = %u\n",
			(page[2] << 8) | page[3]);
		printf("      Alternate Sectors per Zone = %u\n",
			(page[4] << 8) | page[5]);
		printf("      Alternate Tracks per Zone = %u\n",
			(page[6] << 8) | page[7]);
		printf("      Alternate Tracks per Logical Unit = %u\n",
			(page[8] << 8) | page[9]);
		printf("      Sectors per Track = %u\n",
			(page[10] << 8) | page[11]);
		printf("      Data Bytes per Physical Sector = %u\n",
			(page[12] << 8) | page[13]);
		printf("      Interleave = %u\n",
			(page[14] << 8) | page[15]);
		printf("      Track Skew Factor = %u\n",
			(page[16] << 8) | page[17]);
		printf("      Cylinder Skew Factor = %u\n",
			(page[18] << 8) | page[19]);
		printf("      SSEC = %d | HSEC = %d | RMB = %d | SURF = %d\n",
			page[20] >> 7,
			(page[20] >> 6) & 1,
			(page[20] >> 5) & 1,
			(page[20] >> 4) & 1);
		break;
	case 0x0C:					/* 8.3.3.5, 8-54 */
		printf("    Notch Page: [8.3.3.5, 8-54]\n");
		printf("      ND = %d (%sNotched) | LPN = %d\n",
			page[2] >> 7,
			page[2] & 0x80 ? "" : "Not ",
			(page[2] >> 6) & 1);
		printf("      Maximum Number of Notches = %u\n",
			(page[4] << 8) | page[5]);
		printf("      Active Notch = %u\n",
			(page[6] << 8) | page[7]);
		printf("      Starting Boundary = ");
		if (page[2] & 0x40) {
			printf("%lu\n",
				((long) page[8] << 24) | ((long) page[9] << 16)
				| (page[10] << 8) | page[11]);
		} else {
			printf("%lu/%u\n",
				((long) page[8] << 16) | (page[9] << 8)
				| page[10], page[11]);
		}
		printf("      Ending Boundary = ");
		if (page[2] & 0x40) {
			printf("%lu\n",
				((long) page[12] << 24) | ((long) page[13]<<16)
				| (page[14] << 8) | page[15]);
		} else {
			printf("%lu/%u\n",
				((long) page[12] << 16) | (page[13] << 8)
				| page[14], page[15]);
		}
		printf("      Pages Notched = 0x%02x%02x%02x%02x%02x%02x%02x%02x\n",
			page[16], page[17], page[18], page[19],
			page[20], page[21], page[22], page[23]);
		if (page[2] & 0x80) {
			n_notches= (page[4] << 8) | page[5];
			memcpy(notch_page, page, 24);
		}
		break;
	case 0x01:					/* 8.3.3.6, 8-55 */
							/* 9.3.3.4, 9-27 */
		printf("    Read-Write Error Recovery Page: [8.3.3.6, 8-55; 9.3.3.4, 9-27]\n");
		printf("      AWRE = %d | ARRE = %d | TB = %d | RC = %d\n",
			page[2] >> 7,
			(page[2] >> 6) & 1,
			(page[2] >> 5) & 1,
			(page[2] >> 4) & 1);
		printf("      EER = %d | PER = %d | DTE = %d | DCR = %d\n",
			(page[2] >> 3) & 1,
			(page[2] >> 2) & 1,
			(page[2] >> 1) & 1,
			page[2] & 1);
		printf("      Read Retry Count = %d\n", page[3]);
		printf("      Correction Span = %d\n", page[4]);
		printf("      Head Offset Count = %d\n", page[5]);
		printf("      Data Strobe Offset Count = %d\n", page[6]);
		printf("      Write Retry Count = %d\n", page[8]);
		printf("      Recovery Time Limit = %d\n",
			(page[10] << 8) | page[11]);
		break;
	case 0x04:					/* 8.3.3.7, 8-61 */
		printf("    Rigid Disk Drive Geometry Page: [8.3.3.7, 8-61]\n");
		printf("      Number of Cylinders = %lu\n",
			((long) page[2] << 16) | (page[3] << 8) | page[4]);
		printf("      Number of Heads = %d\n", page[5]);
		printf("      Starting Cylinder-Write Precompensation = %lu\n",
			((long) page[6] << 16) | (page[7] << 8) | page[8]);
		printf("      Starting Cylinder-Reduced Write Current = %lu\n",
			((long) page[9] << 16) | (page[10] << 8) | page[11]);
		printf("      Drive Step Rate = %u\n",
			(page[12] << 8) | page[13]);
		printf("      Landing Zone Cylinder = %lu\n",
			((long) page[14] << 16) | (page[15] << 8) | page[16]);
		printf("      RPL = 0x%x\n", page[17] & 0x03);
		printf("      Rotational Offset = %d\n", page[18]);
		printf("      Medium Rotation Rate = %u rpm\n",
			(page[20] << 8) | page[21]);
		break;
	case 0x00:
		printf("    Vendor Specific page.\n");
		break;
	default:
		printf("    Unknown (to me) page:");
		show_hex(6, page + 2, page[1]);
	}
}

void show_pages(int devtype, byte *page, int len, unsigned notch)
{
	if (notch == -1) {
		printf("  Page descriptors: (%d bytes)\n", len);
	} else {
		printf("  Page descriptors at notch %u:\n", notch);
	}

	while (len > 0) {
		if (notch == -1 || isnotched(page[0] & 0x3F)) {
			printf("    PS = %d (%s), Page Code = 0x%02x\n",
				page[0] >> 7,
				page[0] & 0x80 ? "Saveable" : "Volatile",
				page[0] & 0x3F);
			printf("    Page length = %d\n", page[1]);

			show_1page(devtype, page);
		}
		if ((page[0] & 0x3F) == 0) break;  /* last: vendor specific */
		len-= 2 + page[1];
		page+= 2 + page[1];
	}
}

void scsi_mdsense(int fd, char *device)
{
	struct scsicmd scsicmd;
	byte cmd[10];
	inquiry_t inqdata;
	byte buf[8192];
	sense_t sense;
	int devtype;
	int ver;
	int mds_size;
	unsigned notch;

	/* Inquiry, to find out device type and standard conformance. */
							/* 7.2.5, 7-14 */
	cmd[0]= 0x12;
	cmd[1]= 0x00;
	cmd[2]= 0x00;
	cmd[3]= 0x00;
	cmd[4]= sizeof(inqdata);
	cmd[5]= 0x00;
	scsicmd.cmd= cmd;
	scsicmd.cmdlen= 6;
	scsicmd.buf= &inqdata;
	scsicmd.buflen= sizeof(inqdata);
	scsicmd.sense= &sense;
	scsicmd.senselen= sizeof(sense);
	scsicmd.dir= 0;
	if (ioctl(fd, SCIOCCMD, &scsicmd) < 0) scsi_fatal(device, &sense);

	devtype= inqdata.devtype & 0x1F;

	n_notches= 0;

	ver= scsi_ansiver(inqdata.stdver) >= 2 ? 2 : 1;
	mds_size= ver == 2 ? 10 : 6;
	if (mds_size == 10) {
		size_t bl_len;

		/* SCSI-2 mode sense.			7.2.11, 7-27 */
		cmd[0]= 0x5A;
		cmd[1]= 0x00;
		cmd[2]= 0x3F;
		cmd[3]= 0x00;
		cmd[4]= 0x00;
		cmd[5]= 0x00;
		cmd[6]= 0x00;
		cmd[7]= sizeof(buf) >> 8;
		cmd[8]= sizeof(buf) & 0xFF;
		cmd[9]= 0x00;
		scsicmd.cmd= cmd;
		scsicmd.cmdlen= sizeof(cmd);
		scsicmd.buf= buf;
		scsicmd.buflen= sizeof(buf);
		scsicmd.sense= &sense;
		scsicmd.senselen= sizeof(sense);
		scsicmd.dir= 0;
		if (ioctl(fd, SCIOCCMD, &scsicmd) == 0) {
			printf("SCSI device %s, SCSI-2 mode sense: [7.2.11, 7-27]\n", device);
			show_type(devtype, buf[2], buf[3]);
			bl_len= (buf[6] << 8) | buf[7];
			show_blocks(devtype, buf + 8, bl_len);
			show_pages(devtype, buf + 8 + bl_len,
				((buf[0] << 8) | buf[1]) - bl_len - 8, -1);
		} else {
			/* Retry with 6 byte mode sense. */
			mds_size= 6;
		}
	}
	if (mds_size == 6) {
		/* SCSI-2 or SCSI-1 mode sense.		7.2.10, 7-24 */

		cmd[0]= 0x1A;
		cmd[1]= 0x00;
		cmd[2]= ver == 2 ? 0x3F : 0x00;
		cmd[3]= 0x00;
		cmd[4]= 0xFF;
		cmd[5]= 0x00;
		scsicmd.cmd= cmd;
		scsicmd.cmdlen= 6;
		scsicmd.buf= buf;
		scsicmd.buflen= 0xFF;
		scsicmd.sense= &sense;
		scsicmd.senselen= sizeof(sense);
		scsicmd.dir= 0;
		if (ioctl(fd, SCIOCCMD, &scsicmd) < 0)
			scsi_fatal(device, &sense);

		printf("SCSI device %s, SCSI-%d mode sense: [7.2.10, 7-24]\n", device, ver);
		show_type(devtype, buf[1], buf[2]);
		show_blocks(devtype, buf + 4, buf[3]);
		if (ver == 2) {
			show_pages(devtype, buf + 4 + buf[3],
					buf[0] - buf[3] - 8, -1);
		}
	}

	/* Select each notch and show the notch dependend pages. */
	for (notch= 1; n_notches > 0 && notch <= n_notches + 1; notch++) {
		if (notch <= n_notches) {
			/* Set active notch. */
			notch_page[6]= notch >> 8;
			notch_page[7]= notch & 0xFF;
		} else {
			/* Reset to notch 0. */
			notch_page[6]= notch_page[7]= 0;
		}
		if (mds_size == 10) {
			buf[0]= buf[6]= buf[7]= 0;
			memcpy(buf + 8, notch_page, 24);

			/* SCSI-2 mode select.		7.2.11, 7-27 */
			cmd[0]= 0x55;
			cmd[1]= 0x10;
			cmd[2]= 0x00;
			cmd[3]= 0x00;
			cmd[4]= 0x00;
			cmd[5]= 0x00;
			cmd[6]= 0x00;
			cmd[7]= (8 + 24) >> 8;
			cmd[8]= (8 + 24) & 0xFF;
			cmd[9]= 0x00;
			scsicmd.cmd= cmd;
			scsicmd.cmdlen= sizeof(cmd);
			scsicmd.buf= buf;
			scsicmd.buflen= 8 + 24;
			scsicmd.sense= &sense;
			scsicmd.senselen= sizeof(sense);
			scsicmd.dir= 1;
			if (ioctl(fd, SCIOCCMD, &scsicmd) < 0) break;
		} else {
			buf[0]= buf[3]= 0;
			memset(buf, 0, 4);
			memcpy(buf + 4, notch_page, 24);

			/* SCSI-2 mode select.		7.2.8, 7-22 */
			cmd[0]= 0x15;
			cmd[1]= 0x10;
			cmd[2]= 0x00;
			cmd[3]= 0x00;
			cmd[4]= 4 + 24;
			cmd[5]= 0x00;
			scsicmd.cmd= cmd;
			scsicmd.cmdlen= 6;
			scsicmd.buf= buf;
			scsicmd.buflen= 4 + 24;
			scsicmd.sense= &sense;
			scsicmd.senselen= sizeof(sense);
			scsicmd.dir= 1;
			if (ioctl(fd, SCIOCCMD, &scsicmd) < 0) break;
		}

		if (notch > n_notches) break;

		if (mds_size == 10) {
			/* SCSI-2 mode sense.	7.2.11, 7-27 */
			cmd[0]= 0x5A;
			cmd[1]= 0x08;
			cmd[2]= 0x3F;
			cmd[3]= 0x00;
			cmd[4]= 0x00;
			cmd[5]= 0x00;
			cmd[6]= 0x00;
			cmd[7]= sizeof(buf) >> 8;
			cmd[8]= sizeof(buf) & 0xFF;
			cmd[9]= 0x00;
			scsicmd.cmd= cmd;
			scsicmd.cmdlen= sizeof(cmd);
			scsicmd.buf= buf;
			scsicmd.buflen= sizeof(buf);
			scsicmd.sense= &sense;
			scsicmd.senselen= sizeof(sense);
			scsicmd.dir= 0;
			if (ioctl(fd, SCIOCCMD, &scsicmd) < 0)
				scsi_fatal(device, &sense);

			show_pages(devtype, buf + 8,
				((buf[0] << 8) | buf[1]) - 8, notch);
		} else {
			/* SCSI-2 mode sense.	7.2.10, 7-24 */
			cmd[0]= 0x1A;
			cmd[1]= 0x08;
			cmd[2]= 0x3F;
			cmd[3]= 0x00;
			cmd[4]= 0xFF;
			cmd[5]= 0x00;
			scsicmd.cmd= cmd;
			scsicmd.cmdlen= 6;
			scsicmd.buf= buf;
			scsicmd.buflen= 0xFF;
			scsicmd.sense= &sense;
			scsicmd.senselen= sizeof(sense);
			scsicmd.dir= 0;
			if (ioctl(fd, SCIOCCMD, &scsicmd) < 0)
				scsi_fatal(device, &sense);

			show_pages(devtype, buf + 4, buf[0] - 8, notch);
		}
	}
}

void show_1vpd(byte *vpd)
/* Show a Vital Product Data page. */
{
	int i;

	printf("    Qualifier & Device Type = 0x%02x, Page Code = 0x%02x, Length = %d\n",
		vpd[0], vpd[1], vpd[3]);

	if (vpd[1] == 0x82) {				/* 7.3.4.1, 7-73 */
		printf("    ASCII Implemented Operating Definition: [7.3.4.1, 7-73]\n");
		printf("      %.*s\n", vpd[4], (char *) (vpd + 5));
	} else
	if (0x01 <= vpd[1] && vpd[1] <= 0x7F) {		/* 7.3.4.2, 7-74 */
		printf("    ASCII Information Page: [7.3.4.2, 7-74]\n");
		printf("      %.*s\n", vpd[4], (char *) (vpd + 5));
	} else
	if (vpd[1] == 0x81) {				/* 7.3.4.3, 7-75 */
		printf("    Implemented Operating Definition Page: [7.3.4.3, 7-75]\n");
		printf("      Current Operating Definition = 0x%02x\n",
			vpd[4] & 0x7F);
		printf("      SavImp = %d | Default Operating Definition = 0x%02x\n",
			vpd[5] >> 7, vpd[5] & 0x7F);
		for (i = 0; i < vpd[3] - 2; i++) {
			printf("      SavImp = %d | Supported Operating Definition %d = 0x%02x\n",
				vpd[6+i] >> 7, i, vpd[6+i] & 0x7F);
		}
	} else
	if (vpd[1] == 0x80) {				/* 7.3.4.5, 7-77 */
		printf("    Unit Serial Number Page: [7.3.4.5, 7-77]\n");
		printf("      Product Serial Number = %.*s\n",
			vpd[3], (char *) (vpd + 4));
	} else {
		printf("    Unknown (to me) page:");
		show_hex(6, vpd + 4, vpd[3]);
	}
}

void scsi_inquiry(int fd, char *device)
{
	struct scsicmd scsicmd;
	byte cmd[10];
	inquiry_t inqdata;
	byte svpd[255];
	byte vpd[255];
	sense_t sense;
	int type, p;
	char *what;

	/* Prefill with nulls. */
	memset(&inqdata, '\0', sizeof(inqdata));

	/* Do the inquiry.				7.2.5, 7-14 */
	cmd[0]= 0x12;
	cmd[1]= 0x00;
	cmd[2]= 0x00;
	cmd[3]= 0x00;
	cmd[4]= sizeof(inqdata);
	cmd[5]= 0x00;
	scsicmd.cmd= cmd;
	scsicmd.cmdlen= 6;
	scsicmd.buf= &inqdata;
	scsicmd.buflen= sizeof(inqdata);
	scsicmd.sense= &sense;
	scsicmd.senselen= sizeof(sense);
	scsicmd.dir= 0;
	if (ioctl(fd, SCIOCCMD, &scsicmd) < 0) scsi_fatal(device, &sense);

							/* 7.2.5, 7-15 */
	printf("SCSI device %s, Inquiry information: [7.2.5, 7-15]\n", device);
	printf("  Pheripheral Qualifier = %d\n", inqdata.devtype >> 5);
	type= inqdata.devtype & 0x1F;
	printf("  Pheripheral Device Type = 0x%02x (%s)\n", type,
		type > arraysize(devstr) ? "UNKNOWN" : devstr[type]);
	printf("  RMB = %d (%s)\n", scsi_rmb(inqdata.devqual),
		scsi_rmb(inqdata.devqual) ? "Removable" : "Fixed");
	printf("  Device-Type Modifier = 0x%02x\n", inqdata.devqual & 0x7F);
	printf("  ANSI Version = %d (SCSI-%d) | ISO Version = %d | ECMA Version = %d\n",
		scsi_ansiver(inqdata.stdver), scsi_ansiver(inqdata.stdver),
		scsi_isover(inqdata.stdver),
		scsi_ecmaver(inqdata.stdver));
	switch (inqdata.format & 0x0F) {
	case 0:		what= "SCSI-1";		break;
	case 1:		what= "CCS etc.";	break;
	case 2:		what= "SCSI-2";		break;
	default:	what= "???";
	}
	printf("  AENC = %d | TrmIOP = %d | Format = %d (%s)\n",
		inqdata.format >> 7, (inqdata.format >> 6) & 1,
		inqdata.format & 0x0F, what);
	printf("  Additional length = %d\n", inqdata.len);
	if (inqdata.len >= 7 - 4) {
		printf("  RelAdr = %d | WBus32 = %d | WBus16 = %d\n",
			inqdata.flags >> 7,
			(inqdata.flags >> 6) & 1,
			(inqdata.flags >> 5) & 1);
		printf("  Sync = %d (%synchronous transfer)\n",
			(inqdata.flags >> 4) & 1,
			inqdata.flags & 0x10 ? "S" : "As");
		printf("  Linked = %d | CmdQue = %d | SftRe = %d\n",
			(inqdata.flags >> 3) & 1,
			(inqdata.flags >> 1) & 1,
			inqdata.flags & 1);
	}
	if (inqdata.len >= 8 - 4)
		printf("  Vendor:   %.8s\n", inqdata.vendor);
	if (inqdata.len >= 16 - 4)
		printf("  Product:  %.16s\n", inqdata.product);
	if (inqdata.len >= 32 - 4)
		printf("  Revision: %.4s\n", inqdata.revision);
	if (inqdata.len >= 36 - 4)
		printf("  Extra:    %.20s\n", inqdata.extra);

	if (scsi_ansiver(inqdata.stdver) < 2) {
		printf("Note: This is not a SCSI-2 device, further data need not be correct\n");
	}
	printf("  Vital Product Data:\n");

	/* Get the list of supported pages.	7.3.4.4, 7-76 */
	cmd[0]= 0x12;
	cmd[1]= 0x01;
	cmd[2]= 0x00;
	cmd[3]= 0x00;
	cmd[4]= sizeof(svpd);
	cmd[5]= 0x00;
	scsicmd.cmd= cmd;
	scsicmd.cmdlen= 6;
	scsicmd.buf= svpd;
	scsicmd.buflen= sizeof(svpd);
	scsicmd.sense= &sense;
	scsicmd.senselen= sizeof(sense);
	scsicmd.dir= 0;
	if (ioctl(fd, SCIOCCMD, &scsicmd) < 0)
		scsi_report(device, &sense);

	for (p= 0; p < svpd[3]; p++) {
		/* Get each page, except the one above. */
		if (svpd[4+p] == 0x00) continue;
		cmd[0]= 0x12;
		cmd[1]= 0x01;
		cmd[2]= svpd[4+p];
		cmd[3]= 0x00;
		cmd[4]= sizeof(vpd);
		cmd[5]= 0x00;
		scsicmd.cmd= cmd;
		scsicmd.cmdlen= 6;
		scsicmd.buf= vpd;
		scsicmd.buflen= sizeof(vpd);
		scsicmd.sense= &sense;
		scsicmd.senselen= sizeof(sense);
		scsicmd.dir= 0;
		if (ioctl(fd, SCIOCCMD, &scsicmd) < 0)
			scsi_report(device, &sense);

		show_1vpd(vpd);
	}
}

void scsi_strtstp(int fd, char *device, int ss)
{
	struct scsicmd scsicmd;
	byte cmd[10];
	sense_t sense;

	/* Do the start/stop command.			?.?.?, ?-? */
	cmd[0]= 0x1B;
	cmd[1]= 0x00;
	cmd[2]= 0x00;
	cmd[3]= 0x00;
	cmd[4]=   ss;		/* 0 = stop, 1 = start, 2 = stop & eject */
	cmd[5]= 0x00;
	scsicmd.cmd= cmd;
	scsicmd.cmdlen= 6;
	scsicmd.buf= nil;
	scsicmd.buflen= 0;
	scsicmd.sense= &sense;
	scsicmd.senselen= sizeof(sense);
	scsicmd.dir= 0;
	if (ioctl(fd, SCIOCCMD, &scsicmd) < 0) scsi_fatal(device, &sense);
}

void scsi_log(int fd, char *device)
{
	struct scsicmd scsicmd;
	byte cmd[10];
	inquiry_t inqdata;
	byte slog[8192];
	byte log[8192];
	sense_t sense;
	int type, p;
	char *what;

	/* Set up a log sense command.			7.2.7, 7-21 */
	cmd[0]= 0x4D;
	cmd[1]= 0x00;
	cmd[2]= 0x00;	/* get supported log pages */
	cmd[3]= 0x00;
	cmd[4]= 0x00;
	cmd[5]= 0x00;
	cmd[6]= 0x00;
	cmd[7]= sizeof(slog) >> 8;
	cmd[8]= sizeof(slog) & 0xFF;
	cmd[9]= 0x00;
	scsicmd.cmd= cmd;
	scsicmd.cmdlen= sizeof(cmd);
	scsicmd.buf= slog;
	scsicmd.buflen= sizeof(slog);
	scsicmd.sense= &sense;
	scsicmd.senselen= sizeof(sense);
	scsicmd.dir= 0;
	if (ioctl(fd, SCIOCCMD, &scsicmd) != 0) scsi_fatal(device, &sense);
}

void usage(void)
{
	fprintf(stderr, "Usage: scsi device command ...\n");
	exit(1);
}

int main(int argc, char **argv)
{
	char *device;
	char *command;
	int fd;
	int i;

	if (argc < 3) usage();

	device= argv[1];

	if ((fd= open(device, O_RDONLY)) < 0) fatal(device);

	for (i= 2; i < argc; i++) {
		command= argv[i];

		if (strcmp(command, "sense") == 0) {
			scsi_mdsense(fd, device);
		} else
		if (strcmp(command, "inquiry") == 0) {
			scsi_inquiry(fd, device);
		} else
		if (strcmp(command, "start") == 0) {
			scsi_strtstp(fd, device, 1);
		} else
		if (strcmp(command, "stop") == 0) {
			scsi_strtstp(fd, device, 0);
		} else
		if (strcmp(command, "log") == 0) {
			scsi_log(fd, device);
		} else {
			fprintf(stderr, "scsiprobe: unknown command '%s'\n",
				command);
		}
	}
	exit(0);
}
