/*
mkswap.c

Create a swap partition on a block special device.

Created:	Aug 2, 1992 by Philip Homburg

Changed:	Nov 30, 1992 by Kees J. Bot
		To match the mount(8) program.
*/

#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <minix/swap.h>

#if SH_VERSION != 1
#error Swap header version not as expected
#endif

char *prog_name;

void usage(void)
{
	fprintf(stderr,
	"Usage: %s [-fa] [-p priority] <block special> <offset> <size>\n", 
								prog_name);
	exit(1);
}

void main(int argc, char *argv[])
{
	char *special_name;
	char *off_str;
	char *size_str;
	char *priority_str;
	unsigned long offset, size;
	char *check_c, *p_arg;
	int fd, r, c, f_flag, a_flag;
	off_t off;
	char tstchar;
	static char wipe[SWAP_OFFSET];
	static char header_bytes[512];
	swap_hdr_t *hdr;
	long priority;
	
	prog_name= argv[0];

	f_flag= 0;
	a_flag= 0;
	p_arg= NULL;
	while ((c= getopt(argc, argv, "fap:?")) != EOF)
	{
		switch(c)
		{
		case 'f':
			f_flag= 1;
			break;
		case 'a':
			a_flag= 1;
			break;
		case 'p':
			p_arg= optarg;
			break;
		case '?':
			usage();
		default:
			fprintf(stderr, "%s: getopt failed: %c\n", prog_name,
									c);
			exit(1);
		}
	}
			
	if (optind >= argc) usage();		
	special_name= argv[optind++];
	if (optind >= argc) usage();		
	off_str= argv[optind++];
	if (optind >= argc) usage();		
	size_str= argv[optind++];
	if (optind != argc) usage();		
	priority_str= p_arg;

	if (f_flag == a_flag) {
		fprintf(stderr,
		"%s: use -f for a full swap partition, or -x for added swap\n",
			prog_name);
		exit(1);
	}

	/* Parse offset and size */
	offset= strtol(off_str, &check_c, 0);
	if (check_c[0] != '\0')
	{
		fprintf(stderr, "%s: wrong offset '%s'\n", prog_name, off_str);
		exit(1);
	}
	offset *= 1024;				/* The offset is in blocks */

	size= strtol(size_str, &check_c, 0);
	if (check_c[0] != '\0')
	{
		fprintf(stderr, "%s: wrong size '%s'\n", prog_name, size_str);
		exit(1);
	}
	size *= 1024;				/* The size is in blocks */

	/* Let's parse the priority, if any. */
	if (priority_str)
	{
		priority= strtol(priority_str, &check_c, 0);
		if (check_c[0] != '\0')
		{
			fprintf(stderr, "%s: wrong priority '%s'\n", 
						prog_name, priority_str);
			exit(1);
		}
	}
	else
		priority= SH_PRIORITY;
		
	/* Let's open the block special. */
	fd= open(special_name, O_RDWR);
	if (fd == -1)
	{
		fprintf(stderr, "%s: open '%s' failed: %s\n", prog_name, 
						special_name, strerror(errno)); 
		exit(1);
	}
		
	/* Check the offset */
	if (offset < SWAP_OFFSET)
	{
		fprintf(stderr, "%s: offset should more than %d\n", prog_name, 
						SWAP_OFFSET);
		exit(1);
	}
	errno= 0;	/* -1 is a legal value for lseek */
	off= lseek(fd, (off_t)(offset+size)-1, SEEK_SET);
	if (off == (off_t)-1 && errno != 0)
	{
		fprintf(stderr, "%s: lseek %ld failed: %s\n", prog_name, 
					(off_t)(size-1),  strerror(errno));
		exit(1);
	}
	r= read(fd, &tstchar, 1);
	if (r != 1)
	{
		fprintf(stderr, "%s: size check failed: %s\n", prog_name,
			r == -1 ? strerror(errno) : "eof");
		exit(1);
	}

	/* Wipe the device if it is fully swap. */
	if (f_flag) {
		if (lseek(fd, (off_t) 0, SEEK_SET) == -1
				|| write(fd, wipe, sizeof(wipe)) < 0) {
			fprintf(stderr, "%s: wiping device failed: %s\n",
				prog_name, strerror(errno));
			exit(1);
		}
	}

	/* Fill in the header. */
	memset(header_bytes, '\0', sizeof(header_bytes));
	hdr= (swap_hdr_t *) header_bytes;
	hdr->sh_magic[0]= SWAP_MAGIC0;
	hdr->sh_magic[1]= SWAP_MAGIC1;
	hdr->sh_magic[2]= SWAP_MAGIC2;
	hdr->sh_magic[3]= SWAP_MAGIC3;
	hdr->sh_version= SH_VERSION;
	hdr->sh_offset= offset;
	hdr->sh_swapsize= size;
	hdr->sh_priority= priority;
	
	if (f_flag) {
		/* Swap partition. */
		off= SWAP_BOOTOFF;
	} else {
		/* Added swap. */
		off= OPTSWAP_BOOTOFF;
	}
	if (lseek(fd, off, SEEK_SET) == (off_t)-1)
	{
		fprintf(stderr, "%s: lseek %ld failed: %s\n", prog_name, 
					(long) off,  strerror(errno));
		exit(1);
	}
	r= write(fd, header_bytes, sizeof(header_bytes));
	if (r != sizeof(header_bytes))
	{
		fprintf(stderr, "%s: unable to write the header: %s\n", 
				prog_name, r == -1 ? strerror(errno) : "eof");
		exit(1);
	}
	exit(0);
}
