/*	mail 1.0 - mail user agent			Author: Kees J. Bot
 *								30 Apr 1994
 * This is a simple unsophisticated mail reader, the thing
 * that works when you need it.
 *
 * External defines:
 *	CRIPPLE=1		Disable mail reading.
 *	EDOMAIN=string		Name of the email domain.
 */
#define nil 0
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <pwd.h>
#include <limits.h>
#include <time.h>
#if !CRIPPLE
#include <readline/readline.h>
#endif

/* Default mailbox format.  Should be the same as the transport agent uses. */
#ifndef FORMAT
#define FORMAT	MMDF
#endif

/* Name of the mail transport agent, must be sendmail compatible. */
#ifndef MAILER
char MAILER[]= "/usr/lib/sendmail";
#endif

/* Default editor. */
#ifndef EDITOR
#if __minix
char EDITOR[]= "mined";
#else
char EDITOR[]= "vi";
#endif
#endif

/* Default pager. */
#ifndef PAGER
char PAGER[]= "more";
#endif


#define arraysize(a)	(sizeof(a) / sizeof((a)[0]))
#define arraylimit(a)	((a) + arraysize(a))
#define between(a, c, z)  ((unsigned) ((c) - (a)) <= (unsigned) ((z) - (a)))
#define MAX_SIZE	((size_t) ((size_t) -1 < 0 ? \
				(sizeof(size_t) == sizeof(int) ? INT_MAX \
								: LONG_MAX) \
				: (size_t) -1))

typedef enum std_header {	/* Interesting headers. */
	H_TO,
	H_CC,
	H_FROM,
	H_SENDER,
	H_REPLY_TO,
	H_SUBJECT,
	H_DATE,
	H_RESENT_TO,
	H_RESENT_FROM,
	H_RESENT_DATE,
	H_MAX_STD
} std_header_t;

struct mailbox;

typedef struct message {	/* Single mail message. */
	struct mailbox *box;	/* Mailbox this message is found in. */
	int	    flags;	/* Read, modified, saved, deleted, etc. */
	off_t	    offset;	/* Describes the entire message. */
	off_t	    limit;
	off_t	    body;	/* Offset to skip the headers to the body. */
	char	  **headers;	/* Array of headers. */
	size_t	    nhdrs;
	size_t	    std_hdr[H_MAX_STD];	/* References to To:, From:, etc. */
} message_t;

#define M_READ		0x0001	/* Message has been read, etc. */
#define M_EDITED	0x0002
#define M_FORWARDED	0x0004
#define M_REPLIED	0x0008
#define M_DELETED	0x0010
#define M_MOVED		0x0020
#define M_NEW		0x0040
#define M_SENT		0x0080
#define M_SELECTED	0x0100

typedef enum format {
	NOBOX,			/* Not a mailbox, or not tested yet. */
	MBOX,			/* Old and dumb format most mailers use. */
	MMDF			/* Much better. */
} format_t;

typedef struct mailbox {
	char	   *file;	/* File name of the mailbox. */
	int	    draft;	/* File contains one single draft message? */
	FILE	   *fp;		/* File pointer when open. */
	int	    rdonly;	/* Can it be written to? */
	format_t    format;	/* Mailbox format. */
	off_t	    size;	/* Size in bytes after scanning. */
	message_t **messages;	/* The messages in it. */
	size_t	    nmess;
	size_t	    nseen;	/* Number of messages acknowledged. */
	int	    errno;	/* Error code of last operation. */
	struct mailbox *next;	/* List of things to clean up on quit. */
} mailbox_t;

#define NONE_SEEN	(-1)	/* Mailbox not scanned yet. */
#define NO_MESSAGE	(-1)	/* "No message" message number. */
#define NO_HEADER	(-1)	/* You get the picture. */

int	    talk;		/* Interactive. */
mailbox_t  *current;		/* Current open mailbox. */
size_t	    dot;		/* Current message number. */
char	   *home;		/* $HOME */
char	   *editor;		/* $EDITOR */
char	   *pager;		/* $PAGER */

void quit(int excode);		/* Exit with cleanup. */

void *allocate(void *mem, size_t size)
/* Malloc()/realloc() with checking. */
{
	mem= mem == nil ? malloc(size) : realloc(mem, size);
	if (mem == nil && size > 0) {
		fprintf(stderr, "mail: Out of memory\n");
		quit(1);
	}
	return mem;
}

void deallocate(void *mem)
{
	if (mem != nil) free(mem);
}

char *copystr(char *str)
{
	char *copy;

	copy= allocate(nil, (strlen(str) + 1) * sizeof(copy[0]));
	strcpy(copy, str);
	return copy;
}

typedef enum operation { READ, APPEND, WRITE, CLOSE } operation_t;

int lockbox(mailbox_t *box, operation_t op)
/* Obtain a lock on a mailbox. */
{
	struct flock lock;
	int r;

	switch (op) {
	case READ:	lock.l_type= F_RDLCK;	break;
	case APPEND:
	case WRITE:	lock.l_type= F_WRLCK;	break;
	case CLOSE:	lock.l_type= F_UNLCK;	break;
	}
	lock.l_whence= SEEK_SET;
	lock.l_start= 0;
	lock.l_len= 0;

	/* Try without blocking. */
	if (fcntl(fileno(box->fp), F_SETLK, &lock) == 0) return 1;

	/* Tell user and block. */
	printf("Mailbox %s is locked, waiting...", box->file);
	fflush(stdout);
	if ((r= fcntl(fileno(box->fp), F_SETLKW, &lock) < 0)) {
		box->errno= errno;
	}
	fputc('\n', stdout);
	fflush(stdout);
	return r == 0;
}

mailbox_t *allboxes;		/* List of all mailboxes in use. */

mailbox_t *initbox(char *file, int draft)
/* Initialize a mailbox structure. */
{
	mailbox_t *box;

	box= allocate(nil, sizeof(*box));
	box->file= copystr(file);
	box->draft= draft;
	box->fp= nil;
	box->rdonly= 0;
	box->format= NOBOX;
	box->size= 0;
	box->messages= nil;
	box->nmess= 0;
	box->nseen= NONE_SEEN;
	box->errno= 0;
	box->next= allboxes;
	allboxes= box;
	return box;
}

void delbox(mailbox_t *box)
/* Delete a mailbox structure.  Delete the file only if it is a draft. */
{
	mailbox_t **all;
	size_t i;

	for (all= &allboxes; *all != nil; all= &(*all)->next) {
		if (*all == box) {
			*all= box->next;
			break;
		}
	}

	if (box->draft) {
		/* Delete the draft mailbox. */
		if (unlink(box->file) < 0) {
			fprintf(stderr, "mail: can't delete %s: %s\n",
				box->file, strerror(errno));
		}
	}

	/* Delete the message structures. */
	for (i= 0; i < box->nmess; i++) {
		message_t *mess= box->messages[i];
		if (mess->box != box) {
			/* Message is in a draft mailbox. */
			delbox(mess->box);
		} else {
			size_t h;

			for (h= 0; h < mess->nhdrs; h++)
				deallocate(mess->headers[i]);
			deallocate(mess->headers);
			deallocate(mess);
		}
	}
	deallocate(box->messages);
	deallocate(box->file);
}

char *read1boxline(mailbox_t *box, off_t *offset)
/* Read one line from the mailbox file.  Return the offset of the line in
 * the file and a pointer to the line.  Adjust the mailbox size.
 */
{
	static char *line;
	static size_t len;
	size_t i;
	off_t pos= box->size;
	int c;

	*offset= pos;
	i= 0;
	while ((c= getc(box->fp)) != EOF) {
		pos++;
		if (i >= len) {
			line= allocate(line, (len= 2*i+2) * sizeof(line[0]));
		}
		if (c == '\n') break;
		line[i++]= c;
	}
	if (c == EOF) {
		if (ferror(box->fp)) box->errno= errno;
		return nil;
	}
	box->size= pos;
	line[i]= 0;
	return line;
}

int whitespace(int c)
{
	return c == ' ' || c == '\t';
}

int downcase(int c)
/* Mail is case insensitive, but Kees is sensitive to mail. :-) */
{
	return between('A', c, 'Z') ? c - 'A' + 'a' : c;
}

int match(char *pattern, char *line)
/* True if line starts with pattern.  '*' in pattern matches a header. */
{
	int c;

	while (*pattern != 0) {
		if ((c= downcase(*line)) == 0) return 0;

		if (*pattern == '*') {
			if (!between('a', c, 'z') && c != '_') return 0;
			do {
				line++;
				c= downcase(*line);
			} while (between('a', c, 'z') || between('0', c, '9')
					|| c == '_' || c == '-');
			pattern++;
		} else {
			if (downcase(c) != downcase(*pattern)) return 0;
			pattern++;
			line++;
		}
	}
	return 1;
}

int matchfrom(char *line)
/* True iff a line starts with "From<space>". */
{
	return strncmp("From ", line, 5) == 0;
}

int scanbox(mailbox_t *box, operation_t op)
/* Scan a mailbox for messages, among other things.  Leaves the mailbox open. */
{
	enum { SEARCH, HEADER, CONTINUE, BODY } state;
	message_t *mess;
	char **hdr;
	char *line;
	off_t offset;
	struct stat st;
	int fd;

	box->errno= 0;

	if (op == CLOSE) {
		/* Done playing with the mailbox. */
		if (box->fp != nil) {
			if (fclose(box->fp) == EOF) box->errno= errno;
			box->fp= nil;
		}
		return box->errno == 0;
	}

	if (box->fp == nil) {
		/* File not currently open. */

		/* Probe mailbox. */
		if (stat(box->file, &st) < 0
				|| (errno= EISDIR, !S_ISREG(st.st_mode))) {
			if (op == READ || errno != ENOENT) {
				box->errno= errno;
				return 0;
			}
			/* Mailbox doesn't yet exist, create it. */
			if ((fd= open(box->file, O_RDWR|O_CREAT, 0600)) < 0) {
				box->errno= errno;
				return 0;
			}
			close(fd);
		}

		/* Try opening read-write, then read-only. */
		box->rdonly= 0;
		if ((box->fp= fopen(box->file, "r+")) == nil) {
			/* Can't write to the mailbox. */
			box->rdonly= 1;
			if ((box->fp= fopen(box->file, "r")) == nil) {
				/* Can't even read it. */
				box->errno= errno;
				return 0;
			}
		}
	}

	if (op != READ && box->rdonly) {
		box->errno= EACCES;
		return 0;
	}

	if (!lockbox(box, op)) return 0;

	/* New draft? */
	if (box->draft && op == WRITE) return 1;

	/* Seek to the last known position in the mailbox and try to read. */
	if (fseek(box->fp, box->size, SEEK_SET) != 0) {
		box->errno= errno;
		return 0;
	}

	/* Read the mailbox line by line looking for header lines and
	 * message bodies.
	 */
	state= box->draft ? HEADER : SEARCH;
	mess= nil;
	hdr= nil;
	do {
		line= read1boxline(box, &offset);

		if (box->format != MBOX && line != nil
					&& strcmp(line, "\1\1\1\1") == 0) {
			/* MMDF separator found. */

			while ((line= read1boxline(box, &offset)) != nil
					&& strcmp(line, "\1\1\1\1") == 0) {}

			box->format= MMDF;
			state= SEARCH;
			mess= nil;
			hdr= nil;
		} else
		if (box->format != MMDF && line != nil && matchfrom(line)) {
			/* Old mail header found. */
			box->format= MBOX;
			state= SEARCH;
			mess= nil;
			hdr= nil;
		}

		if (line != nil && !box->draft && box->format == NOBOX) {
			/* The first line should have told the mailbox type. */
			box->errno= EINVAL;
			break;
		}

		if (op == APPEND) {
			/* We only want to append a new message. */
			break;
		}

		/* Find reasons to change state. */
		if (line == nil) {
			/* EOF. */
			state= SEARCH;
		} else
		if (state == SEARCH && matchfrom(line)) {
			/* Only the first line may be a "From<space>" line. */
			state= HEADER;
		} else
		if (state != BODY && match("*:", line)) {
			/* A header line. */
			state= HEADER;
		} else
		if ((state == HEADER || state == CONTINUE)
						&& whitespace(line[0])) {
			/* Continuation header line. */
			state= CONTINUE;
		} else {
			/* Anything else is poetry. */
			state= BODY;
		}

		if (state != SEARCH && mess == nil) {
			/* Found a new message. */
			int i;

			mess= allocate(nil, sizeof(*mess));
			box->messages= allocate(box->messages,
				(box->nmess + 1) * sizeof(box->messages[0]));
			box->messages[box->nmess++]= mess;
			mess->box= box;
			mess->flags= 0;
			mess->offset= offset;
		     /* mess->limit= (later) */
			mess->body= offset;
			mess->headers= nil;
			mess->nhdrs= 0;
			for (i= 0; i < arraysize(mess->std_hdr); i++)
				mess->std_hdr[i]= NO_HEADER;
		}

		if (state != CONTINUE && hdr != nil) {
			/* End of a header. */
			hdr= nil;
		}

		if (state == HEADER && hdr == nil) {
			/* Start a new header. */
			std_header_t std;

			mess->headers= allocate(mess->headers,
				(mess->nhdrs + 1) * sizeof(mess->headers[0]));
			hdr= &mess->headers[mess->nhdrs];
			*hdr= copystr(line);
			mess->body= box->size;

			/* Is it a special header? */
			std= H_MAX_STD;
			if (match("To:", line)) std= H_TO;
			if (match("Cc:", line)) std= H_CC;
			if (match("From:", line)) std= H_FROM;
			if (match("Sender:", line)) std= H_SENDER;
			if (match("Reply-To:", line)) std= H_REPLY_TO;
			if (match("Subject:", line)) std= H_SUBJECT;
			if (match("Date:", line)) std= H_DATE;
			if (match("Resent-To:", line)) std= H_RESENT_TO;
			if (match("Resent-From:", line)) std= H_RESENT_FROM;
			if (match("Resent-Date:", line)) std= H_RESENT_DATE;
			if (std != H_MAX_STD
					&& mess->std_hdr[std] == NO_HEADER) {
				/* Record the first of a given header. */
				mess->std_hdr[std]= mess->nhdrs;
			}
			mess->nhdrs++;
		}

		if (state == CONTINUE) {
			/* Enlarge a header with a continuation line. */
			*hdr= allocate(*hdr, (strlen(*hdr) + 1
					+ strlen(line) + 1) * sizeof(**hdr));
			strcat(*hdr, "\n");
			strcat(*hdr, line);
			mess->body= box->size;
		}

		if (state == SEARCH && mess != nil) {
			/* End of a message. */
			mess= nil;
		}

		if (state != SEARCH) {
			/* The message is growing. */
			mess->limit= box->size;
		}
	} while (line != nil);

	if (box->draft && box->nmess == 0) {
		/* File got emptied! */
		mess= allocate(nil, sizeof(*mess));
		box->messages= allocate(box->messages,
						sizeof(box->messages[0]));
		box->messages[box->nmess++]= mess;
		mess->box= box;
		mess->offset= 0;
		mess->limit= 0;
		mess->headers= nil;
		mess->nhdrs= 0;
	}

	if (box->draft) {
		message_t *mess= box->messages[0];
		size_t hdr;
		std_header_t std;
		char *s;

		/* A draft mailbox can only contain one message, but people are
		 * ingenious, and may start lines with "From<space>", or less
		 * likely, add a line with four control-A's.  Let the first
		 * message enclose all.
		 */
		mess->limit= box->messages[box->nmess - 1]->limit;
		/* (The extra messages entries are ignored.) */

		/* Remove empty "fill in" headers. */
		for (hdr= 0; hdr < mess->nhdrs; hdr++) {
			s= strchr(mess->headers[hdr], ':') + 1;
			if (whitespace(*s)) s++;
			if (*s != 0) continue;
			for (std= 0; std < H_MAX_STD; std++) {
				if (mess->std_hdr[std] == hdr)
					mess->std_hdr[std]= NO_HEADER;
				if (mess->std_hdr[std] == NO_HEADER)
					continue;
				if (mess->std_hdr[std] > hdr)
					mess->std_hdr[std]--;
			}
			mess->nhdrs--;
			memmove(mess->headers + hdr,
				mess->headers + hdr + 1,
				(mess->nhdrs - hdr) * sizeof(mess->headers[0]));
			hdr--;
		}
	}

	return box->errno == 0;
}

char *arpa_date(time_t t)
/* Date in the RFC822 format. */
{
	struct tm GTM, *gtm;
	struct tm *ltm;
	static char date[sizeof("ddd, dd mmm yy hh:mm tzn+hh:mm:ss")];
	int tzdiff, tzsign;
	static char *weekday[] = {
		"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat",
	};
	static char *month[] = {
		"Jan", "Feb", "Mar", "Apr", "May", "Jun",
		"Jul", "Aug", "Sep", "Oct", "Nov", "Dec",
	};

	if ((gtm= gmtime(&t)) != NULL) {
		GTM= *gtm;
		gtm= &GTM;
	}
	ltm= localtime(&t);
	if (gtm == NULL) gtm= ltm;
	tzdiff= (ltm->tm_hour*60 + ltm->tm_min)
		- (gtm->tm_hour*60 + gtm->tm_min);
	if (ltm->tm_yday != gtm->tm_yday) {
		if (tzdiff < 0) tzdiff += 24*60; else tzdiff -= 24*60;
	}
	if (tzdiff < 0) {
		tzsign= '-';
		tzdiff= -tzdiff;
	} else {
		tzsign= '+';
	}
	sprintf(date, "%s, %d %s %02d %02d:%02d %c%02d%02d",
		weekday[ltm->tm_wday], ltm->tm_mday,
		month[ltm->tm_mon], ltm->tm_year,
		ltm->tm_hour, ltm->tm_min,
		tzsign, tzdiff / 60, tzdiff % 60);
	return date;
}

void set_date(message_t *mess)
/* Set the date or resent-date on a message that is to be send. */
{
	std_header_t std;
	size_t hdr;
	char date[80];

	std= mess->std_hdr[H_RESENT_TO] != NO_HEADER ? H_RESENT_DATE : H_DATE;
	if (mess->std_hdr[std] == NO_HEADER) {
		mess->headers= allocate(mess->headers,
				(mess->nhdrs + 1) * sizeof(mess->headers[0]));
		mess->headers[mess->nhdrs]= nil;
		mess->std_hdr[std]= mess->nhdrs++;
	}
	strcpy(date, std == H_DATE ? "Date: " : "Resent-Date: ");
	strcat(date, arpa_date(time(nil)));
	hdr= mess->std_hdr[std];
	mess->headers[hdr]= allocate(mess->headers[hdr],
			(strlen(date) + 1) * sizeof(mess->headers[hdr][0]));
	strcpy(mess->headers[hdr], date);
}

int m_pid;
char *m_name;

FILE *start_mailer(char **argv)
/* Start the mail sending program named in argv.  Ye basic pipe(), fork(),
 * exec().
 */
{
	int pfd[2];
	FILE *pfp;

	if (pipe(pfd) < 0) {
		fprintf(stderr, "mail: %s\n", strerror(errno));
		return 0;
	}

	if ((m_pid= fork()) < 0) {
		fprintf(stderr, "mail: can't fork: %s\n", strerror(errno));
		close(pfd[0]);
		close(pfd[1]);
		return nil;
	}

	if (m_pid == 0) {
		/* Child process; start the mailer. */
		close(pfd[1]);
		dup2(pfd[0], 0);
		close(pfd[0]);

		execvp(argv[0], argv);
		fprintf(stderr, "mail: %s: %s\n", argv[0], strerror(errno));
		_exit(127);
	}
	/* Parent process; return an output "file." */
	close(pfd[0]);
	if ((pfp= fdopen(pfd[1], "w")) == nil) {
		fprintf(stderr, "mail: %s\n", strerror(errno));
	}
	return pfp;
}

int stop_mailer(void)
/* Wait for the mail sending program to expire. */
{
	int status;

	if (waitpid(m_pid, &status, 0) < 0) {
		fprintf(stderr, "mail: %s\n", strerror(errno));
		return 0;
	}

	if (status != 0) {
		/* Mailer died. */
		if (WIFSIGNALED(status)) {
			fprintf(stderr, "mail: %s died by signal %d\n",
						m_name, WTERMSIG(status));
		}
	}
	return status == 0;
}

enum what_transport { T_MAIL, T_PAGE };

int transport(enum what_transport how, mailbox_t *box, size_t n)
/* Send out or display a message. */
{
	message_t *mess;
	char *smallv[5];
	char **mailv= smallv;
	size_t i;
	struct passwd *pw;
	void (*hsav)(int), (*isav)(int), (*qsav)(int), (*psav)(int);
	void (*tsav)(int);
	pid_t pid;
	int pfd[2];
	FILE *out;
	off_t length;
	int c;
	int status;
	int ok= 1;
	int direct= 0;

	mess = box->messages[n];

	if (how == T_MAIL) {
		/* Start the mailer and let it figure it out (-t). */
		mailv[0]= MAILER;
		mailv[1]= "-oi";
		mailv[2]= "-t";
		mailv[3]= nil;
	} else {
		if (!talk) {
			/* Display a message directly. */
			out= stdout;
			direct= 1;
		} else {
			/* Use $PAGER. */
			mailv[0]= pager;
			mailv[1]= nil;
		}
	}

	if (!direct) {
		if ((out= start_mailer(mailv)) == nil) ok= 0;

		psav= signal(SIGPIPE, SIG_IGN);
		if (talk) {
			hsav= signal(SIGHUP, SIG_IGN);
			isav= signal(SIGINT, SIG_IGN);
			qsav= signal(SIGQUIT, SIG_IGN);
			tsav= signal(SIGTERM, SIG_IGN);
		}
	}

	if (ok && mess != nil) {
		/* Mail a prepared message. */
		if (how == T_PAGE && talk) {
			fprintf(out, "Message #%lu:\n",
					(unsigned long) (n + 1));
		}
		if (how == T_MAIL) {
			set_date(mess);
		}
		box= mess->box;
		box->errno= 0;
		for (i= 0; i < mess->nhdrs; i++) {
			if (matchfrom(mess->headers[i])) continue;
			fprintf(out, "%s\n", mess->headers[i]);
		}

		if (ok && (box->fp= fopen(box->file, "r")) == nil) {
			box->errno= errno;
			ok= 0;
		}

		if (ok) (void) fseek(box->fp, mess->body, SEEK_SET);
		length= mess->limit - mess->body;
		while (ok && length > 0 && (c= getc(box->fp)) != EOF) {
			if (putc(c, out) == EOF) ok= 0;
			length--;
		}
		if (ok && ferror(box->fp)) {
			box->errno= errno;
			ok= 0;
		}
		if (box->fp != nil) fclose(box->fp);
		box->fp= nil;
	}

	if (!direct) {
		if (out != nil) fclose(out);
		if (!stop_mailer()) ok= 0;

		(void) signal(SIGPIPE, psav);
		if (talk) {
			(void) signal(SIGHUP, hsav);
			(void) signal(SIGINT, isav);
			(void) signal(SIGQUIT, qsav);
			(void) signal(SIGTERM, tsav);
		}
	}
	return ok;
}

char *myaddress(void)
/* My email address. */
{
	static int init= 0;
	static char *address;

	if (!init) {
		struct passwd *pw;
		char host[256];
		char *edomain;

		if ((pw= getpwuid(getuid())) == nil) return nil;

		if ((edomain= getenv("EDOMAIN")) == nil) {
#ifdef EDOMAIN
			edomain= EDOMAIN;
#else
			if (gethostname(host, sizeof(host)) == -1) return nil;
			edomain= host;
#endif
		}
		address= allocate(nil, (strlen(pw->pw_name) + 1
						+ strlen(edomain) + 1)
						* sizeof(address[0]));
		strcpy(address, pw->pw_name);
		strcat(address, "@");
		strcat(address, edomain);
		init= 1;
	}
	return address;
}

char *i_address, *i_name;

int bracketed_address(char *header)
/* Interpret a "name" <address> line. */
{
	int q= 0;
	int p= 0;
	int s= 0;
	size_t aidx= 0, alen= 0;
	size_t nidx= 0, nlen= 0;

	while (*header != 0) {
		if (!q && !p && !s && whitespace(*header)) {
			header++;
			continue;
		}
		if (!q && !p && !s && *header == '"') {
			q= 1;
			header++;
			continue;
		}
		if (!q && !s && *header == '(') {
			p++;
			header++;
			continue;
		}
		if (!p && !s && *header == '\\') {
			s= 1;
			header++;
			continue;
		}
		if (q && !s && *header == '"') {
			q= 0;
			header++;
			continue;
		}
		if (p && *header == ')') {
			p--;
			header++;
			continue;
		}
		if (*header == '\n') {
			/* Continuation line? */
			header++;
			continue;
		}
		if (!q && !p && !s && *header == '<') {
			/* An address! */
			header++;
			while (*header != '>' && *header != 0) {
				if (aidx >= alen) {
					i_address= allocate(i_address,
						(alen= 2*aidx+2)
						* sizeof(i_address[0]));
				}
				i_address[aidx++]= downcase(*header++);
			}
			if (*header++ != '>') {
				aidx= 0;
				break;
			}
			continue;
		}
		s= 0;

		/* A character for the name! */
		if (nidx >= nlen) {
			i_name= allocate(i_name, (nlen= 2*nidx+2)
						* sizeof(i_name[0]));
		}
		i_name[nidx++]= *header++;
	}
	if (aidx == 0) return 0;
	i_address= allocate(i_address, (aidx+1) * sizeof(i_address[0]));
	i_address[aidx]= 0;
	i_name= allocate(i_name, (nidx+1) * sizeof(i_name[0]));
	i_name[nidx]= 0;
	return 1;
}

int normal_address(char *header)
/* Interpret a address (name) line. */
{
	int p= 0;
	size_t aidx= 0, alen= 0;
	size_t nidx= 0, nlen= 0;

	while (*header != 0) {
		if (!p && whitespace(*header)) {
			header++;
			continue;
		}
		if (*header == '(') {
			p++;
			header++;
			continue;
		}
		if (p && *header == ')') {
			p--;
			header++;
			continue;
		}
		if (*header == '\n') {
			/* Continuation line? */
			header++;
			continue;
		}
		if (!p) {
			/* A character for the address. */
			if (aidx >= alen) {
				i_address= allocate(i_address, (alen= 2*aidx+2)
						* sizeof(i_address[0]));
			}
			i_address[aidx++]= downcase(*header++);
		} else {
			/* A character for the name. */
			if (nidx >= nlen) {
				i_name= allocate(i_name, (nlen= 2*nidx+2)
						* sizeof(i_name[0]));
			}
			i_name[nidx++]= *header++;
		}
	}
	if (aidx == 0) return 0;
	i_address= allocate(i_address, (aidx+1) * sizeof(i_address[0]));
	i_address[aidx]= 0;
	i_name= allocate(i_name, (nidx+1) * sizeof(i_name[0]));
	i_name[nidx]= 0;
	return 1;
}

int interpret_address(char *header, char **address, char **name)
/* Losely interpret a To:, From: or whatever mail address.  Only to have
 * something to show the user.  Expect something like
 *	"Kees J. Bot" <kjb@cs.vu.nl>, or
 *	kjb@cs.vu.nl (Kees J. Bot)
 * Return true if interpretation succeeds.
 */
{
	/* Try the <angle bracket> form, then the normal form. */
	if (bracketed_address(header) || normal_address(header)) {
		*address= i_address;
		*name= i_name;
		return 1;
	}
	return 0;
}

int ismyaddress(char *address)
/* True if address is the mail address of the caller of this program. */
{
	static int init= 0;
	static char *name;		/* kjb */
	static char *name_at_domain;	/* kjb@cs.vu.nl */
	static char *name_at_host;	/* kjb@hornet.cs.vu.nl */

	if (!init) {
		struct passwd *pw;
		char host[256], domain[256];

		init= 1;
		if ((pw= getpwuid(getuid())) == nil) return 0;

		name= allocate(nil, (strlen(pw->pw_name) + 1)
							* sizeof(name[0]));
		strcpy(name, pw->pw_name);

#ifndef EDOMAIN
		if (getdomainname(domain, sizeof(domain)) != -1) {
			name_at_domain= allocate(nil, (strlen(pw->pw_name) + 1
						+ strlen(domain) + 1)
						* sizeof(name_at_domain[0]));
			strcpy(name_at_domain, pw->pw_name);
			strcat(name_at_domain, "@");
			strcat(name_at_domain, domain);
		}
		if (gethostname(host, sizeof(host)) != -1) {
			name_at_host= allocate(nil, (strlen(pw->pw_name) + 1
						+ strlen(host) + 1)
						* sizeof(name_at_host[0]));
			strcpy(name_at_host, pw->pw_name);
			strcat(name_at_host, "@");
			strcat(name_at_host, host);
		}
#endif
	}
	if (name != nil && strcmp(address, name) == 0)
		return 1;
	if (name_at_domain != nil && strcmp(address, name_at_domain) == 0)
		return 1;
	if (name_at_host != nil && strcmp(address, name_at_host) == 0)
		return 1;
	if (myaddress() != nil && strcmp(address, myaddress()) == 0)
		return 1;
	return 0;
}

char *expand(char *file)
/* Tilde expansion, current working dir expansion. */
{
	static char *path;
	char cwd[2048];

	if (file[0] == '~' && (file[1] == 0 || file[1] == '/')) {
		path= allocate(path, (strlen(home) + strlen(file + 1) + 1)
						* sizeof(path[0]));
		strcpy(path, home);
		strcat(path, file + 1);
	} else
	if (file[0] != '/') {
		if (getcwd(cwd, sizeof(cwd)) == nil)
			strcpy(cwd, ".");
		path= allocate(path, (strlen(cwd) + 1 + strlen(file) + 1)
						* sizeof(path[0]));
		strcpy(path, home);
		strcat(path, "/");
		strcat(path, file);
	} else {
		path= allocate(path, (strlen(file) + 1) * sizeof(path[0]));
		strcpy(path, file);
	}
	return path;
}

int scanerrors(void)
/* Scan all mailboxes for errors. */
{
	int ok= 1;
	mailbox_t *box;

	for (box= allboxes; box != nil; box= box->next) {
		if (box->errno != 0) {
			fprintf(stderr, "mail: %s: %s\n",
				box->file, strerror(box->errno));
			box->errno= 0;
			ok= 0;
		}
	}
	return ok;
}

char *read1line(char *file, FILE *fp)
/* Read one line from user input. */
{
	static char *line;
	static size_t len;
	size_t i;
	int c;

	i= 0;
	while ((c= getc(fp)) != EOF) {
		if (i >= len) {
			line= allocate(line, (len= 2*i+2) * sizeof(line[0]));
		}
		if (c == '\n') break;
		line[i++]= c;
	}
	if (c == EOF) {
		if (ferror(fp)) {
			fprintf(stderr, "mail: %s: %s\n", file,
							strerror(errno));
		}
		return nil;
	}
	line[i]= 0;
	return line;
}

typedef struct range {		/* Range of message numbers. */
	size_t	    start;
	size_t	    end;
} range_t;

typedef struct command {	/* A command: message list + word list. */
	range_t	   *ranges;
	size_t	    nranges;
	char	  **words;
	size_t	    nwords;
} command_t;

int messaddrchar(int c)
{
	return c != 0 && strchr("01234567890.+-$%", c) != nil;
}

int parsenumber(char **line, size_t *num)
/* Parse a simple number. */
{
	size_t d, n;

	if (!between('0', **line, '9')) return 0;
	n= 0;
	do {
		d= *(*line)++ - '0';
		n= n > MAX_SIZE / 10 ? MAX_SIZE : n * 10 + d;
		if (n < d) n= MAX_SIZE;
	} while (between('0', **line, '9'));
	*num= n;
	return 1;
}

int parseaddress(char **line, size_t *addr)
/* Parse a single message address "a+b". */
{
	int sign;
	size_t num;

	if (**line == '.' || **line == '+' || **line == '-') {
		*addr= dot;
		if (**line == '.') (*line)++;
	} else
	if (**line == '$') {
		*addr= current->nmess - 1;
		(*line)++;
	} else {
		if (!parsenumber(line, addr)) return 0;
		if (*addr > 0) (*addr)--;
	}
	if (**line == '+' || **line == '-') {
		sign= *(*line)++;
		if (!between('0', **line, '9')) {
			num= 1;
		} else {
			(void) parsenumber(line, &num);
		}
		if (sign == '+') {
			*addr= *addr > MAX_SIZE - num ? MAX_SIZE : *addr + num;
		} else {
			*addr= num > *addr ? 0 : *addr - num;
		}
	}
	return 1;
}

int parserange(char **line, range_t *range)
/* Parse a range "a,b". */
{
	if (**line == '%') {
		range->start= 0;
		range->end= MAX_SIZE;
		(*line)++;
	} else {
		if (!parseaddress(line, &range->start)) return 0;
		if (**line != ',') {
			range->end= range->start;
		} else {
			(*line)++;
			if (!parseaddress(line, &range->end)) return 0;
		}
	}
	return 1;
}

int parsecommand(char *line, command_t *cmd)
/* Parse a command line for message ranges and command words. */
{
	char *word;
	size_t i;

	cmd->nranges= 0;
	for (i= 0; i < cmd->nwords; i++) deallocate(cmd->words[i]);
	cmd->nwords= 0;

	/* Parse a list of message numbers. */
	if (messaddrchar(*line)) {
		for (;;) {
			cmd->ranges= allocate(cmd->ranges,
				(cmd->nranges + 1) * sizeof(cmd->ranges[0]));

			if (!parserange(&line, &cmd->ranges[cmd->nranges]))
				return 0;
			cmd->nranges++;

			if (*line != ';') break;
			line++;
		}
	}

	/* Word are things between whitespace. */
	for (;;) {
		while (whitespace(*line)) line++;
		if (*line == 0 || *line == '#') break;
		word= line;
		while (*line != 0 && !whitespace(*line)) line++;
		cmd->words= allocate(cmd->words,
				(cmd->nwords + 1) * sizeof(cmd->words[0]));
		cmd->words[cmd->nwords]= allocate(nil,
				(line - word + 1) * sizeof(cmd->words[0][0]));
		memcpy(cmd->words[cmd->nwords], word, line - word);
		cmd->words[cmd->nwords][line - word]= 0;
		cmd->nwords++;
	}
	return 1;
}

char *standard_header(message_t *mess, std_header_t h, int skip)
/* Return the value of a standard header.  "Skip" tells if the header prefix
 * must be skipped.
 */
{
	char *hdr;

	if (mess->std_hdr[h] == NO_HEADER) return nil;
	hdr= mess->headers[mess->std_hdr[h]];
	if (skip) {
		hdr= strchr(hdr, ':');
		do hdr++; while (whitespace(*hdr));
	}
	return hdr;
}

mailbox_t *make_draft(message_t *mess, int how)
/* Make a draft message using message "mess" as starting point, return the
 * mailbox the draft is put in.  "How" tells if it is to be used for a reply,
 * to be forwarded, resend, or simply edited.
 */
{
	char *file;
	mailbox_t *draft;
	char *hdr, *address, *name;
	int ok= 1;

	file= allocate(nil, (strlen(home) + 1 + 12 + 1) * sizeof(*file));
	strcpy(file, home);
	strcat(file, "/");
	strcat(file, ".draftXXXXXX");
	if (mktemp(file) == nil) {
		fprintf(stderr, "mail: %s: %s\n", file, strerror(errno));
		deallocate(file);
		return nil;
	}
	draft= initbox(file, 1);
	deallocate(file);
	if (!scanbox(draft, WRITE)) {
		scanerrors();
		delbox(draft);
		return nil;
	}

	switch (how) {
	case 'n':
		/* New message. */
		fprintf(draft->fp, "To: \nCc: \nSubject: \n\n");
		break;
	case 'r':
		fprintf(draft->fp, "Resent-To: \nResent-Cc: \n");
		break;
	case 'e':
		break;
	case 'a':
		if ((hdr= standard_header(mess, H_FROM, 1)) == nil) hdr= "";
		fprintf(draft->fp, "To: %s\n", hdr);
		if ((hdr= standard_header(mess, H_CC, 1)) == nil) hdr= "";
		fprintf(draft->fp, "Cc: %s\n", hdr);
		if ((hdr= standard_header(mess, H_SUBJECT, 1)) == nil) hdr= "";
		fprintf(draft->fp, "Subject: %s%s\n",
			match("Re:", hdr) ? "" : "Re: ", hdr);
		if ((hdr= standard_header(mess, H_FROM, 1)) == nil) hdr= "You";
		if (interpret_address(hdr, &address, &name)) {
			if (*name != 0) hdr= name;
		}
		fprintf(draft->fp, "\n%s wrote", hdr);
		if ((hdr= standard_header(mess, H_DATE, 1)) != nil) {
			fprintf(draft->fp, " on %s", hdr);
		}
		fprintf(draft->fp, ":\n");
		break;
	default:
		abort();
	}
	if (ferror(draft->fp)) { draft->errno= errno; ok= 0; }

	if (mess != nil) {
		/* Copy a message for editing, resending, ... */
		mailbox_t *box;
		int c;
		size_t i;
		off_t length;
		int mark;

		box= mess->box;
		for (i= 0; how != 'a' && i < mess->nhdrs; i++) {
			if (how == 'r' && match("Resent-", mess->headers[i]))
				continue;
			fprintf(draft->fp, "%s\n", mess->headers[i]);
			if (ferror(draft->fp)) { draft->errno= errno; ok= 0; }
		}

		if (ok && (box->fp= fopen(box->file, "r")) == nil) {
			box->errno= errno;
			ok= 0;
		}

		if (ok) (void) fseek(box->fp, mess->body, SEEK_SET);
		length= mess->limit - mess->body;
		mark= (how == 'a');
		while (ok && length > 0 && (c= getc(box->fp)) != EOF) {
			if (mark) putc('>', draft->fp);
			putc(c, draft->fp);
			if (ferror(draft->fp)) { draft->errno= errno; ok= 0; }
			length--;
			mark= (how == 'a' && c == '\n');
		}
		if (ok && ferror(box->fp)) { box->errno= errno; ok= 0; }
		if (box->fp != nil) fclose(box->fp);
		box->fp= nil;
	}
	if (ok) ok= scanerrors();
	if (ok) ok= scanbox(draft, CLOSE);
	if (!ok) {
		scanerrors();
		delbox(draft);
		return nil;
	}
	return draft;
}

int edit(char *file)
/* Invoke the editor on a file.  Return true iff the file is updated. */
{
	time_t old_mtime;
	struct stat st;
	struct sigaction isig, qsig;
	int pid;

	old_mtime=  stat(file, &st) < 0 ? -1 : st.st_mtime;

	if ((pid= fork()) < 0) {
		fprintf(stderr, "mail: can't fork: %s\n", strerror(errno));
		return 0;
	}

	if (pid == 0) {
		/* Child process; start the editor. */
		execlp(editor, editor, file, (char *) nil);
		fprintf(stderr, "mail: %s: %s\n", editor, strerror(errno));
		_exit(127);
	}

	(void) sigaction(SIGINT, nil, &isig);
	(void) sigaction(SIGQUIT, nil, &qsig);
	if (waitpid(pid, nil, 0) < 0) {
		fprintf(stderr, "mail: %s\n", strerror(errno));
	}
	(void) sigaction(SIGINT, &isig, nil);
	(void) sigaction(SIGQUIT, &qsig, nil);

	return (stat(file, &st) < 0 ? -1 : st.st_mtime) != old_mtime;
}

#define do_help		nil

int do_del_undel(size_t argc, char **argv, size_t n)
{
	if (argc != 1) {
		fprintf(stderr, "Usage: (.) d\n");
		return 0;
	}
	if (n == NO_MESSAGE) return 1;

	if (argv[0][0] == 'd') {
		current->messages[n]->flags|= M_DELETED;
	} else {
		current->messages[n]->flags&= ~M_DELETED;
	}
	return 1;
}

int do_edit_etc(size_t argc, char **argv, size_t n)
{
	mailbox_t *newbox;
	message_t *newmess;
	mailbox_t *oldbox;

	if (argc != 1) {
		fprintf(stderr, "Usage: (.) %s\n", argv[0]);
		return 0;
	}
	if (n == NO_MESSAGE) return 1;

	if ((newbox= make_draft(current->messages[n], argv[0][0])) == nil)
		return 0;
	if (!edit(newbox->file)) {
		if (argv[0][0] != 'e' && talk) {
			fprintf(stderr, "Draft message discarded\n");
		}
		delbox(newbox);
		return 1;
	}
	if (!scanbox(newbox, READ)) {
		scanerrors();
		delbox(newbox);
		return 0;
	}

	switch (argv[0][0]) {
	case 'e':
		newbox->messages[0]->flags= current->messages[n]->flags
							| M_EDITED | M_READ;
		oldbox= current->messages[n]->box;
		current->messages[n]= newbox->messages[0];
		if (oldbox->draft) delbox(oldbox);
	case 'r':
	case 'a':
		dot= n+1;
		current->messages= allocate(current->messages,
			(current->nmess + 1) * sizeof(current->messages[0]));
		memmove(current->messages + dot + 1,
			current->messages + dot,
			(current->nmess - dot) * sizeof(current->messages[0]));
		current->messages[dot]= newbox->messages[0];
		current->nmess++;
		current->nseen++;
		current->messages[n]->flags|= M_READ
			| (argv[0][0] == 'r' ? M_FORWARDED : M_REPLIED);
		current->messages[dot]->flags|= M_NEW | M_EDITED | M_READ;
	}
	return 1;
}

#define do_forward	nil

int do_header(size_t argc, char **argv, size_t n)
{
	message_t *mess;
	char *date, *party, *subject;
	char *address, *name;
	char flags[16];

	if (argc != 1) {
		fprintf(stderr, "Usage: (.) h\n");
		return 0;
	}
	if (n == NO_MESSAGE) return 1;

	mess= current->messages[n];
	date= standard_header(mess, H_DATE, 1);
	party= standard_header(mess, H_FROM, 1);
	subject= standard_header(mess, H_SUBJECT, 1);

	if (date != nil) {
		while (*date != 0 && *date != ' ') date++;
		while (*date == ' ') date++;
	}

	if (party == nil || (interpret_address(party, &address, &name)
						&& ismyaddress(address))) {
		char *to;
		if ((to= standard_header(mess, H_TO, 0)) != nil) party= to;
	}

	memset(flags, ' ', sizeof(flags));
	if (mess->flags & M_READ)	flags[0]= 'R';
	if (mess->flags & M_EDITED)	flags[0]= 'E';
	if (mess->flags & M_FORWARDED)	flags[1]= 'F';
	if (mess->flags & M_REPLIED)	flags[1]= 'A';
	if (mess->flags & M_DELETED)	flags[2]= 'D';
	if (mess->flags & M_MOVED)	flags[3]= 'M';
	if (mess->flags & M_NEW)	flags[4]= 'N';
	if (mess->flags & M_SENT)	flags[4]= 'S';

	printf("%4lu %.5s  %-9.9s  %-20.20s  %.33s\n",
		(unsigned long) (n+1),
		flags,
		date == nil ? "" : date,
		party == nil ? "" : party,
		subject == nil ? "" : subject);
	return 1;
}

#define do_move		nil	/* move to box */

int do_new(size_t argc, char **argv, size_t n)
{
	mailbox_t *newbox;
	message_t *newmess;

	if (argc != 1) {
		fprintf(stderr, "Usage: n\n");
		return 0;
	}
	if (n != NO_MESSAGE) return 1;

	if ((newbox= make_draft(nil, 'n')) == nil) return 0;
	if (!edit(newbox->file)) {
		delbox(newbox);
		return 1;
	}
	if (!scanbox(newbox, READ)) {
		scanerrors();
		delbox(newbox);
		return 0;
	}
	current->messages= allocate(current->messages,
		(current->nmess + 1) * sizeof(current->messages[0]));
	dot= current->nmess++;
	current->nseen++;
	current->messages[dot]= newbox->messages[0];
	current->messages[dot]->flags= M_NEW | M_EDITED | M_READ;
	return 1;
}

#define do_open		nil	/* open new box */

int do_print(size_t argc, char **argv, size_t n)
{
	if (argc != 1) {
		fprintf(stderr, "Usage: (.) p\n");
		return 0;
	}
	if (n == NO_MESSAGE) return 1;
	(void) transport(T_PAGE, current, n);
	current->messages[n]->flags|= M_READ;
	dot= n;
	return 1;
}

int do_send(size_t argc, char **argv, size_t n)
{
	if (argc != 1) {
		fprintf(stderr, "Usage: (.) s\n");
		return 0;
	}
	if (n == NO_MESSAGE) return 1;
	if (transport(T_MAIL, current, n)) {
		if (talk) printf("Message %lu sent\n", (unsigned long) (n + 1));
		current->messages[n]->flags|= M_SENT;
	} else {
		if (talk) printf("Sending message %lu failed\n",
						(unsigned long) (n + 1));
	}
	return 1;
}

int do_cd(size_t argc, char **argv, size_t n)
{
	char *dir;

	if (argc > 2) {
		fprintf(stderr, "Usage: cd [directory]\n");
		return 0;
	}
	if (n != NO_MESSAGE) return 1;

	dir= argc == 2 ? argv[1] : home;
	if (chdir(dir[0] == '~' ? expand(dir) : dir) < 0) {
		fprintf(stderr, "cd: %s: %s\n", dir, strerror(errno));
		return 0;
	}
	return 1;
}

int write_box(mailbox_t *box)
{
	char *file;
	mailbox_t *tmpbox, *msgbox;
	size_t i, j;
	int ok= 1;

	if (!scanbox(box, WRITE)) {
		scanerrors();
		return 0;
	}
	if (box->nmess != box->nseen) {
		fprintf(stderr, "mail: %s is active, write aborted\n");
		scanbox(box, CLOSE);
		return 0;
	}
	file= allocate(nil, (strlen(home) + 1 + 12 + 1) * sizeof(*file));
	strcpy(file, home);
	strcat(file, "/");
	strcat(file, ".draftXXXXXX");
	if (mktemp(file) == nil) {
		fprintf(stderr, "mail: %s: %s\n", file, strerror(errno));
		deallocate(file);
		return nil;
	}
	tmpbox= initbox(file, 0);
	deallocate(file);
	tmpbox->format= box->format;
	if (!scanbox(tmpbox, WRITE)) {
		scanerrors();
		delbox(tmpbox);
		return 0;
	}

	for (i= 0; ok && i <= box->nmess; i++) {
		message_t *mess= box->messages[i];
		mailbox_t *msgbox= mess->box;
		off_t length;
		int c;

		if (tmpbox->format == MMDF) {
			fprintf(tmpbox->fp, "\1\1\1\1\n");
		} else
		if (mess->nhdrs == 0 || !matchfrom(mess->headers[0])) {
			fprintf(tmpbox->fp, "From somewhere!someone\n");
		}

		for (j= 0; j < mess->nhdrs; j++) {
			fprintf(tmpbox->fp, "%s\n", mess->headers[j]);
		}
		if (ferror(tmpbox->fp)) { tmpbox->errno= errno; ok= 0; }

		if (ok && msgbox != box
			&& (msgbox->fp= fopen(msgbox->file, "r")) == nil) {
			msgbox->errno= errno;
			ok= 0;
		}
		if (ok) (void) fseek(msgbox->fp, mess->body, SEEK_SET);
		length= mess->limit - mess->body;
		while (ok && length > 0 && (c= getc(msgbox->fp)) != EOF) {
			if (putc(c, tmpbox->fp) == EOF) {
				tmpbox->errno= errno; ok= 0;
			}
			length--;
		}
		if (ok && ferror(msgbox->fp)) { msgbox->errno= errno; ok= 0; }
		if (ok && tmpbox->format == MMDF) {
			fprintf(tmpbox->fp, "\1\1\1\1\n");
			if (ferror(tmpbox->fp)) { tmpbox->errno= errno; ok= 0; }
		}
		if (msgbox != box && msgbox->fp != nil) {
			fclose(msgbox->fp);
			msgbox->fp= nil;
		}
	}
	if (ok) ok= scanerrors();
	if (ok) ok= scanbox(tmpbox, CLOSE);
	if (!ok) {
		scanerrors();
		tmpbox->draft= 1;
		delbox(tmpbox);
		return nil;
	}
	/*XXX*/
}

#define do_write	nil
#define do_quit		nil
#define do_inbox	nil
#define do_savebox	nil

int do_command(char *file, FILE *fp)
/* Read and execute one command. */
{
	char *line, *word0;
	static command_t cmd;
	int (*do_cmd)(size_t argc, char **argv, size_t n);
	size_t i, j;

#if !CRIPPLE
	if (talk && fp == stdin) {
		static char *oldline;

		if (oldline != nil) {
			free(oldline);
			oldline= nil;
		}

		if ((line= readline("? ")) == nil) {
			if (talk) fputc('\n', stderr);
			return 0;
		}
		add_history(line);
		oldline= line;
	} else {
#endif
		if (talk) fprintf(stderr, "? ");

		if ((line= read1line(file, fp)) == nil) {
			if (talk) fputc('\n', stderr);
			return 0;
		}
#if !CRIPPLE
	}
#endif
	while (whitespace(*line)) line++;

	if (current == nil && messaddrchar(*line)) {
		fprintf(stderr,
		"mail: no current mailbox, so no message list possible.\n");
		return talk;
	}

	if (!parsecommand(line, &cmd)) {
		if (!talk) fprintf(stderr, "mail: %s: ", line);
		fprintf(stderr, "address list syntax error\n");
		if (!talk) quit(1);
		return 1;
	}

	/* An empty command is an alias for "+p". */
	if (cmd.nwords == 0) {
		if (!talk && cmd.nranges == 0) return 1;
		cmd.words= allocate(cmd.words, sizeof(cmd.words[0]));
		cmd.words[0]= copystr("p");
		cmd.nwords= 1;
		if (cmd.nranges == 0) {
			if (dot == current->nmess - 1) return 1;
			dot++;
		}
	}
	if (dot == NO_MESSAGE) dot= 0;

	word0= cmd.words[0];

	/* A command that has no address range and is not a single letter is
	 * shell command when interactive or if it starts with a '!'.
	 */
	if (((talk && word0[1] != 0) || line[0] == '!') && cmd.nranges == 0
					&& strcmp(word0, "cd") != 0) {
		if (*line == '!') line++;
		(void) system(line);
		return 1;
	}

	/* An empty address range means "." */
	if (cmd.nranges == 0) {
		cmd.ranges= allocate(cmd.ranges, sizeof(cmd.ranges[0]));
		cmd.ranges[0].start= cmd.ranges[0].end= dot;
		cmd.nranges= 1;
	}

	/* Select the proper command function. */
	do_cmd= nil;
	if (strcmp(word0, "?") == 0) do_cmd= do_help;
	if (strcmp(word0, "a") == 0) do_cmd= do_edit_etc;
	if (strcmp(word0, "d") == 0) do_cmd= do_del_undel;
	if (strcmp(word0, "e") == 0) do_cmd= do_edit_etc;
	if (strcmp(word0, "f") == 0) do_cmd= do_forward;
	if (strcmp(word0, "h") == 0) do_cmd= do_header;
	if (strcmp(word0, "m") == 0) do_cmd= do_move;
	if (strcmp(word0, "n") == 0) do_cmd= do_new;
	if (strcmp(word0, "o") == 0) do_cmd= do_open;
	if (strcmp(word0, "p") == 0) do_cmd= do_print;
	if (strcmp(word0, "s") == 0) do_cmd= do_send;
	if (strcmp(word0, "u") == 0) do_cmd= do_del_undel;
	if (strcmp(word0, "r") == 0) do_cmd= do_edit_etc;
	if (strcmp(word0, "w") == 0) do_cmd= do_write;
	if (strcmp(word0, "q") == 0) do_cmd= do_quit;
	if (strcmp(word0, "cd") == 0) do_cmd= do_cd;
	if (strcmp(word0, "inbox") == 0) do_cmd= do_inbox;
	if (strcmp(word0, "savebox") == 0) do_cmd= do_savebox;

	if (do_cmd == nil) {
		if (!talk) fprintf(stderr, "mail: ");
		fprintf(stderr, "unknown command: '%s'\n", word0);
		return talk;
	}

	/* Select the addressed messages. */
	for (i= 0; current != nil && i < current->nmess; i++) {
		current->messages[i]->flags&= ~M_SELECTED;
		for (j= 0; j < cmd.nranges; j++) {
			if (cmd.ranges[j].start <= i
					&& i <= cmd.ranges[j].end) {
				current->messages[i]->flags|= M_SELECTED;
			}
		}
	}

	/* Try the command for each selected message, and then with no
	 * message to do some cleanup.  Commands that are not message related
	 * wait for that last call.
	 */
	do {
		j= NO_MESSAGE;
		for (i= 0; current != nil && i < current->nmess; i++) {
			if (current->messages[i]->flags & M_SELECTED) {
				current->messages[i]->flags&= ~M_SELECTED;
				j= i;
				break;
			}
		}
		if (!(*do_cmd)(cmd.nwords, cmd.words, j)) {
			return talk;
		}
	} while (j != NO_MESSAGE);
	return 1;
}

void quit(int excode)
{
	mailbox_t *box;

	/* Delete all draft mailboxes. */
	for (box= allboxes; box != nil; box= box->next) {
		if (box->draft && unlink(box->file) < 0) {
			fprintf(stderr, "mail: can't delete %s: %s\n",
				box->file, strerror(errno));
			excode= 1;
		}
	}
	exit(excode);
}

void mailsubject(char *subject, char **addresses)
/* Mail standard input as a mail message.  Only send if non-empty. */
{
	char **argv;
	int i;
	int c;
	int ex_code= 0;
	FILE *mailer;

	/* Contruct a sendmail command. */
	for (i= 0; addresses[i] != nil; i++) {}
	argv= allocate(nil, (i + 4) * sizeof(argv[0]));
	argv[0]= MAILER;
	argv[1]= "-oi";
	argv[2]= "--";
	memcpy(argv + 3, addresses, (i + 1) * sizeof(addresses[0]));

	mailer= nil;	/* No mailer started yet. */

	while ((c= getchar()) != EOF) {
		if (mailer == nil) {
			if ((mailer= start_mailer(argv)) == nil) {
				ex_code= 1;
				break;
			}
			if (subject != nil) {
				fprintf(mailer, "Subject: %s\n\n", subject);
				if (ferror(mailer)) {
					fprintf(stderr,
					"mail: error writing to mailer: %s\n",
						strerror(errno));
					ex_code= 1;
					break;
				}
			}
		}
		if (putc(c, mailer) == EOF) {
			fprintf(stderr, "mail: error writing to mailer: %s\n",
						strerror(errno));
			ex_code= 1;
			break;
		}
	}
	if (ferror(stdin)) {
		fprintf(stderr, "mail: error reading message: %s\n",
							strerror(errno));
		ex_code= 1;
	}
	if (mailer != nil) {
		if (fclose(mailer) == EOF) {
			fprintf(stderr, "mail: error writing to mailer: %s\n",
						strerror(errno));
			ex_code= 1;
		}
		if (!stop_mailer()) ex_code= 1;
	}
	exit(ex_code);
}

void resendmail(char **addresses)
/* Resend standard input to new receipients.  Add Resent-* headers. */
{
	char **argv;
	int i;
	int c;
	int ex_code= 0;
	FILE *mailer;
	char **headers, *line;
	size_t hi, hn;

	/* Read the message headers. */
	hi= 0;
	hn= 32;
	headers= allocate(nil, hn * sizeof(headers[0]));

	while ((line= read1line("standard input", stdin)) != nil) {
		if (hi == 0 && matchfrom(line)) {
			/* Special "From<space>" header.  Get rid of it. */
			continue;
		} else
		if (hi > 0 && whitespace(line[0])) {
			/* Continuation header line. */
		} else
		if (match("*:", line)) {
			/* Normal header line. */
		} else {
			/* Message body. */
			break;
		}
		if (hi == hn) {
			hn*= 2;
			headers= allocate(headers, hn * sizeof(headers[0]));
		}
		headers[hi++]= copystr(line);
	}

	/* Contruct a sendmail command. */
	for (i= 0; addresses[i] != nil; i++) {}
	argv= allocate(nil, (i + 4) * sizeof(argv[0]));
	argv[0]= MAILER;
	argv[1]= "-oi";
	argv[2]= "--";
	memcpy(argv + 3, addresses, (i + 1) * sizeof(addresses[0]));

	mailer= start_mailer(argv);

	/* Output the headers. */
	for (i= 0; i < hi; i++) fprintf(mailer, "%s\n", headers[i]);

	/* Add Resent-* headers. */
	if (myaddress() != nil) {
		fprintf(mailer, "Resent-From: %s\n", myaddress());
	}
	fprintf(mailer, "Resent-To:");
	for (i= 0; addresses[i] != nil; i++) {
		fprintf(mailer, " %s%c", addresses[i],
			addresses[i+1] != nil ? ',' : '\n');
	}
	fprintf(mailer, "Resent-Date: %s\n", arpa_date(time(nil)));

	/* The line that wasn't a header. */
	fprintf(mailer, "%s\n", line);
	if (ferror(mailer)) {
		fprintf(stderr, "mail: error writing to mailer: %s\n",
						strerror(errno));
		ex_code= 1;
	}

	/* The rest of the body. */
	while (ex_code == 0 && (c= getchar()) != EOF) {
		if (putc(c, mailer) == EOF) {
			fprintf(stderr, "mail: error writing to mailer: %s\n",
						strerror(errno));
			ex_code= 1;
		}
	}

	if (ferror(stdin)) {
		fprintf(stderr, "mail: error reading message: %s\n",
						strerror(errno));
		ex_code= 1;
	}

	if (fclose(mailer) == EOF) {
		fprintf(stderr, "mail: error writing to mailer: %s\n",
						strerror(errno));
		ex_code= 1;
	}
	if (!stop_mailer()) ex_code= 1;
	exit(ex_code);
}

void usage(void)
{
	fprintf(stderr, "Usage: mail [sendmail-options]\n");
	fprintf(stderr, "       mail -s subject address ...\n");
#if CRIPPLE
	fprintf(stderr, "       Use 'man -k mail' to find mail readers\n");
#endif
	exit(1);
}

void main(int argc, char **argv)
{
	FILE *rcfp;
	char *name;
	enum action { MAIL, RESEND, SENDMAIL } action;

	if ((name= strrchr(argv[0], '/')) == nil) name= argv[0]; else name++;

	if (strcmp(name, "mail") == 0) action= MAIL;
	else
	if (strcmp(name, "resend") == 0) action= RESEND;
	else
		action= SENDMAIL;

	if (action == MAIL) {
		char *subject= nil;
		int i= 1;

		if (argc > 1 && argv[1][0] == '-') {
			if (argv[1][1] == 's') {
				subject= argv[1] + 2;
				if (*subject == 0) {
					if (argc < 3) usage();
					subject= argv[2];
					i++;
				}
				if (++i == argc) usage();
			} else
			if (argv[1][1] == '?') usage();
		}
		if (subject != nil) mailsubject(subject, argv + i);

		/* Any arguments?  Let the mailer figure it out. */
		if (argc > 1) action= SENDMAIL;
	}

	if (action == RESEND) {
		/* Resend mail to the addresses given. */
		if (argc < 2) {
			fprintf(stderr, "Usage: resend address ...\n");
			exit(1);
		}
		resendmail(argv + 1);
	}

	if (action == SENDMAIL) {
		/* Not simply called as "mail".  Let the transport agent
		 * figure it out.
		 */
		execv(MAILER, argv);
		fprintf(stderr, "mail: can't execute %s: %s\n",
			MAILER, strerror(errno));
		_exit(127);
	}
#if CRIPPLE
	usage();
#endif

	if ((home= getenv("HOME")) == nil) {
		fprintf(stderr, "mail: $HOME not set?!\n");
		exit(1);
	}

	if ((editor= getenv("EDITOR")) == nil) editor= EDITOR;

	if ((pager= getenv("PAGER")) == nil) pager= PAGER;

	/* Read .mailrc. */
	if ((rcfp= fopen(expand("~/.mailrc"), "r")) == nil) {
		if (errno != ENOENT) {
			fprintf(stderr, "mail: ~/.mailrc: %s\n",
							strerror(errno));
			exit(1);
		}
	} else {
		while (do_command("~/.mailrc", rcfp)) {}
		fclose(rcfp);
	}

	/* No addresses, go read incoming mail. */
	talk= isatty(0) && isatty(1);

	current= initbox(expand("~/mailbox"), 0);
	dot= NO_MESSAGE;
	if (talk) fprintf(stderr, "Current mailbox is '%s'\n", current->file);
	do {
		scanerrors();
		if (!scanbox(current, READ) && current->errno == ENOENT)
			current->errno= 0;
		scanerrors();
		scanbox(current, CLOSE);

		if (current->nseen != current->nmess && talk) {
			if (current->nseen != NONE_SEEN) {
				size_t n= current->nmess - current->nseen;
				fprintf(stderr,
					"%u new message%s arrived, now ",
					n, n == 1 ? "" : "s");
			}
			fprintf(stderr, "%u message%s\n",
					current->nmess,
					current->nmess == 1 ? "" : "s");
		}
		current->nseen= current->nmess;
	} while (do_command("input", stdin));

	quit(0);
}
