/*
rip.c

A Routing Information Protocol (RFC-1058) implementation.

Created:	Aug 5, 1993 by Philip Homburg <philip@cs.vu.nl>
*/

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/syslog.h>
#include <net/gen/in.h>
#include <net/gen/rip.h>
#include <net/gen/udp.h>

#include "ansi.h"
#include "interface.h"
#include "itab.h"
#include "otab.h"
#include "rip.h"
#include "staticr.h"
#include "update.h"

char *prog_name;

int rip_ifdist;
int debug;

static void usage ARGS(( void ));
static void rip_init ARGS(( void ));
static void fwait_until ARGS(( time_t to, asio_fd_set_t *fdset_p ));
static void timeout ARGS(( int sig ));

DEFUN
(int main, (argc, argv),
	int argc AND
	char *argv[]
)
{
	int c, i;
	int d_flag;
	time_t prev_time, curr_time, next_time;
	asio_fd_set_t fd_set;

	prog_name= argv[0];

	d_flag= 0;
	while((c= getopt(argc, argv, "?d")) != -1)
	{
		switch(c)
		{
		case 'd':
			if (d_flag)
				usage();
			d_flag= 1;
			break;
		case '?':
			usage();
		}
	}

	debug= d_flag;

	if (debug)
	{
		openlog(prog_name, LOG_PID|LOG_PERROR, LOG_DAEMON);
		setlogmask(LOG_UPTO(LOG_DEBUG));
	}
	else
	{
		openlog(prog_name, LOG_PID, LOG_DAEMON);
		setlogmask(LOG_UPTO(LOG_INFO));
	}

	rip_init();
	interface_init();
	itab_init();
	otab_init();
	staticr_init();

	if ((argc-optind) % 2 != 0 || argc-optind == 0)
		usage();
	for (i= optind; i<argc; i += 2)
		interface_add(argv[i], argv[i+1]);
	staticr_load();

	dynr_loadarr();
	staticr_arrived();

	syslog(LOG_DEBUG, "itab:");
	itab_print_all(LOG_DEBUG);
	syslog(LOG_DEBUG, "otab:");
	otab_print_all(LOG_DEBUG);

	otab_send(0);

	prev_time= time(NULL);
	next_time= prev_time + RIP_FREQUENCY - random() % RIP_FUZZ;
	for (;;)
	{
		curr_time= time(NULL);
		if (curr_time < prev_time)
		{
			syslog(LOG_WARNING,
				"main: time ran backwards (from %ld to %ld)",
					prev_time, curr_time);
			prev_time= curr_time;
			next_time= prev_time + RIP_FREQUENCY -
				random() % RIP_FUZZ;
			continue;
		}
		if (curr_time >= next_time)
		{
			staticr_arrived();
			itab_timeout();
			otab_send(0);
			prev_time= curr_time;
			next_time= prev_time + RIP_FREQUENCY -
				random() % RIP_FUZZ;
		}
		interface_read(&fd_set);
		fwait_until(next_time, &fd_set);
	}
	exit(0);
}

DEFUN_VOID (static void usage)
{
	fprintf(stderr, "USAGE: %s <ip dev> <udp dev>...\n", prog_name);
	exit(1);
}

DEFUN_VOID (static void rip_init)
{
	syslog(LOG_DEBUG, "rip_init()");

	rip_ifdist= 1;
}

DEFUN
(static void fwait_until, (to, fdset_p),
	time_t to AND
	asio_fd_set_t *fdset_p
)
{
	fwait_t fw;
	struct timeval tv;
	int r;

	fw.fw_flags= 0;
	fw.fw_bits= fdset_p->afds_bits;
	fw.fw_maxfd= ASIO_FD_SETSIZE;

	signal(SIGALRM, timeout);

	tv.tv_sec= to;
	tv.tv_usec= 0;
	sysutime(UTIME_SETALARM, &tv);

	for (;;)
	{
		r= fwait(&fw);
		alarm(0);
		if (r == -1 && errno == EINTR)
			return;
		if (r == -1)
		{
			syslog(LOG_ERR, "fwait_until: fwait failed: %m");
			return;
		}
		interface_arrived(fw.fw_fd, fw.fw_operation, fw.fw_result, 
			fw.fw_errno);
		if (!(fw.fw_flags & FWF_MORE))
			break;
	}
}

DEFUN
(static void timeout, (sig),
	int sig
)
{
	signal(SIGALRM, timeout);
	alarm(1);
}
