/*
itab.c

Created:	Aug 18, 1993 by Philip Homburg <philip@cs.vu.nl>
*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/syslog.h>
#include <net/hton.h>
#include <net/gen/in.h>
#include <net/gen/inet.h>
#include <net/gen/rip.h>
#include <net/gen/udp.h>

#include "ansi.h"
#include "interface.h"
#include "ip_misc.h"
#include "itab.h"
#include "otab.h"
#include "update.h"

itab_t *i_table;

static void add ARGS(( itab_t *prev, itab_t *curr, update_t update ));
static void update ARGS(( itab_t *prev, itab_t *curr, update_t update ));
static int compare ARGS(( itab_t *itab_p, update_t *update_p ));
static int check_dest ARGS(( itab_t *ent ));

DEFUN_VOID( void itab_init)
{
	i_table= NULL;
}

DEFUN
(void itab_update, (update_p, update_no),
	update_t *update_p AND
	int update_no
)
{
	itab_t *prev, *curr;
	int r;

	if (update_no < 0)
	{
		syslog(LOG_ERR, "itab_update: update_no = %d", update_no);
		return;
	}
	if (update_no == 0)
		return;
	prev= NULL;
	curr= i_table;
	for (;;)
	{
		if (curr == NULL)
		{
			/* Add a new entry a the end of the (possibly empty) 
			 * table.
			 */
			add(prev, curr, *update_p);
			update_p++;
			update_no--;
			if (update_no == 0)
				return;

			/* Relook up curr */
			if (prev == NULL)
				curr= i_table;
			else
				curr= prev->it_next;
			continue;
		}
		r= compare(curr, update_p);
		if (r == -1)
		{
			prev= curr;
			curr= curr->it_next;
			continue;
		}
		if (r != 0 && r != 1)
		{
			syslog(LOG_ERR, "itab_update: compare returned %d",
					r);
			return;
		}
		if (r == 0)
			update(prev, curr, *update_p);
		else
			add(prev, NULL, *update_p);
		update_p++;
		update_no--;
		if (update_no == 0)
			return;

		/* Relook up curr */
		if (prev == NULL)
			curr= i_table;
		else
			curr= prev->it_next;
	}
}

DEFUN
(static int compare, (itab_p, update_p),
	itab_t *itab_p AND
	update_t *update_p
)
{
	/* Warning, this function should use the same algorithm as
	 * update_compare, except that distances compare equal.
	 */

	if (itab_p->it_dest > update_p->u_dest)
		return 1;
	else if (itab_p->it_dest < update_p->u_dest)
		return -1;

	if (itab_p->it_subnetmask > update_p->u_subnetmask)
		return 1;
	else if (itab_p->it_subnetmask < update_p->u_subnetmask)
		return -1;

	if (itab_p->it_ifno > update_p->u_ifno)
		return 1;
	else if (itab_p->it_ifno < update_p->u_ifno)
		return -1;

	if (itab_p->it_gateway > update_p->u_gateway)
		return 1;
	else if (itab_p->it_gateway < update_p->u_gateway)
		return -1;
	return 0;
}

DEFUN
(static void add, (prev, curr, update),
	itab_t *prev AND
	itab_t *curr AND
	update_t update
)
{
	itab_t i_tmp;
	interface_t *ifp;
	int r, type;

	if (curr == NULL)
	{
		/* Create an invalid entry just in case something goes wrong.
		 * If everything turns out to be alright we can simply turn
		 * it into a valid one.
		 */
		if ((curr= malloc(sizeof(*curr))) == NULL)
		{
			syslog(LOG_ERR, "ibuf.add: out of memory");
			return;
		}
		curr->it_dest= update.u_dest;
		curr->it_subnetmask= update.u_subnetmask;
		curr->it_ifno= update.u_ifno;
		curr->it_gateway= update.u_gateway;
		curr->it_dist= update.u_dist;
		if (prev != NULL)
		{
			curr->it_next= prev->it_next;
			prev->it_next= curr;
		}
		else
		{
			curr->it_next= NULL;
			i_table= curr;
		}
	}
	else if (curr->it_type != ITT_DELETED)
	{
		syslog(LOG_ERR, "itab.add: got strange type: %d",
			curr->it_type);
		itab_print(LOG_ERR, curr);
		return;
	}
	curr->it_type= ITT_INVALID;

	/* Update the time */
	curr->it_time= time(NULL);

	if ((type= check_dest(curr)) == ITT_INVALID)
		return;

	if (curr->it_ifno < 0 || curr->it_ifno >= interface_no)
	{
		syslog(LOG_ERR, "itab.add: got strange ifno: %d",
			curr->it_ifno);
		itab_print(LOG_ERR, curr);
		return;
	}
	ifp= &interface_table[curr->it_ifno];
	if (curr->it_gateway != (ipaddr_t)0 &&
		(((curr->it_gateway ^ ifp->if_addr) & ifp->if_subnetmask) != 0 ||
		(curr->it_gateway & ~ifp->if_subnetmask) == 0 ||
		(curr->it_gateway & ~ifp->if_subnetmask) ==
		~ifp->if_subnetmask))
	{
		syslog(LOG_ERR, "itab.add: got strange gateway: %s",
			inet_ntoa(htonl(curr->it_gateway)));
		itab_print(LOG_ERR, curr);
		return;
	}
	if (curr->it_dist < 1 || curr->it_dist >= RIP_INFINITY)
	{
		if (curr->it_dist == RIP_INFINITY)
		{
			syslog(LOG_DEBUG,
				"itab.add: ignoring unreachable route");
			itab_print(LOG_DEBUG, curr);
		}
		else
		{
			syslog(LOG_ERR, "itab.add: got strange distance: %d",
				curr->it_dist);
			itab_print(LOG_ERR, curr);
		}
		return;
	}

	/* Entry seems to be OK, try to add it to the output table. */
	if (type == ITT_NORMAL)
		r= otab_add_normal(curr);
	else if (type == ITT_SUBNET)
	{
		i_tmp= *curr;
		i_tmp.it_dest &= ifp->if_netmask;
		i_tmp.it_subnetmask= ifp->if_netmask;
		r= otab_add_agregate(&i_tmp);

		if (r == 0)
			r= otab_add_subnet(curr);
	}
	else 
	{
		syslog(LOG_ERR,
			"itab.add: got strange type from check_dest: %d",
				type);
		itab_print(LOG_ERR, curr);
		return;
	}

	if (r == -1)
	{
		/* Kernel routing table is full, or out of memory */
		curr->it_type= ITT_DELETED;
		return;
	}
	curr->it_type= type;
}

DEFUN
(static int check_dest, (itab_t *ent),
	itab_t *ent
)
{
	ipaddr_t ipaddr, subnetmask, netmask, subnetpart, hostpart;
	interface_t *ifp;
	int i, nettype;

	ipaddr= ent->it_dest;
	subnetmask= ent->it_subnetmask;

	/* Check if zeros in the subnetmask correspond to zeros in the
	 * address
	 */
	if ((ipaddr & ~subnetmask) != 0)
	{
		syslog(LOG_ERR,
		"itab.check_dest: address and subnetmask don't match");
		itab_print(LOG_ERR, ent);
		return ITT_INVALID;
	}

	nettype= ip_nettype(ipaddr);
	if (nettype != IPNT_ZERO && nettype != IPNT_CLASS_A &&
		nettype != IPNT_CLASS_B && nettype != IPNT_CLASS_C &&
		nettype != IPNT_CLASS_D && nettype != IPNT_CLASS_E)
	{
		syslog(LOG_ERR,
"itab.check_dest: destination %s belongs to wrong network class: %s",
				inet_ntoa(htonl(ipaddr)), ip_nettoa(nettype));
		itab_print(LOG_ERR, ent);
		return ITT_INVALID;
	}
	if (nettype == IPNT_CLASS_D || nettype == IPNT_CLASS_E)
	{
		/* Routes to networks of class D or E are to be ignored
		 * (RPC-1058, page 27)
		 */
		syslog(LOG_DEBUG,
			"itab.check_dest: got distination to a %s network",
				ip_nettoa(nettype));
		return ITT_INVALID;
	}
	netmask= ip_netmask(nettype);
	if (subnetmask < netmask)
	{
		syslog(LOG_ERR, "itab.check_dest: got bad netmask: %s",
			inet_ntoa(htonl(subnetmask)));
		itab_print(LOG_ERR, ent);
		return ITT_INVALID;
	}
	else if (subnetmask != netmask)
	{
		/* Check the subnet or host part of the address. */
		subnetpart= ipaddr & ~netmask;
		if (subnetpart == 0 || subnetpart == (subnetmask & ~netmask))
		{
			syslog(LOG_ERR,
			"itab.check_dest: invalid subnetpart of address %s",
					inet_ntoa(htonl(ipaddr)));
			itab_print(LOG_ERR, ent);
			return ITT_INVALID;
		}
	}

	/* Look for an interface on the same network as this route */
	for (i= 0, ifp= interface_table; i<interface_no; i++, ifp++)
	{
		if (((ipaddr ^ ifp->if_addr) & netmask) == 0)
			break;
	}
	if (i == interface_no || nettype == IPNT_ZERO)
	{
		/* No interface found, arbitrary subnetting is allowed, so
		 * no further checks are needed.
		 */
		return ITT_NORMAL;
	}
	if (ifp->if_subnetmask == netmask)
	{
		/* A not subnetted network, only host routes and network
		 * routes are accepted.
		 */
		if (subnetmask == netmask || subnetmask == (ipaddr_t)-1)
			return ITT_NORMAL;
		syslog(LOG_ERR, "itab.check_dest: subnetting not allowed");
		itab_print(LOG_ERR, ent);
		return ITT_INVALID;
	}

	/* Host and subnet routes are allowed, network routes are silently
	 * ignored.
	 */
	if (subnetmask != netmask && subnetmask != ifp->if_subnetmask &&
		subnetmask != (ipaddr_t)-1)
	{
		syslog(LOG_ERR, "itab.check_dest: invalid subnetting");
		itab_print(LOG_ERR, ent);
		return ITT_INVALID;
	}
	if (subnetmask == netmask)
	{
		syslog(LOG_DEBUG, "itab.check_dest: ignoring network route");
		itab_print(LOG_DEBUG, ent);
		return ITT_INVALID;
	}

	if (subnetmask == ifp->if_subnetmask)
		return ITT_SUBNET;

	/* Check the subnet part of the address. */
	subnetpart= (ipaddr & ~netmask) & ifp->if_subnetmask;
	if (subnetpart == 0 || subnetpart == (ifp->if_subnetmask & ~netmask))
	{
		syslog(LOG_ERR,
		"itab.check_dest: invalid subnetpart of address %s",
				inet_ntoa(htonl(ipaddr)));
		itab_print(LOG_ERR, ent);
		return ITT_INVALID;
	}

	/* Check the host part of the address. */
	hostpart= ipaddr & ~ifp->if_subnetmask;
	if (subnetpart == 0 || subnetpart == ~ifp->if_subnetmask)
	{
		syslog(LOG_ERR,
		"itab.check_dest: invalid host part of address %s",
				inet_ntoa(htonl(ipaddr)));
		itab_print(LOG_ERR, ent);
		return ITT_INVALID;
	}
	return ITT_NORMAL;
}

DEFUN
(void itab_print, (logtype, ent),
	int logtype AND
	itab_t *ent
)
{
	char *cp1, *cp2, *cp3, *cp4;
	long time_diff;

	cp1= strdup(inet_ntoa(htonl(ent->it_dest)));
	cp2= strdup(inet_ntoa(htonl(ent->it_subnetmask)));
	cp3= strdup(inet_ntoa(htonl(ent->it_gateway)));
	switch(ent->it_type)
	{
	case ITT_INVALID:	cp4= "invalid"; break;
	case ITT_DELETED:	cp4= "deleted"; break;
	case ITT_NORMAL:	cp4= "normal"; break;
	case ITT_SUBNET:	cp4= "subnet"; break;
	default:		cp4= "<unknown>"; break;
	}
	time_diff= (long)ent->it_time - (long)time(NULL);

	syslog(logtype,
	"route: to %s[%s] via %d(%s) at distance %d type %s time %lds",
		cp1, cp2, ent->it_ifno, cp3, ent->it_dist, cp4, time_diff);
	free(cp1);
	free(cp2);
	free(cp3);
}

DEFUN
(void itab_print_all, (logtype),
	int logtype
)
{
	itab_t *itab_p;

	for (itab_p= i_table; itab_p; itab_p= itab_p->it_next)
		itab_print(logtype, itab_p);
}

DEFUN
(static void update, (prev, curr, update),
	itab_t *prev AND
	itab_t *curr AND
	update_t update
)
{
	itab_t i_tmp;
	ipaddr_t netmask;
	int r;

	syslog(LOG_DEBUG, "itab.update(0x%x, 0x%x, {})", prev, curr);

	if (curr->it_type == ITT_DELETED)
	{
		/* This is infact an add. */
		add(prev, curr, update);
		return;
	}
	if (curr->it_type == ITT_INVALID)
	{
		if (curr->it_dist != update.u_dist)
		{
			syslog(LOG_DEBUG,
			"itab.update: got an update for an invalid route");
			curr->it_type= ITT_DELETED;
			add(prev, curr, update);
			return;
		}

		/* Update the time */
		curr->it_time= time(NULL);
		return;
	}

	if (curr->it_dist == update.u_dist)
	{
		/* Nothing changed, update the time if the distance is not
		 * RIP_INFINITY.
		 */
		if (curr->it_dist != RIP_INFINITY)
			curr->it_time= time(NULL);
		return;
	}
	curr->it_dist= update.u_dist;
	curr->it_time= time(NULL);
	if (curr->it_type == ITT_NORMAL)
	{
		r= otab_update_normal(NULL, curr);
		if (r == -1)
			curr->it_type= ITT_DELETED;
		return;
	}
	else if (curr->it_type == ITT_SUBNET)
	{
		i_tmp= *curr;
		netmask= ip_netmask(ip_nettype(i_tmp.it_dest));
		i_tmp.it_dest &= netmask;
		i_tmp.it_subnetmask= netmask;
		r= otab_update_agregate(NULL, &i_tmp);
		if (r == 0)
			r= otab_update_subnet(NULL, curr);
		if (r == -1)
			curr->it_type= ITT_DELETED;
		return;
	}
	else
	{
		syslog(LOG_ERR, "itab.update: entry with strange type %d",
			curr->it_type);
		itab_print(LOG_ERR, curr);
	}
}

DEFUN_VOID( void itab_timeout )
{
	itab_t *prev, *ent;
	itab_t i_tmp;
	ipaddr_t netmask;
	time_t now;

	now= time(NULL);
	for (prev= NULL, ent= i_table; ent; prev= ent, ent= ent->it_next)
	{
		if (ent->it_type == ITT_INVALID)
		{
			if (now >= ent->it_time + RIP_TIMEOUT)
				ent->it_type= ITT_DELETED;
		}
		else if (ent->it_type == ITT_NORMAL)
		{
			if (ent->it_dist == RIP_INFINITY && 
				now >= ent->it_time + RIP_DELETE_TO)
			{
				ent->it_type= ITT_DELETED;
				otab_remove_normal(ent);
			}
			if (ent->it_dist != RIP_INFINITY &&
				now >= ent->it_time + RIP_TIMEOUT)
			{
				ent->it_time= now;
				ent->it_dist= RIP_INFINITY;
				otab_update_normal(NULL, ent);
			}
		}
		else if (ent->it_type == ITT_SUBNET)
		{
			if (ent->it_dist == RIP_INFINITY && 
				now >= ent->it_time + RIP_DELETE_TO)
			{
				ent->it_type= ITT_DELETED;
				i_tmp= *ent;
				netmask= ip_netmask(ip_nettype(i_tmp.it_dest));
				i_tmp.it_dest &= netmask;
				i_tmp.it_subnetmask= netmask;
				otab_remove_subnet(ent);
				otab_remove_agregate(&i_tmp);
			}
			if (ent->it_dist != RIP_INFINITY &&
				now >= ent->it_time + RIP_TIMEOUT)
			{
				ent->it_time= now;
				ent->it_dist= RIP_INFINITY;
				i_tmp= *ent;
				netmask= ip_netmask(ip_nettype(i_tmp.it_dest));
				i_tmp.it_dest &= netmask;
				i_tmp.it_subnetmask= netmask;
				otab_update_normal(NULL, ent);
				otab_update_agregate(NULL, &i_tmp);
			}
		}
		if (ent->it_type == ITT_DELETED)
		{
			if (prev != NULL)
			{
				prev->it_next= ent->it_next;
				ent->it_next= NULL;
				free(ent);
				ent= prev;
			}
			/* else, bad luck for now */

			continue;
		}

	}
	if (i_table != NULL && i_table->it_type == ITT_DELETED)
	{
		ent= i_table;
		i_table= ent->it_next;
		free(ent);
	}
}
