/*
vjhc.c

Van Jacobson header compression based on RFC-1144

Created:	Nov 11, 1993 by Philip Homburg <philip@cs.vu.nl>
*/

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <net/hton.h>
#include <net/gen/in.h>
#include <net/gen/ip_hdr.h>
#include <net/gen/oneCsum.h>
#include <net/gen/tcp.h>
#include <net/gen/tcp_hdr.h>
#include <net/gen/vjhc.h>

#include "ansi.h"
#include "debug.h"
#include "ip.h"
#include "pkt.h"
#include "ppp.h"
#include "vjhc.h"

typedef struct snd_state
{
	int s_indx;
	ipaddr_t s_src_ip;
	ipaddr_t s_dst_ip;
	u32_t s_srcdst_port;
	char s_data[IP_MAX_HDR_SIZE + TCP_MAX_HDR_SIZE];
	struct snd_state *s_next;
} snd_state_ut;

typedef struct rcv_state
{
	int s_ip_hdr_len;
	int s_tot_len;
	char s_data[IP_MAX_HDR_SIZE + TCP_MAX_HDR_SIZE];
} rcv_state_ut;

int ip_snd_vjhc= 0;
int ip_snd_vjhc_compress_cid= 0;
int ip_snd_vjhc_state_nr= 16;
int ip_rcv_vjhc= 0;
int ip_rcv_vjhc_compress_cid= 0;
int ip_rcv_vjhc_state_nr= 16;

static int xmit_last;
static snd_state_ut *xmit_state= NULL;
static snd_state_ut *xmit_head;

static int rcv_toss;
static rcv_state_ut *rcv_state;
static rcv_state_ut *rcv_last;

DEFUN_VOID( void ip_vjhc_init )
{
	int i;

	if (ip_snd_vjhc)
	{
		xmit_last= -1;
		if (xmit_state != NULL)
		{
			free(xmit_state);
			xmit_state= NULL;
		}
		xmit_state= calloc(ip_snd_vjhc_state_nr, sizeof(snd_state_ut));
		assert(xmit_state != NULL);

		xmit_head= 0;
		for (i= 0; i<ip_snd_vjhc_state_nr; i++)
		{
			xmit_state[i].s_indx= i;
			xmit_state[i].s_next= xmit_head;
			xmit_head= &xmit_state[i];
		}
	}
	else
	{
		DPRINTF(1,
	("ip_vjhc_init: sending of compressed packets is not enabled\n"));
	}

	if (ip_rcv_vjhc)
	{
		if (rcv_state != NULL)
		{
			free(rcv_state);
			rcv_state= NULL;
		}
		rcv_state= calloc(ip_rcv_vjhc_state_nr, sizeof(snd_state_ut));
		assert(rcv_state != NULL);
	}
	else
	{
		DPRINTF(1,
	("ip_vjhc_init: reception of compressed packets is not enabled\n"));
	}
}

DEFUN
(int ip_vjhc_compress, (pkt),
	pkt_ut *pkt
)
{
	ip_hdr_t *ip_hdr, *oip_hdr;
	tcp_hdr_t *tcp_hdr, *otcp_hdr;
	int ip_hdr_len, tcp_hdr_len, tot_len;
	snd_state_ut *prev, *state, *next;
	int changes;
	u8_t new_hdr[16], *cp;
	u32_t delta, deltaA, deltaS;
	u16_t cksum;

	if (pkt->p_size < IP_MIN_HDR_SIZE + TCP_MIN_HDR_SIZE)
	{
		DPRINTF(1, ("ip_vjhc_compress: packet too small %d\n",
			pkt->p_size));
		return PPP_TYPE_IP;
	}
	ip_hdr= (ip_hdr_t *)(pkt->p_data + pkt->p_offset);
	if ((ntohs(ip_hdr->ih_flags_fragoff) &
		(IH_MORE_FRAGS | IH_FRAGOFF_MASK)) != 0)
	{
		DPRINTF(1, ("ip_vjhc_compress: fragmented packet\n"));
		return PPP_TYPE_IP;
	}
	ip_hdr_len= (ip_hdr->ih_vers_ihl & IH_IHL_MASK) * 4;
	tcp_hdr= (tcp_hdr_t *)(pkt->p_data + pkt->p_offset + ip_hdr_len);
	if ((tcp_hdr->th_flags & (THF_SYN | THF_FIN | THF_RST | THF_ACK)) !=
		THF_ACK)
	{
		DPRINTF(5, ("ip_vjhc_compress: wrong flags in TCP header\n"));
		return PPP_TYPE_IP;
	}
	tcp_hdr_len= (tcp_hdr->th_data_off & TH_DO_MASK) >> 2;	/* !!! */
	tot_len= ip_hdr_len + tcp_hdr_len;

#if 0
	DPRINTF(1, ("ip_vjhc_compress: packet with size %d\n\t",
		pkt->p_size));
	for (cp= (u8_t *)pkt->p_data+pkt->p_offset; cp < (u8_t *)pkt->p_data +
		pkt->p_offset + ip_hdr_len; cp++)
	{
		DPRINTF_CONT(1, ("%x ", *cp));
	}
	DPRINTF_CONT(1, ("\n\t"));
	for (; cp < (u8_t *)pkt->p_data + pkt->p_offset + tot_len; cp++)
	{
		DPRINTF_CONT(1, ("%x ", *cp));
	}
	DPRINTF_CONT(1, ("\n\t"));
	for (; cp < (u8_t *)pkt->p_data + pkt->p_offset + pkt->p_size; cp++)
	{
		DPRINTF_CONT(1, ("%x ", *cp));
	}
	DPRINTF_CONT(1, ("\n"));
#endif

	for (prev= NULL, state= xmit_head;; )
	{
		if (ip_hdr->ih_src == state->s_src_ip &&
			ip_hdr->ih_dst == state->s_dst_ip &&
			*(u32_t *)&tcp_hdr->th_srcport ==
			state->s_srcdst_port)
		{
			DPRINTF(5, ("ip_vjhc_compress: found entry\n"));
			if (prev != NULL)
			{
				DPRINTF(2,
				("ip_vjhc_compress: moving entry to front\n"));
				prev->s_next= state->s_next;
				state->s_next= xmit_head;
				xmit_head= state;
			}
			break;
		}
		next= state->s_next;
		if (next != NULL)
		{
			prev= state;
			state= next;
			continue;
		}

		/* Not found */
		if (prev != NULL)
		{
			prev->s_next= state->s_next;
			state->s_next= xmit_head;
			xmit_head= state;
		}
		state->s_src_ip= ip_hdr->ih_src;
		state->s_dst_ip= ip_hdr->ih_dst;
		state->s_srcdst_port= *(u32_t *)&tcp_hdr->th_srcport;
		DPRINTF(5, ("ip_vjhc_compress: new entry: %x, %x, %x\n",
			state->s_src_ip, state->s_dst_ip,
			state->s_srcdst_port ));
		memcpy(state->s_data, ip_hdr, tot_len);
		xmit_last= ip_hdr->ih_proto= state->s_indx;
		return PPP_TYPE_VJHC_UNCOMPR;
	}
	oip_hdr= (ip_hdr_t *)(state->s_data);
	otcp_hdr= (tcp_hdr_t *)(state->s_data + ip_hdr_len);

	if (*(u16_t *)&ip_hdr->ih_vers_ihl !=
			*(u16_t *)&oip_hdr->ih_vers_ihl ||
		*(u16_t *)&ip_hdr->ih_flags_fragoff !=
			*(u16_t *)&oip_hdr->ih_flags_fragoff ||
		*(u16_t *)&ip_hdr->ih_ttl !=
			*(u16_t *)&oip_hdr->ih_ttl ||
		tcp_hdr->th_data_off != otcp_hdr->th_data_off ||
		(ip_hdr_len > 20 && memcmp(&ip_hdr[1], &oip_hdr[1],
			ip_hdr_len-20) != 0) ||
		(tcp_hdr_len > 20 && memcmp(&tcp_hdr[1], &otcp_hdr[1],
			tcp_hdr_len-20) != 0))
	{
		DPRINTF(2, ("ip_vjhc_compress: ip or tcp header changed\n"));
		memcpy(state->s_data, ip_hdr, tot_len);
		xmit_last= ip_hdr->ih_proto= state->s_indx;
		return PPP_TYPE_VJHC_UNCOMPR;
	}

	changes= 0;
	cp= new_hdr;

	if (tcp_hdr->th_flags & THF_URG)
	{
		delta= ntohs(tcp_hdr->th_urgptr);
		VJHC_ENCODEZ(cp, delta);
		changes |= VJHC_FLAG_U;
	}
	else if (tcp_hdr->th_urgptr != otcp_hdr->th_urgptr)
	{
		DPRINTF(1, (
		"ip_vjhc_compress: unexpected urgent pointer change\n"));
		memcpy(state->s_data, ip_hdr, tot_len);
		xmit_last= ip_hdr->ih_proto= state->s_indx;
		return PPP_TYPE_VJHC_UNCOMPR;
	}

	if ((delta= (u16_t)(ntohs(tcp_hdr->th_window) -
		ntohs(otcp_hdr->th_window))) != 0)
	{
		VJHC_ENCODE(cp, delta);
		changes |= VJHC_FLAG_W;
	}
	if ((deltaA= (u32_t)(ntohl(tcp_hdr->th_ack_nr) -
		ntohl(otcp_hdr->th_ack_nr))) != 0)
	{
		if (deltaA > 0xffff)
		{
			DPRINTF(5, ("ip_vjhc_compress: bad ack change\n"));
			memcpy(state->s_data, ip_hdr, tot_len);
			xmit_last= ip_hdr->ih_proto= state->s_indx;
			return PPP_TYPE_VJHC_UNCOMPR;
		}
		DPRINTF(5,
	("ip_vjhc_compress: ack= 0x%08x, oack= 0x%08x, deltaA= 0x%x\n",
			tcp_hdr->th_ack_nr, otcp_hdr->th_ack_nr, deltaA));
		VJHC_ENCODE(cp, deltaA);
		changes |= VJHC_FLAG_A;
	}
	if ((deltaS= (u32_t)(ntohl(tcp_hdr->th_seq_nr) -
		ntohl(otcp_hdr->th_seq_nr))) != 0)
	{
		if (deltaS > 0xffff)
		{
			DPRINTF(2, ("ip_vjhc_compress: bad seq change\n"));
			memcpy(state->s_data, ip_hdr, tot_len);
			xmit_last= ip_hdr->ih_proto= state->s_indx;
			return PPP_TYPE_VJHC_UNCOMPR;
		}
		VJHC_ENCODE(cp, deltaS);
		changes |= VJHC_FLAG_S;
	}

	switch(changes)
	{
	case 0:
	case VJHC_FLAG_U:
		if (ip_hdr->ih_length != oip_hdr->ih_length &&
			ntohs(oip_hdr->ih_length) == tot_len)
		{
			break;
		}
		DPRINTF(5, ("ip_vjhc_compress: retransmission\n"));

		/* Fall through */

	case VJHC_SPEC_I:
	case VJHC_SPEC_D:
		if (changes != 0)
			DPRINTF(1, ("ip_vjhc_compress: bad special case\n"));
		memcpy(state->s_data, ip_hdr, tot_len);
		xmit_last= ip_hdr->ih_proto= state->s_indx;
		return PPP_TYPE_VJHC_UNCOMPR;

	case VJHC_FLAG_S | VJHC_FLAG_A:
		if (deltaS == deltaA &&
			deltaS == ntohs(oip_hdr->ih_length) - tot_len)
		{
			DPRINTF(1, ("ip_vjhc_compress: special case I\n"));
			changes= VJHC_SPEC_I;
			cp= new_hdr;
		}
		break;

	case VJHC_FLAG_S:
		if (deltaS == ntohs(oip_hdr->ih_length) - tot_len)
		{
			DPRINTF(5, ("ip_vjhc_compress: special case D\n"));
			changes= VJHC_SPEC_D;
			cp= new_hdr;
		}
		break;
	}

	if ((delta= ntohs(ip_hdr->ih_id) - ntohs(oip_hdr->ih_id)) != 1)
	{
		VJHC_ENCODEZ(cp, delta);
		changes |= VJHC_FLAG_I;
	}
	if (tcp_hdr->th_flags & THF_PSH)
		changes |= VJHC_FLAG_P;

	memcpy(state->s_data, ip_hdr, tot_len);
	delta= cp - new_hdr;

	cp= (u8_t *)pkt->p_data + pkt->p_offset + tot_len -delta;
	if (delta != 0)
		memcpy(cp, new_hdr, delta);
	cksum= ntohs(otcp_hdr->th_chksum);
	*--cp= cksum;
	*--cp= (cksum >> 8);
	if (!ip_snd_vjhc_compress_cid || xmit_last != state->s_indx)
	{
		xmit_last= state->s_indx;
		delta++;
		changes |= VJHC_FLAG_C;
		*--cp= xmit_last;
	}
	*--cp= changes;
	delta += 3;
	pkt->p_offset += tot_len - delta;
	pkt->p_size -= tot_len - delta;
	return PPP_TYPE_VJHC_COMPR;
}

DEFUN
(void ip_vjhc_arr_uncompr, (pkt),
	pkt_ut *pkt
)
{
	rcv_state_ut *state;
	ip_hdr_t *ip_hdr;
	tcp_hdr_t *tcp_hdr;
	int ip_hdr_len, tcp_hdr_len, tot_len;
	int slot;

	if (!ip_rcv_vjhc)
	{
		DPRINTF(0,
	("ip_vjhc_arr_uncompr: got packet but compression is not enabled\n"));
		pkt_free(pkt);
		return;
	}

	ip_hdr= (ip_hdr_t *)(pkt->p_data + pkt->p_offset);
	slot= ip_hdr->ih_proto;
	ip_hdr->ih_proto= IPPROTO_TCP;
	DPRINTF(5, ("ip_vjhc_arr_uncompr: packet at slot %d\n", slot)); 
	if (slot >= ip_rcv_vjhc_state_nr)
	{
		DPRINTF(1, ("ip_vjhc_arr_uncompr: bad slot\n")); 
		rcv_toss= 1;
		pkt_free(pkt);
		return;
	}
	state= &rcv_state[slot];

	ip_hdr_len= (ip_hdr->ih_vers_ihl & IH_IHL_MASK) * 4;
	tcp_hdr= (tcp_hdr_t *)(pkt->p_data + pkt->p_offset + ip_hdr_len);
	tcp_hdr_len= (tcp_hdr->th_data_off & TH_DO_MASK) >> 2;	/* !!! */
	tot_len= ip_hdr_len + tcp_hdr_len;
	memcpy(state->s_data, ip_hdr, tot_len);
	ip_hdr= (ip_hdr_t *)state->s_data;
	ip_hdr->ih_hdr_chk= 0;
	state->s_ip_hdr_len= ip_hdr_len;
	state->s_tot_len= tot_len;
	rcv_toss= 0;
	rcv_last= state;

	ip_arrived(pkt);
}

DEFUN
(void ip_vjhc_arr_compr, (pkt),
	pkt_ut *pkt
)
{
	ip_hdr_t *ip_hdr;
	tcp_hdr_t *tcp_hdr;
	rcv_state_ut *state;
	int slot;
	int changes;
	u8_t *cp;
	u32_t delta;
	int tot_len;

	DPRINTF(5, ("ip_vjhc_arr_compr\n")); 

	if (!ip_rcv_vjhc)
	{
		DPRINTF(0,
	("ip_vjhc_arr_compr: got packet but compression is not enabled\n"));
		pkt_free(pkt);
		return;
	}

	cp= (u8_t *)pkt->p_data + pkt->p_offset;
	changes= *cp++;
	DPRINTF(5, ("ip_vjhc_arr_compr: changes= 0x%x\n", changes)); 

	if (changes & VJHC_FLAG_C)
	{
		slot= *cp++;
		DPRINTF(5, ("ip_vjhc_arr_compr: packet at slot %d\n", slot)); 
		if (slot >= ip_rcv_vjhc_state_nr)
		{
			DPRINTF(1, ("ip_vjhc_arr_compr: bad slot\n")); 
			rcv_toss= 1;
			pkt_free(pkt);
			return;
		}
		state= rcv_last= &rcv_state[slot];
		rcv_toss= 0;
	}
	else
	{
		if (!ip_rcv_vjhc_compress_cid)
		{
			DPRINTF(0,
("ip_vjhc_arr_compr: got compressed id but id compression is not enabled\n"));
			pkt_free(pkt);
			return;
		}
		if (rcv_toss)
		{
			DPRINTF(1, ("ip_vjhc_arr_compr: tossing packets\n")); 
			pkt_free(pkt);
			return;
		}
		state= rcv_last;
	}

	ip_hdr= (ip_hdr_t *)state->s_data;
	tcp_hdr= (tcp_hdr_t *)(state->s_data+state->s_ip_hdr_len);
	tot_len= state->s_tot_len;
	tcp_hdr->th_chksum= htons((cp[0] << 8) | cp[1]);
	cp += 2;
	if (changes & VJHC_FLAG_P)
		tcp_hdr->th_flags |= THF_PSH;
	else
		tcp_hdr->th_flags &= ~THF_PSH;

	switch (changes & VJHC_SPEC_MASK)
	{
	case VJHC_SPEC_I:
		delta= ntohs(ip_hdr->ih_length) - tot_len;
		tcp_hdr->th_ack_nr= htonl(ntohl(tcp_hdr->th_ack_nr) + delta);
		tcp_hdr->th_seq_nr= htonl(ntohl(tcp_hdr->th_seq_nr) + delta);
		break;
	case VJHC_SPEC_D:
		delta= ntohs(ip_hdr->ih_length) - tot_len;
		tcp_hdr->th_seq_nr= htonl(ntohl(tcp_hdr->th_seq_nr) + delta);
		break;
	default:
		if (changes & VJHC_FLAG_U)
		{
			tcp_hdr->th_flags |= THF_URG;
			VJHC_DECODEU(cp, tcp_hdr->th_urgptr);
		}
		else
			tcp_hdr->th_flags &= ~THF_URG;
		if (changes & VJHC_FLAG_W)
		{
			VJHC_DECODES(cp, tcp_hdr->th_window);
		}
		if (changes & VJHC_FLAG_A)
		{
			DPRINTF(5, ("ip_vjhc_arr_compr: oack= 0x%08x\n",
				tcp_hdr->th_ack_nr)); 
			VJHC_DECODEL(cp, tcp_hdr->th_ack_nr);
			DPRINTF(5, ("ip_vjhc_arr_compr: ack= 0x%08x\n",
				tcp_hdr->th_ack_nr)); 
		}
		if (changes & VJHC_FLAG_S)
		{
			VJHC_DECODEL(cp, tcp_hdr->th_seq_nr);
		}
		break;
	}

	if (changes & VJHC_FLAG_I)
	{
		VJHC_DECODES(cp, ip_hdr->ih_id);
	}
	else
		ip_hdr->ih_id= htons(ntohs(ip_hdr->ih_id) + 1);

	delta= cp - (u8_t *)(pkt->p_data+pkt->p_offset);
	if (delta > pkt->p_size)
	{
		DPRINTF(1, ("ip_vjhc_arr_compr: truncated packet\n")); 
		rcv_toss= 1;
		pkt_free(pkt);
		return;
	}
	pkt->p_offset += delta;
	pkt->p_size -= delta;
	if (pkt->p_offset < tot_len)
	{
		DPRINTF(1, (
		"ip_vjhc_arr_compr: not enough space to insert header\n")); 
		abort();
	}
	pkt->p_offset -= tot_len;
	pkt->p_size += tot_len;
	cp= (u8_t *)pkt->p_data+pkt->p_offset;
	ip_hdr->ih_length= htons(pkt->p_size);
	memcpy(cp, ip_hdr, tot_len);
	ip_hdr= (ip_hdr_t *)cp;
	ip_hdr->ih_hdr_chk= ~oneC_sum((u16_t)0, (u16_t *)ip_hdr,
		state->s_ip_hdr_len);

#if 0
	DPRINTF(1, ("ip_vjhc_arr_compr: packet with size %d\n\t",
		pkt->p_size));
	for (cp= (u8_t *)pkt->p_data+pkt->p_offset; cp < (u8_t *)pkt->p_data +
		pkt->p_offset + state->s_ip_hdr_len; cp++)
	{
		DPRINTF_CONT(1, ("%x ", *cp));
	}
	DPRINTF_CONT(1, ("\n\t"));
	for (; cp < (u8_t *)pkt->p_data + pkt->p_offset + tot_len; cp++)
	{
		DPRINTF_CONT(1, ("%x ", *cp));
	}
	DPRINTF_CONT(1, ("\n\t"));
	for (; cp < (u8_t *)pkt->p_data + pkt->p_offset + pkt->p_size; cp++)
	{
		DPRINTF_CONT(1, ("%x ", *cp));
	}
	DPRINTF_CONT(1, ("\n"));
#endif

	ip_arrived(pkt);
}

/*
 * $PchHeader: /mount/hd2/minix/local/cmd/ppp/RCS/vjhc.c,v 1.2 1995/05/23 08:12:13 philip Exp $
 */
