/*
ipcp_option.c

Parse and generate IPCP options.

Created:	June 16, 1993 by Philip Homburg <philip@cs.vu.nl>
*/

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include "ansi.h"
#include "clock.h"
#include "debug.h"
#include "ipcp.h"
#include "pkt.h"
#include "optlib.h"
#include "ppp.h"
#include "sm.h"
#include "vjhc.h"

#define CHUNK_SIZE	64

static int debug_addresses= 0;

static int ipcpo_compression ARGS(( int action, void *data ));

static option_ent_ut option_table[]=
{
	IPCP_OPTION_COMPRESSION, ipcpo_compression,
	0, 0,
};

DEFUN
(void ipcpo_parse_arg, (action, option),
	int action AND
	char *option
)
{
	opt_vjhc_ut *opt_vjhc_snd_p, *opt_vjhc_rcv_p;
	char *oe, *vorig;
	int ol, min, max;

	DPRINTF(0, ("ipcpo_parse_arg(%d, '%s')\n", action, option));
	if (strncmp(option, OPT_VJHC, strlen(OPT_VJHC)) == 0)
	{
		opt_vjhc_snd_p= NULL;
		opt_vjhc_rcv_p= NULL;

		if ((oe= strchr(option, '=')) != NULL)
			ol= oe-option;
		else
			ol= strlen(option);

		if (ol == strlen(OPT_VJHC))
		{
			opt_vjhc_snd_p= &opt_vjhc_snd;
			opt_vjhc_rcv_p= &opt_vjhc_rcv;
		}
		else if (ol == strlen(OPT_VJHC_SND) &&
			strncmp(option, OPT_VJHC_SND,
			strlen(OPT_VJHC_SND)) == 0)
		{
			opt_vjhc_snd_p= &opt_vjhc_snd;
		}
		else if (ol == strlen(OPT_VJHC_RCV) &&
			strncmp(option, OPT_VJHC_RCV,
			strlen(OPT_VJHC_RCV)) == 0)
		{
			opt_vjhc_rcv_p= &opt_vjhc_rcv;
		}
		else
		{
		DPRINTF(0, ("ipcpo_parse_arg: unrecognized option '%.*s'\n",
				ol, option));
			exit(1);
		}
		if (action != PPP_OPT_NOCHANGE)
		{
			if (opt_vjhc_snd_p)
				opt_vjhc_snd_p->action= action;
			if (opt_vjhc_rcv_p)
				opt_vjhc_rcv_p->action= action;
		}
		if (option[ol] != '=')
			return;
		vorig= option += ol+1;

		if ((oe= strchr(option, ',')) != NULL)
			ol= oe-option;
		else
			ol= strlen(option);
		if (ol == 0 && *option == '\0')
		{
		DPRINTF(0, ("ipcpo_parse_arg, %s: malformed value '%s'\n",
				OPT_VJHC, option));
			exit(1);
		}
		if (ol != 0)
		{
			max= strtol(option, &oe, 0);
			if (oe[0] == '-')
			{
				min= max;
				if (min < OPT_VJHC_MIN_SLOT_NR ||
					min > OPT_VJHC_MAX_SLOT_NR)
				{
					DPRINTF(0,
		("ipcpo_parse_arg, %s: illegal value for min # slots %d\n",
						OPT_VJHC, min));
					exit(1);
				}
				if (opt_vjhc_snd_p)
					opt_vjhc_snd_p->min_slot_nr= min;
				if (opt_vjhc_rcv_p)
					opt_vjhc_rcv_p->min_slot_nr= min;
				option= oe+1;
				max= strtol(option, &oe, 0);
			}
			if (oe[0] != ',' && oe[0] != '\0')
			{
				DPRINTF(0,
			("ipcpo_parse_arg, %s: unable to parse value '%s'\n",
					OPT_VJHC, vorig));
				exit(1);
			}
			if (max < OPT_VJHC_MIN_SLOT_NR ||
				max > OPT_VJHC_MAX_SLOT_NR)
			{
				DPRINTF(0,
		("ipcpo_parse_arg, %s: illegal value for max # slots %d\n",
					OPT_VJHC, max));
				exit(1);
			}
			if (opt_vjhc_snd_p)
			{
				opt_vjhc_snd_p->max_slot_nr= max;
				if (opt_vjhc_snd_p->min_slot_nr > max)
				{
					DPRINTF(0,
		("ipcpo_parse_arg, %s: min # slots (=%d) > max # slots (=%d)\n",
				OPT_VJHC, opt_vjhc_snd_p->min_slot_nr, max));
					exit(1);
				}
			}
			if (opt_vjhc_rcv_p)
			{
				opt_vjhc_rcv_p->max_slot_nr= max;
				if (opt_vjhc_rcv_p->min_slot_nr > max)
				{
					DPRINTF(0,
	("ipcpo_parse_arg, %s: min # slots (=%d) > max # slots (=%d)\n",
				OPT_VJHC, opt_vjhc_rcv_p->min_slot_nr, max));
					exit(1);
				}
			}
			option= oe;
			ol= strlen(option);
		}
		if (option[0] == ',')
		{
			option++;
			ol= strlen(option);

			max= strtol(option, &oe, 0);
			if (oe[0] == '-')
			{
				min= max;
				if (min < 0 || min > 1)
				{
					DPRINTF(0,
			("ipcpo_parse_arg, %s: illegal value for min cid %d\n",
						OPT_VJHC, min));
					exit(1);
				}
				if (opt_vjhc_snd_p)
					opt_vjhc_snd_p->min_cid= min;
				if (opt_vjhc_rcv_p)
					opt_vjhc_rcv_p->min_cid= min;
				option= oe+1;
				max= strtol(option, &oe, 0);
			}
			if (oe[0] != '\0')
			{
				DPRINTF(0,
			("ipcpo_parse_arg, %s: unable to parse value '%s'\n",
					OPT_VJHC, vorig));
				exit(1);
			}
			if (max < 0 || max > 1)
			{
				DPRINTF(0,
		("ipcpo_parse_arg, %s: illegal value for max cid %d\n",
					OPT_VJHC, max));
				exit(1);
			}
			if (opt_vjhc_snd_p)
			{
				opt_vjhc_snd_p->max_cid= max;
				if (opt_vjhc_snd_p->min_cid > max)
				{
					DPRINTF(0,
		("ipcpo_parse_arg, %s: min cid (=%d) > max cid (=%d)\n",
				OPT_VJHC, opt_vjhc_snd_p->min_cid, max));
					exit(1);
				}
			}
			if (opt_vjhc_rcv_p)
			{
				opt_vjhc_rcv_p->max_cid= max;
				if (opt_vjhc_rcv_p->min_cid > max)
				{
					DPRINTF(0,
		("ipcpo_parse_arg, %s: min cid (=%d) > max cid (=%d)\n",
				OPT_VJHC, opt_vjhc_rcv_p->min_cid, max));
					exit(1);
				}
			}
		}
		if (opt_vjhc_snd_p)
		{
			DPRINTF(1,
		("ipcpo_parse_arg: opt_vjhc_snd= { %d, %d, %d, %d, %d }\n",
				opt_vjhc_snd_p->action, 
				opt_vjhc_snd_p->min_slot_nr, 
				opt_vjhc_snd_p->max_slot_nr, 
				opt_vjhc_snd_p->min_cid, 
				opt_vjhc_snd_p->max_cid));
		}
		if (opt_vjhc_rcv_p)
		{
			DPRINTF(1,
		("ipcpo_parse_arg: opt_vjhc_rcv= { %d, %d, %d, %d, %d }\n",
				opt_vjhc_rcv_p->action, 
				opt_vjhc_rcv_p->min_slot_nr, 
				opt_vjhc_rcv_p->max_slot_nr, 
				opt_vjhc_rcv_p->min_cid, 
				opt_vjhc_rcv_p->max_cid));
		}
	}
	else
	{
		DPRINTF(0, ("ipcpo_parse_arg: unrecognized option '%s'\n",
			option));
		exit(1);
	}
}


DEFUN_VOID (void ipcpo_init)
{
	ol_init(option_table);
}


DEFUN_VOID (pkt_ut *ipcpo_mk_request)
{
	return ol_mk_request(option_table);
}


DEFUN
(int ipcpo_got_request, (length, data, ack_p, nak_p, rej_p),
size_t length AND
u8_t *data AND
pkt_ut **ack_p AND
pkt_ut **nak_p AND
pkt_ut **rej_p
)
{
	return ol_got_request(&ipcp_state, option_table, length, data,
		ack_p, nak_p, rej_p);
}

#if 0
DEFUN
(int ipcpo_got_request, (length, data, ack_p, nak_p, rej_p),
size_t length AND
u8_t *data AND
pkt_ut **ack_p AND
pkt_ut **nak_p AND
pkt_ut **rej_p
)
{
	option_ent_ut *ent_p;
	u8_t *dptr;
	size_t l;
	int optcnt;
	int res;

	ipcpo_ack_pkt_p= ack_p;
	ipcpo_nak_pkt_p= nak_p;
	ipcpo_rej_pkt_p= rej_p;

	for (ent_p= option_table; ent_p->oe_func != 0; ent_p++)
	{
		(void)ent_p->oe_func(OEC_RCV_REQ, NULL);
	}

	while (length > 0)
	{
		optcnt= 0;
		for (ent_p= option_table; length > 0; ent_p++)
		{
			if (*data != ent_p->oe_code && ent_p->oe_func != 0)
			{
				continue;
			}
			if (length == 1)
			{
				ipcpo_reject_truncated_option(data);
				return 0;
			}
			l= data[1];
			if (l > length)
			{
				ipcpo_reject_truncated_option(data);
				return 0;
			}
			if (ent_p->oe_func == 0)
			{
				if (optcnt == 0)
				{
					ipcpo_reject_option(data, l);
					length -= l;
					data += l;
				}
				break;
			}
			res= ent_p->oe_func(OEC_RCV_OPTION, data);
			if (res == -1)
				return -1;
			optcnt++;
			length -= l;
			data += l;
		}
	}
	assert(length == 0);
	return 0;
}
#endif


DEFUN_VOID (void ipcpo_got_ack)
{
	ol_got_ack(option_table);
}


#if 0
DEFUN_VOID (void ipcpo_got_ack)
{
	option_ent_ut *ent_p;

	for (ent_p= option_table; ent_p->oe_func != 0; ent_p++)
	{
		(void)ent_p->oe_func(OEC_GOT_ACK, NULL);
	}
}
#endif


DEFUN
(int ipcpo_got_nak, (length, data, req_pkt),
size_t length AND
u8_t *data AND
pkt_ut *req_pkt
)
{
	return ol_got_nak(option_table, length, data, req_pkt);
}


#if 0
DEFUN
(int ipcpo_got_nak, (length, data, req_pkt),
size_t length AND
u8_t *data AND
pkt_ut *req_pkt
)
{
	option_ent_ut *ent_p;
	u8_t *dptr;
	size_t l;
	int optcnt;
	int res;

	while (length > 0)
	{
		optcnt= 0;
		for (ent_p= option_table; length > 0; ent_p++)
		{
			if (*data != ent_p->oe_code && ent_p->oe_func != 0)
			{
				continue;
			}
			if (length == 1)
			{
				DPRINTF(0,
			("ipcpo_got_nak: illegal option length: 1\n"));
				return 0;
			}
			l= data[1];
			if (l > length)
			{
				DPRINTF(0,
			("ipcpo_got_nak: truncated option: %d > %d\n",
					l, length));
				return 0;
			}
			if (ent_p->oe_func == 0)
			{
				if (optcnt == 0)
				{
					DPRINTF(0,
			("ipcpo_got_nak: nak for unknown option: %d\n",
						*data));
					length -= l;
					data += l;
				}
				break;
			}
			res= ent_p->oe_func(OEC_GOT_NAK, data);
			if (res == -1)
				return -1;
			optcnt++;
			length -= l;
			data += l;
		}
	}
	assert(length == 0);
	return 0;
}
#endif


DEFUN
(int ipcpo_got_reject, (length, data, req_pkt),
size_t length AND
u8_t *data AND
pkt_ut *req_pkt
)
{
	ol_got_reject(option_table, length, data, req_pkt);
}


#if 0
DEFUN
(int ipcpo_got_rej, (length, data, req_pkt),
size_t length AND
u8_t *data AND
pkt_ut *req_pkt
)
{
	u8_t *rdata;
	size_t rlength, rl;
	int i, res;
	int rop;
	option_ent_ut *ent_p;

	rlength= req_pkt->p_size;
	rdata= (u8_t *)(req_pkt->p_data + req_pkt->p_offset);
	ent_p= option_table;

	while (length > 0 && rlength > 0)
	{
		rop= rdata[0];
		rl= rdata[1];
		assert(rl >= 2 && rl <= rlength);

		if (rop != data[0])
		{
			/* This option is not rejected */
			rlength -= rl;
			rdata += rl;
			continue;
		}

		/* Option should be a copy of the original option in the
		 * configuration request.
		 */
		if (length < rl ||memcmp(data, rdata, rl) != 0)
			break;

		for (; ent_p->oe_code != rop && ent_p->oe_func != 0; ent_p++)
		{
			;	/* Do nothing */
		}
		if (ent_p->oe_func == 0)
			break;
		res= ent_p->oe_func(OEC_GOT_REJ, NULL);
		if (res == -1)
			return -1;
		rlength -= rl;
		rdata += rl;
		length -= rl;
		data += rl;
	}
	if (length != 0)
	{
		DPRINTF(1, ( "ipcpo_got_rej: bad data, req = " ));
		for (i= 0; i<rlength; i++)
			DPRINTF_CONT(1, ( "%02x", rdata[i] ));
		DPRINTF_CONT(1, ( " repl= " ));
		for (i= 0; i<length; i++)
			DPRINTF_CONT(1, ( "%02x", data[i] ));
		DPRINTF_CONT(1, ( "\n" ));
	}
	return 0;
}
#endif


DEFUN_VOID (void ipcpo_sending_ack)
{
	ol_sending_ack(option_table);
}


#if 0
DEFUN_VOID (void ipcpo_sending_ack)
{
	option_ent_ut *ent_p;

	for (ent_p= option_table; ent_p->oe_func != 0; ent_p++)
	{
		(void)ent_p->oe_func(OEC_SENT_ACK, NULL);
	}
}
#endif


#if 0
DEFUN
(static void ipcpo_ack_option, (data, lenght),
	void *data AND
	size_t length
)
{
	int i;

	DPRINTF(1, ("ipcpo_ack_option: ack option type %d, data: ",
		*(u8_t *)data ));
	for (i= 0; i < length; i++)
		DPRINTF_CONT(1, ( "%02x", ((u8_t *)data)[i] ));
	DPRINTF_CONT(1, ( "\n" ));
	if (*(ipcp_state.sms_ack_pkt_p) == NULL)
		*(ipcp_state.sms_ack_pkt_p)= pkt_alloc(CHUNK_SIZE);
	pkt_add(*(ipcp_state.sms_ack_pkt_p), data, length, CHUNK_SIZE);
}
#endif


#if 0
DEFUN
(static void ipcpo_nak_option, (data, lenght),
	void *data AND
	size_t length
)
{
	int i;

	DPRINTF(1, ("ipcpo_nak_option: nak option type %d, data: ",
		*(u8_t *)data ));
	for (i= 0; i < length; i++)
		DPRINTF_CONT(1, ( "%02x", ((u8_t *)data)[i] ));
	DPRINTF_CONT(1, ( "\n" ));
	if (*ipcp_state.sms_nak_pkt_p == NULL)
		*ipcp_state.sms_nak_pkt_p= pkt_alloc(CHUNK_SIZE);
	pkt_add(*(ipcp_state.sms_nak_pkt_p), data, length, CHUNK_SIZE);
}
#endif


#if 0
DEFUN
(static void ipcpo_reject_option, (data, lenght),
	void *data AND
	size_t length
)
{
	int i;

	DPRINTF(1, ("ipcpo_reject_option: reject option type %d, data: ",
		*(u8_t *)data ));
	for (i= 0; i < length; i++)
		DPRINTF_CONT(1, ( "%02x", ((u8_t *)data)[i] ));
	DPRINTF_CONT(1, ( "\n" ));
	if (*ipcpo_rej_pkt_p == NULL)
		*ipcpo_rej_pkt_p= pkt_alloc(CHUNK_SIZE);
	pkt_add(*ipcpo_rej_pkt_p, data, length, CHUNK_SIZE);
}
#endif


#if 0
DEFUN
(static void ipcpo_reject_truncated_option, (data),
	void *data
)
{
	u8_t bytes[2];

	DPRINTF(1, ("ipcpo_reject_truncated_option: option %d\n",
		*(u8_t *)data));
	bytes[0]= *(u8_t *)data;
	bytes[1]= 2;
	ipcpo_reject_option(bytes, 2);
}
#endif


DEFUN
(static int ipcpo_compression, (action, data),
	int action AND
	void *data
)
{
	pkt_ut *pkt;
	u8_t bytes[6];
	u8_t *option;
	int slot_nr, cid, min_slot, max_slot, min_cid, max_cid;

	switch(action)
	{
	case OEC_INIT:
		DPRINTF(1, ("ipcpo_compression(init)\n"));
		opt_vjhc_snd.curr_action= opt_vjhc_snd.action;
		opt_vjhc_snd.curr_slot_nr= opt_vjhc_snd.max_slot_nr;
		opt_vjhc_snd.curr_cid= opt_vjhc_snd.max_cid;
		opt_vjhc_snd.rej_retry= IPCP_DFLT_MAX_REJ;
		opt_vjhc_snd.nak_retry= IPCP_DFLT_MAX_NAK;
		opt_vjhc_rcv.curr_action= opt_vjhc_rcv.action;
		opt_vjhc_rcv.rej_retry= IPCP_DFLT_MAX_REJ;
		opt_vjhc_rcv.nak_retry= IPCP_DFLT_MAX_NAK;
		return 0;
	case OEC_SND_REQ:
		DPRINTF(1, ("ipcpo_compression(snd_req)\n"));
		opt_vjhc_snd.curr_opt= NULL;
		pkt= data;
		action= opt_vjhc_snd.curr_action;
		if (action == PPP_OPT_FORCED_OFF || action == PPP_OPT_OFF)
			return 0;
		assert (action == PPP_OPT_FORCED_ON || action == PPP_OPT_ON);
		bytes[0]= IPCP_OPTION_COMPRESSION;
		bytes[1]= 6;
		bytes[2]= (PPP_TYPE_VJHC_COMPR >> 8);
		bytes[3]= (PPP_TYPE_VJHC_COMPR & 0xFF);
		bytes[4]= opt_vjhc_snd.curr_slot_nr-1;
		bytes[5]= opt_vjhc_snd.curr_cid;
		pkt_add(pkt, bytes, 6, CHUNK_SIZE);
		return 0;
	case OEC_RCV_REQ:
		DPRINTF(1, ("ipcpo_compression(rcv_req)\n"));
		opt_vjhc_rcv.curr_opt= NULL;
		return 0;
	case OEC_RCV_OPTION:
		DPRINTF(1, ("ipcpo_compression(rcv_option)\n"));
		option= data;
		action= opt_vjhc_rcv.curr_action;
		if (action == PPP_OPT_FORCED_OFF)
		{
			ol_reject_option(&ipcp_state, option, option[1]);
			return 0;
		}
		if (action == PPP_OPT_OFF)
		{
			action= opt_vjhc_rcv.curr_action= PPP_OPT_ON;
		}
		if (opt_vjhc_rcv.curr_opt != NULL)
		{
			if (memcmp(option, opt_vjhc_rcv.curr_opt,
				option[1]) != 0)
			{
				DPRINTF(0,
("ipcpo_compression: rejecting duplicate option with different values\n"));
				ol_reject_option(&ipcp_state, option,
					option[1]);
				return 0;
			}
			DPRINTF(1,
				("ipcpo_compression: got duplicate option\n"));
		}
		else
			opt_vjhc_rcv.curr_opt= option;

		min_slot= opt_vjhc_rcv.min_slot_nr;
		max_slot= opt_vjhc_rcv.max_slot_nr;
		min_cid= opt_vjhc_rcv.min_cid;
		max_cid= opt_vjhc_rcv.max_cid;

		/* Try the ACK case first */
		if (option[1] == 6 &&
			option[2] == (PPP_TYPE_VJHC_COMPR >> 8) &&
			option[3] == (PPP_TYPE_VJHC_COMPR & 0xFF))
		{
			slot_nr= option[4]+1;
			cid= option[5];
			if (slot_nr >= min_slot && slot_nr <= max_slot &&
				cid >= min_cid && cid <= max_cid)
			{
				opt_vjhc_rcv.curr_slot_nr= slot_nr;
				opt_vjhc_rcv.curr_cid= cid;
				ol_ack_option(&ipcp_state, option, option[1]);
				return 0;
			}
		}

		/* Check the retry counter */
		if (opt_vjhc_rcv.nak_retry > 0)
			opt_vjhc_rcv.nak_retry--;
		else
		{
			DPRINTF(0, ("ipcpo_compression: value conflict\n"));
			if (action == PPP_OPT_FORCED_ON)
				return -1;
			assert(action == PPP_OPT_ON);
			action= opt_vjhc_rcv.curr_action= PPP_OPT_OFF;
			ol_reject_option(&ipcp_state, option, option[1]);
			return 0;
		}

		/* Standard NAK */
		bytes[0]= IPCP_OPTION_COMPRESSION;
		bytes[1]= 6;
		bytes[2]= (PPP_TYPE_VJHC_COMPR >> 8);
		bytes[3]= (PPP_TYPE_VJHC_COMPR & 0xFF);
		bytes[4]= max_slot-1;
		bytes[5]= max_cid;
		if (option[1] != 6 ||
			option[2] != (PPP_TYPE_VJHC_COMPR >> 8) &&
			option[3] != (PPP_TYPE_VJHC_COMPR & 0xFF))
		{
			DPRINTF(1,
			("ipcpo_compression: strange length or type\n"));
			ol_nak_option(&ipcp_state, bytes, bytes[1]);
			return 0;
		}
		if (slot_nr < min_slot)
			bytes[4]= min_slot-1;
		else if (slot_nr >= min_slot && slot_nr <= max_slot)
			bytes[4]= slot_nr-1;
		if (cid < min_cid)
			bytes[5]= min_cid;
		else if (cid >= min_cid && cid <= max_cid)
			bytes[5]= cid;

		ol_nak_option(&ipcp_state, bytes, bytes[1]);
		return 0;
	case OEC_RCV_REQ_END:
		DPRINTF(1, ("ipcpo_compression(rcv_req_end)\n"));
		if (opt_vjhc_rcv.curr_opt != NULL)
		{
			/* Option received and processed. */
			return 0;
		}
		action= opt_vjhc_rcv.curr_action;
		if (action == PPP_OPT_FORCED_OFF || action == PPP_OPT_OFF)
			return 0;	/* We do not care */

		assert (action == PPP_OPT_FORCED_ON || action == PPP_OPT_ON);

		/* Standard NAK */
		bytes[0]= IPCP_OPTION_COMPRESSION;
		bytes[1]= 6;
		bytes[2]= (PPP_TYPE_VJHC_COMPR >> 8);
		bytes[3]= (PPP_TYPE_VJHC_COMPR & 0xFF);
		bytes[4]= opt_vjhc_rcv.curr_slot_nr-1;
		bytes[5]= opt_vjhc_rcv.curr_cid;

		if (opt_vjhc_rcv.nak_retry > 0)
		{
			opt_vjhc_rcv.nak_retry--;
			ol_nak_option(&ipcp_state, bytes, bytes[1]);
			return 0;
		}
		if (action == PPP_OPT_ON)
		{
			opt_vjhc_rcv.curr_action= PPP_OPT_OFF;
			return 0;
		}
		DPRINTF(0, ("lcpo_compression: on/off conflict\n"));
		return -1;
	case OEC_GOT_NAK:
		DPRINTF(1, ("ipcpo_compression(got_nak)\n"));
		option= data;
		action= opt_vjhc_snd.curr_action;
		if (action == PPP_OPT_FORCED_OFF)
		{
			return 0;
		}
		if (action == PPP_OPT_OFF)
		{
			action= opt_vjhc_snd.curr_action= PPP_OPT_ON;
		}
		if (opt_vjhc_snd.curr_opt == NULL)
			opt_vjhc_snd.curr_opt= option;

		min_slot= opt_vjhc_snd.min_slot_nr;
		max_slot= opt_vjhc_snd.max_slot_nr;
		min_cid= opt_vjhc_snd.min_cid;
		max_cid= opt_vjhc_snd.max_cid;

		/* Try the acceptable case first */
		if (option[1] == 6 &&
			option[2] == (PPP_TYPE_VJHC_COMPR >> 8) &&
			option[3] == (PPP_TYPE_VJHC_COMPR & 0xFF))
		{
			slot_nr= option[4]+1;
			cid= option[5];
			if (slot_nr >= min_slot && slot_nr <= max_slot &&
				cid >= min_cid && cid <= max_cid)
			{
				opt_vjhc_snd.curr_slot_nr= slot_nr;
				opt_vjhc_snd.curr_cid= cid;
				return 0;
			}
		}

		if (option == opt_vjhc_snd.curr_opt)
		{
			/* Check the retry counter */
			if (opt_vjhc_snd.nak_retry > 0)
				opt_vjhc_snd.nak_retry--;
			else
			{
				DPRINTF(0,
				("ipcpo_compression: value conflict\n"));
				if (action == PPP_OPT_FORCED_ON)
					return -1;
				assert(action == PPP_OPT_ON);
				action= opt_vjhc_snd.curr_action= PPP_OPT_OFF;
				return 0;
			}

			if (option[1] != 6 ||
				option[2] != (PPP_TYPE_VJHC_COMPR >> 8) &&
				option[3] != (PPP_TYPE_VJHC_COMPR & 0xFF))
			{
				DPRINTF(1,
			("ipcpo_compression: strange length or type\n"));
				return 0;
			}
			if (slot_nr < min_slot)
				opt_vjhc_snd.curr_slot_nr= min_slot;
			else if (slot_nr >= min_slot && slot_nr <= max_slot)
				opt_vjhc_snd.curr_slot_nr= slot_nr;
			if (cid < min_cid)
				opt_vjhc_snd.curr_cid= min_cid;
			else if (cid >= min_cid && cid <= max_cid)
				opt_vjhc_snd.curr_cid= cid;
		}
		return 0;
	case OEC_GOT_REJ:
		DPRINTF(1, ("ipcpo_compression(got_rej)\n"));
		if (opt_vjhc_snd.curr_action == PPP_OPT_FORCED_ON)
		{
			if (opt_vjhc_snd.rej_retry > 0)
			{
				opt_vjhc_snd.rej_retry--;
				return 0;
			}
			DPRINTF(0, ("ipcpo_compression: on/off conflict\n"));
			return -1;
		}
		assert(opt_vjhc_snd.curr_action == PPP_OPT_ON);
		opt_vjhc_snd.curr_action= PPP_OPT_OFF;
		return 0;
	case OEC_SENT_ACK:
		DPRINTF(1, ("ipcpo_compression(sent_ack)\n"));
		action= opt_vjhc_rcv.curr_action;
		if (action == PPP_OPT_ON || action == PPP_OPT_FORCED_ON)
		{
			ip_snd_vjhc= 1;
			ip_snd_vjhc_state_nr= opt_vjhc_rcv.curr_slot_nr;
			ip_snd_vjhc_compress_cid= opt_vjhc_rcv.curr_cid;
		}
		else
		{
			assert(action == PPP_OPT_OFF ||
				action == PPP_OPT_FORCED_OFF);
			ip_snd_vjhc= 0;
		}
		return 0;
	case OEC_GOT_ACK:
		DPRINTF(1, ("ipcpo_compression(got_ack)\n"));
		action= opt_vjhc_snd.curr_action;
		if (action == PPP_OPT_ON || action == PPP_OPT_FORCED_ON)
		{
			ip_rcv_vjhc= 1;
			ip_rcv_vjhc_state_nr= opt_vjhc_snd.curr_slot_nr;
			ip_rcv_vjhc_compress_cid= opt_vjhc_snd.curr_cid;
		}
		else
		{
			assert(action == PPP_OPT_OFF ||
				action == PPP_OPT_FORCED_OFF);
			ip_rcv_vjhc= 0;
		}
		return 0;
	default:
		DPRINTF(0, ("ipcpo_compression: unknown action %d\n", action));
		abort();
	}
}

/*
 * $PchHeader: /mount/hd2/minix/local/cmd/ppp/RCS/ipcp_option.c,v 1.1 1995/02/14 07:52:41 philip Exp $
 */
