/*
ipcp.c

Implementation of the IP control protocol.

Created:	May 28, 1993 by Philip Homburg <philip@cs.vu.nl>
*/

#include <sys/types.h>
#include <assert.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "ansi.h"
#include "clock.h"
#include "debug.h"
#include "ip.h"
#include "ipcp.h"
#include "ncp.h"
#include "optlib.h"
#include "pkt.h"
#include "ppp.h"
#include "sm.h"

sm_state_ut ipcp_state;
sm_action_ut ipcp_action;

#define IPCP_PKT_Q_SIZE	64

static pkt_ut *ipcp_pkt_q[IPCP_PKT_Q_SIZE];
static pkt_ut **ipcp_pkt_nq_ptr;

static ppp_snd_callback_ut ipcp_snd_callback;
static pkt_ut *ipcp_code_rej_pkt;

static void ipcp_snd_restart ARGS(( struct ppp_snd_callback *cb, int arg ));
static void ipcp_send_data ARGS(( int type, void *data, size_t size ));

static void ipcp_action_tlu ARGS(( void ));
static void ipcp_action_tld ARGS(( void ));
static void ipcp_action_tls ARGS(( void ));
static void ipcp_action_tlf ARGS(( void ));

static void ipcp_action_irc ARGS(( int retry_cnt ));
static void ipcp_action_zrc ARGS(( void ));

static void ipcp_action_scr ARGS(( void ));

static void ipcp_action_sca ARGS(( void ));
static void ipcp_action_scn ARGS(( void ));

static void ipcp_action_str ARGS(( void ));
static void ipcp_action_sta ARGS(( void ));

static void ipcp_action_scj ARGS(( void ));

static void ipcp_configure_req ARGS(( int identifier, size_t length,
						u8_t *data, pkt_ut *pkt ));
static void ipcp_configure_ack ARGS(( int identifier, size_t length,
						u8_t *data, pkt_ut *pkt ));
static void ipcp_configure_nak ARGS(( int identifier, size_t length,
						u8_t *data, pkt_ut *pkt ));
static void ipcp_configure_reject ARGS(( int identifier, size_t length,
						u8_t *data, pkt_ut *pkt ));
static void ipcp_terminate_req ARGS(( int identifier, size_t length,
						u8_t *data, pkt_ut *pkt ));
static void ipcp_terminate_ack ARGS(( int identifier, size_t length,
						u8_t *data, pkt_ut *pkt ));
static void ipcp_code_reject ARGS(( int identifier, size_t length,
						u8_t *data, pkt_ut *pkt ));

DEFUN_VOID (void ipcp_init)
{
	int i;

	DPRINTF(1, ("ipcp_init()\n"));

	ipcp_action.sma_tlu= ipcp_action_tlu;
	ipcp_action.sma_tld= ipcp_action_tld;
	ipcp_action.sma_tls= ipcp_action_tls;
	ipcp_action.sma_tlf= ipcp_action_tlf;
	ipcp_action.sma_irc= ipcp_action_irc;
	ipcp_action.sma_zrc= ipcp_action_zrc;
	ipcp_action.sma_scr= ipcp_action_scr;
	ipcp_action.sma_sca= ipcp_action_sca;
	ipcp_action.sma_scn= ipcp_action_scn;
	ipcp_action.sma_str= ipcp_action_str;
	ipcp_action.sma_sta= ipcp_action_sta;
	ipcp_action.sma_scj= ipcp_action_scj;

	sm_init(&ipcp_state, "ipcp", &ipcp_action);
	ipcp_state.sms_max_configure= IPCP_DFLT_MAX_CONFIGURE;
	ipcp_state.sms_max_terminate= IPCP_DFLT_MAX_TERMINATE;
	ipcp_state.sms_restart_timer= IPCP_DFLT_RESTART_TIMER;
	ipcp_state.sms_type_req= IPCP_TYPE_CONFIGURE_REQ;
	ipcp_state.sms_type_ack= IPCP_TYPE_CONFIGURE_ACK;
	ipcp_state.sms_type_nak= IPCP_TYPE_CONFIGURE_NAK;
	ipcp_state.sms_type_reject= IPCP_TYPE_CONFIGURE_REJECT;
	ipcp_state.sms_last_snd_seq= 0;
	ipcp_state.sms_option_init= ipcpo_init;
	ipcp_state.sms_mk_request= ipcpo_mk_request;
	ipcp_state.sms_send_data= ipcp_send_data;
	ipcp_state.sms_got_request= ipcpo_got_request;
	ipcp_state.sms_sending_ack= ipcpo_sending_ack;
	ipcp_state.sms_got_ack= ipcpo_got_ack;
	ipcp_state.sms_got_nak= ipcpo_got_nak;
	ipcp_state.sms_got_reject= ipcpo_got_reject;

	for (i= 0; i<IPCP_PKT_Q_SIZE; i++)
		ipcp_pkt_q[i]= NULL;
	ipcp_pkt_nq_ptr= ipcp_pkt_q;

	ppp_snd_init_callback(&ipcp_snd_callback, ipcp_snd_restart, 0);
}


DEFUN_VOID (static void ipcp_action_tlu)
{
	DPRINTF(1, ("ipcp_action_tlu()\n"));
	ip_up();
}


DEFUN_VOID (static void ipcp_action_tld)
{
	DPRINTF(1, ("ipcp_action_tld()\n"));
	ip_down();
}


DEFUN_VOID (static void ipcp_action_tls)
{
	DPRINTF(1, ("ipcp_action_tls()\n"));
	ncp_tls();
}


DEFUN_VOID (static void ipcp_action_tlf)
{
	DPRINTF(1, ("ipcp_action_tlf()\n"));
	ncp_tlf();
}


DEFUN
(static void ipcp_action_irc, (retry_cnt),
	int retry_cnt
)
{
	DPRINTF(1, ("ipcp_action_irc()\n"));
	ipcp_state.sms_restart_counter= retry_cnt;
}


DEFUN_VOID (static void ipcp_action_zrc)
{
	DPRINTF(1, ("ipcp_action_zrc()\n"));
	ipcp_state.sms_restart_counter= 0;
}


DEFUN_VOID (static void ipcp_action_scr)
{
	DPRINTF(1, ("ipcp_action_scr()\n"));

	ol_send_cr(&ipcp_state);
}


#if 0
DEFUN_VOID (static void ipcp_action_scr)
{
	pkt_ut *pkt, *optpkt;
	u8_t *dptr;
	int error;
	size_t option_size;

	DPRINTF(1, ("ipcp_action_scr()\n"));

	assert(ipcp_state.sms_restart_counter > 0);
	ipcp_state.sms_restart_counter--;
	clck_set_timer(&ipcp_state.sms_timer, clck_get_time() + 
		ipcp_state.sms_restart_timer);

	if (ipcp_scr_pkt != NULL)
	{
		pkt_free(ipcp_scr_pkt);
	}

	if (!ipcp_state.sms_options_inited)
	{
		ipcpo_init();
		ipcp_state.sms_options_inited= 1;
	}

	optpkt= ipcpo_mk_request();
	ipcp_scr_pkt= optpkt;
	option_size= optpkt->p_size;

	pkt= pkt_alloc(PPP_MAX_HDR_SIZE+4+option_size);
	pkt->p_offset= PPP_MAX_HDR_SIZE;
	pkt->p_size= 4+option_size;
	assert(pkt->p_offset + pkt->p_size <= pkt->p_maxsize);
	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);
	dptr[0]= IPCP_TYPE_CONFIGURE_REQ;
	dptr[1]= ++ipcp_last_snd_seq;
	dptr[2]= 0;
	dptr[3]= 4+option_size;
	memcpy(dptr+4, optpkt->p_data+optpkt->p_offset, option_size);

	ppp_snd_sethdr(pkt, PPP_TYPE_IPCP);
	error= ppp_snd(pkt, &ipcp_snd_callback);
	if (error == 0)
		return;
	assert(error == EAGAIN);
	if (*ipcp_pkt_nq_ptr != NULL)
		pkt_free(*ipcp_pkt_nq_ptr);
	*ipcp_pkt_nq_ptr= pkt;
	ipcp_pkt_nq_ptr++;
	if (ipcp_pkt_nq_ptr == &ipcp_pkt_q[IPCP_PKT_Q_SIZE])
		ipcp_pkt_nq_ptr= ipcp_pkt_q;
}
#endif


DEFUN_VOID (static void ipcp_action_sca)
{
	ol_sca(&ipcp_state);
}


#if 0
DEFUN_VOID (static void ipcp_action_sca)
{
	pkt_ut *pkt;
	u8_t *dptr;
	int error;
	size_t option_size;

	DPRINTF(1, ("ipcp_action_sca()\n"));

	ipcpo_sending_ack();

	option_size= ipcp_state.sms_sca_pkt->p_size;

	pkt= pkt_alloc(PPP_MAX_HDR_SIZE+4+option_size);
	pkt->p_offset= PPP_MAX_HDR_SIZE;
	pkt->p_size= 4+option_size;
	assert(pkt->p_offset + pkt->p_size <= pkt->p_maxsize);
	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);
	dptr[0]= IPCP_TYPE_CONFIGURE_ACK;
	dptr[1]= ipcp_state.sms_last_rcvd_seq;
	dptr[2]= 0;
	dptr[3]= 4+option_size;
	if (option_size != 0)
	{
		memcpy(dptr+4, ipcp_state.sms_sca_pkt->p_data +
			ipcp_state.sms_sca_pkt->p_offset, option_size);
	}
	ppp_snd_sethdr(pkt, PPP_TYPE_IPCP);
	error= ppp_snd(pkt, &ipcp_snd_callback);
	if (error == 0)
		return;
	assert(error == EAGAIN);
	if (*ipcp_pkt_nq_ptr != NULL)
		pkt_free(*ipcp_pkt_nq_ptr);
	*ipcp_pkt_nq_ptr= pkt;
	ipcp_pkt_nq_ptr++;
	if (ipcp_pkt_nq_ptr == &ipcp_pkt_q[IPCP_PKT_Q_SIZE])
		ipcp_pkt_nq_ptr= ipcp_pkt_q;
}
#endif


DEFUN_VOID (static void ipcp_action_scn)
{
	ol_scn(&ipcp_state);
}


#if 0
DEFUN_VOID (static void ipcp_action_scn)
{
	pkt_ut *pkt;
	u8_t *dptr;
	int error;
	size_t option_size;

	DPRINTF(1, ("ipcp_action_scn()\n"));

	option_size= ipcp_state.sms_scn_pkt->p_size;

	pkt= pkt_alloc(PPP_MAX_HDR_SIZE+4+option_size);
	pkt->p_offset= PPP_MAX_HDR_SIZE;
	pkt->p_size= 4+option_size;
	assert(pkt->p_offset + pkt->p_size <= pkt->p_maxsize);
	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);
	dptr[0]= ipcp_state.sms_scn_type;
	dptr[1]= ipcp_state.sms_last_rcvd_seq;
	dptr[2]= 0;
	dptr[3]= 4+option_size;
	memcpy(dptr+4, ipcp_state.sms_scn_pkt->p_data +
		ipcp_state.sms_scn_pkt->p_offset, option_size);
	pkt_free(ipcp_state.sms_scn_pkt);
	ipcp_state.sms_scn_pkt= NULL;

	ppp_snd_sethdr(pkt, PPP_TYPE_IPCP);
	error= ppp_snd(pkt, &ipcp_snd_callback);
	if (error == 0)
		return;
	assert(error == EAGAIN);
	if (*ipcp_pkt_nq_ptr != NULL)
		pkt_free(*ipcp_pkt_nq_ptr);
	*ipcp_pkt_nq_ptr= pkt;
	ipcp_pkt_nq_ptr++;
	if (ipcp_pkt_nq_ptr == &ipcp_pkt_q[IPCP_PKT_Q_SIZE])
		ipcp_pkt_nq_ptr= ipcp_pkt_q;
}
#endif


DEFUN_VOID (static void ipcp_action_str)
{
	pkt_ut *pkt;
	u8_t *dptr;
	int error;

	DPRINTF(1, ("ipcp_action_str()\n"));

	assert(ipcp_state.sms_restart_counter > 0);
	ipcp_state.sms_restart_counter--;
	clck_set_timer(&ipcp_state.sms_timer, clck_get_time() + 
						ipcp_state.sms_restart_timer);

	pkt= pkt_alloc(PPP_MAX_HDR_SIZE+4);
	pkt->p_offset= PPP_MAX_HDR_SIZE;
	pkt->p_size= 4;
	assert(pkt->p_offset + pkt->p_size <= pkt->p_maxsize);
	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);
	dptr[0]= IPCP_TYPE_TERMINATE_REQ;
	dptr[1]= ++ipcp_state.sms_last_snd_seq;
	dptr[2]= 0;
	dptr[3]= 4;
	ppp_snd_sethdr(pkt, PPP_TYPE_IPCP);
	error= ppp_snd(pkt, &ipcp_snd_callback);
	if (error == 0)
		return;
	assert(error == EAGAIN);
	if (*ipcp_pkt_nq_ptr != NULL)
		pkt_free(*ipcp_pkt_nq_ptr);
	*ipcp_pkt_nq_ptr= pkt;
	ipcp_pkt_nq_ptr++;
	if (ipcp_pkt_nq_ptr == &ipcp_pkt_q[IPCP_PKT_Q_SIZE])
		ipcp_pkt_nq_ptr= ipcp_pkt_q;
}


DEFUN_VOID (static void ipcp_action_sta)
{
	pkt_ut *pkt;
	u8_t *dptr;
	int error;

	DPRINTF(1, ("ipcp_action_sta()\n"));

	pkt= pkt_alloc(PPP_MAX_HDR_SIZE+4);
	pkt->p_offset= PPP_MAX_HDR_SIZE;
	pkt->p_size= 4;
	assert(pkt->p_offset + pkt->p_size <= pkt->p_maxsize);
	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);
	dptr[0]= IPCP_TYPE_TERMINATE_ACK;
	dptr[1]= ipcp_state.sms_last_rcvd_seq;
	dptr[2]= 0;
	dptr[3]= 4;
	ppp_snd_sethdr(pkt, PPP_TYPE_IPCP);
	error= ppp_snd(pkt, &ipcp_snd_callback);
	if (error == 0)
		return;
	assert(error == EAGAIN);
	if (*ipcp_pkt_nq_ptr != NULL)
		pkt_free(*ipcp_pkt_nq_ptr);
	*ipcp_pkt_nq_ptr= pkt;
	ipcp_pkt_nq_ptr++;
	if (ipcp_pkt_nq_ptr == &ipcp_pkt_q[IPCP_PKT_Q_SIZE])
		ipcp_pkt_nq_ptr= ipcp_pkt_q;
}


DEFUN_VOID (static void ipcp_action_scj)
{
	size_t length, sendlength;
	u8_t *dptr;
	int error;
	pkt_ut *pkt, *rej_pkt;

	DPRINTF(1, ("ipcp_action_scj\n"));

	rej_pkt= ipcp_code_rej_pkt;
	length= rej_pkt->p_size;

	if (4 + length > ppp_snd_maxsize)
		sendlength= ppp_snd_maxsize-4;
	else
		sendlength= length;

	pkt= pkt_alloc(PPP_MAX_HDR_SIZE+4+sendlength);
	pkt->p_offset= PPP_MAX_HDR_SIZE;
	pkt->p_size= 4+sendlength;
	assert(pkt->p_offset + pkt->p_size <= pkt->p_maxsize);
	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);
	dptr[0]= IPCP_TYPE_CODE_REJECT;
	dptr[1]= ++ipcp_state.sms_last_snd_seq;
	dptr[2]= (4+sendlength) >> 8;
	dptr[3]= 4+sendlength;
	memcpy(dptr+4, rej_pkt->p_data + rej_pkt->p_offset, sendlength);

	pkt_free(ipcp_code_rej_pkt);
	ipcp_code_rej_pkt= NULL;

	ppp_snd_sethdr(pkt, PPP_TYPE_IPCP);
	error= ppp_snd(pkt, &ipcp_snd_callback);
	if (error == 0)
		return;
	assert(error == EAGAIN);
	if (*ipcp_pkt_nq_ptr != NULL)
		pkt_free(*ipcp_pkt_nq_ptr);
	*ipcp_pkt_nq_ptr= pkt;
	ipcp_pkt_nq_ptr++;
	if (ipcp_pkt_nq_ptr == &ipcp_pkt_q[IPCP_PKT_Q_SIZE])
		ipcp_pkt_nq_ptr= ipcp_pkt_q;
}


DEFUN
(void ipcp_reject_proto, (pkt),
	pkt_ut *pkt
)
{
	DPRINTF(1, ("ipcp_reject_proto: not implemented\n"));
	abort();
}


DEFUN
(void ipcp_arrived, (pkt),
	pkt_ut *pkt
)
{
	int code, identifier;
	size_t length;
	u8_t *dptr;

	if (!ipcp_state.sms_isup)
	{
		DPRINTF(1, ("ipcp_arrived: got packet while down\n"));
		pkt_free(pkt);
		return;
	}

	if (pkt->p_size < 4)
	{
		DPRINTF(1, ("ipcp_arrived: packet with invalid length (%d)\n",
			pkt->p_size));
		pkt_free(pkt);
		return;
	}
	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);
	code= dptr[0];
	identifier= dptr[1];
	length= (dptr[2] << 8) | dptr[3];
	if (length < 4 || length > pkt->p_size)
	{
		DPRINTF(1, (
			"ipcp_arrived: packet with invalid length field (%d)\n",
				length));
		pkt_free(pkt);
		return;
	}
	pkt->p_size= length;

	switch(code)
	{
	case IPCP_TYPE_CONFIGURE_REQ:
		ipcp_configure_req(identifier, length, dptr, pkt);
		break;
	case IPCP_TYPE_CONFIGURE_ACK:
		ipcp_configure_ack(identifier, length, dptr, pkt);
		break;
	case IPCP_TYPE_CONFIGURE_NAK:
		ipcp_configure_nak(identifier, length, dptr, pkt);
		break;
	case IPCP_TYPE_CONFIGURE_REJECT:
		ipcp_configure_reject(identifier, length, dptr, pkt);
		break;
	case IPCP_TYPE_TERMINATE_REQ:
		ipcp_terminate_req(identifier, length, dptr, pkt);
		break;
	case IPCP_TYPE_TERMINATE_ACK:
		ipcp_terminate_ack(identifier, length, dptr, pkt);
		break;
	case IPCP_TYPE_CODE_REJECT:
		ipcp_code_reject(identifier, length, dptr, pkt);
		break;
	default:
		assert(ipcp_code_rej_pkt == NULL);
		ipcp_code_rej_pkt= pkt;
		sm_ruc(&ipcp_state);
		break;
	}
}


DEFUN
(static void ipcp_configure_req, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	ol_configure_req(&ipcp_state, identifier, length, data, pkt);
}
#if 0
DEFUN
(static void ipcp_configure_req, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	pkt_ut *ack, *nak, *rej;
	int res;

	if (!ipcp_state.sms_options_inited)
	{
		ipcpo_init();
		ipcp_state.sms_options_inited= 1;
	}

	ipcp_last_rcvd_seq= identifier;
	length -= 4;
	data += 4;

	if (ipcp_sca_pkt != NULL)
	{
		pkt_free(ipcp_sca_pkt);
		ipcp_sca_pkt= NULL;
	}
	if (ipcp_scn_pkt != NULL)
	{
		pkt_free(ipcp_scn_pkt);
		ipcp_scn_pkt= NULL;
	}
	ack= NULL;
	nak= NULL;
	rej= NULL;
	res= ipcpo_got_request(length, data, &ack, &nak, &rej);
	pkt_free(pkt);

	if (res == -1)
	{
		if (ack)
			pkt_free(ack);
		if (nak)
			pkt_free(nak);
		if (rej)
			pkt_free(rej);
		DPRINTF(0,
		("ipcp_configure_req: options do not converge, closing\n"));
		sm_close(&ipcp_state);
		return;
	}
	if (rej != NULL)
	{
		ipcp_scn_pkt= rej;
		ipcp_scn_type= IPCP_TYPE_CONFIGURE_REJECT;
		if (ack != NULL)
			pkt_free(ack);
		if (nak != NULL)
			pkt_free(nak);
		sm_rcr_minus(&ipcp_state);
		return;
	}
	if (nak != NULL)
	{
		ipcp_scn_pkt= nak;
		ipcp_scn_type= IPCP_TYPE_CONFIGURE_NAK;
		if (ack != NULL)
			pkt_free(ack);
		sm_rcr_minus(&ipcp_state);
		return;
	}
	if (ack == NULL)
		ack= pkt_alloc(1);
	ipcp_sca_pkt= ack;
	sm_rcr_plus(&ipcp_state);
}
#endif


DEFUN
(static void ipcp_configure_ack, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	ol_configure_ack(&ipcp_state, identifier, length, data, pkt);
}


#if 0
(static void ipcp_configure_ack, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	if (identifier != ipcp_last_snd_seq)
	{
		DPRINTF(1, ( 
"ipcp_configure_ack: got packet with invalid sequence number (%d != %d)\n",
				identifier, ipcp_last_snd_seq));
		pkt_free(pkt);
		return;
	}

	if (length != 4+ipcp_state.sms_scr_pkt->p_size ||
		(ipcp_state.sms_scr_pkt->p_size != 0 && 
		memcmp(data+4, ipcp_state.sms_scr_pkt->p_data +
		ipcp_state.sms_scr_pkt->p_offset, length-4) != 0))
	{
		DPRINTF(1, ( 
		"ipcp_configure_ack: packet didn't ack the right options\n" ));
		pkt_free(pkt);
		return;
	}
	ipcpo_got_ack();
	pkt_free(pkt);
	sm_rca(&ipcp_state);
}
#endif


DEFUN
(static void ipcp_configure_nak, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	ol_configure_nak(&ipcp_state, identifier, length, data, pkt);
}


#if 0
DEFUN
(static void ipcp_configure_nak, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	int res;
	if (identifier != ipcp_state.sms_last_snd_seq)
	{
		DPRINTF(1, (
"ipcp_configure_nak: got packet with invalid sequence number (%d != %d)\n",
				identifier, ipcp_state.sms_last_snd_seq));
		pkt_free(pkt);
		return;
	}

	length -= 4;
	data += 4;

	res= ipcpo_got_nak(length, data, ipcp_state.sms_scr_pkt);
	pkt_free(pkt);

	if (res == -1)
	{
		DPRINTF(0, ("options do not converge, closing\n"));
		sm_close(&ipcp_state);
	}
	else
		sm_rcn(&ipcp_state);
}
#endif


DEFUN
(static void ipcp_configure_reject, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	ol_configure_reject(&ipcp_state, identifier, length, data, pkt);
}


#if 0
DEFUN
(static void ipcp_configure_reject, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	int res;
	if (identifier != ipcp_state.sms_last_snd_seq)
	{
		DPRINTF(1, (
"ipcp_configure_reject: got packet with invalid sequence number (%d != %d)\n",
				identifier, ipcp_state.sms_last_snd_seq));
		pkt_free(pkt);
		return;
	}

	length -= 4;
	data += 4;

	res= ipcpo_got_rej(length, data, ipcp_state.sms_scr_pkt);
	pkt_free(pkt);

	if (res == -1)
	{
		DPRINTF(0, ("options do not converge, closing\n"));
		sm_close(&ipcp_state);
	}
	else
		sm_rcn(&ipcp_state);
}
#endif


DEFUN
(static void ipcp_terminate_req, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	DPRINTF(1, ("ipcp_terminate_req\n"));
	ipcp_state.sms_last_rcvd_seq= identifier;

	pkt_free(pkt);
	sm_rtr(&ipcp_state);
}


DEFUN
(static void ipcp_terminate_ack, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	DPRINTF(1, ("ipcp_terminate_ack\n"));
	if (identifier != ipcp_state.sms_last_snd_seq)
	{
		DPRINTF(1, (
"ipcp_terminate_ack: got packet with invalid sequence number (%d != %d)\n",
				identifier, ipcp_state.sms_last_snd_seq));
		pkt_free(pkt);
		return;
	}

	pkt_free(pkt);
	sm_rta(&ipcp_state);
}


DEFUN
(static void ipcp_code_reject, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	int i;

	DPRINTF(1, ("ipcp_code_reject: length= %d, data=", length-4));
	for (i= 4; i<length && i<20; i++)
		DPRINTF_CONT(1, (" %02x", data[i]));
	if (i<length)
		DPRINTF_CONT(1, (" ..."));
	DPRINTF_CONT(1, ("\n"));
	pkt_free(pkt);
	sm_rxj_minus(&ipcp_state);
}


DEFUN
(static void ipcp_snd_restart, (cb, arg),
	ppp_snd_callback_ut *cb AND
	int arg

)
{
	int i, error;

	assert(cb == &ipcp_snd_callback);
	assert(arg == 0);

	for (i= 0; i<IPCP_PKT_Q_SIZE; i++)
	{
		if (*ipcp_pkt_nq_ptr != NULL)
		{
			DPRINTF(0, ("ipcp_snd_restart: sending packet\n"));
			error= ppp_snd(*ipcp_pkt_nq_ptr, &ipcp_snd_callback);
			if (error == EAGAIN)
				return;
		}
		*ipcp_pkt_nq_ptr= NULL;
		ipcp_pkt_nq_ptr++;
		if (ipcp_pkt_nq_ptr == &ipcp_pkt_q[IPCP_PKT_Q_SIZE])
			ipcp_pkt_nq_ptr= ipcp_pkt_q;
	}
}


DEFUN
(static void ipcp_send_data, (type, data, size),
	int type AND
	void *data AND
	size_t size
)
{
	pkt_ut *pkt;
	int error;
	u8_t *dptr;

	pkt= pkt_alloc(PPP_MAX_HDR_SIZE+4+size);
	pkt->p_offset= PPP_MAX_HDR_SIZE;
	pkt->p_size= 4+size;
	assert(pkt->p_offset + pkt->p_size <= pkt->p_maxsize);
	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);
	switch (type)
	{
	case IPCP_TYPE_CONFIGURE_REQ:
		dptr[0]= type;
		dptr[1]= ++ipcp_state.sms_last_snd_seq;
		dptr[2]= (4 + size) >> 8;
		dptr[3]= 4+size;
		if (size != 0)
			memcpy(dptr+4, data, size);
		break;
	case IPCP_TYPE_CONFIGURE_ACK:
	case IPCP_TYPE_CONFIGURE_NAK:
	case IPCP_TYPE_CONFIGURE_REJECT:
		dptr[0]= type;
		dptr[1]= ipcp_state.sms_last_rcvd_seq;
		dptr[2]= (4 + size) >> 8;
		dptr[3]= 4+size;
		if (size != 0)
			memcpy(dptr+4, data, size);
		break;
	default:
		DPRINTF(0, ("ipcp_send_data: illegal type %d\n", type));
		abort();
	}

	ppp_snd_sethdr(pkt, PPP_TYPE_IPCP);
	error= ppp_snd(pkt, &ipcp_snd_callback);
	if (error == 0)
		return;
	assert(error == EAGAIN);
	if (*ipcp_pkt_nq_ptr != NULL)
		pkt_free(*ipcp_pkt_nq_ptr);
	*ipcp_pkt_nq_ptr= pkt;
	ipcp_pkt_nq_ptr++;
	if (ipcp_pkt_nq_ptr == &ipcp_pkt_q[IPCP_PKT_Q_SIZE])
		ipcp_pkt_nq_ptr= ipcp_pkt_q;
}

/*
 * $PchHeader: /mount/hd2/minix/local/cmd/ppp/RCS/ipcp.c,v 1.2 1995/04/28 07:58:53 philip Exp $
 */
