/*	patchone.c - patch or diff one file		Author: Kees J. Bot
 *								23 Jan 1994
 */
#define nil 0
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <ctype.h>
#include "patchtree.h"
#include "inspect.h"

char *patchone(char *original, char *target,
	unsigned ncrc, off_t nsize, unsigned ocrc, off_t osize, off_t *rsize)
/* Run the patch program on the original with target as the patch file. */
{
	int binary;
	unsigned crc;
	off_t size;
	char *result;
	pid_t r, pid;
	int status;

	inspect(original, &binary, &crc, &size);

	/* The original must match the crc exactly. */
	if (binary || ocrc != crc || osize != size) return nil;

	/* Copy the original file to our playground. */
	result= tmpdirfile(basename(original));
	if (link(original, result) < 0) copyfile(original, result);

	/* Try to patch it. */
	switch ((pid= fork())) {
	case -1:
		fatal("fork()");
	case 0:
		execlp("patch", "patch", "-f", "-s",
					result, target, (char *) nil);
		report("patch");
		exit(127);
	default:
		while ((r= wait(&status)) != pid)
			if (r == -1) fatal("wait()");

		if (WIFSIGNALED(status)) {
			fprintf(stderr,
				"patchtree: patch died by signal %d%s\n",
				WTERMSIG(status),
				status & 0200 ? " - core dumped" : "");
			quit(1);
		}
		if (WEXITSTATUS(status) == 127) quit(1);
	}

	if (WEXITSTATUS(status) != 0) {
		deallocate(result);
		result= nil;
	} else {
		inspect(result, &binary, &crc, &size);
		if (binary || crc != ncrc && size != nsize) {
			deallocate(result);
			result= nil;
		}
	}
	if (result == nil) {
		/* Patch failed unexpectedly. */
		fprintf(stderr, "patchtree: patching %s with %s failed\n",
			original, target);
	}
	*rsize= 0;		/* Minimum size. */
	return result;
}

char *diffone(char *original, char *target,
	unsigned ncrc, off_t nsize, unsigned ocrc, off_t osize, off_t *rsize)
/* Run the diff program on the original and the target file. */
{
	int binary;
	FILE *fp;
	off_t diffpos;
	char *result;
	char opt[2 + 3 * sizeof(unsigned)];
	pid_t r, pid;
	int status;
	struct stat st;
	int sane;
	int c;

	inspect(original, &binary, &ocrc, &osize);
	if (binary) return nil;
	inspect(target, &binary, &ncrc, &nsize);
	if (binary) return nil;

	/* Create a target diff file. */
	result= tmpdirfile(basename(target));
	if ((fp= fopen(result, "w+")) == nil) fatal(result);
	fprintf(fp, PTREE_MAGIC, ncrc, (unsigned long) nsize,
				ocrc, (unsigned long) osize);
	fprintf(fp, "%s\n", original);
	fflush(fp);
	if (ferror(fp)) fatal(result);

	diffpos= ftell(fp);

	/* Try to send a diff to it. */
	switch ((pid= fork())) {
	case -1:
		fatal("fork()");
	case 0:
		/* Redirect diff output to the diff file. */
		if (dup2(fileno(fp), 1) < 0) fatal(result);
		close(fileno(fp));

		if (context == DEFCONTEXT) {
			opt[0]= '-';
			opt[1]= action == CDIFF ? 'c' : 'u';
			opt[2]= 0;
		} else {
			sprintf(opt, "-%c%u",
				action == CDIFF ? 'C' : 'U', context);
		}

		if (action == DIFF) {
			execlp("diff", "diff", original, target, (char *) nil);
		} else {
			execlp("diff", "diff", opt,
					original, target, (char *) nil);
		}
		report("diff");
		exit(127);
	default:
		while ((r= wait(&status)) != pid)
			if (r == -1) fatal("wait()");

		if (WIFSIGNALED(status)) {
			fprintf(stderr,
				"patchtree: diff died by signal %d%s\n",
				WTERMSIG(status),
				status & 0200 ? " - core dumped" : "");
			quit(1);
		}
		if (WEXITSTATUS(status) == 127) quit(1);
	}

	if (WEXITSTATUS(status) > 1) {
		/* 0: no differences, 1: diff created, 2: bad. */
		/* Diff has complained about the situation, we hope. */
		quit(1);
	}

	if (lstat(result, &st) < 0) fatal(result);

	/* Time for some sanity checks.  We've got to be sure it's a proper
	 * diff.
	 */
	sane= 1;

	/* You normally don't want the diff to be larger then the target file,
	 * unless you are only allowed to distribute diffs to copyrighted
	 * sources...
	 */
	if (!fflag && st.st_size >= nsize) sane= 0;

	/* Don't count the patchtree magic header. */
	st.st_size-= diffpos;

	/* If the diff is enormous then the files were probably not
	 * comparable.
	 */
	if (st.st_size >= osize + nsize) sane= 0;

	/* If the diff has lenght zero then files must crc the same. */
	if (st.st_size == 0 && (ocrc != ncrc || osize != nsize)) sane= 0;

	/* A diff can't be smaller than the size difference. */
	if (osize < nsize && st.st_size < (nsize - osize)) sane= 0;
	if (nsize < osize && st.st_size < (osize - nsize)) sane= 0;

	/* If the diff is small then check the very first character.  Some of
	 * those smart diff programs declare a file to be "binary" and run
	 * cmp(1) on it.  "Binary files ... differ ...".
	 */
	if (sane && st.st_size > 0 && st.st_size <
				80 + strlen(original) + strlen(target)) {
		fseek(fp, diffpos, SEEK_SET);
		c= fgetc(fp);
		if (action == DIFF && !isdigit(c)) sane= 0;
		if (action == CDIFF && c != '*') sane= 0;
		if (action == UDIFF && c != '-') sane= 0;
	}
	fclose(fp);

	if (!sane) {
		deallocate(result);
		result= nil;
	}
	*rsize= st.st_size;
	return result;
}
