/*	maptree.c - map the names in a directory tree	Author: Kees J. Bot
 *								23 Jan 1994
 */
#define nil 0
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <stddef.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/stat.h>
#include "patchtree.h"

typedef struct map {
	struct map	*next;		/* Next on the hash chain. */
	struct map	*up;		/* The name is part of this dir. */
	char		name[1];	/* (see footnote) */
} map_t;

#define HASH_SIZE	(256 << sizeof(char *))

static map_t *maptab[HASH_SIZE];	/* Hash list of names. */
static map_t EXCLUDE[1];		/* Special "exclude this" up marker. */
#define excluded(mp)	((mp)->up == EXCLUDE)

static unsigned hash(const char *name)
/* From a name to a hash value. */
{
	unsigned h= 0;

	while (*name != 0) {
		h= h * 0x1111 + (unsigned char) *name++;
	}

	return h % HASH_SIZE;
}

static map_t **search(const char *name)
/* Return a hook to where a name is, or could be added. */
{
	map_t **pmp= &maptab[hash(name)];

	while (*pmp != nil && strcmp(name, (*pmp)->name) > 0)
		pmp= &(*pmp)->next;

	/* Note that if more than one file of the same name is put in the
	 * database, then we've just found the first of them in this sorted
	 * hash bucket.
	 */

	return pmp;
}

void map_exclude(const char *file)
/* Add one to a list of files to exclude. */
{
	map_t **pmp, *new;

	new= allocate(nil, offsetof(map_t, name)
				+ (strlen(file) + 1) * sizeof(file[0]));
	strcpy(new->name, file);
	new->up= EXCLUDE;

	pmp= search(file);
	new->next= *pmp;
	*pmp= new;
}

static char *path;			/* A pathname and its max length. */
static size_t path_len= 0;		/* Its current length (excl null). */
static size_t path_size= 0;		/* Current allocated length + null. */

static int map_down(map_t *up)
/* Map all the files in the directory named by path. */
{
	DIR *dp;
	struct dirent *entry;
	map_t *mp, **pmp;
	size_t up_len= path_len;
	static struct stat st;

	/* First read all the files, strung together with their hash links. */
	if ((dp= opendir(path)) == nil) {
		if (errno != EACCES) fatal(path);
		report(path);
		return 0;
	}

	mp= nil;
	while ((entry= readdir(dp)) != nil) {
		map_t *new, *excl;

		if (isdots(entry->d_name)) continue;

		/* Don't do files or directories that are excluded. */
		if ((excl= *search(entry->d_name)) != nil && excluded(excl))
			continue;

		new= allocate(nil, offsetof(map_t, name)
				+ (strlen(entry->d_name) + 1)
						* sizeof(entry->d_name[0]));
		strcpy(new->name, entry->d_name);
		new->up= up;
		new->next= mp;
		mp= new;
	}
	(void) closedir(dp);

	/* Recurse into directories, add files to the database. */
	while (mp != nil) {
		map_t *next= mp->next;

		path_len= up_len + 1 + strlen(mp->name);
		if (path_len >= path_size) {
			path_size= 2 * path_len;
			path= allocate(path, path_size * sizeof(path[0]));
		}
		path[up_len]= '/';
		strcpy(path + up_len + 1, mp->name);

		if (lstat(path, &st) < 0) {
			if (errno != ENOENT) fatal(path);
			st.st_mode= 0;
		}

		if (S_ISDIR(st.st_mode)) {
			/* It's a directory, map the files in it. */
			(void) map_down(mp);
		} else
		if (S_ISREG(st.st_mode)) {
			/* A plain file; map it. */
			pmp= search(mp->name);
			mp->next= *pmp;
			*pmp= mp;
		} else {
			/* Device, pipe, whatever; drop it. */
			deallocate(mp);
		}
		mp= next;
	}
	return;
}

void map_tree(const char *dir)
{
	map_t *root;

	/* Initialize the path name. */
	path_len= strlen(dir);
	if (path_len >= path_size) {
		path_size= 2 * path_len;
		path= allocate(path, path_size * sizeof(path[0]));
	}

	strcpy(path, dir);

	/* Allocate a most upward "up" for all the names to map. */
	root= allocate(nil, offsetof(map_t, name)
				+ (path_len + 1) * sizeof(dir[0]));
	strcpy(root->name, dir);
	root->up= nil;
	root->next= nil;

	if (!map_down(root)) quit(1);
}

static map_t *next_file;		/* Next file with a given name. */
static char *next_name;			/* The given name. */

void map_search(const char *name)
/* Find the name in the database, so map_next_file may find it. */
{
	map_t *mp;

	if ((mp= *search(name)) != nil && !excluded(mp)) {
		next_file= mp;
		next_name= mp->name;
	}
}

static void map_up(map_t *mp)
/* Recurse up the directory tree, and create a path on your way down. */
{
	if (mp->up == nil) {
		/* This is the root. */
		path_len= 0;
	} else {
		map_up(mp->up);
		path[path_len++]= '/';
	}
	strcpy(path + path_len, mp->name);
	path_len+= strlen(mp->name);
}

int map_next_file(char **file)
/* Construct the full path name of one of the files found by map_search(). */
{
	if (next_file == nil || strcmp(next_name, next_file->name) != 0)
		return 0;

	map_up(next_file);

	next_file= next_file->next;

	*file= path;
	return 1;
}

void map_release(void)
/* Release all allocated memory, mainly because we may be out of it. */
{
	map_t **pmp, *junk;

	for (pmp= maptab; pmp < maptab + HASH_SIZE; pmp++) {
		while ((junk= *pmp) != nil) {
			*pmp= junk->next;
			deallocate(junk);
		}
	}
}

/* Footnote:  It is bad style to allocate name as a one byte array instead
 * of a pointer to a string.  New revisions to the C standard may even outlaw
 * it.  But it saves lots of storage, a scarce resource under Minix-86.
 */
