/*
dis386util.c

Created:	Aug 27, 1992 by Philip Homburg
*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "ansi.h"

#include "types.h"
#include "dis386.h"
#include "dis386int.h"

int i386_data_32;
int i386_addr_32;
int i386_opsize_prefix;
int i386_addrsize_prefix;

static int allocated_nodes;

static n386_t *make_const_u8 ARGS(( u8_t u8 ));
static n386_t *make_const16 ARGS(( i16_t i16 ));
static n386_t *make_const32 ARGS(( i32_t i32 ));

DEFUN_VOID (n386_t *(n386_alloc))
{
	n386_t *node;

	node= malloc(sizeof(*node));
	if (node == 0)
	{
		fprintf(stderr, "n386_alloc: out of memory\n");
		exit(1);
	}
	allocated_nodes++;
	return node;
}

DEFUN
(void (n386_free), (node),
	n386_t *node
)
{
	if (node)
	{
		allocated_nodes--;
		free(node);
	}
}

DEFUN
(void (n386_freetree), (node),
	n386_t *node
)
{
	int type;
	
	if (!node)
		return;
	type= node->n_type;
#if 0
	printf("got type %d\n", type);
#endif
	switch(type)
	{
	case I386_DAA_27:
	case I386_DAS_2F:
	case I386_AAA_37:
	case I386_AAS_3F:
	case I386_PUSHA_60:
	case I386_POPA_61:
	case I386_NOP_90:
	case I386_CBW_98:
	case I386_CWD_99:
	case I386_WAIT_9B:
	case I386_PUSHF_9C:
	case I386_POPF_9D:
	case I386_SAHF_9E:
	case I386_LAHF_9F:
	case I386_LEAVE_C9:
	case I386_INT3_CC:
	case I386_INTO_CE:
	case I386_IRET_CF:
	case I386_AAM_D4:
	case I386_AAD_D5:
	case I386_XLAT_D7:
	case I386_FNOP_D9_D0:
	case I386_FCHS_D9_E0:
	case I386_FABS_D9_E1:
	case I386_FTST_D9_E4:
	case I386_FXAM_D9_E5:
	case I386_FLD1_D9_E8:
	case I386_FLDL2T_D9_E9:
	case I386_FLDL2E_D9_EA:
	case I386_FLDPI_D9_EB:
	case I386_FLDLG2_D9_EC:
	case I386_FLDLN2_D9_ED:
	case I386_FLDZ_D9_EE:
	case I386_F2XM1_D9_F0:
	case I386_FYL2X_D9_F1:
	case I386_FPTAN_D9_F2:
	case I386_FPATAN_D9_F3:
	case I386_FXTRACT_D9_F4:
	case I386_FPREM1_D9_F5:
	case I386_FDECSTP_D9_F6:
	case I386_FINCSTP_D9_F7:
	case I386_FPREM_D9_F8:
	case I386_FYL2XP1_D9_F9:
	case I386_FSQRT_D9_FA:
	case I386_FSINCOS_D9_FB:
	case I386_FRNDINT_D9_FC:
	case I386_FSCALE_D9_FD:
	case I386_FSIN_D9_FE:
	case I386_FCOS_D9_FF:
	case I386_FUCOMPP_DA_E9:
	case I386_FCLEX_DB_E2:
	case I386_FINIT_DB_E3:
	case I386_FCOMPP_DE_D9:
	case I386_LOCK_F0:
	case I386_REPNE_F2:
	case I386_REPE_F3:
	case I386_HLT_F4:
	case I386_CMC_F5:
	case I386_CLC_F8:
	case I386_STC_F9:
	case I386_CLI_FA:
	case I386_STI_FB:
	case I386_CLD_FC:
	case I386_STD_FD:
	case I386_CLTS_F_6:
	case I486_INVD_F_8:
	case I486_WBINVD_F_9:
	case I386_REG8:
	case I386_REG16:
	case I386_REG32:
	case I386_REG_ST0:
	case I386_REG_ST:
	case I386_CONST_I8:
	case I386_CONST_U8:
	case I386_CONST16:
	case I386_CONST32:
	case I386_SEG:
	case I386_CREG:
	case I386_DREG:
	case I386_TREG:
	case I386_ILLEGAL:
		break;
	case I386_DEC_4x:
	case I386_INC_4x:
	case I386_OPS16_66:
	case I386_OPS32_66:
	case I386_JO_70:
	case I386_JNO_71:
	case I386_JB_72:
	case I386_JNB_73:
	case I386_JZ_74:
	case I386_JNZ_75:
	case I386_JBE_76:
	case I386_JNBE_77:
	case I386_JS_78:
	case I386_JNS_79:
	case I386_JP_7A:
	case I386_JNP_7B:
	case I386_JL_7C:
	case I386_JNL_7D:
	case I386_JLE_7E:
	case I386_JNLE_7F:
	case I386_INT_CD:
	case I386_RET_Cx:
	case I386_RETF_Cx:
	case I386_LDENV_D9_4:
	case I386_FLDCW_D9_5:
	case I386_STENV_D9_6:
	case I386_FSTCW_D9_7:
	case I386_FRSTOR_DB_4:
	case I386_FFREE_DD_0:
	case I386_FUCOM_DD_4:
	case I386_FUCOMP_DD_5:
	case I386_FSAVE_DD_6:
	case I386_FBLD_DF_4:
	case I386_FBSTP_DF_6:
	case I386_FCOM_Dx:
	case I386_FCOMP_Dx:
	case I386_FXCH_Dx:
	case I386_FSTSW_Dx:
	case I386_FLD_Dx:
	case I386_FST_Dx:
	case I386_FSTP_Dx:
	case I386_FILD_Dx:
	case I386_FIST_Dx:
	case I386_FISTP_Dx:
	case I386_FIMUL_Dx:
	case I386_FIADD_Dx:
	case I386_FICOM_Dx:
	case I386_FICOMP_Dx:
	case I386_FISUB_Dx:
	case I386_FISUBR_Dx:
	case I386_FIDIV_Dx:
	case I386_FIDIVR_Dx:
	case I386_LOOPNE_E0:
	case I386_LOOPE_E1:
	case I386_LOOP_E2:
	case I386_JCXZ_E3:
	case I386_CALL_E8:
	case I386_NOT_G3_2:
	case I386_NEG_G3_3:
	case I386_INC_G4_0:
	case I386_DEC_G4_1:
	case I386_INC_G5_0:
	case I386_DEC_G5_1:
	case I386_CALL_G5_2:
	case I386_CALLF_G5_3:
	case I386_JMPF_G5_5:
	case I386_SLDT_G6_0:
	case I386_STR_G6_1:
	case I386_LLDT_G6_2:
	case I386_LTR_G6_3:
	case I386_VERR_G6_4:
	case I386_VERW_G6_5:
	case I386_SGDT_G7_0:
	case I386_SIDT_G7_1:
	case I386_LGDT_G7_2:
	case I386_LIDT_G7_3:
	case I386_SMSW_G7_4:
	case I386_LMSW_G7_6:
	case I386_JO_F_80:
	case I386_JNO_F_81:
	case I386_JB_F_82:
	case I386_JNB_F_83:
	case I386_JZ_F_84:
	case I386_JNZ_F_85:
	case I386_JBE_F_86:
	case I386_JNBE_F_87:
	case I386_JS_F_88:
	case I386_JNS_F_89:
	case I386_JP_F_8A:
	case I386_JNP_F_8B:
	case I386_JL_F_8C:
	case I386_JNL_F_8D:
	case I386_JLE_F_8E:
	case I386_JNLE_F_8F:
	case I386_SETO_F_90:
	case I386_SETNO_F_91:
	case I386_SETB_F_92:
	case I386_SETNB_F_93:
	case I386_SETZ_F_94:
	case I386_SETNZ_F_95:
	case I386_SETBE_F_96:
	case I386_SETNBE_F_97:
	case I386_SETS_F_98:
	case I386_SETNS_F_99:
	case I386_SETP_F_9A:
	case I386_SETNP_F_9B:
	case I386_SETL_F_9C:
	case I386_SETNL_F_9D:
	case I386_SETLE_F_9E:
	case I386_SETNLE_F_9F:
	case I486_BSWAP_F_Cx:
	case I386_JMP:
	case I386_POP:
	case I386_PUSH:
	case I386_SEG_PREF:
	case I386_PCREL8:
	case I386_PCREL16:
	case I386_PCREL32:
		n386_freetree(node->n_u.n_1addr);
		break;
	case I386_ADD_0x:
	case I386_OR_0x:
	case I386_ADC_1x:
	case I386_SBB_1x:
	case I386_AND_2x:
	case I386_SUB_2x:
	case I386_CMP_3x:
	case I386_XOR_3x:
	case I386_BOUND_62:
	case I386_ARPL_63:
	case I386_INS_6x:
	case I386_OUTS_6x:
	case I386_LEA_8D:
	case I386_MOV_8x:
	case I386_TEST_8x:
	case I386_XCHG_8x:
	case I386_CALLF_9A:
	case I386_XCHG_9x:
	case I386_MOV_Ax:
	case I386_MOVS_Ax:
	case I386_CMPS_Ax:
	case I386_STOS_Ax:
	case I386_LODS_Ax:
	case I386_SCAS_Ax:
	case I386_TEST_Ax:
	case I386_MOV_Bx:
	case I386_ENTER_C8:
	case I386_LES_C4:
	case I386_LDS_C5:
	case I386_MOV_Cx:
	case I386_FADD_Dx:
	case I386_FMUL_Dx:
	case I386_FSUBR_Dx:
	case I386_FSUB_Dx:
	case I386_FDIVR_Dx:
	case I386_FDIV_Dx:
	case I386_FADDP_Dx:
	case I386_FMULP_Dx:
	case I386_FSUBRP_Dx:
	case I386_FSUBP_Dx:
	case I386_FDIVRP_Dx:
	case I386_FDIVP_Dx:
	case I386_IN_Ex:
	case I386_JMPF_EA:
	case I386_OUT_Ex:
	case I386_ADD_G1_0:
	case I386_OR_G1_1:
	case I386_ADC_G1_2:
	case I386_SBB_G1_3:
	case I386_AND_G1_4:
	case I386_SUB_G1_5:
	case I386_XOR_G1_6:
	case I386_CMP_G1_7:
	case I386_ROL_G2_0:
	case I386_ROR_G2_1:
	case I386_RCL_G2_2:
	case I386_RCR_G2_3:
	case I386_SHL_G2_4:
	case I386_SHR_G2_5:
	case I386_SAR_G2_7:
	case I386_TEST_G3_0:
	case I386_MUL_G3_4:
	case I386_IMUL_G3_5:
	case I386_DIV_G3_6:
	case I386_IDIV_G3_7:
	case I386_BT_G8_4:
	case I386_BTS_G8_5:
	case I386_BTR_G8_6:
	case I386_BTC_G8_7:
	case I386_LAR_F_2:
	case I386_LSL_F_3:
	case I386_MOV_F_2x:
	case I386_BT_F_A3:
	case I386_BTS_F_AB:
	case I386_IMUL_F_AF:
	case I486_CMPXCHG_F_Ax:
	case I386_LSS_F_B2:
	case I386_BTR_F_B3:
	case I386_LFS_F_B4:
	case I386_LGS_F_B5:
	case I386_BTC_F_BB:
	case I386_BSF_F_BC:
	case I386_BSR_F_BD:
	case I386_MOVSX_F_Bx:
	case I386_MOVZX_F_Bx:
	case I486_XADD_F_Cx:
		n386_freetree(node->n_u.n_2addr.n_first);
		n386_freetree(node->n_u.n_2addr.n_second);
		break;
	case I386_IMUL3_6x:
	case I386_SHLD_F_Ax:
	case I386_SHRD_F_Ax:
		n386_freetree(node->n_u.n_3addr.n_first);
		n386_freetree(node->n_u.n_3addr.n_second);
		n386_freetree(node->n_u.n_3addr.n_third);
		break;
	case I386_ADDRESS8:
	case I386_ADDRESS16:
	case I386_ADDRESS32:
	case I386_ADDRESS64:
	case I386_ADDRESS80:
	case I386_ADDRESS14B:
	case I386_ADDRESS28B:
	case I386_ADDRESS94B:
	case I386_ADDRESS108B:
	case I386_ADDRESS_FAR16:
	case I386_ADDRESS_FAR32:
	case I386_ADDRESS16_16:
	case I386_ADDRESS16_32:
	case I386_ADDRESS32_32:
		n386_freetree(node->n_u.n_address.n_segment);
		n386_freetree(node->n_u.n_address.n_base);
		n386_freetree(node->n_u.n_address.n_index);
		n386_freetree(node->n_u.n_address.n_disp);
		break;
	default:
		printf("n386_freetree: illegal type %d\n", type);
		exit(1);
	}
	n386_free(node);
#if 0
	printf("n386_freetree: end type %d\n", type);
#endif
}

DEFUN_VOID( void n386_check )
{
	if (allocated_nodes != 0)
	{
		printf("n386_check: allocated_node= %d\n", allocated_nodes);
		exit(1);
	}
}
		
DEFUN
(n386_t *i386_make_segment, (segment),
	r386_t segment
)
{
	n386_t *n;

	n= n386_alloc();
	n->n_type= I386_SEG;
	n->n_u.n_reg= segment;
	return n;
}

DEFUN
(n386_t *i386_make_register, (reg),
	r386_t reg
)
{
	n386_t *n;

	n= n386_alloc();
	if (reg > I386_REG8_FIRST && reg < I386_REG8_LAST)
		n->n_type= I386_REG8;
	else if (reg > I386_REG16_FIRST && reg < I386_REG16_LAST)
		n->n_type= I386_REG16;
	else if (reg > I386_REG32_FIRST && reg < I386_REG32_LAST)
		n->n_type= I386_REG32;
	else
	{
		printf("i386_make_register: illegal register %d\n", reg);
		exit(1);
	}
	n->n_u.n_reg= reg;
	return n;
}

DEFUN
(n386_t *i386_make_const_i8, (i8),
	i8_t i8
)
{
	n386_t *n;

	n= n386_alloc();
	n->n_type= I386_CONST_I8;
	n->n_u.n_const_i8= i8;
	return n;
}

DEFUN
(n386_t *i386_reg8_node, (reg_no),
	int reg_no
)
{
	r386_t reg;

	switch(reg_no)
	{
	case 0: reg= I386_AL; break;
	case 1: reg= I386_CL; break;
	case 2: reg= I386_DL; break;
	case 3: reg= I386_BL; break;
	case 4: reg= I386_AH; break;
	case 5: reg= I386_CH; break;
	case 6: reg= I386_DH; break;
	case 7: reg= I386_BH; break;
	default:
		printf("i386_reg8_node: illegal reg_no %d\n", reg_no);
		exit(1);
	}
	return i386_make_register(reg);
}

DEFUN
(n386_t *i386_reg16_node, (reg_no),
	int reg_no
)
{
	r386_t reg;

	switch(reg_no)
	{
	case 0: reg= I386_AX; break;
	case 1: reg= I386_CX; break;
	case 2: reg= I386_DX; break;
	case 3: reg= I386_BX; break;
	case 4: reg= I386_SP; break;
	case 5: reg= I386_BP; break;
	case 6: reg= I386_SI; break;
	case 7: reg= I386_DI; break;
	default:
		printf("i386_reg16_node: illegal reg_no %d\n", reg_no);
		exit(1);
	}
	return i386_make_register(reg);
}

DEFUN
(n386_t *i386_reg32_node, (reg_no),
	int reg_no
)
{
	r386_t reg;

	switch(reg_no)
	{
	case 0: reg= I386_EAX; break;
	case 1: reg= I386_ECX; break;
	case 2: reg= I386_EDX; break;
	case 3: reg= I386_EBX; break;
	case 4: reg= I386_ESP; break;
	case 5: reg= I386_EBP; break;
	case 6: reg= I386_ESI; break;
	case 7: reg= I386_EDI; break;
	default:
		printf("i386_reg32_node: illegal reg_no %d\n", reg_no);
		exit(1);
	}
	return i386_make_register(reg);
}

DEFUN
(n386_t *i386_data_reg_node, (reg_no, word),
	int reg_no AND
	int word
)
{
	if (word == 0)
		return i386_reg8_node(reg_no);
	else if ((i386_data_32 ^ i386_opsize_prefix) == 0)
		return i386_reg16_node(reg_no);
	else
		return i386_reg32_node(reg_no);
}

DEFUN
(n386_t *i386_const_i8, (block, size, actsize_p),
	char *block AND
	unsigned size AND
	unsigned *actsize_p
)
{
	i8_t value;
	
	if (size < 1)
		return 0;
	value= *(i8_t *)(block+0);
	*actsize_p= 1;
	return i386_make_const_i8(value);
}

DEFUN
(n386_t *i386_const_u8, (block, size, actsize_p),
	char *block AND
	unsigned size AND
	unsigned *actsize_p
)
{
	u8_t value;
	
	if (size < 1)
		return 0;
	value= *(u8_t *)(block+0);
	*actsize_p= 1;
	return make_const_u8(value);
}

DEFUN
(n386_t *i386_const16, (block, size, actsize_p),
	char *block AND
	unsigned size AND
	unsigned *actsize_p
)
{
	i16_t value;
	
	if (size < 2)
		return 0;
	value= *(u8_t *)(block+1);
	value= (value << 8) | *(u8_t *)(block+0);
	*actsize_p= 2;
	return make_const16(value);
}

DEFUN
(n386_t *i386_const32, (block, size, actsize_p),
	char *block AND
	unsigned size AND
	unsigned *actsize_p
)
{
	i32_t value;
	
	if (size < 4)
		return 0;
	value= *(u8_t *)(block+3);
	value= (value << 8) | *(u8_t *)(block+2);
	value= (value << 8) | *(u8_t *)(block+1);
	value= (value << 8) | *(u8_t *)(block+0);
	*actsize_p= 4;
	return make_const32(value);
}

DEFUN
(static n386_t *make_const_u8, (u8),
	u8_t u8
)
{
	n386_t *n;

	n= n386_alloc();
	n->n_type= I386_CONST_U8;
	n->n_u.n_const_u8= u8;
	return n;
}

DEFUN
(static n386_t *make_const16, (i16),
	i16_t i16
)
{
	n386_t *n;

	n= n386_alloc();
	n->n_type= I386_CONST16;
	n->n_u.n_const16= i16;
	return n;
}

DEFUN
(static n386_t *make_const32, (i32),
	i32_t i32
)
{
	n386_t *n;

	n= n386_alloc();
	n->n_type= I386_CONST32;
	n->n_u.n_const32= i32;
	return n;
}

DEFUN
(n386_t *i386_data_const, (block, size, actsize_p, word),
	char *block AND
	unsigned size AND
	unsigned *actsize_p AND
	int word
)
{
	n386_t *o;
	unsigned actsize;
	
	if (!word)
		o= i386_const_i8(block, size, &actsize);
	else if ((i386_data_32 ^ i386_opsize_prefix) == 0)
		o= i386_const16(block, size, &actsize);
	else
		o= i386_const32(block, size, &actsize);
	if (o == 0)
		return 0;
	*actsize_p= actsize;
	return o;
}

DEFUN
(n386_t *i386_addr_const, (block, size, actsize_p, word),
	char *block AND
	unsigned size AND
	unsigned *actsize_p AND
	int word
)
{
	n386_t *o;
	unsigned actsize;
	
	if (!word)
		o= i386_const_i8(block, size, &actsize);
	else if ((i386_addr_32 ^ i386_addrsize_prefix) == 0)
		o= i386_const16(block, size, &actsize);
	else
		o= i386_const32(block, size, &actsize);
	if (o == 0)
		return 0;
	*actsize_p= actsize;
	return o;
}

