/*
ack.c

Created:	Aug 21, 1992 by Philip Homburg
*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ansi.h"

#include "types.h"
#include "ack.h"
#include "dis386.h"

#define PRINT_ALL	DEBUG

int n_strings;
ack_label_func_t ack_text_func, ack_data_func, ack_const_func;
unsigned long ack_pc;
unsigned ack_size;
char *ack_block, *ack_block1;
char *comment;

static char *node2string ARGS(( n386_t *node, char *block ));
static char *exprnode2string ARGS(( n386_t *node ));
static char *dump ARGS(( unsigned long pc, char *block ));
static char *one_expr ARGS(( char *name, n386_t *expr ));
static char *one_expr_b ARGS(( char *name, n386_t *expr ));
static char *one_expr_fi ARGS(( char *name, n386_t *expr ));
static char *one_expr_f ARGS(( char *name, n386_t *expr ));
static char *two_expr ARGS(( char *name, n386_t *first, n386_t *second ));
static char *two_expr_2nd ARGS(( char *name, n386_t *first, n386_t *second ));
static char *two_expr_b ARGS(( char *name, n386_t *first, n386_t *second ));
static char *two_expr_b_0 ARGS(( char *name, n386_t *first, n386_t *second ));
static char *two_expr_b2 ARGS(( char *name, n386_t *first, n386_t *second ));
static char *two_expr_b_2nd ARGS(( char *name, n386_t *first, n386_t *second ));
static char *two_expr_b2nd ARGS(( char *name, n386_t *first, n386_t *second ));
static char *three_expr ARGS(( char *name, n386_t *first, n386_t *second, 
						n386_t *third ));
static char *seg_prefix ARGS(( n386_t *node ));
static char *far_expr ARGS(( char *name, n386_t *offset, n386_t *segment ));
static char *stralloc ARGS(( void ));
static void strfree ARGS(( char *s ));

#if DEBUG & 2
#define stralloc() (printf("stralloc: %s, %d\n", __FILE__, __LINE__), \
	(stralloc)())
#define strfree(x) (printf("strfree: %s, %d\n", __FILE__, __LINE__), \
	(strfree)(x))
#endif

DEFUN(
char *ack_dis386, (pc, block, size, actsize_p),
	unsigned long pc AND
	char *block AND
	unsigned size AND
	unsigned *actsize_p
)
{
	n386_t *node;
	char *s, *o;
	int i;

	if (size == 0)
		return 0;
		
#if DEBUG & 2
	ack_block= block;
	ack_size= size>16 ? 16 : size;
	s= dump(pc, block);
	puts(s);
	strfree(s);
#endif
	
	for (i= 0; i<=size; i++)
	{
#if DEBUG
#else
i= size;
#endif
		node= dis386(block, i, actsize_p);
		if (node != 0 || i>16)
			break;
	}
	if (!node)
		return 0;
#if DEBUG
#else
i= *actsize_p;
#endif
	ack_pc= pc;
	ack_size= *actsize_p;
	ack_block= block;
	ack_block1= block;
	
	s= node2string(node, block);
	o= dump(pc, block);
	strcat(s, o);
	strfree(o);
#if DEBUG & 2
fflush(stdout);	
fprintf(stderr, "%s\n", s);
#endif
	assert(*actsize_p == i || (printf("%d, %d\n", *actsize_p, i), 0));
	n386_freetree(node);
	n386_check();
	n_strings--;
#if DEBUG & 2
fflush(stdout);
#endif
	assert((n_strings == 0) || (printf("%d\n", n_strings), 0));
	return s;
}

DEFUN
(static char *node2string, (node, block),
	n386_t *node AND
	char *block
)
{
	char *o;
	char *s;
	char num[10];
	int i, first;

	s= stralloc();
	
	switch(node->n_type)
	{
	case I386_ADD_0x:
	case I386_ADD_G1_0:
		strfree(s);
		return two_expr_b("add", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_OR_0x:
	case I386_OR_G1_1:
		strfree(s);
		return two_expr_b("or", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_ADC_1x:
	case I386_ADC_G1_2:
		strfree(s);
		return two_expr_b("adc", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_SBB_1x:
	case I386_SBB_G1_3:
		strfree(s);
		return two_expr_b("sbb", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_AND_2x:
	case I386_AND_G1_4:
		strfree(s);
		return two_expr_b("and", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_SUB_2x:
	case I386_SUB_G1_5:
		strfree(s);
		return two_expr_b("sub", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_DAA_27:
		strcpy(s, "daa");
		return s;				
	case I386_DAS_2F:
		strcpy(s, "das");
		return s;				
	case I386_AAA_37:
		strcpy(s, "aaa");
		return s;				
	case I386_AAS_3F:
		strcpy(s, "aas");
		return s;				
	case I386_XOR_3x:
	case I386_XOR_G1_6:
		strfree(s);
		return two_expr_b("xor", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_CMP_3x:
	case I386_CMP_G1_7:
		strfree(s);
		return two_expr_b("cmp", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_INC_4x:
	case I386_INC_G4_0:
	case I386_INC_G5_0:
		strfree(s);
		return one_expr_b("inc", node->n_u.n_1addr);
	case I386_DEC_4x:	
	case I386_DEC_G4_1:
	case I386_DEC_G5_1:
		strfree(s);
		return one_expr_b("dec", node->n_u.n_1addr);
	case I386_BOUND_62:
		strfree(s);	
		return two_expr("bound", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_ARPL_63:
		strfree(s);	
		return two_expr("arpl", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_PUSHA_60:
		strcpy(s, "pusha");
		return s;
	case I386_POPA_61:
		strcpy(s, "popa");
		return s;
	case I386_OPS16_66:
		strcpy(s, "o16 ");
		ack_size--;
		ack_block1++;
		o= node2string(node->n_u.n_1addr, block+1);
		ack_block1--;
		ack_size++;
		strcat(s, o);
		strfree(o);
		return s;
	case I386_OPS32_66:
		strcpy(s, "o32 ");
		ack_size--;
		ack_block1++;
		o= node2string(node->n_u.n_1addr, block+1);
		ack_size++;
		ack_block1--;
		strcat(s, o);
		strfree(o);
		return s;
	case I386_IMUL3_6x:
		strfree(s);
		return three_expr("imul", node->n_u.n_3addr.n_first,
			node->n_u.n_3addr.n_second, node->n_u.n_3addr.n_third);
	case I386_INS_6x:
		strfree(s);
		return two_expr_b_0("ins", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_OUTS_6x:
		strfree(s);
		return two_expr_b_0("outs", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_JO_70:
	case I386_JO_F_80:
		strfree(s);
		return one_expr("jo", node->n_u.n_1addr);
	case I386_JNO_71:
	case I386_JNO_F_81:
		strfree(s);
		return one_expr("jno", node->n_u.n_1addr);
	case I386_JB_72:
	case I386_JB_F_82:
		strfree(s);
		return one_expr("jb", node->n_u.n_1addr);
	case I386_JNB_73:
	case I386_JNB_F_83:
		strfree(s);
		return one_expr("jnb", node->n_u.n_1addr);
	case I386_JZ_74:
	case I386_JZ_F_84:
		strfree(s);
		return one_expr("jz", node->n_u.n_1addr);
	case I386_JNZ_75:
	case I386_JNZ_F_85:
		strfree(s);
		return one_expr("jnz", node->n_u.n_1addr);
	case I386_JBE_76:
	case I386_JBE_F_86:
		strfree(s);
		return one_expr("jbe", node->n_u.n_1addr);
	case I386_JNBE_77:
	case I386_JNBE_F_87:
		strfree(s);
		return one_expr("jnbe", node->n_u.n_1addr);
	case I386_JS_78:
	case I386_JS_F_88:
		strfree(s);
		return one_expr("js", node->n_u.n_1addr);
	case I386_JNS_79:
	case I386_JNS_F_89:
		strfree(s);
		return one_expr("jns", node->n_u.n_1addr);
	case I386_JP_7A:
	case I386_JP_F_8A:
		strfree(s);
		return one_expr("jp", node->n_u.n_1addr);
	case I386_JNP_7B:
	case I386_JNP_F_8B:
		strfree(s);
		return one_expr("jnp", node->n_u.n_1addr);
	case I386_JL_7C:
	case I386_JL_F_8C:
		strfree(s);
		return one_expr("jl", node->n_u.n_1addr);
	case I386_JNL_7D:
	case I386_JNL_F_8D:
		strfree(s);
		return one_expr("jnl", node->n_u.n_1addr);
	case I386_JLE_7E:
	case I386_JLE_F_8E:
		strfree(s);
		return one_expr("jle", node->n_u.n_1addr);
	case I386_JNLE_7F:
	case I386_JNLE_F_8F:
		strfree(s);
		return one_expr("jnle", node->n_u.n_1addr);
	case I386_TEST_8x:
	case I386_TEST_Ax:
	case I386_TEST_G3_0:
		strfree(s);
		return two_expr_b("test", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_XCHG_8x:
	case I386_XCHG_9x:
		strfree(s);
		return two_expr_b("xchg", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_MOV_8x:
	case I386_MOV_Bx:
	case I386_MOV_Cx:
	case I386_MOV_Ax:
	case I386_MOV_F_2x:
		strfree(s);
		return two_expr_b("mov", node->n_u.n_mov.n_dst,
						node->n_u.n_mov.n_src);
	case I386_LEA_8D:
		strfree(s);
		return two_expr("lea", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_NOP_90:
		strcpy(s, "nop");
		return s;
	case I386_CBW_98:
		strcpy(s, "cbw");
		return s;
	case I386_CWD_99:
		strcpy(s, "cwd");
		return s;
	case I386_CALLF_9A:
		strfree(s);
		return far_expr("callf", node->n_u.n_far.n_offset,
					node->n_u.n_far.n_segment);
	case I386_WAIT_9B:
		strcpy(s, "wait");
		return s;
	case I386_PUSHF_9C:
		strcpy(s, "pushf");
		return s;
	case I386_POPF_9D:
		strcpy(s, "popf");
		return s;
	case I386_SAHF_9E:
		strcpy(s, "sahf");
		return s;
	case I386_LAHF_9F:
		strcpy(s, "lahf");
		return s;
	case I386_MOVS_Ax:
		strfree(s);
		return two_expr_b_0("movs", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_CMPS_Ax:
		strfree(s);
		return two_expr_b_0("cmps", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_STOS_Ax:
		strfree(s);
		return two_expr_b_0("stos", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_LODS_Ax:
		strfree(s);
		return two_expr_b_0("lods", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_SCAS_Ax:
		strfree(s);
		return two_expr_b_0("scas", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_LES_C4:
		strfree(s);
		return two_expr("les", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_LDS_C5:
		strfree(s);
		return two_expr("lds", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_RET_Cx:
		if (node->n_u.n_1addr)
		{
			strfree(s);
			return one_expr("ret", node->n_u.n_1addr);
		}
		strcpy(s, "ret");
		return s;
	case I386_RETF_Cx:
		if (node->n_u.n_1addr)
		{
			strfree(s);
			return one_expr("retf", node->n_u.n_1addr);
		}
		strcpy(s, "retf");
		return s;
	case I386_ENTER_C8:
		strfree(s);
		return two_expr("enter", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_LEAVE_C9:
		strcpy(s, "leave");
		return s;
	case I386_INT3_CC:
		strcpy(s, "int\t3");
		return s;
	case I386_INT_CD:
		strfree(s);
		return one_expr("int", node->n_u.n_1addr);
	case I386_INTO_CE:
		strcpy(s, "into");
		return s;
	case I386_IRET_CF:
		strcpy(s, "iret");
		return s;
	case I386_AAM_D4:
		strcpy(s, "aam");
		return s;
	case I386_AAD_D5:
		strcpy(s, "aad");
		return s;
	case I386_XLAT_D7:
		strcpy(s, "xlat");
		return s;
	case I386_FNOP_D9_D0:
		strcpy(s, "fnop");
		return s;
	case I386_FCHS_D9_E0:
		strcpy(s, "fchs");
		return s;
	case I386_FABS_D9_E1:
		strcpy(s, "fabs");
		return s;
	case I386_FTST_D9_E4:
		strcpy(s, "ftst");
		return s;
	case I386_FXAM_D9_E5:
		strcpy(s, "fxam");
		return s;
	case I386_FLD1_D9_E8:
		strcpy(s, "fld1");
		return s;
	case I386_FLDL2T_D9_E9:
		strcpy(s, "fldl2t");
		return s;
	case I386_FLDL2E_D9_EA:
		strcpy(s, "fldl2e");
		return s;
	case I386_FLDPI_D9_EB:
		strcpy(s, "fldpi");
		return s;
	case I386_FLDLG2_D9_EC:
		strcpy(s, "fldlg2");
		return s;
	case I386_FLDLN2_D9_ED:
		strcpy(s, "fldln2");
		return s;
	case I386_FLDZ_D9_EE:
		strcpy(s, "fldz");
		return s;
	case I386_F2XM1_D9_F0:
		strcpy(s, "f2xm1");
		return s;
	case I386_FYL2X_D9_F1:
		strcpy(s, "fyl2x");
		return s;
	case I386_FPTAN_D9_F2:
		strcpy(s, "fptan");
		return s;
	case I386_FPATAN_D9_F3:
		strcpy(s, "fpatan");
		return s;
	case I386_FXTRACT_D9_F4:
		strcpy(s, "fxtract");
		return s;
	case I386_FPREM1_D9_F5:
		strcpy(s, "fprem1");
		return s;
	case I386_FDECSTP_D9_F6:
		strcpy(s, "fdecstp");
		return s;
	case I386_FINCSTP_D9_F7:
		strcpy(s, "fincstp");
		return s;
	case I386_FPREM_D9_F8:
		strcpy(s, "fprem");
		return s;
	case I386_FYL2XP1_D9_F9:
		strcpy(s, "fyl2xp1");
		return s;
	case I386_FSQRT_D9_FA:
		strcpy(s, "fsqrt");
		return s;
	case I386_FSINCOS_D9_FB:
		strcpy(s, "fsincos");
		return s;
	case I386_FRNDINT_D9_FC:
		strcpy(s, "frndint");
		return s;
	case I386_FSCALE_D9_FD:
		strcpy(s, "fscale");
		return s;
	case I386_FSIN_D9_FE:
		strcpy(s, "fsin");
		return s;
	case I386_FCOS_D9_FF:
		strcpy(s, "fcos");
		return s;
	case I386_LDENV_D9_4:
		strfree(s);
		return one_expr("fldenv", node->n_u.n_1addr);
	case I386_FLDCW_D9_5:
		strfree(s);
		return one_expr("fldcw", node->n_u.n_1addr);
	case I386_STENV_D9_6:
		strfree(s);
		return one_expr("fstenv", node->n_u.n_1addr);
	case I386_FSTCW_D9_7:
		strfree(s);
		return one_expr("fstcw", node->n_u.n_1addr);
	case I386_FUCOMPP_DA_E9:
		strcpy(s, "fucompp");
		return s;
	case I386_FCLEX_DB_E2:
		strcpy(s, "fclex");
		return s;
	case I386_FINIT_DB_E3:
		strcpy(s, "finit");
		return s;
	case I386_FRSTOR_DB_4:
		strfree(s);
		return one_expr("frstor", node->n_u.n_1addr);
	case I386_FFREE_DD_0:
		strfree(s);
		return one_expr("ffree", node->n_u.n_1addr);
	case I386_FUCOM_DD_4:
		strfree(s);
		return one_expr("fucom", node->n_u.n_1addr);
	case I386_FUCOMP_DD_5:
		strfree(s);
		return one_expr("fucomp", node->n_u.n_1addr);
	case I386_FSAVE_DD_6:
		strfree(s);
		return one_expr("fsave", node->n_u.n_1addr);
	case I386_FCOMPP_DE_D9:
		strcpy(s, "fcompp");
		return s;
	case I386_FBLD_DF_4:
		strfree(s);
		return one_expr("fbld", node->n_u.n_1addr);
	case I386_FBSTP_DF_6:
		strfree(s);
		return one_expr("fbstp", node->n_u.n_1addr);
	case I386_FXCH_Dx:
		strfree(s);
		return one_expr("fxch", node->n_u.n_1addr);
	case I386_FCOM_Dx:
		strfree(s);
		return one_expr_f("fcom", node->n_u.n_1addr);
	case I386_FCOMP_Dx:
		strfree(s);
		return one_expr_f("fcomp", node->n_u.n_1addr);
	case I386_FADD_Dx:
		strfree(s);
		if (node->n_u.n_2addr.n_second == NULL)
			return one_expr_f("fadd", node->n_u.n_2addr.n_first);
		return two_expr("fadd", node->n_u.n_2addr.n_first,
			node->n_u.n_2addr.n_second);
	case I386_FMUL_Dx:
		strfree(s);
		if (node->n_u.n_2addr.n_second == NULL)
			return one_expr_f("fmul", node->n_u.n_2addr.n_first);
		return two_expr("fmul", node->n_u.n_2addr.n_first,
			node->n_u.n_2addr.n_second);
	case I386_FSUBR_Dx:
		strfree(s);
		if (node->n_u.n_2addr.n_second == NULL)
			return one_expr_f("fsubr", node->n_u.n_2addr.n_first);
		return two_expr("fsubr", node->n_u.n_2addr.n_first,
			node->n_u.n_2addr.n_second);
	case I386_FSUB_Dx:
		strfree(s);
		if (node->n_u.n_2addr.n_second == NULL)
			return one_expr_f("fsub", node->n_u.n_2addr.n_first);
		return two_expr("fsub", node->n_u.n_2addr.n_first,
			node->n_u.n_2addr.n_second);
	case I386_FDIVR_Dx:
		strfree(s);
		if (node->n_u.n_2addr.n_second == NULL)
			return one_expr_f("fdivr", node->n_u.n_2addr.n_first);
		return two_expr("fdivr", node->n_u.n_2addr.n_first,
			node->n_u.n_2addr.n_second);
	case I386_FDIV_Dx:
		strfree(s);
		if (node->n_u.n_2addr.n_second == NULL)
			return one_expr_f("fdiv", node->n_u.n_2addr.n_first);
		return two_expr("fdiv", node->n_u.n_2addr.n_first,
			node->n_u.n_2addr.n_second);
	case I386_FSTSW_Dx:
		strfree(s);
		return one_expr("fstsw", node->n_u.n_1addr);
	case I386_FLD_Dx:
		strfree(s);
		return one_expr_f("fld", node->n_u.n_1addr);
	case I386_FST_Dx:
		strfree(s);
		return one_expr_f("fst", node->n_u.n_1addr);
	case I386_FSTP_Dx:
		strfree(s);
		return one_expr_f("fstp", node->n_u.n_1addr);
	case I386_FILD_Dx:
		strfree(s);
		return one_expr_fi("fild", node->n_u.n_1addr);
	case I386_FIST_Dx:
		strfree(s);
		return one_expr_fi("fist", node->n_u.n_1addr);
	case I386_FISTP_Dx:
		strfree(s);
		return one_expr_fi("fistp", node->n_u.n_1addr);
	case I386_FIMUL_Dx:
		strfree(s);
		return one_expr_fi("fimul", node->n_u.n_1addr);
	case I386_FIADD_Dx:
		strfree(s);
		return one_expr_fi("fiadd", node->n_u.n_1addr);
	case I386_FICOM_Dx:
		strfree(s);
		return one_expr_fi("ficom", node->n_u.n_1addr);
	case I386_FICOMP_Dx:
		strfree(s);
		return one_expr_fi("ficomp", node->n_u.n_1addr);
	case I386_FISUB_Dx:
		strfree(s);
		return one_expr_fi("fisub", node->n_u.n_1addr);
	case I386_FISUBR_Dx:
		strfree(s);
		return one_expr_fi("fisubr", node->n_u.n_1addr);
	case I386_FIDIV_Dx:
		strfree(s);
		return one_expr_fi("fidiv", node->n_u.n_1addr);
	case I386_FIDIVR_Dx:
		strfree(s);
		return one_expr_fi("fidivr", node->n_u.n_1addr);
	case I386_FADDP_Dx:
		strfree(s);
		return two_expr("faddp", node->n_u.n_2addr.n_first,
			node->n_u.n_2addr.n_second);
	case I386_FMULP_Dx:
		strfree(s);
		return two_expr("fmulp", node->n_u.n_2addr.n_first,
			node->n_u.n_2addr.n_second);
	case I386_FSUBRP_Dx:
		strfree(s);
		return two_expr("fsubrp", node->n_u.n_2addr.n_first,
			node->n_u.n_2addr.n_second);
	case I386_FSUBP_Dx:
		strfree(s);
		return two_expr("fsubp", node->n_u.n_2addr.n_first,
			node->n_u.n_2addr.n_second);
	case I386_FDIVRP_Dx:
		strfree(s);
		return two_expr("fdivrp", node->n_u.n_2addr.n_first,
			node->n_u.n_2addr.n_second);
	case I386_FDIVP_Dx:
		strfree(s);
		return two_expr("fdivp", node->n_u.n_2addr.n_first,
			node->n_u.n_2addr.n_second);
	case I386_LOOPNE_E0:
		strfree(s);
		return one_expr("loopne", node->n_u.n_1addr);
	case I386_LOOPE_E1:
		strfree(s);
		return one_expr("loope", node->n_u.n_1addr);
	case I386_LOOP_E2:
		strfree(s);
		return one_expr("loop", node->n_u.n_1addr);
	case I386_JCXZ_E3:
		strfree(s);
		return one_expr("jcxz", node->n_u.n_1addr);
	case I386_IN_Ex:
		strfree(s);
		if (node->n_u.n_2addr.n_first->n_type == I386_REG8)
		{
#if PRINT_ALL
			return two_expr("inb", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
#else /* PRINT_ALL */
			return one_expr("inb", node->n_u.n_2addr.n_second);
#endif /* PRINT_ALL */
		}
#if PRINT_ALL
		return two_expr("in", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
#else /* PRINT_ALL */
		return one_expr("in", node->n_u.n_2addr.n_second);
#endif /* PRINT_ALL */
	case I386_OUT_Ex:
		strfree(s);
		if (node->n_u.n_2addr.n_second->n_type == I386_REG8)
		{
#if PRINT_ALL
			return two_expr("outb", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
#else /* PRINT_ALL */
			return one_expr("outb", node->n_u.n_2addr.n_first);
#endif /* PRINT_ALL */
		}
#if PRINT_ALL
		return two_expr("out", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
#else /* PRINT_ALL */
		return one_expr("out", node->n_u.n_2addr.n_first);
#endif /* PRINT_ALL */
	case I386_CALL_E8:
	case I386_CALL_G5_2:
		strfree(s);
		return one_expr("call", node->n_u.n_1addr);
	case I386_JMPF_EA:
		strfree(s);
		return far_expr("jmpf", node->n_u.n_far.n_offset,
					node->n_u.n_far.n_segment);
	case I386_LOCK_F0:
		strcpy(s, "lock");
		return s;
	case I386_REPNE_F2:
		strcpy(s, "repne");
		return s;
	case I386_REPE_F3:
		strcpy(s, "repe");
		return s;
	case I386_HLT_F4:
		strcpy(s, "hlt");
		return s;
	case I386_CMC_F5:
		strcpy(s, "cmc");
		return s;
	case I386_CLC_F8:
		strcpy(s, "clc");
		return s;
	case I386_STC_F9:
		strcpy(s, "stc");
		return s;
	case I386_CLI_FA:
		strcpy(s, "cli");
		return s;
	case I386_STI_FB:
		strcpy(s, "sti");
		return s;
	case I386_CLD_FC:
		strcpy(s, "cld");
		return s;
	case I386_STD_FD:
		strcpy(s, "std");
		return s;
	case I386_ROL_G2_0:
		strfree(s);
		return two_expr_b("rol", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_ROR_G2_1:
		strfree(s);
		return two_expr_b("ror", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_RCL_G2_2:
		strfree(s);
		return two_expr_b("rcl", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_RCR_G2_3:
		strfree(s);
		return two_expr_b("rcr", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_SHL_G2_4:
		strfree(s);
		return two_expr_b("shl", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_SHR_G2_5:
		strfree(s);
		return two_expr_b("shr", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_SAR_G2_7:
		strfree(s);
		return two_expr_b("sar", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_NOT_G3_2:
		strfree(s);
		return one_expr_b("not", node->n_u.n_1addr);
	case I386_NEG_G3_3:
		strfree(s);
		return one_expr_b("neg", node->n_u.n_1addr);
	case I386_MUL_G3_4:
		strfree(s);
		return two_expr_b_2nd("mul", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_IMUL_G3_5:
	case I386_IMUL_F_AF:
		strfree(s);
		return two_expr_b2nd("imul", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_DIV_G3_6:
		strfree(s);
		return two_expr_b_2nd("div", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_IDIV_G3_7:
		strfree(s);
		return two_expr_b_2nd("idiv", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_CALLF_G5_3:
		strfree(s);
		return one_expr("callf", node->n_u.n_1addr);
	case I386_JMPF_G5_5:
		strfree(s);
		return one_expr("jmpf", node->n_u.n_1addr);
	case I386_SLDT_G6_0:
		strfree(s);
		return one_expr("sldt", node->n_u.n_1addr);
	case I386_STR_G6_1:
		strfree(s);
		return one_expr("str", node->n_u.n_1addr);
	case I386_LLDT_G6_2:
		strfree(s);
		return one_expr("lldt", node->n_u.n_1addr);
	case I386_LTR_G6_3:
		strfree(s);
		return one_expr("ltr", node->n_u.n_1addr);
	case I386_VERR_G6_4:
		strfree(s);
		return one_expr("verr", node->n_u.n_1addr);
	case I386_VERW_G6_5:
		strfree(s);
		return one_expr("verw", node->n_u.n_1addr);
	case I386_SGDT_G7_0:
		strfree(s);
		return one_expr("sgdt", node->n_u.n_1addr);
	case I386_SIDT_G7_1:
		strfree(s);
		return one_expr("sidt", node->n_u.n_1addr);
	case I386_LGDT_G7_2:
		strfree(s);
		return one_expr("lgdt", node->n_u.n_1addr);
	case I386_LIDT_G7_3:
		strfree(s);
		return one_expr("lidt", node->n_u.n_1addr);
	case I386_SMSW_G7_4:
		strfree(s);
		return one_expr("smsw", node->n_u.n_1addr);
	case I386_LMSW_G7_6:
		strfree(s);
		return one_expr("lmsw", node->n_u.n_1addr);
	case I386_BT_G8_4:
	case I386_BT_F_A3:
		strfree(s);
		return two_expr("bt", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_BTS_G8_5:
	case I386_BTS_F_AB:
		strfree(s);
		return two_expr("bts", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_BTR_G8_6:
	case I386_BTR_F_B3:
		strfree(s);
		return two_expr("btr", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_BTC_G8_7:
	case I386_BTC_F_BB:
		strfree(s);
		return two_expr("btc", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_LAR_F_2:
		strfree(s);
		return two_expr("lar", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_LSL_F_3:
		strfree(s);
		return two_expr("lsl", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_CLTS_F_6:
		strcpy(s, "clts");
		return s;
	case I486_INVD_F_8:
		strcpy(s, "invd");
		return s;
	case I486_WBINVD_F_9:
		strcpy(s, "wbinvd");
		return s;
	case I386_SETO_F_90:
		strfree(s);
		return one_expr("seto", node->n_u.n_1addr);
	case I386_SETNO_F_91:
		strfree(s);
		return one_expr("setno", node->n_u.n_1addr);
	case I386_SETB_F_92:
		strfree(s);
		return one_expr("setb", node->n_u.n_1addr);
	case I386_SETNB_F_93:
		strfree(s);
		return one_expr("setnb", node->n_u.n_1addr);
	case I386_SETZ_F_94:
		strfree(s);
		return one_expr("setz", node->n_u.n_1addr);
	case I386_SETNZ_F_95:
		strfree(s);
		return one_expr("setnz", node->n_u.n_1addr);
	case I386_SETBE_F_96:
		strfree(s);
		return one_expr("setbe", node->n_u.n_1addr);
	case I386_SETNBE_F_97:
		strfree(s);
		return one_expr("setnbe", node->n_u.n_1addr);
	case I386_SETS_F_98:
		strfree(s);
		return one_expr("sets", node->n_u.n_1addr);
	case I386_SETNS_F_99:
		strfree(s);
		return one_expr("setns", node->n_u.n_1addr);
	case I386_SETP_F_9A:
		strfree(s);
		return one_expr("setp", node->n_u.n_1addr);
	case I386_SETNP_F_9B:
		strfree(s);
		return one_expr("setnp", node->n_u.n_1addr);
	case I386_SETL_F_9C:
		strfree(s);
		return one_expr("setl", node->n_u.n_1addr);
	case I386_SETNL_F_9D:
		strfree(s);
		return one_expr("setnl", node->n_u.n_1addr);
	case I386_SETLE_F_9E:
		strfree(s);
		return one_expr("setle", node->n_u.n_1addr);
	case I386_SETNLE_F_9F:
		strfree(s);
		return one_expr("setnle", node->n_u.n_1addr);
	case I386_SHLD_F_Ax:
		strfree(s);
		return three_expr("shld", node->n_u.n_3addr.n_first,
			node->n_u.n_3addr.n_second, node->n_u.n_3addr.n_third);
	case I386_SHRD_F_Ax:
		strfree(s);
		return three_expr("shrd", node->n_u.n_3addr.n_first,
			node->n_u.n_3addr.n_second, node->n_u.n_3addr.n_third);
	case I486_CMPXCHG_F_Ax:
		strfree(s);
		return two_expr_b("cmpxchg", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_LSS_F_B2:
		strfree(s);
		return two_expr("lss", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_LFS_F_B4:
		strfree(s);
		return two_expr("lfs", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_LGS_F_B5:
		strfree(s);
		return two_expr("lgs", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_BSF_F_BC:
		strfree(s);
		return two_expr("bsf", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_BSR_F_BD:
		strfree(s);
		return two_expr("bsr", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_MOVZX_F_Bx:
		strfree(s);
		return two_expr_b2("movzx", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I386_MOVSX_F_Bx:
		strfree(s);
		return two_expr_b2("movsx", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I486_XADD_F_Cx:
		strfree(s);
		return two_expr_b("xadd", node->n_u.n_2addr.n_first,
					node->n_u.n_2addr.n_second);
	case I486_BSWAP_F_Cx:
		strfree(s);
		return one_expr("bswap", node->n_u.n_1addr);
	case I386_JMP:
		strfree(s);
		return one_expr("jmp", node->n_u.n_1addr);
	case I386_POP:
		strfree(s);
		return one_expr("pop", node->n_u.n_1addr);
	case I386_PUSH:
		strfree(s);
		return one_expr("push", node->n_u.n_1addr);
	case I386_SEG_PREF:
		strfree(s);
		return seg_prefix(node->n_u.n_1addr);
	case I386_ILLEGAL:
		strcpy(s, ".data1\t");
		for (i= 0, first= 1; i<ack_size; i++, first= 0)
		{
			sprintf(num, "%s0x%x", first ? "" : ", ", 
						*(unsigned char *)(block+i));
			strcat(s, num);
		}
		return s;
	default:
		printf("node2string: illegal node type %d\n", node->n_type);
		exit(1);
	}
}

DEFUN
(static char *exprnode2string, (node),
	n386_t *node
)
{
	char *o;
	char *s;
	char num[16];
	i16_t i16;
	i32_t i32;
	unsigned long offset, target, label_value;
	char *label_name;

	s= stralloc();
	
	switch(node->n_type)
	{
	case I386_REG8:
		switch(node->n_u.n_reg)
		{
		case I386_AL: strcpy(s, "al"); break;
		case I386_CL: strcpy(s, "cl"); break;
		case I386_DL: strcpy(s, "dl"); break;
		case I386_BL: strcpy(s, "bl"); break;
		case I386_AH: strcpy(s, "ah"); break;
		case I386_CH: strcpy(s, "ch"); break;
		case I386_DH: strcpy(s, "dh"); break;
		case I386_BH: strcpy(s, "bh"); break;
		default:
			printf("exprnode2string: illegal reg8 %d\n",
				node->n_u.n_reg);
			exit(1);
		}
		break;
	case I386_REG16:
		switch(node->n_u.n_reg)
		{
		case I386_AX: strcpy(s, "ax"); break;
		case I386_CX: strcpy(s, "cx"); break;
		case I386_DX: strcpy(s, "dx"); break;
		case I386_BX: strcpy(s, "bx"); break;
		case I386_SP: strcpy(s, "sp"); break;
		case I386_BP: strcpy(s, "bp"); break;
		case I386_SI: strcpy(s, "si"); break;
		case I386_DI: strcpy(s, "di"); break;
		default:
			printf("exprnode2string: illegal reg16 %d\n",
				node->n_u.n_reg);
			exit(1);
		}
		break;
	case I386_REG32:
		switch(node->n_u.n_reg)
		{
		case I386_EAX: strcpy(s, "eax"); break;
		case I386_ECX: strcpy(s, "ecx"); break;
		case I386_EDX: strcpy(s, "edx"); break;
		case I386_EBX: strcpy(s, "ebx"); break;
		case I386_ESP: strcpy(s, "esp"); break;
		case I386_EBP: strcpy(s, "ebp"); break;
		case I386_ESI: strcpy(s, "esi"); break;
		case I386_EDI: strcpy(s, "edi"); break;
		default:
			printf("exprnode2string: illegal reg32 %d\n",
				node->n_u.n_reg);
			exit(1);
		}
		break;
	case I386_REG_ST0:
		strcpy(s, "st");
		break;
	case I386_REG_ST:
		sprintf(s, "st(%d)", node->n_u.n_freg);
		break;
	case I386_ADDRESS8:
	case I386_ADDRESS16:
	case I386_ADDRESS32:
	case I386_ADDRESS64:
	case I386_ADDRESS80:
	case I386_ADDRESS14B:
	case I386_ADDRESS28B:
	case I386_ADDRESS94B:
	case I386_ADDRESS108B:
	case I386_ADDRESS_FAR16:
	case I386_ADDRESS_FAR32:
	case I386_ADDRESS16_16:
	case I386_ADDRESS16_32:
	case I386_ADDRESS32_32:
		strcpy(s, "");
#if PRINT_ALL
		o= exprnode2string(node->n_u.n_address.n_segment);
		strcat(s, o);
		strfree(o);
		strcat(s, ":");
#endif /* PRINT_ALL */
		if (node->n_u.n_address.n_disp)
		{
			o= exprnode2string(node->n_u.n_address.n_disp);
			if (node->n_u.n_address.n_base == 0 &&
				node->n_u.n_address.n_index == 0)
			{
				strcat(s, "(");
				strcat(s, o);
				strcat(s, ")");
			}
			else
				strcat(s, o);
			strfree(o);
		}
		if (node->n_u.n_address.n_base)
		{
			strcat(s,"(");
			o= exprnode2string(node->n_u.n_address.n_base);
			strcat(s, o);
			strfree(o);
			strcat(s,")");
		}
		if (node->n_u.n_address.n_index)
		{
			strcat(s,"(");
			o= exprnode2string(node->n_u.n_address.n_index);
			strcat(s, o);
			strfree(o);
			if (node->n_u.n_address.n_scale != 1)
			{
				sprintf(num, "*%d",
						node->n_u.n_address.n_scale);
				strcat(s, num);
			}
			strcat(s,")");
		}
#if PRINT_ALL
		switch(node->n_type)
		{
		case I386_ADDRESS8: strcat(s, ":8"); break;
		case I386_ADDRESS16: strcat(s, ":16"); break;
		case I386_ADDRESS32: strcat(s, ":32"); break;
		case I386_ADDRESS64: strcat(s, ":64"); break;
		case I386_ADDRESS80: strcat(s, ":80"); break;
		case I386_ADDRESS14B: strcat(s, ":14b"); break;
		case I386_ADDRESS28B: strcat(s, ":28b"); break;
		case I386_ADDRESS94B: strcat(s, ":94b"); break;
		case I386_ADDRESS108B: strcat(s, ":108b"); break;
		case I386_ADDRESS_FAR16: strcat(s, ":16:16"); break;
		case I386_ADDRESS_FAR32: strcat(s, ":32:16"); break;
		case I386_ADDRESS16_16: strcat(s, ":16:16"); break;
		case I386_ADDRESS16_32: strcat(s, ":16:32"); break;
		case I386_ADDRESS32_32: strcat(s, ":32:32"); break;
		default:
			printf("exprnode2string: illegal type %d\n", 
				node->n_type);
			exit(1);
		}
#endif /* PRINT_ALL */
		break;
	case I386_CONST_I8:
		sprintf(s, "%d", node->n_u.n_const_i8);
		break;
	case I386_CONST_U8:
		sprintf(s, "%d", node->n_u.n_const_u8);
		break;
	case I386_CONST16:
		i16= node->n_u.n_const16;
		sprintf(s, "%s0x%x", i16 < 0 ? "-" : "",i16 < 0 ? -i16 : i16);
		break;
	case I386_CONST32:
		i32= node->n_u.n_const32;
		sprintf(s, "%s0x%x", i32 < 0 ? "-" : "",i32 < 0 ? -i32 : i32);
		break;
	case I386_PCREL8:
	case I386_PCREL16:
	case I386_PCREL32:
		sprintf(num, ".+%d+", ack_size);
		strcpy(s, num);
		o= exprnode2string(node->n_u.n_1addr);
		strcat(s,o);
		strfree(o);
		switch(node->n_u.n_1addr->n_type)
		{
		case I386_CONST_I8:
			offset= node->n_u.n_1addr->n_u.n_const_i8; break;
		case I386_CONST_U8:
			offset= node->n_u.n_1addr->n_u.n_const_u8; break;
		case I386_CONST16:
			offset= node->n_u.n_1addr->n_u.n_const16; break;
		case I386_CONST32:
			offset= node->n_u.n_1addr->n_u.n_const32; break;
		default:
			return s;
		}	
		target= ack_pc+(ack_block1-ack_block)+ack_size+offset;
		label_name= NULL;
		if (ack_text_func)
		{
			(*ack_text_func)(target, &label_name, &label_value,
				NULL, NULL);
		}		
		if (label_name == NULL)
		{
			o= stralloc();
			sprintf(o, "(addr: 0x%x)", target);
			if (comment)
			{
				strcat(comment, o);
				strfree(o);
			}
			else
				comment= o;	
			return s;
		}
		o= stralloc();
		sprintf(o, "(addr: %s = 0x%x)", s, target);
		strfree(s);
		if (comment)
		{
			strcat(comment, o);
			strfree(o);
		}
		else
			comment= o;
		s= stralloc();
		if (target == label_value)
			sprintf(s, "%s", label_name);
		else	
			sprintf(s, "%s+%ld", label_name, target-label_value);
		return s;
	case I386_SEG:
		switch(node->n_u.n_reg)
		{
		case I386_ES: strcpy(s, "es"); break;
		case I386_CS: strcpy(s, "cs"); break;
		case I386_SS: strcpy(s, "ss"); break;
		case I386_DS: strcpy(s, "ds"); break;
		case I386_FS: strcpy(s, "fs"); break;
		case I386_GS: strcpy(s, "gs"); break;
		default:
			printf("exprnode2string: illegal seg %d\n",
				node->n_u.n_reg);
			exit(1);
		}
		break;
	case I386_CREG:
		switch(node->n_u.n_reg)
		{
		case I386_CR0: strcpy(s, "cr0"); break;
		case I386_CR1: strcpy(s, "cr1"); break;
		case I386_CR2: strcpy(s, "cr2"); break;
		case I386_CR3: strcpy(s, "cr3"); break;
		case I386_CR4: strcpy(s, "cr4"); break;
		case I386_CR5: strcpy(s, "cr5"); break;
		case I386_CR6: strcpy(s, "cr6"); break;
		case I386_CR7: strcpy(s, "cr7"); break;
		default:
			printf("exprnode2string: illegal creg %d\n",
				node->n_u.n_reg);
			exit(1);
		}
		break;
	case I386_DREG:
		switch(node->n_u.n_reg)
		{
		case I386_DR0: strcpy(s, "dr0"); break;
		case I386_DR1: strcpy(s, "dr1"); break;
		case I386_DR2: strcpy(s, "dr2"); break;
		case I386_DR3: strcpy(s, "dr3"); break;
		case I386_DR4: strcpy(s, "dr4"); break;
		case I386_DR5: strcpy(s, "dr5"); break;
		case I386_DR6: strcpy(s, "dr6"); break;
		case I386_DR7: strcpy(s, "dr7"); break;
		default:
			printf("exprnode2string: illegal dreg %d\n",
				node->n_u.n_reg);
			exit(1);
		}
		break;
	case I386_TREG:
		switch(node->n_u.n_reg)
		{
		case I386_TR0: strcpy(s, "tr0"); break;
		case I386_TR1: strcpy(s, "tr1"); break;
		case I386_TR2: strcpy(s, "tr2"); break;
		case I486_TR3: strcpy(s, "tr3"); break;
		case I486_TR4: strcpy(s, "tr4"); break;
		case I486_TR5: strcpy(s, "tr5"); break;
		case I386_TR6: strcpy(s, "tr6"); break;
		case I386_TR7: strcpy(s, "tr7"); break;
		default:
			printf("exprnode2string: illegal treg %d\n",
				node->n_u.n_reg);
			exit(1);
		}
		break;
	default:
		printf("exprnode2string: illegal node type %d\n", 
								node->n_type);
		exit(1);
	}
	return s;
}

DEFUN
(static char *dump, (pc, block),
	unsigned long pc AND
	char *block
)
{
	char *o;
	char *s;
	char num[16], num1[16];
	int i;

	s= stralloc();
	
	strcpy(s, "\t! ");
	if (comment)
	{
		strcat(s, comment);
		strfree(comment);
		comment= NULL;
		strcat(s, " ");
	}
	sprintf(num1, "0x%lx", pc);
	sprintf(num, "%6s: ", num1);
	strcat(s, num);
	for (i= 0; i<ack_size; i++, block++)
	{
		if (i != 0)
			strcat(s, " ");
		sprintf(num, "%02x", *(unsigned char *)block);
		strcat(s, num);
	}
	return s;
}

DEFUN
(static char *one_expr, (name, expr),
	char *name AND
	n386_t *expr
)
{
	char *s, *o;
	
	s= stralloc();
	
	strcpy(s, name);
	strcat(s, "\t");
	o= exprnode2string(expr);
	strcat(s, o);
	strfree(o);
	return s;
}

DEFUN
(static char *one_expr_b, (name, expr),
	char *name AND
	n386_t *expr
)
{
	char name_buf[16];
	
	if (expr->n_type == I386_REG8 || expr->n_type == I386_ADDRESS8)
	{
		strcpy(name_buf, name);
		strcat(name_buf, "b");
		name= name_buf;
	}
	return one_expr(name, expr);
}

DEFUN
(static char *one_expr_fi, (name, expr),
	char *name AND
	n386_t *expr
)
{
	char name_buf[16];
	
	strcpy(name_buf, name);
	name= name_buf;
	switch(expr->n_type)
	{
	case I386_ADDRESS16:
		strcat(name_buf, "s");
		break;
	case I386_ADDRESS32:
		strcat(name_buf, "l");
		break;
	case I386_ADDRESS64:
		strcat(name_buf, "q");
		break;
	default:
		printf("one_expr_fi: unknown type %d\n", expr->n_type);
		exit(1);
	}
	return one_expr(name, expr);
}

DEFUN
(static char *one_expr_f, (name, expr),
	char *name AND
	n386_t *expr
)
{
	char name_buf[16];
	
	strcpy(name_buf, name);
	name= name_buf;
	switch(expr->n_type)
	{
	case I386_ADDRESS32:
		strcat(name_buf, "s");
		break;
	case I386_ADDRESS64:
		strcat(name_buf, "d");
		break;
	case I386_ADDRESS80:
		strcat(name_buf, "x");
		break;
	case I386_REG_ST:
		break;
	default:
		printf("one_expr_f: unknown type %d\n", expr->n_type);
		exit(1);
	}
	return one_expr(name, expr);
}

DEFUN
(static char *two_expr, (name, first, second),
	char *name AND
	n386_t *first AND
	n386_t *second
)
{
	char *s, *o;
	
	s= stralloc();
	
	strcpy(s, name);
	strcat(s, "\t");
	o= exprnode2string(first);
	strcat(s, o);
	strfree(o);
	strcat(s, ", ");
	o= exprnode2string(second);
	strcat(s, o);
	strfree(o);
	return s;
}

DEFUN
(static char *two_expr_2nd, (name, first, second),
	char *name AND
	n386_t *first AND
	n386_t *second
)
{
#if PRINT_ALL
	return two_expr(name, first, second);
#else /* PRINT_ALL */
	return one_expr(name, second);
#endif /* PRINT_ALL */
}

DEFUN
(static char *seg_prefix, (node),
	n386_t *node
)
{
	char *s, *o;
	
	s= stralloc();
	
	switch(node->n_u.n_reg)
	{
	case I386_ES: strcpy(s, "eseg"); break;
	case I386_CS: strcpy(s, "cseg"); break;
	case I386_SS: strcpy(s, "sseg"); break;
	case I386_DS: strcpy(s, "dseg"); break;
	case I386_FS: strcpy(s, "fseg"); break;
	case I386_GS: strcpy(s, "gseg"); break;
	default:
		printf("seg_prefix: illegal selector %d\n", node->n_u.n_reg);
		exit(1);
	}
	return s;
}

DEFUN
(static char *far_expr, (name, offset, segment),
	char *name AND
	n386_t *offset AND
	n386_t *segment
)
{
	char *s, *o;
	
	s= stralloc();
	
	strcpy(s, name);
	strcat(s, "\t");
	o= exprnode2string(segment);
	strcat(s, o);
	strfree(o);
	strcat(s, ":");
	o= exprnode2string(offset);
	strcat(s, o);
	strfree(o);
	return s;
}

DEFUN
(static char *two_expr_b, (name, first, second),
	char *name AND
	n386_t *first AND
	n386_t *second
)
{
	char name_buf[16];
	
	if (first->n_type == I386_REG8 || first->n_type == I386_ADDRESS8)
	{
		strcpy(name_buf, name);
		strcat(name_buf, "b");
		name= name_buf;
	}
	return two_expr(name, first, second);
}

DEFUN
(static char *two_expr_b_0, (name, first, second),
	char *name AND
	n386_t *first AND
	n386_t *second
)
{
	char *s, name_buf[16];
	
	if (first->n_type == I386_REG8 || first->n_type == I386_ADDRESS8)
	{
		strcpy(name_buf, name);
		strcat(name_buf, "b");
		name= name_buf;
	}
#if PRINT_ALL
	return two_expr(name, first, second);
#else /* PRINT_ALL */
	s= stralloc();
	strcpy(s, name);
	return s;
#endif /* PRINT_ALL */	
}

DEFUN
(static char *two_expr_b2, (name, first, second),
	char *name AND
	n386_t *first AND
	n386_t *second
)
{
	char name_buf[16];
	
	if (second->n_type == I386_REG8 || second->n_type == I386_ADDRESS8)
	{
		strcpy(name_buf, name);
		strcat(name_buf, "b");
		name= name_buf;
	}
	return two_expr(name, first, second);
}

DEFUN
(static char *two_expr_b_2nd, (name, first, second),
	char *name AND
	n386_t *first AND
	n386_t *second
)
{
	char name_buf[16];
	
	if (first->n_type == I386_REG8 || first->n_type == I386_ADDRESS8)
	{
		strcpy(name_buf, name);
		strcat(name_buf, "b");
		name= name_buf;
	}
#if PRINT_ALL
	return two_expr(name, first, second);
#else /* PRINT_ALL */
	return one_expr(name, second);
#endif /* PRINT_ALL */
}

DEFUN
(static char *two_expr_b2nd, (name, first, second),
	char *name AND
	n386_t *first AND
	n386_t *second
)
{
	char name_buf[16];
	
	if (first->n_type == I386_REG8 || first->n_type == I386_ADDRESS8)
	{
		strcpy(name_buf, name);
		strcat(name_buf, "b");
		name= name_buf;
#if PRINT_ALL
		return two_expr(name, first, second);
#else /* PRINT_ALL */
		return one_expr(name, second);
#endif /* PRINT_ALL */
	}
	return two_expr(name, first, second);
}

DEFUN
(static char *three_expr, (name, first, second, third),
	char *name AND
	n386_t *first AND
	n386_t *second AND
	n386_t *third
)
{
	char *s, *o;
	
	s= stralloc();
	
	strcpy(s, name);
	strcat(s, "\t");
	o= exprnode2string(first);
	strcat(s, o);
	strfree(o);
	strcat(s, ", ");
	o= exprnode2string(second);
	strcat(s, o);
	strfree(o);
	strcat(s, ", ");
	o= exprnode2string(third);
	strcat(s, o);
	strfree(o);
	return s;
}

DEFUN_VOID( static char *(stralloc) )
{
	char *s;
	
	s= malloc(1024);	/* Enough */
	if (s == 0)
	{
		fprintf(stderr, "stralloc: out of memory\n");
		exit(1);
	}
	n_strings++;
	return s;
}

DEFUN
(static void (strfree), (s),
	char *s
)
{
	if (s != NULL)
	{
		n_strings--;
		free(s);
	}
}

DEFUN
(char *ack_text_label, (label),
	char *label
)
{
	char *s;

	s= stralloc();
	sprintf(s, "%.512s:", label);
	n_strings--;
	return s;
}

DEFUN
(void ack_label_funcs, (text_func, data_func, const_func),
	ack_label_func_t text_func AND 
	ack_label_func_t data_func AND
	ack_label_func_t const_func
)
{
	ack_text_func= text_func;	
	ack_data_func= data_func;	
	ack_const_func= const_func;	
}	
