/* This file handles signals, which are asynchronous events and are generally
 * a messy and unpleasant business.  Signals can be generated by the KILL
 * system call, or from the keyboard (SIGINT) or from the clock (SIGALRM).
 * In all cases control eventually passes to check_sig() to see which processes
 * can be signaled.  The actual signaling is done by sig_proc().
 *
 * The entry points into this file are:
 *   do_signal:	perform the SIGNAL system call
 *   do_kill:	perform the KILL system call
 *   do_sigaction:   perform the SIGACTION system call
 *   do_sigpending:  perform the SIGPENDING system call
 *   do_sigprocmask: perform the SIGPROCMASK system call
 *   do_sigreturn:   perform the SIGRETURN system call
 *   do_sigsuspend:  perform the SIGSUSPEND system call
 *   do_ksig:	accept a signal originating in the kernel (e.g., SIGINT)
 *   sig_proc:	interrupt or terminate a signaled process
 *   do_alarm:	perform the ALARM system call by calling set_alarm()
 *   set_alarm:	tell the clock task to start or stop a timer
 *   do_pause:	perform the PAUSE system call
 */

#include "mm.h"
#include <sys/stat.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include <sys/core.h>
#include <signal.h>
#include <sys/sigcontext.h>
#include <unistd.h>
#include "assert.h"
INIT_ASSERT
#include "mproc.h"
#include "param.h"

#define CORE_MODE	0777	/* mode to use on core image files */
#define DUMPED          0200	/* bit set in status when core dumped */
#define DUMP_SIZE	((INT_MAX / BLOCK_SIZE) * BLOCK_SIZE)
				/* buffer size for core dumps */

FORWARD _PROTOTYPE( void check_pending, (void)				);
FORWARD _PROTOTYPE( int check_sig, (pid_t proc_id, int signo)		);
FORWARD _PROTOTYPE( void dump_core, (struct mproc *rmp)			);
FORWARD _PROTOTYPE( void unpause, (int pro)				);
FORWARD _PROTOTYPE( int fs_write, (int fd, unsigned nbytes, int proc,
						int seg, vir_bytes ptr)	);

/*===========================================================================*
 *				do_signal				     *
 *===========================================================================*/
PUBLIC int do_signal()
{
/* Perform the signal(sig, func) call by setting bits to indicate that a 
 * signal is to be caught or ignored.
 *
 * This function exists only for compatibility with old binaries.  In the
 * new implementation, signal(2) is a library function built on top of the
 * POSIX sigaction(2) system call.  The present system level function ties
 * into the mechanism used to implement sigaction().
 *
 * Since the process invoking this system call is oblivious to signal
 * masks, it is assumed that the signal is not in the process signal
 * mask, and is therefore also not pending. 
 */

  sigset_t old_ignore;

  assert(mm_call == SIGNAL);

  if (signal_sig < 1 || signal_sig > _NSIG) return(EINVAL);
  if (signal_sig == SIGKILL) return(OK);	/* SIGKILL may not ignored/caught */
  old_ignore = mp->mp_ignore;

  sigemptyset(&mp->mp_sigact[signal_sig].sa_mask);
  sigdelset(&mp->mp_sigmask, signal_sig);
  mp->mp_sigact[signal_sig].sa_flags = SA_COMPAT | SA_NODEFER;
  mp->mp_sigact[signal_sig].sa_handler = signal_func;

  if (signal_func == SIG_IGN) {
	sigaddset(&mp->mp_ignore, signal_sig);
	sigdelset(&mp->mp_catch, signal_sig);
  } else {
	sigdelset(&mp->mp_ignore, signal_sig);
	if (signal_func == SIG_DFL)
		sigdelset(&mp->mp_catch, signal_sig);
  	else {
		sigaddset(&mp->mp_catch, signal_sig);
		if (signal_sig != SIGILL && signal_sig != SIGTRAP)
			mp->mp_sigact[signal_sig].sa_flags |= SA_RESETHAND;
		mp->mp_func = signal_func;
  	}
  }
  if (sigismember(&old_ignore, signal_sig)) return(1);
  return(OK);
}


/*===========================================================================*
 *				do_kill					     *
 *===========================================================================*/
PUBLIC int do_kill()
{
/* Perform the kill(pid, signo) system call. */

  assert(mm_call == KILL);

  return check_sig(kill_pid, kill_sig);
}


/*===========================================================================*
 *			       do_sigaction				     *
 *===========================================================================*/
PUBLIC int do_sigaction()
{
  int r;
  struct sigaction svec;
  struct sigaction *svp;

  assert(mm_call == SIGACTION);

  if (sigaction_sig < 1 || sigaction_sig > _NSIG) return (EINVAL);
  svp = &mp->mp_sigact[sigaction_sig];
  if (sigaction_osa != NULL) {
	r = sys_copy(MM_PROC_NR, SEG_D, (phys_bytes) svp, who, SEG_D,
		     (phys_bytes) sigaction_osa, (phys_bytes) sizeof(svec));
	if (r != OK) return(r);
  }

  if (sigaction_sig == SIGKILL) return OK;
  if (sigaction_nsa == NULL) return OK;

  /* Read in the sigaction structure. */
  r = sys_copy(who, SEG_D, (phys_bytes) sigaction_nsa, MM_PROC_NR, SEG_D,
	     (phys_bytes) &svec, (phys_bytes) sizeof(svec));
  if (r != OK) return(r);

  if (svec.sa_handler == SIG_IGN) {
	sigaddset(&mp->mp_ignore, sigaction_sig);
	sigdelset(&mp->mp_sigpending, sigaction_sig);
	sigdelset(&mp->mp_catch, sigaction_sig);
  } else {
	sigdelset(&mp->mp_ignore, sigaction_sig);
	if (svec.sa_handler == SIG_DFL)
		sigdelset(&mp->mp_catch, sigaction_sig);
	else
		sigaddset(&mp->mp_catch, sigaction_sig);
  }
  mp->mp_sigact[sigaction_sig].sa_handler = svec.sa_handler;
  sigdelset(&svec.sa_mask, SIGKILL);
  mp->mp_sigact[sigaction_sig].sa_mask = svec.sa_mask;
  mp->mp_sigact[sigaction_sig].sa_flags = svec.sa_flags;
  mp->mp_sigreturn = (vir_bytes) sigaction_ret;
  return(OK);
}

/*===========================================================================*
 *                            do_sigpending                                  *
 *===========================================================================*/
PUBLIC int do_sigpending()
{
  assert(mm_call == SIGPENDING);

  ret_mask = (long) mp->mp_sigpending;
  return OK;
}

/*===========================================================================*
 *                            do_sigprocmask                                 *
 *===========================================================================*/
PUBLIC int do_sigprocmask()
{
/* Note that the library interface passes the actual mask in sigmask_set,
 * not a pointer to the mask, in order to save a sys_copy.  Similarly,
 * the old mask is placed in the return message which the library
 * interface copies (if requested) to the user specified address.
 *
 * The library interface must set SIG_INQUIRE if the 'set' argument
 * is NULL.
 */

  int i;

  assert(mm_call == SIGPROCMASK);

  ret_mask = (long) mp->mp_sigmask;

  switch (sigprocmask_how) {
      case SIG_BLOCK:
	sigdelset((sigset_t *)&sigprocmask_set, SIGKILL);
	for (i = 1; i < _NSIG; i++) {
		if (sigismember((sigset_t *)&sigprocmask_set, i))
			sigaddset(&mp->mp_sigmask, i);
	}
	break;

      case SIG_UNBLOCK:
	for (i = 1; i < _NSIG; i++) {
		if (sigismember((sigset_t *)&sigprocmask_set, i))
			sigdelset(&mp->mp_sigmask, i);
	}
	check_pending();
	break;

      case SIG_SETMASK:
	sigdelset((sigset_t *)&sigprocmask_set, SIGKILL);
	mp->mp_sigmask = (sigset_t)sigprocmask_set;
	check_pending();
	break;

      case SIG_INQUIRE:
	break;

      default:
	return EINVAL;
	break;
  }
  return OK;
}

/*===========================================================================*
 *                               do_sigreturn				     *
 *===========================================================================*/
PUBLIC int do_sigreturn()
{
/* A user signal handler is done.  Restore context and check for
 * pending unblocked signals.
 */
  int r;

  assert(mm_call == SIGRETURN);

  mp->mp_sigmask = (sigset_t) sigreturn_set;
  sigdelset(&mp->mp_sigmask, SIGKILL);

  r= sys_sigreturn(who, (vir_bytes)sigreturn_context, sigreturn_flags);
  check_pending();
  return r;
}

/*===========================================================================*
 *                            do_sigsuspend                                  *
 *===========================================================================*/
PUBLIC int do_sigsuspend()
{
  assert(mm_call == SIGSUSPEND);

  mp->mp_sigmask2 = mp->mp_sigmask;	/* save the old mask */
  mp->mp_sigmask = (sigset_t) sigsuspend_set;
  sigdelset(&mp->mp_sigmask, SIGKILL);
  mp->mp_flags |= SIGSUSPENDED;
  dont_reply = TRUE;
  check_pending();
  return OK;
}


/*===========================================================================*
 *				check_sig				     *
 *===========================================================================*/
PRIVATE int check_sig(proc_id, signo)
pid_t proc_id;			/* pid of proc to sig, or 0 or -1, or -pgrp */
int signo;			/* signal to send to process (0 to _NSIG) */
{
/* Check to see if it is possible to send a signal.  The signal may have to be
 * sent to a group of processes.  This routine is invoked by the KILL system
 * call, and also when the kernel catches a DEL or other signal.
 */

  register struct mproc *rmp, *rmp_next;
  int count;			/* count # of signals sent */
  int error_code;

  if (signo < 0 || signo > _NSIG) return(EINVAL);

  /* Search the proc table for processes to signal. */
  /* XXX - see forkexit.c about pid magic. */
  count = 0;
  error_code = ESRCH;

  /* First check normal, life processes */
  for (rmp = mproc_inuse; rmp; rmp= rmp_next) {

  	/* Fetch the next process before the current process is moved to a
  	 * different queue
  	 */
  	rmp_next= rmp->mp_next;

	assert (rmp->mp_flags & IN_USE);
	assert (!(rmp->mp_flags & HANGING));

	/* Check for selection. */
	if (proc_id > 0 && proc_id != rmp->mp_pid) continue;
	if (proc_id == 0 && mp->mp_procgrp != rmp->mp_procgrp) continue;
	if (proc_id < -1 && rmp->mp_procgrp != -proc_id) continue;

	/* Check for permission. */
	if (mp->mp_effuid != SUPER_USER
	    && mp->mp_realuid != rmp->mp_realuid
	    && mp->mp_effuid != rmp->mp_realuid
	    && mp->mp_realuid != rmp->mp_effuid
	    && mp->mp_effuid != rmp->mp_effuid) {
		error_code = EPERM;
		continue;
	}

	count++;
	if (signo == 0) continue;

	/* 'sig_proc' will handle the disposition of the signal.  The
	 * signal may be caught, blocked, ignored, or cause process
	 * termination, possibly with core dump.
	 */

	sig_proc(rmp, signo);

	if (proc_id > 0) break;	/* only one process being signaled */
  }
  if (signo == 0 && count == 0)
  {
  	/* A signal 0 is also "delivered" to a dead process */
	for (rmp = mproc_deadlist; rmp; rmp= rmp->mp_next) {
		assert (rmp->mp_flags & IN_USE);
		assert (rmp->mp_flags & HANGING);

		/* Check for selection. */
		if (proc_id > 0 && proc_id != rmp->mp_pid) continue;
		if (proc_id == 0 && mp->mp_procgrp != rmp->mp_procgrp) continue;
		if (proc_id < -1 && rmp->mp_procgrp != -proc_id) continue;

		/* Check for permission. */
		if (mp->mp_effuid != SUPER_USER
		    && mp->mp_realuid != rmp->mp_realuid
		    && mp->mp_effuid != rmp->mp_realuid
		    && mp->mp_realuid != rmp->mp_effuid
		    && mp->mp_effuid != rmp->mp_effuid) {
			error_code = EPERM;
			continue;
		}

		count++;
	}
  }

  /* We have one special case left; a signal directed to init */
  if (proc_id == INIT_PID)
  {
  	rmp= &mproc[INIT_PROC_NR];

	/* Return EINVAL for attempts to send SIGKILL to INIT. */
  	if (signo == SIGKILL)
  		return(EINVAL);

	/* Check for permission. */
	if (mp->mp_effuid != SUPER_USER
	    && mp->mp_realuid != rmp->mp_realuid
	    && mp->mp_effuid != rmp->mp_realuid
	    && mp->mp_realuid != rmp->mp_effuid
	    && mp->mp_effuid != rmp->mp_effuid) {
		return EPERM;
	}

	count++;
	if (signo == 0)
		return OK;

	/* 'sig_proc' will handle the disposition of the signal.  The
	 * signal may be caught, blocked, ignored, or cause process
	 * termination, possibly with core dump.
	 */
	sig_proc(rmp, signo);
  }

  /* If the calling process has killed itself, don't reply. */
  if ((mp->mp_flags & IN_USE) == 0 || (mp->mp_flags & HANGING))
	dont_reply = TRUE;
  return(count > 0 ? OK : error_code);
}


/*===========================================================================*
 *				do_ksig					     *
 *===========================================================================*/
PUBLIC int do_ksig()
{
/* Certain signals, such as segmentation violations and DEL, originate in the
 * kernel.  When the kernel detects such signals, it sets bits in a bit map.
 * As soon as MM is awaiting new work, the kernel sends MM a message containing
 * the process slot and bit map.  That message comes here.  The File System
 * also uses this mechanism to signal writing on broken pipes (SIGPIPE).
 */

#if DEBUG
	static int count= 0;
#endif
	int loop_count= 0;

	/* The kernel sends only signals out of a small set. Checking only this
	 * set is faster and gives a better consistency check.
	 */
	static int kernel_sigs[]= { SIGTRAP, SIGALRM, SIGINT, SIGQUIT, SIGILL,
		SIGFPE, SIGSEGV, SIGWINCH, SIGPIPE, SIGHUP, SIGABRT, -1 };

	register struct mproc *rmp;
	int i, *ip, proc_nr, r;
	pid_t proc_id, id;
	sigset_t sig_map;

#if DEBUG
	count++;
	if ((count & 65535) == 0) printf("%d calls to do_ksig\n", count);
#endif

	assert(mm_call == KSIG);

	/* Only kernel may make this call. */
	if (who != synal_tasknr) 
	{
		if (who < INIT_PROC_NR)
			panic("KSIG from", who);
		where(); printf("returning EPERM\n");
		return(EPERM);
	}
	dont_reply = TRUE;		/* don't reply to the kernel */

	for(;;)
	{
		r= sys_getsig(&proc_nr, &sig_map);
		if (r != 0)
		{
			if (r == ESRCH)
				break;
			panic("sys_getsig failed", r);
		}
#if DEBUG
		if ((count & 65535) == 0)
		{
			printf("do_ksig[%d]: sigset 0x%lx for process %d\n",
				count, sig_map, proc_nr);
		}
#endif
		loop_count++;
		if (loop_count == 256)
		{
			printf(
		"do_ksig: signals looping: sigset %lx for process %d\n",
				sig_map, proc_nr);
		}

		rmp = &mproc[proc_nr];
		if ( (rmp->mp_flags & IN_USE) == 0 ||
			(rmp->mp_flags & HANGING) )
		{
			panic("got signal for unused slot ", proc_nr);
		}
		proc_id = rmp->mp_pid;
		mp = &mproc[0];	 /* pretend kernel signals are from MM */
		mp->mp_procgrp = rmp->mp_procgrp; /* get process group right */

		/* Check each bit in turn to see if a signal is to be sent.
		 * Unlike kill(), the kernel may collect several unrelated
		 * signals for a process and pass them to MM in one blow.
		 * Thus loop on the bit map. For SIGINT and SIGQUIT, use
		 * proc_id 0 to indicate a broadcast to the recipient's
		 * process group.
		 */
		for (ip= kernel_sigs; (i= *ip) != -1; ip++)
		{
			if (!sigismember(&sig_map, i))
				continue;
			sigdelset(&sig_map, i);
			switch (i) {
			case SIGINT: 
			case SIGQUIT: 
			case SIGWINCH:
				id = 0; break;	/* broadcast to process group */
			default:
				id = proc_id; 
				break;	
			}
			check_sig(id, i);
			if ((rmp->mp_flags & IN_USE) == 0 || 
				(rmp->mp_flags & HANGING))
			{
				/* Quit processing signals for this process. */
				sigemptyset(&sig_map);
			}
		}
		if (sig_map != 0)
		{
			printf(
			"do_ksig: unknown kernel signal, sig_map= 0x%lx\n",
				sig_map);
			panic(NULL, NO_NUM);
		}
	}
	return(OK);
}


/*===========================================================================*
 *				sig_proc				     *
 *===========================================================================*/
PUBLIC void sig_proc(rmp, signo)
register struct mproc *rmp;	/* pointer to the process to be signaled */
int signo;			/* signal to send to process (1 to _NSIG) */
{
/* Send a signal to a process.  Check to see if the signal is to be caught,
 * ignored, or blocked.  If the signal is to be caught, coordinate with
 * KERNEL to push a sigcontext structure and a sigframe structure onto
 * the catcher's stack.  Also, KERNEL will reset the program counter and
 * stack pointer, so that when the process next runs, it will be executing
 * the signal handler.  When the signal handler returns,  sigreturn(2)
 * will be called.  Then KERNEL will restore the signal context from the
 * sigcontext structure.
 *
 * If there is insufficient stack space, kill the process.
 */

  vir_bytes new_sp;
  int slot;
  int sigflags;
  struct sigmsg sm;

  slot = (int) (rmp - mproc);
  if (!(rmp->mp_flags & IN_USE)) {
	printf("MM: signal %d sent to dead process %d\n", signo, slot);
	panic("", NO_NUM);
  }
  if (rmp->mp_flags & HANGING) {
	printf("MM: signal %d sent to HANGING process %d\n", signo, slot);
	panic("", NO_NUM);
  }
  if (rmp->mp_flags & TRACED && signo != SIGKILL) {
	/* A traced process has special handling. */
	stop_proc(rmp, signo);	/* a signal causes it to stop */
	return;
  }
  if (sigismember(&rmp->mp_ignore, signo)) return;
  if (sigismember(&rmp->mp_sigmask, signo)) {
	/* Signal should be blocked. */
	sigaddset(&rmp->mp_sigpending, signo);
	return;
  }
  sigflags = rmp->mp_sigact[signo].sa_flags;
  if (sigismember(&rmp->mp_catch, signo)) {
	if (rmp->mp_flags & SIGSUSPENDED)
		sm.sm_mask = rmp->mp_sigmask2;
	else
		sm.sm_mask = rmp->mp_sigmask;
	sm.sm_signo = signo;
	sm.sm_sighandler = (vir_bytes) rmp->mp_sigact[signo].sa_handler;
	sm.sm_sigreturn = rmp->mp_sigreturn;
	sys_getsp(slot, &new_sp);
	sm.sm_stkptr = new_sp;
	if (sigflags & SA_COMPAT)
		/* make room for an old style stack frame */
		new_sp -= SIG_PUSH_BYTES;
	else
		/* make room for the sigcontext and sigframe struct */
		new_sp = new_sp - sizeof(struct sigcontext) 
			 - 3 * sizeof(char *) - 2 * sizeof(int);

	if (sys_adj_stack((int)(rmp-mproc), new_sp) != OK)
		goto dodefault;

	rmp->mp_sigmask |= rmp->mp_sigact[signo].sa_mask;
	if (sigflags & SA_NODEFER)
		sigdelset(&rmp->mp_sigmask, signo);
	else
		sigaddset(&rmp->mp_sigmask, signo);

	if (sigflags & SA_RESETHAND) {
		sigdelset(&rmp->mp_catch, signo);
		rmp->mp_sigact[signo].sa_handler = SIG_DFL;
	}

  	if (sigflags & SA_COMPAT)
		sys_oldsig(slot, signo, rmp->mp_func);
	else
		sys_sendsig(slot, &sm);

	sigdelset(&rmp->mp_sigpending, signo);
	/* If process is hanging on PAUSE, WAIT, SIGSUSPEND, tty, pipe, etc.,
	 * release it.
 	 */
	unpause(slot);
	return;
  }
dodefault:
  /* Signal should not or cannot be caught.  Take default action. */
  if (sigismember(&default_ignore, signo)) return;

  rmp->mp_sigstatus = (char) signo;
  if (sigismember(&core_bits, signo)) {
	/* Switch to the user's FS environment and dump core. */
	tell_fs(CHDIR, slot, FALSE, 0);
	dump_core(rmp);
  }
  mm_exit(rmp, 0);		/* terminate process */
}


/*===========================================================================*
 *				unpause					     *
 *===========================================================================*/
PRIVATE void unpause(pro)
int pro;			/* which process number */
{
/* A signal is to be sent to a process.  If that process is hanging on a
 * system call, the system call must be terminated with EINTR.  Possible
 * calls are PAUSE, WAIT, READ and WRITE, the latter two for pipes and ttys.
 * First check if the process is hanging on an MM call.  If not, tell FS,
 * so it can check for READs and WRITEs from pipes, ttys and the like.
 */

  register struct mproc *rmp;

  rmp = &mproc[pro];

  /* Check to see if process is hanging on a PAUSE call. */
  if ( (rmp->mp_flags & PAUSED) && (rmp->mp_flags & HANGING) == 0) {
	rmp->mp_flags &= ~PAUSED;
	reply(pro, EINTR, 0, NIL_PTR);
	return;
  }

  /* Check to see if process is hanging on a WAIT call. */
  if ( (rmp->mp_flags & WAITING) && (rmp->mp_flags & HANGING) == 0) {
	rmp->mp_flags &= ~WAITING;
	reply(pro, EINTR, 0, NIL_PTR);
	return;
  }

  /* Check to see if process is hanging on a SIGSUSPEND call. */
  if ((rmp->mp_flags & SIGSUSPENDED) && (rmp->mp_flags & HANGING) == 0) {
	rmp->mp_flags &= ~SIGSUSPENDED;
	reply(pro, EINTR, 0, NIL_PTR);
	return;
  }

  /* Process is not hanging on an MM call.  Ask FS to take a look. */
	tell_fs(UNPAUSE, pro, 0, 0);
}


/*===========================================================================*
 *                               check_pending				     *
 *===========================================================================*/
PRIVATE void check_pending()
{
  /* Check to see if any pending signals have been unblocked.  The
   * first such signal found is delivered.
   *
   * If multiple pending unmasked signals are found, they will be
   * delivered sequentially.
   *
   * There are several places in this file where the signal mask is
   * changed.  At each such place, check_pending() should be called to
   * check for newly unblocked signals.
   */

  int i;

  for (i = 1; i < _NSIG; i++) {
	if (sigismember(&mp->mp_sigpending, i) &&
		!sigismember(&mp->mp_sigmask, i)) {
		sigdelset(&mp->mp_sigpending, i);
		sig_proc(mp, i);
		break;
	}
  }
}


/*===========================================================================*
 *				do_alarm				     *
 *===========================================================================*/
PUBLIC int do_alarm()
{
/* Perform the alarm(seconds) system call. */

  assert(mm_call == ALARM);

  return(set_alarm(who, alarm_seconds));
}


/*===========================================================================*
 *				set_alarm				     *
 *===========================================================================*/
PUBLIC int set_alarm(proc_nr, sec)
int proc_nr;			/* process that wants the alarm */
int sec;			/* how many seconds delay before the signal */
{
/* This routine is used by do_alarm() to set the alarm timer.  It is also used
 * to turn the timer off when a process exits with the timer still on.
 */

  message m_sig;
  int remaining;

  /* Tell the clock task to provide a signal message when the time comes.
   *
   * Large delays cause a lot of problems.  First, the alarm system call
   * takes an unsigned seconds count and the library has cast it to an int.
   * That probably works, but on return the library will convert "negative"
   * unsigneds to errors.  Presumably no one checks for these errors, so
   * force this call through.  Second, If unsigned and long have the same
   * size, converting from seconds to ticks can easily overflow.  Finally,
   * the kernel has similar overflow bugs adding ticks.
   *
   * Fixing this requires a lot of ugly casts to fit the wrong interface
   * types and to avoid overflow traps.  DELTA_TICKS has the right type
   * (clock_t) although it is declared as long.  How can variables like
   * this be declared properly without combinatorial explosion of message
   * types?
   */
  m_sig.m_type = SET_ALARM;
  m_sig.CLOCK_PROC_NR = proc_nr;
  m_sig.DELTA_TICKS = (clock_t) (HZ * (unsigned long) (unsigned) sec);
  if ( (unsigned long) m_sig.DELTA_TICKS / HZ != (unsigned) sec)
	m_sig.DELTA_TICKS = LONG_MAX;	/* eternity (really CLOCK_T_MAX) */
  if (sendrec(clck_tasknr, &m_sig) != OK) panic("alarm err", NO_NUM);
  remaining = (int) m_sig.SECONDS_LEFT;
  if (remaining != m_sig.SECONDS_LEFT || remaining < 0)
	remaining = INT_MAX;	/* true value is not representable */
  return(remaining);
}


/*===========================================================================*
 *				do_pause				     *
 *===========================================================================*/
PUBLIC int do_pause()
{
/* Perform the pause() system call. */

  assert(mm_call == PAUSE);

  mp->mp_flags |= PAUSED;
  dont_reply = TRUE;
  return(OK);
}


/*===========================================================================*
 *				dump_core				     *
 *===========================================================================*/
PRIVATE void dump_core(rmp)
register struct mproc *rmp;	/* whose core is to be dumped */
{
/* Make a core dump on the file "core", if possible. */

  int fd, seg, slot;
  char *buf;
#if (CHIP == INTEL) && (WORD_SIZE == 2)
  core_86_t core;
#endif /* CHIP == INTEL && WORD_SIZE == 2 */
#if (CHIP == INTEL) && (WORD_SIZE == 4)
  core_386_t core;
#endif /* CHIP == INTEL && WORD_SIZE == 4 */
  phys_bytes left;		/* careful; 64K might overflow vir_bytes */
  unsigned nr_to_write;		/* unsigned for arg to write() but < INT_MAX */

  slot = (int) (rmp - mproc);

  /* Can core file be written?  We are operating in the user's FS environment,
   * so no special permission checks are needed.
   */
  if (rmp->mp_realuid != rmp->mp_effuid) return;
  if ( (fd = creat(core_name, CORE_MODE)) < 0) return;
  rmp->mp_sigstatus |= DUMPED;

   /* The kernel will keep the stack segment up to date. */

  /* Ask the kernel for the core header and write it out to begin the core. */
  sys_core(slot, &core, (int) sizeof core);	/* XXX - take size_t not int*/
  write(fd, (char *) &core, (unsigned) sizeof core);

  /* Loop through segments and write the segments themselves out. */
  for (seg = 0; seg < NR_SEGS; seg++) {
	switch (seg) {
	    case SEG_T:
		buf = (char *) core.c_text.core_vir;
		left = (phys_bytes) core.c_text.core_len;
		break;
	    case SEG_D:
		buf = (char *) core.c_data.core_vir;
		left = (phys_bytes) core.c_data.core_len;
		break;
	    case SEG_S:
		buf = (char *) core.c_stack.core_vir;
		left = (phys_bytes) core.c_stack.core_len;
		break;
	}

	/* Loop through a segment, dumping it. */
	while (left != 0) {
		nr_to_write = (unsigned) MIN(left, DUMP_SIZE);
		if (fs_write(fd, nr_to_write, slot, seg == SEG_S ? SEG_D : seg,
			(vir_bytes)buf) != OK)
		{
			close(fd);
			return;
		}
		buf += nr_to_write;
		left -= nr_to_write;
	}
  }
  close(fd);
}


/*===========================================================================*
 *				fs_write				     *
 *===========================================================================*/
PRIVATE int fs_write(fd, nbytes, proc, seg, ptr)
int fd;
unsigned nbytes;
int proc;
int seg;
vir_bytes ptr;
{
	message m;
	int r;

	assert(seg >= 0 && seg < NR_SEGS);

	m.m1_i1 = fd;
	m.m1_i2 = nbytes;
	m.m1_i3= (proc << 4) | seg;
	m.m1_p1 = (char *) ptr;

	/* Clear unused fields */
	m.m1_p2= 0;
	m.m1_p3= 0;

	r= _taskcall(FS_PROC_NR, WRITE, &m);
	if (r < 0)
		return r;
	if (r != nbytes)
		return EIO;
	return OK;
}

/*
 * $PchId: signal.c,v 1.6 1996/02/29 23:28:59 philip Exp $
 */
