/*
misc.c

miscellaneous system calls.

Created:	Aug 9, 1992 by Philip Homburg
*/

#include "mm.h"
#include <string.h>
#include <minix/callnr.h>
#include <minix/queryparam.h>
#include <signal.h>
#include <sys/svrctl.h>
#include "assert.h"
INIT_ASSERT
#include "mproc.h"
#include "param.h"

FORWARD _PROTOTYPE( int mmqueryparam, (vir_bytes argp)			    );
FORWARD _PROTOTYPE( int mq_getc, (void)					    );
FORWARD _PROTOTYPE( void mq_flush, (void)				    );

/*=====================================================================*
 *  			    do_nice				       *
 *=====================================================================*/
PUBLIC int do_nice()
{
  /* nice_table maps a nice value in the range [-20..20] to a schedule
   * quantum. We treat the nice value as a logarithmic value. The table has
   * to satisfy the following constraints:
   * - all values should be positive
   * - use at most 31 bits to avoid overflows
   * - the value that corresponds to nice value 0 should be equal to
   *   DEF_SCHEDQUANT in the kernel (currently, this value is 0x10000).
   *
   * This table is based on the following program:
   *    #include <stdio.h>
   *    #include <math.h>
   *    
   *    int main(void)
   *    {
   *    	int i;
   *    	unsigned long ul;
   *    	double v;
   *    
   *    	for (i= -20; i<= 20; i++)
   *    	{
   *    		v= 0x10000 * pow(1024, -i/20.0);
   *    		ul= v;
   *    		printf("0x%lx, ", ul);
   *    		if ((i+20) % 5 == 4)
   *    			printf("\n");
   *    	}
   *    	printf("\n");
   *    	return 0;
   *    }
   */

  static unsigned long nice_table[]=
  {
	0x4000000, 0x2d413cd, 0x2000001, 0x16a09e7, 0x1000001, 
	0xb504f4, 0x800000, 0x5a827a, 0x400000, 0x2d413d, 
	0x200000, 0x16a09f, 0x100001, 0xb5050, 0x80000, 
	0x5a828, 0x40000, 0x2d414, 0x20000, 0x16a0a, 
	0x10000,
	0xb505, 0x8000, 0x5a83, 0x4000, 0x2d42,
	0x2000, 0x16a1, 0x1000, 0xb51, 0x800,
	0x5a9, 0x401, 0x2d5, 0x201, 0x16b,
	0x100, 0xb6, 0x80, 0x5b, 0x40, 
  };

  register struct mproc *rmp = mp;
  int n;

  assert(mm_call == NICE);

  if (nice_type != 0)
	return EINVAL;
  if (nice_incr < 0 && rmp->mp_effuid != SUPER_USER)
  	return EPERM;

  n= rmp->mp_nice + nice_incr;
  if (n < -20)
	n= -20;
  else if (n > 20)
	n= 20;

  rmp->mp_nice= n;

  sys_nice(who, nice_table[n+20]);
  result2= n;
  return OK;
}

/*=====================================================================*
 *  			    do_reboot				       *
 *=====================================================================*/
PUBLIC int do_reboot()
{
  register struct mproc *rmp = mp;
  char monitor_code[256];

  assert(mm_call == REBOOT);

  if (rmp->mp_effuid != SUPER_USER)   return EPERM;

  switch (reboot_flag) {
  case RBT_HALT:
  case RBT_REBOOT:
  case RBT_PANIC:
  case RBT_RESET:
	break;
  case RBT_MONITOR:
	if (reboot_size > sizeof(monitor_code)) return EINVAL;
	memset(monitor_code, 0, sizeof(monitor_code));
	if (sys_copy(who, SEG_D, (phys_bytes) reboot_code,
		MM_PROC_NR, SEG_D, (phys_bytes) monitor_code,
		(phys_bytes) reboot_size) != OK) return EFAULT;
	if (monitor_code[sizeof(monitor_code)-1] != 0) return EINVAL;
	break;
  default:
	return EINVAL;
  }

  while (mproc_inuse)
  {
	rmp= mproc_inuse;
	assert((rmp->mp_flags & (IN_USE|HANGING)) == IN_USE);
	sig_proc(rmp, 9);	/* Kill all user processes except init */
	assert(rmp != mproc_inuse);
  }

  tell_fs(EXIT, INIT_PROC_NR, 0, 0);	/* cleanup init */

#if VMDEXT_NEWMOUNT
  tell_fs(REBOOT,0,0,0);
#else /* !VMDEXT_NEWMOUNT */
  tell_fs(SYNC,0,0,0);
#endif /* !VMDEXT_NEWMOUNT */

  sys_abort(reboot_flag, monitor_code);
  /* NOTREACHED */
}

/*=====================================================================*
 *  			    do_sysenv                                  *
 *=====================================================================*/
PUBLIC int do_sysenv()
{
  assert(mm_call == SYSENV);

  return sys_sysenv(who, sysenv_key, sysenv_keylen, sysenv_val, sysenv_vallen);
}


#if PAGING_VM
/*=====================================================================*
 *  			    do_swapoff                                  *
 *=====================================================================*/
PUBLIC int do_swapoff()
{
  register struct mproc *rmp = mp;
  int r;

  assert(mm_call == SWAPOFF);

  if (rmp->mp_effuid != SUPER_USER)   return EPERM;

  r= sys_swapoff(swapoff_segm);
  return r;
}
#endif /* PAGING_VM */

/*=====================================================================*
 *  			    do_svrctl                                  *
 *=====================================================================*/
PUBLIC int do_svrctl()
{
	ioreq_t req;
	vir_bytes ptr;
	struct mmexeccache mmexeccache;
	int r;
	struct mproc *rmp;
	pid_t pidarg;

	assert(mm_call == SVRCTL);

	req= mm_in.m2_l1;
	if (req == 0) req= mm_in.m2_i1;

	ptr= (vir_bytes)mm_in.m2_p1;

	if (mp->mp_effuid != SUPER_USER)
		return EPERM;

	/* Is the request for the kernel? */
	if (((req >> 8) & 0xFF) == 'S') {
		return sys_sysctl(who, req, ptr);
	}

	switch(req)
	{
	case MMGEXECCACHE:
		mmexeccache.mec_limit= ec_limit;
		mmexeccache.mec_usage= ec_usage;
		r= sys_copy(MM_PROC_NR, SEG_D, (phys_bytes) &mmexeccache,
			who, SEG_D, (phys_bytes) ptr,
			(phys_bytes) sizeof(mmexeccache));
		return r;

	case MMSEXECCACHE:
		r= sys_copy(who, SEG_D, (phys_bytes) ptr,
			MM_PROC_NR, SEG_D, (phys_bytes) &mmexeccache,
			(phys_bytes) sizeof(mmexeccache));
		if (r != OK)
			return r;
		ec_limit= mmexeccache.mec_limit;
		ec_adjust_usage();
		return OK;

	case MMQUERYPARAM:
		return mmqueryparam(ptr);

	case MMSIGNON:
		/* A user process becomes a task.  Simulate an exit by
		 * releasing a waiting parent and disinheriting children.
		 */
		rmp = &mproc[mp->mp_parent];
		tell_fs(EXIT, who, 1, 0);

		pidarg = rmp->mp_wpid;
		if ((rmp->mp_flags & WAITING) && (pidarg == -1
			|| pidarg == mp->mp_pid || -pidarg == mp->mp_procgrp))
		{
			/* Wake up the parent. */
			reply(mp->mp_parent, mp->mp_pid, 0, NIL_PTR);
			rmp->mp_flags &= ~WAITING;
		}

		/* Disinherit children. */
		for (rmp = mproc_inuse; rmp; rmp= rmp->mp_next)
		{
			if (rmp->mp_parent == who) {
				rmp->mp_parent = INIT_PROC_NR;
			}
		}

		/* Become invisible. */
		mp->mp_pid = mp->mp_procgrp = 0;
		mp->mp_parent = 0;
		if (mproc_inuse == mp) mproc_inuse= mp->mp_next;
		if (mp->mp_next != NULL) mp->mp_next->mp_prev= mp->mp_prev;
		if (mp->mp_prev != NULL) mp->mp_prev->mp_next= mp->mp_next;
		mproc_syslist->mp_prev= mp;
		mp->mp_next= mproc_syslist;
		mp->mp_prev= NULL;
		mproc_syslist= mp;
		return OK;

	default:
		return EINVAL;
	}
}


/*===========================================================================*
 *				mmqueryparam				     * 
 *===========================================================================*/
PRIVATE struct queryvars {
	struct svrqueryparam qpar;
	char parbuf[256], valbuf[256];
	char *param, *value;
	size_t vcount;
	int r;
} *qvars;

PRIVATE int mmqueryparam(argp)
vir_bytes argp;
{
	/* Return values, sizes, or addresses of variables in MM space. */

	struct queryvars qv;
	void *addr;
	size_t n, size;
	int byte;
	int more;
	static char hex[]= "0123456789ABCDEF";
#define mq_putc(c) ((void)((qv.vcount == 0 ? mq_flush() : (void) 0), \
					*qv.value++= (c), qv.vcount--))

	qv.r= sys_copy(who, SEG_D, (phys_bytes) argp,
		MM_PROC_NR, SEG_D, (phys_bytes) &qv.qpar,
		(phys_bytes) sizeof(qv.qpar));

	/* Export these to mq_getc() and mq_flush(). */
	qvars= &qv;
	qv.param= qv.parbuf + sizeof(qv.parbuf);
	qv.value= qv.valbuf;
	qv.vcount= sizeof(qv.valbuf);

	do {
		more= queryparam(mq_getc, &addr, &size);
		for (n= 0; n < size; n++) {
			byte= ((u8_t *) addr)[n];
			mq_putc(hex[byte >> 4]);
			mq_putc(hex[byte & 0x0F]);
		}
		mq_putc(more ? ',' : 0);
	} while (more);
	mq_flush();
	return qv.r;
}


/*===========================================================================*
 *				mq_getc					     *
 *===========================================================================*/
PRIVATE int mq_getc()
{
	/* Return one character of the names to search for. */
	struct queryvars *qv= qvars;
	size_t n;

	if (qv->r != OK || qv->qpar.psize == 0) return 0;
	if (qv->param == qv->parbuf + sizeof(qv->parbuf)) {
		/* Need to fill the parameter buffer. */
		n= qv->qpar.psize;
		if (n > sizeof(qv->parbuf)) n= sizeof(qv->parbuf);
		qv->r= sys_copy(who, SEG_D, (phys_bytes) qv->qpar.param,
			MM_PROC_NR, SEG_D, (phys_bytes) qv->parbuf,
			(phys_bytes) n);
		if (qv->r != OK) return 0;
		qv->qpar.param+= n;
		qv->param= qv->parbuf;
	}
	qv->qpar.psize--;
	return (u8_t) *qv->param++;
}


/*===========================================================================*
 *				mq_flush				     *
 *===========================================================================*/
PRIVATE void mq_flush()
{
	/* Send gathered characters back to the user. */
	struct queryvars *qv= qvars;
	size_t n;

	if ((n= qv->value - qv->valbuf) > qv->qpar.vsize) n= qv->qpar.vsize;
	if (n > 0 && qv->r == OK) {
		/* Copy the value buffer to user space. */
		qv->r= sys_copy(MM_PROC_NR, SEG_D, (phys_bytes) qv->valbuf,
			who, SEG_D, (phys_bytes) qv->qpar.value,
			(phys_bytes) n);
		qv->qpar.value+= n;
		qv->qpar.vsize-= n;
	}
	qv->value= qv->valbuf;
	qv->vcount= sizeof(qv->valbuf);
}


#if !NDEBUG
/*=========================================================================*
 *				bad_assertion				   *
 *=========================================================================*/
PUBLIC void bad_assertion(file, line, what)
char *file;
int line;
char *what;
{
  printf("mm panic at %s(%d): assertion \"%s\" failed\n", file, line, what);
  panic(NULL, NO_NUM);
}


/*=========================================================================*
 *				bad_compare				   *
 *=========================================================================*/
PUBLIC void bad_compare(file, line, lhs, what, rhs)
char *file;
int line;
int lhs;
char *what;
int rhs;
{
  printf("mm panic at %s(%d): compare (%d) %s (%d) failed\n",
	file, line, lhs, what, rhs);
  panic(NULL, NO_NUM);
}
#endif /* !NDEBUG */

/*
 * $PchId: misc.c,v 1.4 1995/11/28 07:13:30 philip Exp $
 */
