/* This file contains the main program of the memory manager and some related
 * procedures.  When MINIX starts up, the kernel runs for a little while,
 * initializing itself and its tasks, and then it runs MM and FS.  Both MM
 * and FS initialize themselves as far as they can.  MM then makes a call to
 * FS, because MM has to wait for FS to acquire a RAM disk.  MM asks the
 * kernel for all free memory and starts serving requests.
 *
 * The entry points into this file are:
 *   main:	starts MM running
 *   reply:	reply to a process making an MM system call
 */

#include "mm.h"
#include <minix/callnr.h>
#include <minix/com.h>
#include <minix/queryparam.h>
#include <signal.h>
#include <stddef.h>
#include "mproc.h"
#include "param.h"

/* Make these variables available to inquisitive utilities. */
PRIVATE struct export_param_list mm_ex_param_list[] = {
  QP_VARIABLE(mproc),
  QP_ARRAY(mproc),
  QP_FIELD(mp_pid, struct mproc),
  QP_FIELD(mp_procgrp, struct mproc),
  QP_FIELD(mp_parent, struct mproc),
  QP_FIELD(mp_realuid, struct mproc),
  QP_FIELD(mp_effuid, struct mproc),
  QP_FIELD(mp_realgid, struct mproc),
  QP_FIELD(mp_effgid, struct mproc),
  QP_FIELD(mp_flags, struct mproc),
  QP_FIELD(mp_next, struct mproc),
  QP_VARIABLE(mproc_inuse),
  QP_VARIABLE(mproc_syslist),
  QP_VARIABLE(mproc_deadlist),
  QP_END()
};
PRIVATE struct export_params mm_ex_params = { mm_ex_param_list };

FORWARD _PROTOTYPE( void get_work, (void)				);
FORWARD _PROTOTYPE( void mm_init, (void)				);

/*===========================================================================*
 *				main					     *
 *===========================================================================*/
PUBLIC void main()
{
/* Main routine of the memory manager. */

  int error;

  mm_init();			/* initialize memory manager tables */

  qp_export(&mm_ex_params);

  /* This is MM's main loop-  get work and do it, forever and forever. */
  while (TRUE) {
	/* Wait for message. */
	get_work();		/* wait for an MM system call */
	mp = &mproc[who];

  	/* Set some flags. */
	error = OK;
	dont_reply = FALSE;
	err_code = -999;

	/* If the call number is valid, perform the call. */
	if (mm_call < 0 || mm_call >= NCALLS)
		error = no_sys();
	else
		error = (*scall_table[mm_call])();

	/* Send the results back to the user to indicate completion. */
	if (dont_reply) continue;	/* no reply for EXIT and WAIT */
	if (mm_call == EXEC && error == OK) continue;
	reply(who, error, result2, res_ptr);
  }
}


/*===========================================================================*
 *				get_work				     *
 *===========================================================================*/
PRIVATE void get_work()
{
/* Wait for the next message and extract useful information from it. */

  if (receive(ANY, &mm_in) != OK) panic("MM receive error", NO_NUM);
  who = mm_in.m_source;		/* who sent the message */
  mm_call = mm_in.m_type;	/* system call number */

}


/*===========================================================================*
 *				reply					     *
 *===========================================================================*/
PUBLIC void reply(proc_nr, result, res2, respt)
int proc_nr;			/* process to reply to */
int result;			/* result of the call (usually OK or error #)*/
int res2;			/* secondary result */
char *respt;			/* result if pointer */
{
/* Send a reply to a user process. */

  register struct mproc *proc_ptr;

  proc_ptr = &mproc[proc_nr];
  /* 
   * To make MM robust, check to see if destination is still alive.  This
   * validy check must be skipped if the caller is a task.
   */
  if ((who >=0) && ((proc_ptr->mp_flags&IN_USE) == 0 || 
	(proc_ptr->mp_flags&HANGING))) return;

  reply_type = result;
  reply_i1 = res2;
  reply_p1 = respt;
  if (send(proc_nr, &mm_out) != OK) panic("MM can't reply", NO_NUM);
}


/*===========================================================================*
 *				mm_init					     *
 *===========================================================================*/
PRIVATE void mm_init()
{
/* Initialize the memory manager. */
  static char core_sigs[] = {
	SIGQUIT, SIGILL, SIGTRAP, SIGABRT,
	SIGEMT, SIGFPE, SIGUSR1, SIGSEGV,
	SIGUSR2, 0 };
  static char ignore_sigs[] = { SIGCHLD, SIGWINCH, 0 };
  register struct mproc *rmp, **pmp_list;
  register char *sig_ptr;
  int proc_nr, r;
  extern int revision;
  message mess;
  struct mem_map dummy_map[NR_SEGS];

  /* Build the set of signals which cause core dumps.
   * (core_bits is now misnamed.  DEBUG.)
   */
  sigemptyset(&core_bits);
  for (sig_ptr = core_sigs; *sig_ptr != 0; sig_ptr++)
	sigaddset(&core_bits, *sig_ptr);
  sigemptyset(&default_ignore);
  for (sig_ptr = ignore_sigs; *sig_ptr != 0; sig_ptr++)
	sigaddset(&default_ignore, *sig_ptr);

  /* Initialize MM's tables. */
  mproc_freelist= NULL;
  mproc_deadlist= NULL;
  mproc_syslist= NULL;
  mproc_inuse= NULL;
  for (proc_nr = 0; proc_nr < NR_PROCS; proc_nr++)
  {
  	rmp= &mproc[proc_nr];
  	if (sys_getmap(proc_nr, dummy_map) == OK)
  	{
		rmp->mp_flags |= IN_USE;
		if (proc_nr >= INIT_PROC_NR)
		{
			rmp->mp_pid = (proc_nr - INIT_PROC_NR) + INIT_PID;
			rmp->mp_procgrp = INIT_PID;
			rmp->mp_parent = INIT_PROC_NR;
			sigemptyset(&rmp->mp_ignore);
			sigemptyset(&rmp->mp_catch);
		}
		pmp_list= &mproc_syslist;
	}
	else
	{
		pmp_list= &mproc_freelist;
	}
  	rmp->mp_next= *pmp_list;
  	if (rmp->mp_next)
  		rmp->mp_next->mp_prev= rmp;
  	rmp->mp_prev= NULL;
  	*pmp_list= rmp;
  }
  mproc_exec= mproc_freelist;
  mproc_freelist= mproc_freelist->mp_next;
  mproc_freelist->mp_prev= NULL;

  /* Find two tasks we need to communicate with. */
  r= sys_findproc(CLOCK_NAME, &clck_tasknr, 0);
  if (r != OK)
	panic("unable to find clock task:", r);
  r= sys_findproc(SYN_AL_NAME, &synal_tasknr, 0);
  if (r != OK)
	panic("unable to find synchronous alarm task:", r);

  /* Send FS a message telling it that we are ready to go "on-line".  Add the
   * revision number as a bonus.  Continue when FS replies.
   */
  mess.m1_i1= revision;
  if (sendrec(FS_PROC_NR, &mess) != OK)
	panic("MM can't sync up with FS", NO_NUM);

  mem_init();		/* initialize tables to all physical mem */
}


/*===========================================================================*
 *				get_mem					     *
 *===========================================================================*/
PUBLIC int get_mem(base, size)
phys_clicks *base, *size;
{
/* Ask kernel for the next chunk of memory.  Return true iff the chunk is OK. */
  mm_out.m_type = SYS_MEM;
  if (sendrec(SYSTASK, &mm_out) != OK)
	panic("kernel didn't respond to get_mem", NO_NUM);
  if (mm_out.m1_i2 == 0) return(0);
  *base = mm_out.m1_i1;
  *size = mm_out.m1_i2;
  return(1);
}

/*
 * $PchId: main.c,v 1.6 1995/12/22 07:58:12 philip Exp $
 */
