/* This file handles the 4 system calls that get and set uids and gids.
 * It also handles getpid().  The code for each one is so tiny that it hardly
 * seemed worthwhile to make each a separate function.
 */
 /* Further more the code for setsid() is located here. */

#include "mm.h"
#include <minix/callnr.h>
#include <signal.h>
#include "mproc.h"
#include "param.h"
#include "assert.h"
INIT_ASSERT

/*===========================================================================*
 *				do_getset				     *
 *===========================================================================*/
PUBLIC int do_getset()
{
/* Handle GETUID, GETGID, GETPID, SETUID, SETGID.  The three GETs return
 * their primary results in 'r'.  GETUID and GETGID also return secondary
 * results (the effective IDs) in 'result2', which is returned to the user.
 */

  register struct mproc *rmp = mp;
  register int r;

  assert(mm_call == GETPID || mm_call == SETUID || mm_call == GETUID || 
  	mm_call == SETGID || mm_call == GETGID || mm_call == GSGROUPS);

  switch(mm_call) {
	case GETUID:
		r = rmp->mp_realuid;
		result2 = rmp->mp_effuid;
		break;

	case GETGID:
		r = rmp->mp_realgid;
		result2 = rmp->mp_effgid;
		break;

	case GETPID:
		r = mproc[who].mp_pid;
		result2 = mproc[rmp->mp_parent].mp_pid;
		break;

	case SETUID:
		if (rmp->mp_saveuid[0] != SUPER_USER
			&& rmp->mp_saveuid[1] != SUPER_USER
			&& rmp->mp_saveuid[0] != gsuid_id
			&& rmp->mp_saveuid[1] != gsuid_id) return(EPERM);
		rmp->mp_realuid = gsuid_id;
		rmp->mp_effuid = gsuid_id;
		tell_fs(SETUID, who, gsuid_id, gsuid_id);
		r = OK;
		break;

	case SETGID:
		for (r = 0; r < rmp->mp_ngroups; r++) {
			if (rmp->mp_groups[r] == gsgid_id) break;
		}
		if (rmp->mp_saveuid[0] != SUPER_USER
			&& rmp->mp_saveuid[1] != SUPER_USER
			&& rmp->mp_savegid[0] != gsgid_id
			&& rmp->mp_savegid[1] != gsgid_id
			&& r == rmp->mp_ngroups) return(EPERM);
		rmp->mp_realgid = gsgid_id;
		rmp->mp_effgid = gsgid_id;
		tell_fs(SETGID, who, gsgid_id, gsgid_id);
		r = OK;
		break;

	case GSGROUPS:
		if (gsgrps_setgrp) {
			/* Set one supplementary group id. */
			if (rmp->mp_effuid != SUPER_USER) return(EPERM);
			if (gsgrps_index < -1 || gsgrps_index >= NGROUPS_MAX)
				return(EINVAL);
			if ((rmp->mp_ngroups= gsgrps_index+1) > 0)
				rmp->mp_groups[gsgrps_index]= gsgrps_group;
			tell_fs(GSGROUPS, who, gsgrps_index, gsgrps_group);
			r = OK;
		} else {
			/* Get one supplementary group id. */
			if ((unsigned) gsgrps_index >= rmp->mp_ngroups)
				return(ESRCH);
			r = rmp->mp_groups[gsgrps_index];
		}
		break;

	default:
		r = EINVAL;
		break;	
  }

  return(r);
}


/*===========================================================================*
 *				do_setsid				     *
 *===========================================================================*/
PUBLIC int do_setsid()
{
/* Handle SETSID. This call sets the process group of the calling process to 
 * its pid, and tells FS to release the controlling terninal for that process.
 */

  register struct mproc *rmp = mp;
  int proc_nr;

  assert(mm_call == SETSID);

  proc_nr = (int) (rmp - mproc);	/* get process slot number */

  if (rmp->mp_procgrp == rmp->mp_pid)
	return (EPERM);

  /* We assume that no process group with pid rmp->mp_pid exits allready. */
  rmp->mp_procgrp= rmp->mp_pid;

  /* Let's direct FS to release the controlling tty for this process. */
  tell_fs(SETSID, proc_nr, 0, 0);

  return(rmp->mp_procgrp);
}


/*===========================================================================*
 *				do_getpgrp				     *
 *===========================================================================*/
PUBLIC int do_getpgrp()
{
/* Handle GETPGRP. This call returns the process group id of the calling
 * process.
 */

  register struct mproc *rmp = mp;

  assert(mm_call == GETPGRP);

  return(rmp->mp_procgrp);
}

/*
 * $PchId: getset.c,v 1.4 1995/11/28 07:14:14 philip Exp $
 */
