/*	vdisk - virtual disk driver			Author: Kees J. Bot
 *								16 Dec 1995
 * The vdisk driver allows one to concatenate disk devices
 * to form one new disk.  (This driver is to be extended to offer
 * striping once the author finds time.  Not that striping is very
 * useful with Minix still being single threaded.)
 */
#define nil 0
#include "kernel.h"
#include "driver.h"
#if ENABLE_VDISK
#include <sys/ioctl.h>
#include <minix/vdisk.h>
#include "assert.h"
INIT_ASSERT

#define NR_VDISKS	 8		/* This many virtual disks */
#define NR_SUBDISKS	32		/* Of this many subdisks total */

typedef struct subdisk {
	struct subdisk	*next;
	u64_t		size;		/* Size of the subdisk in bytes */
	short		task_nr;	/* Subdisk = <task_nr, minor> */
	short		minor;
} subdisk_t;

typedef struct vdisk {
	unsigned	open_ct;
	struct device	geom;
	subdisk_t	*subdisks;
} vdisk_t;

static int vdisk_tasknr= ANY;
static subdisk_t subdisk[NR_SUBDISKS];	/* Subdisk administration */
static subdisk_t *v_sdfree;		/* List of free subdisks */
static vdisk_t vdisk[NR_VDISKS];	/* Vdisk administration */
static vdisk_t *v_vp;			/* Current disk struct */

_PROTOTYPE( static char *v_name, (void)					);
_PROTOTYPE( static int v_do_opcl, (struct driver *dp, message *m_ptr)	);
_PROTOTYPE( static int v_do_ioctl, (struct driver *dp, message *m_ptr)	);
_PROTOTYPE( static struct device *v_prepare, (int device)		);
_PROTOTYPE( static int v_transfer, (int proc_nr, int opcode,
			u64_t position, iovec_t *iov, unsigned nr_req)	);
_PROTOTYPE( static void v_geometry, (struct partition *entry)		);

static struct driver v_dtab= {
	v_name,		/* Current device's name */
	v_do_opcl,	/* Open or mount request, initialize device */
	v_do_opcl,	/* Release device */
	v_do_ioctl,	/* Attach to or disable virtual disks (&etc) */
	v_prepare,	/* Prepare for I/O on a given minor device */
	v_transfer,	/* Do the I/O */
	nop_cleanup,	/* Nothing to clean up */
	v_geometry,	/* Tell the geometry of the disk */
};

void vdisk_task(void)
{
	struct subdisk *sdp;

	vdisk_tasknr= proc_number(proc_ptr);

	/* Put all subdisks on the free list. */
	v_sdfree= nil;
	for (sdp= subdisk; sdp < subdisk + NR_SUBDISKS; sdp++) {
		sdp->next= v_sdfree;
		v_sdfree= sdp;
	}

	driver_task(&v_dtab);
}

static char *v_name()
/* Return the name of the current virtual disk. */
{
	static char name[]= "vd7";

	name[2]= '0' + (v_vp - vdisk);
	return name;
}

static struct device *v_prepare(int device)
/* Prepare for I/O on the virtual disk numbered 'device'. */
{
	if (device >= NR_VDISKS) return NIL_DEV;

	return &(v_vp= &vdisk[device])->geom;
}

static int v_do_opcl(struct driver *dp, message *m_ptr)
/* Open or close a virtual disk device. */
{
	if (v_prepare(m_ptr->DEVICE) == NIL_DEV) return ENXIO;

	v_vp->open_ct += m_ptr->m_type == DEV_OPEN ? +1 : -1;
	return OK;
}

static int v_do_ioctl(struct driver *dp, message *m_ptr)
/* Attach to or disable virtual disks (&etc) */
{
	struct vdisk_attach vdatt;
	phys_bytes vdatt_phys;
	int r, index;
	message m;
	vdisk_t *vp;
	subdisk_t **psdp, *sdp;
	struct partition sub_geom;

	if (m_ptr->REQUEST != DIOCVDATTACH) return do_diocntl(dp, m_ptr);

	if (v_prepare(m_ptr->DEVICE) == NIL_DEV) return ENXIO;
	vp= v_vp;

	if (vp->open_ct != 1) return EBUSY;

	if ((vdatt_phys= numap(m_ptr->PROC_NR, (vir_bytes) m_ptr->ADDRESS,
				(vir_bytes) sizeof(vdatt))) == 0) return EFAULT;
	phys_copy(vdatt_phys, vir2phys(&vdatt), (phys_bytes) sizeof(vdatt));

	if ((index= vdatt.index) == 0) {
		/* Reinitializing the subdisk list; close the old devices. */
		while ((sdp= vp->subdisks) != nil) {
			vp->subdisks= sdp->next;
			sdp->next= v_sdfree;
			v_sdfree= sdp;
			m.m_type= DEV_CLOSE;
			m.DEVICE= sdp->minor;
			m.PROC_NR= vdisk_tasknr;
			if (sdp->task_nr == vdisk_tasknr) {
				(void) v_do_opcl(dp, &m);
			} else {
				r= sendrec(sdp->task_nr, &m);
				assert(r == OK);
			}
			/* ignore status */
		}
		vp->geom.dv_size= cvu64(0);
	}

	/* The index must indicate the next subdisk. */
	psdp= &vp->subdisks;
	while (index > 0 && *psdp != nil) {
		index--;
		psdp= &(*psdp)->next;
	}
	if (index != 0 && *psdp != nil) return EINVAL;
	if ((sdp= v_sdfree) == nil) return ENFILE;

	/* Open the new subdisk. */
	m.m_type= DEV_OPEN;
	m.DEVICE= vdatt.minor;
	m.PROC_NR= vdisk_tasknr;
	m.COUNT= R_BIT|W_BIT;
	if (vdatt.task_nr == vdisk_tasknr) {
		if (vdatt.minor == (vp - vdisk)) return EBUSY;
		r= v_do_opcl(dp, &m);
	} else {
		r= sendrec(vdatt.task_nr, &m);
		assert(r == OK);
		r= m.REP_STATUS;
	}
	if (r != OK) return EIO;

	/* Query the size of the subdisk. */
	m.m_type= DEV_IOCTL;
	m.DEVICE= vdatt.minor;
	m.PROC_NR= vdisk_tasknr;
	m.REQUEST= DIOCGETP;
	m.ADDRESS= (char *) &sub_geom;
	if (vdatt.task_nr == vdisk_tasknr) {
		r= do_diocntl(dp, &m);
	} else {
		r= sendrec(vdatt.task_nr, &m);
		assert(r == OK);
		r= m.REP_STATUS;
	}
	if (r != OK) {
		/* Query failed, close and get out. */
		m.m_type= DEV_CLOSE;
		m.DEVICE= vdatt.minor;
		m.PROC_NR= vdisk_tasknr;
		if (vdatt.task_nr == vdisk_tasknr) {
			(void) v_do_opcl(dp, &m);
		} else {
			r= sendrec(vdatt.task_nr, &m);
			assert(r == OK);
		}
		return EIO;
	}

	/* Attach a new subdisk. */
	sdp->size= sub_geom.size;
	sdp->task_nr= vdatt.task_nr;
	sdp->minor= vdatt.minor;
	v_sdfree= sdp->next;
	sdp->next= *psdp;
	*psdp= sdp;
	vp->geom.dv_size= add64(vp->geom.dv_size, sub_geom.size);
	return OK;
}

static int v_transfer(int proc_nr, int opcode, u64_t position, iovec_t *iov,
								unsigned nr_req)
{
	vdisk_t *vp= v_vp;
	subdisk_t *sdp= vp->subdisks;
	vir_bytes size;
	message m;
	int r;

	for (;;) {
		/* Which of the subdisks? */
		for (;;) {
			if (sdp == nil) return OK;		/* EOF */
			if (cmp64(position, sdp->size) < 0) break;
			position= sub64(position, sdp->size);
			sdp= sdp->next;
		}

		/* Remember how many bytes iov[0] has. */
		size = iov[0].iov_size;

		/* Let the subdisk driver work on the vector.  We are happy
		 * if it handles at least the first element.
		 */
		if (sdp->task_nr == vdisk_tasknr) {
			/* Recursion: see "recursion". */
			v_vp= &vdisk[sdp->minor];
			r= v_transfer(proc_nr, opcode, position, iov, nr_req);
		} else {
			m.m_type= opcode;
			m.DEVICE= sdp->minor;
			m.PROC_NR= proc_nr;
			m.POSITION= ex64lo(position);
			m.HIGHPOS= ex64hi(position);
			m.ADDRESS= (char *) iov;
			m.COUNT= nr_req;
			r= sendrec(sdp->task_nr, &m);
			assert(r == OK);
			r= m.REP_STATUS;
		}
		if (iov[0].iov_size == 0 || r != OK) return r;

		/* We must have hit EOF on this subdisk.  Update position
		 * and continue with the next subdisk.
		 */
		if (iov[0].iov_size == size) return EIO;     /* early EOF??? */
		position= add64u(position, size - iov[0].iov_size);
	}
}

static void v_geometry(struct partition *entry)
/* There are no partitions, so it doesn't matter what lies we lie. */
{
	entry->cylinders= entry->heads= entry->sectors= 1;
}

#endif /* ENABLE_VDISK */
