/*
kernel/tty.c

Created:	Feb 27, 1992 by Philip Homburg <philip@cs.vu.nl>
*/

/*
 * Requests:
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_MODE
 * -------------------------------------------------------------
 * | DEV_OPEN    |minor dev  | proc nr   |  fd       |   mode   |
 * |-------------+-----------+-----------+-----------+----------+
 * | DEV_CLOSE   |minor dev  | proc nr   |  fd       |          |
 * |-------------+-----------+-----------+-----------+----------+
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_COUNT NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_READ    |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_WRITE   |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_IOCTL NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_IOCTL3  |minor dev  | proc nr   |  fd       |  command  | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_OPERATION
 * -------------------------------------------------------------------|
 * | DEV_CANCEL  |minor dev  | proc nr   |  fd       | which operation|
 * |-------------+-----------+-----------+-----------+----------------|
 *
 * Replies:
 *
 *    m_type        REP_PROC_NR   REP_STATUS   REP_REF    REP_OPERATION
 * ----------------------------------------------------------------------|
 * | DEVICE_REPLY |   proc nr   |  status    |  fd     | which operation |
 * |--------------+-------------+------------+---------+-----------------|
 */

#include "kernel.h"

#include <sys/ioctl.h>
#include <fcntl.h>
#include <signal.h>
#include <termios.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include "assert.h"
INIT_ASSERT
#include "mq.h"
#include "timer.h"
#include "tty.h"

/* Ioctl type encoding. */
#define IOCPARM_MASK	_IOCPARM_MASK
#define IOC_VOID	_IOC_VOID
#define IOCTYPE_MASK	_IOCTYPE_MASK
#define IOC_IN		_IOC_IN
#define IOC_OUT		_IOC_OUT
#define IOC_INOUT	_IOC_INOUT

PRIVATE u8_t buffer[256];	/* Used for transfers from user space,
				 * eg. writes */
PRIVATE int char_size_table[256];	/* Sizes of characters. */
PRIVATE struct termios default_termios;
PRIVATE struct winsize default_winsize;

#define is_canon(t) ((t)->t_termios.c_lflag & ICANON)

/* sgtty.h can't be included here. Fortunately, a size mismatch results
 * in ENOTTY.
 */
struct sgttyb
{
	char sg_ispeed;
	char sg_ospeed;
	char sg_erase;
	char sg_kill;
	int  sg_flags;
};

struct tchars
{
	char t_intrc;
	char t_quitc;
	char t_startc;
	char t_stopc;
	char t_eofc;
	char t_brkc;
};

FORWARD _PROTOTYPE( int do_open, (tty_t *pty, message *m )		);
FORWARD _PROTOTYPE( int do_close, (tty_t *pty, message *m )		);
FORWARD _PROTOTYPE( int do_read, (tty_t *pty, mq_t *mq )		);
FORWARD _PROTOTYPE( int do_write, (tty_t *pty, mq_t *mq )		);
FORWARD _PROTOTYPE( int do_ioctl, (tty_t *pty, mq_t *mq )		);
FORWARD _PROTOTYPE( void do_ioctl_compat, (tty_t *pty, message *m )	);
FORWARD _PROTOTYPE( int compat_getp, (tty_t *tty, struct sgttyb *sg)	);
FORWARD _PROTOTYPE( int compat_getc, (tty_t *tty, struct tchars *sg)	);
FORWARD _PROTOTYPE( int compat_setp, (tty_t *tty, struct sgttyb *sg)	);
FORWARD _PROTOTYPE( int compat_setc, (tty_t *tty, struct tchars *sg)	);
FORWARD _PROTOTYPE( int tspd2sgspd, (speed_t tspd)			);
FORWARD _PROTOTYPE( speed_t sgspd2tspd, (int sgspd)			);
FORWARD _PROTOTYPE( void setattr, (tty_t *tty, struct termios *termios)	);
FORWARD _PROTOTYPE( int do_cancel, (tty_t *pty, message *m)		);
FORWARD _PROTOTYPE( void read_reply, (tty_t *tty, int result)		);
FORWARD _PROTOTYPE( void write_reply, (tty_t *tty, int result)		);
FORWARD _PROTOTYPE( void reply, (tty_t *tty, mq_t *mq, int result,
						int can_enqueue)	);
FORWARD _PROTOTYPE( void setup_read, (tty_t *tty)			);
FORWARD _PROTOTYPE( void setup_write, (tty_t *tty)			);
FORWARD _PROTOTYPE( void restart_read_canon, (tty_t *tty)		);
FORWARD _PROTOTYPE( void restart_read_noncanon, (tty_t *tty)		);
FORWARD _PROTOTYPE( void restart_write, (tty_t *tty)			);
FORWARD _PROTOTYPE( void restart_ioctl, (tty_t *tty)			);
FORWARD _PROTOTYPE( void output, (tty_t *tty, int chr)			);
FORWARD _PROTOTYPE( void echo, (tty_t *tty, int chr)			);
FORWARD _PROTOTYPE( void echonl, (tty_t *tty)				);
FORWARD _PROTOTYPE( void echo_eof, (tty_t *tty)				);
FORWARD _PROTOTYPE( int input_canon_zero_space, (tty_t *tty, int chr)	);
FORWARD _PROTOTYPE( int input_canon, (tty_t *tty, int *chr_p)		);
FORWARD _PROTOTYPE( int output_pos, (tty_t *tty, int pos, int chr)	);
FORWARD _PROTOTYPE( void do_reprint, (tty_t *tty)			);
FORWARD _PROTOTYPE( void do_sig, (tty_t *tty, int sig)			);
FORWARD _PROTOTYPE( void def_get, (int dev_ref)				);
FORWARD _PROTOTYPE( int def_put, (int dev_ref, char *buf, int siz)	);
FORWARD _PROTOTYPE( void def_hup, (int dev_ref)				);
FORWARD _PROTOTYPE( void def_reply, (int dev_ref, mq_t *mq)		);
FORWARD _PROTOTYPE( void def_setattr, (int dev_ref, 
						struct termios *tmiop)	);
FORWARD _PROTOTYPE( void def_break, (int dev_ref, clock_t dur)		);
FORWARD _PROTOTYPE( void set_rd_timer, (tty_t *tty)			);
FORWARD _PROTOTYPE( void rd_timer_to, (struct tmrs *tp, tmr_arg_ut arg)	);
FORWARD _PROTOTYPE( int put_canon, (tty_t *tty, char *buf, int siz)	);
FORWARD _PROTOTYPE( int put_noncanon, (tty_t *tty, char *buf, int siz)	);
FORWARD _PROTOTYPE( int put_timed_raw, (tty_t *tty, char *buf, int siz)	);
FORWARD _PROTOTYPE( int put_raw, (tty_t *tty, char *buf, int siz)	);
FORWARD _PROTOTYPE( int put_discard, (tty_t *tty, char *buf, int siz)	);
FORWARD _PROTOTYPE( void ib_flush, (tty_t *tty)				);
FORWARD _PROTOTYPE( int ib_compl_ln, (tty_t *tty)			);
FORWARD _PROTOTYPE( int ib_get_head, (tty_t *tty, u8_t **cp1,
					int *sz1, u8_t **cp2, int *sz2)	);
FORWARD _PROTOTYPE( int ib_get_tail, (tty_t *tty, u8_t **cp1,
					int *sz1, u8_t **cp2, int *sz2)	);
FORWARD _PROTOTYPE( int ib_get_space, (tty_t *tty, u8_t **cp,
							int *linsz)	);
FORWARD _PROTOTYPE( int ib_delete, (tty_t *tty, int size)		);
FORWARD _PROTOTYPE( int ib_grow, (tty_t *tty, int size)			);
FORWARD _PROTOTYPE( int ib_shrink, (tty_t *tty, int size)		);
FORWARD _PROTOTYPE( int ib_newline, (tty_t *tty)			);
FORWARD _PROTOTYPE( void ib_pack, (tty_t *tty)				);
FORWARD _PROTOTYPE( int ib_check, (tty_t *tty)				);
FORWARD _PROTOTYPE( void ib_print, (tty_t *tty)				);
FORWARD _PROTOTYPE( int ob_empty, (tty_t *tty)				);
FORWARD _PROTOTYPE( void ob_flush, (tty_t *tty)				);
FORWARD _PROTOTYPE( int ob_space, (tty_t *tty, u8_t **cp1, int *sz1,
						u8_t **cp2, int *sz2)	);
FORWARD _PROTOTYPE( int ob_get_head, (tty_t *tty, u8_t **cp)		);
FORWARD _PROTOTYPE( void ob_grow, (tty_t *tty, int n)			);
FORWARD _PROTOTYPE( void ob_delete, (tty_t *tty, int n)			);

/*===========================================================================*
 *				tty_receive				     *
 *===========================================================================*/
PUBLIC void tty_receive(tty, mq)
struct tty *tty;
struct mq *mq;
{
	int result, send_reply, free_mess;

	switch (mq->mq_mess.m_type)
	{
	case DEV_OPEN:
		result= do_open(tty, &mq->mq_mess);
		send_reply= 1;
		free_mess= 1;
		break;
	case DEV_CLOSE:
		result= do_close(tty, &mq->mq_mess);
		assert(result == 0 || result == 1);
		send_reply= 1;
		free_mess= 1;
		break;
	case DEV_READ:
		result= do_read(tty, mq);
		assert(result == OK || result == SUSPEND);
		send_reply= (result == SUSPEND);
		free_mess= 0;
		break;
	case DEV_WRITE:
		result= do_write(tty, mq);
		assert(result == OK || result == SUSPEND);
		send_reply= (result == SUSPEND);
		free_mess= 0;
		break;
	case DEV_IOCTL3:
		result= do_ioctl(tty, mq);
		assert(result == OK || result == SUSPEND);
		send_reply= (result == SUSPEND);
		free_mess= 0;
		break;
	case DEV_CANCEL:
		result= do_cancel(tty, &mq->mq_mess);
		send_reply= 1;
		free_mess= 1;
		mq->mq_mess.m_type= mq->mq_mess.NDEV_OPERATION;
		break;
	case DEV_IOCTL:
		do_ioctl_compat(tty, &mq->mq_mess);
		send_reply= 0;
		free_mess= 1;
		break;
	default:
		panic("got unknown message, type= ", 
			mq->mq_mess.m_type);
	}
	if (send_reply)
	{
		reply(tty, mq, result, FALSE);
	}
	if (free_mess)
		mq_free(mq);
}


/*===========================================================================*
 *				do_open					     *
 *===========================================================================*/
PRIVATE int do_open(tty, m)
tty_t *tty;
message *m;
{
	int r, ctl;

	if (tty->t_opencnt == 0)
	{
		tty->t_flags &= ~(TF_DEV_CLOSED|TF_TTY_CLOSED);
		tty->t_flags |= TF_READ_INITIATIVE | TF_WRITE_INITIATIVE;
		tty->t_rd_lnext= 0;
		tty->t_putf= put_canon;
		tty->t_termios.c_lflag |=
			(TLOCAL_DEF & (ECHO|ECHOE|ECHOK|ECHONL));
	}

	/* If caller meets the following conditions, it becomes 
	 * controlling tty.
	 */
	r= TRUE;
	if (tty->t_sesldr != 0) r = FALSE;	/* we already got a ctl tty */
	if (m->NDEV_MODE & O_NOCTTY) r = FALSE;	/* O_NOCTTY set */

	ctl= 0;
	if (r)
	{
		tty->t_sesldr = m->NDEV_PROC;
		ctl= 1;
	}
	tty->t_opencnt++;
	return ctl;
}


/*===========================================================================*
 *				do_read					     *
 *===========================================================================*/
PRIVATE int do_read(tty, mq)
tty_t *tty;
mq_t *mq;
{
	assert(!(tty->t_flags & TF_TTY_CLOSED));
	if (tty->t_flags & TF_DEV_CLOSED)
	{
		reply(tty, mq, 0, TRUE);
		return OK;
	}
	if (tty->t_rd_queue)
	{
		mq->mq_next= 0;
		tty->t_rd_queue_tail->mq_next= mq;
		tty->t_rd_queue_tail= mq;
		assert (!(tty->t_flags & TF_READ_INITIATIVE));
		return SUSPEND;
	}
	mq->mq_next= 0;
	tty->t_rd_queue= mq;
	tty->t_rd_queue_tail= mq;
	setup_read(tty);
	if (tty->t_rd_queue == NULL)
	{
		/* umap failed, setup_read already replied */
		return OK;		
	}

	if (is_canon(tty))
		restart_read_canon(tty);
	else
	{
		restart_read_noncanon(tty);
		if (tty->t_termios.c_cc[VTIME] != 0)
			set_rd_timer(tty);
	}

	if (tty->t_flags & TF_READ_INITIATIVE)
	{
		tty->t_flags &= ~TF_READ_INITIATIVE;
		(*tty->t_get)(tty->t_line);
	}	
	if (tty->t_rd_queue)
		return SUSPEND;
	return OK;
}


/*===========================================================================*
 *				do_write				     *
 *===========================================================================*/
PRIVATE int do_write(tty, mq)
tty_t *tty;
mq_t *mq;
{
	assert(!(tty->t_flags & TF_TTY_CLOSED));
	if (tty->t_flags & TF_DEV_CLOSED)
	{
		reply(tty, mq, 0, TRUE);
		return OK;
	}
	if (tty->t_wr_queue)
	{
		tty->t_wr_queue_tail->mq_next= mq;
		tty->t_wr_queue_tail= mq;
		assert (!(tty->t_flags & TF_WRITE_INITIATIVE));
		return SUSPEND;
	}
	mq->mq_next= 0;
	tty->t_wr_queue= mq;
	tty->t_wr_queue_tail= mq;
	setup_write(tty);
	if (tty->t_wr_queue == NULL)
	{
		/* umap failed, setup_write already replied */
		return OK;		
	}
	if (tty->t_flags & TF_WRITE_INITIATIVE)
	{
		tty_get(tty);	/* This function is normally called */
				/* by the device to get some data. */
	}
	else
		restart_write(tty);	/* Buffer the output. */
	if (tty->t_wr_queue)
		return SUSPEND;
	return OK;
}


/*===========================================================================*
 *				do_ioctl				     *
 *===========================================================================*/
PRIVATE int do_ioctl(tty, mq)
tty_t *tty;
mq_t *mq;
{
	phys_bytes user_phys, local_phys;
	u32_t cmd;
	size_t size;
	clock_t dur;
	union
	{
		struct termios termios;
		struct sgttyb sg;
		struct tchars tc;
		struct winsize winsize;
		int i;
	} param;
	int c, r;

	cmd= mq->mq_mess.NDEV_IOCTL;

	if (cmd & (IOC_IN|IOC_OUT))
	{
		size= (cmd >> 16) & IOCPARM_MASK;
		user_phys= numap(mq->mq_mess.NDEV_PROC, 
			(vir_bytes)mq->mq_mess.NDEV_BUFFER, size);
		if (user_phys == 0)
		{
			reply(tty, mq, EFAULT, TRUE);
			return OK;
		}
		local_phys= vir2phys(&param);
		if (size > sizeof(param)) size= sizeof(param);
	}
	if (cmd & IOC_IN)
	{
		/* Copy in */
		phys_copy(user_phys, local_phys, size);
	}

	r= OK;
	switch(cmd)
	{
	case TCGETS:
		param.termios= tty->t_termios;
		break;
	case TCSETSW:
	case TCSETSF:
		mq->mq_next= NULL;
		if (tty->t_ioc_queue != NULL)
		{
			tty->t_ioc_queue_tail->mq_next= mq;
			tty->t_ioc_queue_tail= mq;
			return SUSPEND;
		}
		if (!ob_empty(tty))
		{
			tty->t_flags |= TF_DRAIN;
			tty->t_ioc_queue= mq;
			tty->t_ioc_queue_tail= mq;
			return SUSPEND;
		}
		if (mq->mq_mess.NDEV_IOCTL == TCSETSF)
			ib_flush(tty);
		/* Fall through */
	case TCSETS:
		setattr(tty, &param.termios);
		break;
	case TCDRAIN:
		mq->mq_next= NULL;
		if (tty->t_ioc_queue != NULL)
		{
			tty->t_ioc_queue_tail->mq_next= mq;
			tty->t_ioc_queue_tail= mq;
			return SUSPEND;
		}
		if (ob_empty(tty))
		{
			reply(tty, mq, OK, TRUE);
			return OK;
		}
		tty->t_flags |= TF_DRAIN;
		tty->t_ioc_queue= mq;
		tty->t_ioc_queue_tail= mq;
		return SUSPEND;
	case TCFLSH:
		switch(param.i)
		{
		case TCIFLUSH:	ib_flush(tty); break;
		case TCOFLUSH:	ob_flush(tty); break;
		case TCIOFLUSH: ib_flush(tty); ob_flush(tty); break;
		default:	r= EINVAL; break;
		}
		break;
	case TCFLOW:
		switch(param.i)
		{
		case TCOOFF:
			tty->t_flags |= TF_XON_OUTPUT_OFF;
			break;
		case TCOON:
			tty->t_flags &= ~TF_XON_OUTPUT_OFF;
			tty_get(tty);
			break;
		case TCIOFF:
			c= tty->t_termios.c_cc[VSTOP];
			if (c == _POSIX_VDISABLE)
				c= '\23'; /* ^S */
			output(tty, c);
			break;
		case TCION:
			c= tty->t_termios.c_cc[VSTART];
			if (c == _POSIX_VDISABLE)
				c= '\21'; /* ^Q */
			output(tty, c);
			break;
		default:
			r= EINVAL;
			break;
		}
		break;
	case TCSBRK:
		if (param.i == 0)
			param.i= 4;
		dur= param.i*HZ/10;
		(*tty->t_break)(tty->t_line, dur);
		break;
	case TIOCGETP:
		compat_getp(tty, &param.sg);
		break;
	case TIOCSETP:
		compat_setp(tty, &param.sg);
		break;
	case TIOCGETC:
		compat_getc(tty, &param.tc);
		break;
	case TIOCSETC:
		compat_setc(tty, &param.tc);
		break;
	case TIOCGWINSZ:
		param.winsize= tty->t_winsize;
		break;
	case TIOCSWINSZ:
		tty->t_winsize= param.winsize;
		if (tty->t_sesldr)
			cause_sig(tty->t_sesldr, SIGWINCH);
		break;
	default:
		reply(tty, mq, ENOTTY, TRUE);
		return OK;
	}

	if (cmd & IOC_OUT)
	{
		/* Copy out */
		phys_copy(local_phys, user_phys, size);
	}
	reply(tty, mq, r, TRUE);
	return OK;
}


/*===========================================================================*
 *				do_ioctl_compat				     *
 *===========================================================================*/
PRIVATE void do_ioctl_compat(tty, m)
tty_t *tty;
message *m;
{
	int minor, proc, func, result, status;
	long flags, erki, spek;
	u8_t erase, kill, intr, quit, xon, xoff, brk, eof, ispeed, ospeed;
	message reply_mess;
	struct sgttyb sg;
	struct tchars tc;

#if DEBUG
if (debug) { printW(); printf("in do_ioctl_compat\n"); }
#endif

	minor= m->m2_i1;	/* TTY_LINE */
	proc= m->m2_i2;		/* PROC_NR */
	func= m->m2_i3;		/* COUNT */
	spek= m->m2_l1;
	flags= m->m2_l2;

	switch(func)
	{
	case (('t'<<8) | 8):	/* TIOCGETP */
		status= compat_getp(tty, &sg);
		erase= sg.sg_erase;
		kill= sg.sg_kill;
		ispeed= sg.sg_ispeed;
		ospeed= sg.sg_ospeed;
		flags= sg.sg_flags;

		if (status != OK)
			break;
		erki= ((long)ospeed << 24) |
			((long)ispeed << 16) | (erase << 8) | kill;
		break;
	case (('t'<<8) | 18):	/* TIOCGETC */
		status= compat_getc(tty, &tc);
		intr= tc.t_intrc;
		quit= tc.t_quitc;
		xon= tc.t_startc;
		xoff= tc.t_stopc;
		brk= tc.t_brkc;
		eof= tc.t_eofc;

		if (status != OK)
			break;
		erki= ((long)intr << 24) | ((long)quit << 16) |
							(xon << 8) | xoff;
		flags= (eof << 8) | brk;
		break;
	case (('t'<<8) | 17):	/* TIOCSETC */
		tc.t_stopc= (spek >> 0) & 0xff;
		tc.t_startc= (spek >> 8) & 0xff;
		tc.t_quitc= (spek >> 16) & 0xff;
		tc.t_intrc= (spek >> 24) & 0xff;
		tc.t_brkc= (flags >> 0) & 0xff;
		tc.t_eofc= (flags >> 8) & 0xff;
		status= compat_setc(tty, &tc);
		break;
	case (('t'<<8) | 9):	/* TIOCSETP */
		sg.sg_erase= (spek >> 8) & 0xff;
		sg.sg_kill= (spek >> 0) & 0xff;
		sg.sg_ispeed= (spek >> 16) & 0xff;
		sg.sg_ospeed= (spek >> 24) & 0xff;
		sg.sg_flags= flags;
		status= compat_setp(tty, &sg);
		break;
	case (('t'<<8) | 16):	/* TIOCFLUSH */
	printf("got a TIOCFLUSH\n");
	default:
		status= ENOTTY;
	}
	reply_mess.m_type= TASK_REPLY;
	reply_mess.REP_PROC_NR= proc;	/* REP_PROC_NR */
	reply_mess.REP_STATUS= status;	/* REP_STATUS */
	reply_mess.m2_l1= erki;	/* TTY_SPEK */
	reply_mess.m2_l2= flags;	/* TTY_FLAGS */
	result= send(m->m_source, &reply_mess);
	assert(result == OK);
}


/*===========================================================================*
 *				compat_getp				     *
 *===========================================================================*/
PRIVATE int compat_getp(tty, sg)
tty_t *tty;
struct sgttyb *sg;
{
	int ispd, ospd, flgs;
	sg->sg_erase= tty->t_termios.c_cc[VERASE];
	sg->sg_kill= tty->t_termios.c_cc[VKILL];
	ospd= tspd2sgspd(cfgetospeed(&tty->t_termios));
	ispd= tspd2sgspd(cfgetispeed(&tty->t_termios));
	sg->sg_ospeed= ospd;
	sg->sg_ispeed= ispd;
		
	flgs= 0;

	/* XTABS	- if OPOST and XTABS */

	if ((tty->t_termios.c_oflag & (OPOST|XTABS)) == (OPOST|XTABS))
		flgs |= 0006000;

	/* BITS5..BITS8	- map directly to CS5..CS8 */

	switch(tty->t_termios.c_cflag & CSIZE)
	{
	case CS5:	flgs |= 0000000;	break;
	case CS6:	flgs |= 0000400;	break;
	case CS7:	flgs |= 0001000;	break;
	case CS8:	flgs |= 0001400;	break;
	}

	/* EVENP	- if PARENB and not PARODD */

	if ((tty->t_termios.c_cflag & (PARENB|PARODD)) == PARENB)
		flgs |= 0000200;

	/* ODDP		- if PARENB and PARODD */

	if ((tty->t_termios.c_cflag & (PARENB|PARODD)) == (PARENB|PARODD))
		flgs |= 0000100;

	/* RAW		- if not ICANON and not ISIG */

	if (!(tty->t_termios.c_lflag & (ICANON|ISIG)))
		flgs |= 0000040;

	/* CRMOD	- if ICRNL */

	if (tty->t_termios.c_iflag & ICRNL)
		flgs |= 0000020;

	/* ECHO		- if ECHO */

	if (tty->t_termios.c_lflag & ECHO)
		flgs |= 0000010;

	/* CBREAK	- if not ICANON and ISIG */

	if ((tty->t_termios.c_lflag & (ICANON|ISIG)) == ISIG)
		flgs |= 0000002;

	sg->sg_flags= flgs;
	return OK;
}


/*===========================================================================*
 *				compat_getc				     *
 *===========================================================================*/
PRIVATE int compat_getc(tty, tc)
tty_t *tty;
struct tchars *tc;
{
	tc->t_intrc= tty->t_termios.c_cc[VINTR];
	tc->t_quitc= tty->t_termios.c_cc[VQUIT];
	tc->t_startc= tty->t_termios.c_cc[VSTART];
	tc->t_stopc= tty->t_termios.c_cc[VSTOP];
	tc->t_brkc= tty->t_termios.c_cc[VEOL];
	tc->t_eofc= tty->t_termios.c_cc[VEOF];
	return OK;
}


/*===========================================================================*
 *				compat_setp				     *
 *===========================================================================*/
PRIVATE int compat_setp(tty, sg)
tty_t *tty;
struct sgttyb *sg;
{
	struct termios termios;
	int flags;

	termios= tty->t_termios;

	termios.c_cc[VERASE]= sg->sg_erase;
	termios.c_cc[VKILL]= sg->sg_kill;
	cfsetispeed(&termios, sgspd2tspd(sg->sg_ispeed & BYTE));
	cfsetospeed(&termios, sgspd2tspd(sg->sg_ospeed & BYTE));
	flags= sg->sg_flags;

	/* Input flags */

	/* BRKINT	- not changed */
	/* ICRNL	- set if CRMOD is set and not RAW */
	/*		  (CRMOD also controls output) */

	if ((flags & 0000020) && !(flags & 0000040))
		termios.c_iflag |= ICRNL;
	else
		termios.c_iflag &= ~ICRNL;

	/* IGNBRK	- not changed */
	/* IGNCR	- forced off (ignoring cr's is not supported) */

	termios.c_iflag &= ~IGNCR;

	/* IGNPAR	- not changed */
	/* INLCR	- forced off (mapping nl's to cr's is not supported) */

	termios.c_iflag &= ~INLCR;

	/* INPCK	- not changed */
	/* ISTRIP	- not changed */
	/* IXOFF	- not changed */
	/* IXON		- forced on if not RAW */

	if (!(flags & 0000040))
		termios.c_iflag |= IXON;
	else
		termios.c_iflag &= ~IXON;

	/* PARMRK	- not changed */

	/* Output flags */

	/* OPOST	- forced on if not RAW */

	if (!(flags & 0000040))
		termios.c_oflag |= OPOST;
	else
		termios.c_oflag &= ~OPOST;

	/* ONLCR	- forced on if CRMOD */

	if (flags & 0000020)
		termios.c_oflag |= ONLCR;
	else
		termios.c_oflag &= ~ONLCR;

	/* XTABS	- forced on if XTABS */

	if (flags & 0006000)
		termios.c_oflag |= XTABS;
	else
		termios.c_oflag &= ~XTABS;

	/* CLOCAL	- not changed */
	/* CREAD	- forced on (receiver is always enabled) */

	termios.c_cflag |= CREAD;

	/* CSIZE	- CS5-CS8 correspond directly to BITS5-BITS8 */

	termios.c_cflag &= ~CSIZE;
	switch(flags & 0001400)
	{
	case 0000000:	termios.c_cflag |= CS5; break;
	case 0000400:	termios.c_cflag |= CS6; break;
	case 0001000:	termios.c_cflag |= CS7; break;
	case 0001400:	termios.c_cflag |= CS8; break;
	}

	/* CSTOPB	- not changed */
	/* HUPCL	- not changed */
	/* PARENB	- set if EVENP or ODDP is set */

	if (flags & (0000200|0000100))
		termios.c_cflag |= PARENB;
	else
		termios.c_cflag &= ~PARENB;

	/* PARODD	- set if ODDP is set */

	if (flags & 0000100)
		termios.c_cflag |= PARODD;
	else
		termios.c_cflag &= ~PARODD;

	/* Local flags */

	/* ECHO		- set if ECHO is set */

	if (flags & 0000010)
		termios.c_lflag |= ECHO;
	else	
		termios.c_lflag &= ~ECHO;
	
	/* ECHOE	- not changed */
	/* ECHOK	- not changed */
	/* ECHONL	- not changed */
	/* ICANON	- set if neither CBREAK nor RAW */

	if (!(flags & (0000002|0000040)))
		termios.c_lflag |= ICANON;
	else
		termios.c_lflag &= ~ICANON;

	/* IEXTEN	- set if not RAW */
	if (!(flags & 0000040))
		termios.c_lflag |= IEXTEN;
	else
		termios.c_lflag &= ~IEXTEN;

	/* ISIG		- set if not RAW */
	if (!(flags & 0000040))
		termios.c_lflag |= ISIG;
	else
		termios.c_lflag &= ~ISIG;

	/* NOFLSH	- not changed */
	/* TOSTOP	- not changed */
	
	setattr(tty, &termios);
	return OK;
}


/*===========================================================================*
 *				compat_setc				     *
 *===========================================================================*/
PRIVATE int compat_setc(tty, tc)
tty_t *tty;
struct tchars *tc;
{
	struct termios termios;

	termios= tty->t_termios;

	termios.c_cc[VINTR]= tc->t_intrc;
	termios.c_cc[VQUIT]= tc->t_quitc;
	termios.c_cc[VSTART]= tc->t_startc;
	termios.c_cc[VSTOP]= tc->t_stopc;
	termios.c_cc[VEOL]= tc->t_brkc;
	termios.c_cc[VEOF]= tc->t_eofc;
	setattr(tty, &termios);
	return OK;
}


/*===========================================================================*
 *				tspd2sgspd				     *
 *===========================================================================*/
PRIVATE int tspd2sgspd(tspd)
speed_t tspd;
{
	switch(tspd)
	{
	/* Standard speeds */
	case B0:	return 0;
	case B110:	return 1;
	case B300:	return 3;
	case B1200:	return 12;
	case B2400:	return 24;
	case B4800:	return 48;
	case B9600:	return 96;
	case B19200:	return 192;

	/* Extra speeds, 0 values are remapped to prevent hup on restore */
	case B50:	return 50;
	case B75:	return 75;
	case B134:	return 134;
	case B200:	return 2;
	case B600:	return 6;
	case B1800:	return 18;
	case B38400:	return 195;
	case B57600:	return 194;
	case B115200:	return 193;

	default:	return 96;
	}
}


/*===========================================================================*
 *				sgspd2tspd				     *
 *===========================================================================*/
PRIVATE speed_t sgspd2tspd(sgspd)
int sgspd;
{
	switch(sgspd)
	{
	/* Standard speeds */
	case 0:		return B0;
	case 1:		return B110;
	case 3:		return B300;
	case 12:	return B1200;
	case 24:	return B2400;
	case 48:	return B4800;
	case 96:	return B9600;
	case 192:	return B19200;

	/* Extra speeds, that were remapped to prevent hup on restore */
	case 50:	return B50;
	case 75:	return B75;
	case 134:	return B134;
	case 2:		return B200;
	case 6:		return B600;
	case 18:	return B1800;
	case 195:	return B38400;
	case 194:	return B57600;
	case 193:	return B115200;

	default:	return B9600;
	}
}


/*===========================================================================*
 *				do_close				     *
 *===========================================================================*/
PRIVATE int do_close(tty, m)
tty_t *tty;
message *m;
{
	int last;			/* last close? */
	assert(!(tty->t_flags & TF_TTY_CLOSED));

	last= 0;
	if (tty->t_opencnt <= 0)
		panic("tty line closed too many times", tty->t_opencnt);
	tty->t_opencnt--;
	if (tty->t_opencnt == 0) 
	{
		last= 1;
		tty->t_sesldr= 0;
		assert(!tty->t_rd_queue);
		assert(!tty->t_wr_queue);
		assert(!tty->t_ioc_queue);
		if (tty->t_termios.c_cflag & HUPCL)
			(*tty->t_hup)(tty->t_line);
		tty->t_flags &= ~(TF_XON_OUTPUT_OFF|TF_DRAIN);
		setattr(tty, &default_termios);
		tty->t_termios= default_termios;
		tty->t_winsize= default_winsize;

		/* Flush any leftover input and output. */
		ib_flush(tty);
		ob_flush(tty);

		/* We are now closed */
		tty->t_flags |= TF_TTY_CLOSED;
	}
	return last;
}


/*===========================================================================*
 *				do_cancel				     *
 *===========================================================================*/
PRIVATE int do_cancel(tty, m)
tty_t *tty;
message *m;
{
	int operation;
	int status;
	mq_t *mq, *mq_prev;

	operation= m->NDEV_OPERATION;

	if (operation == CANCEL_ANY || operation == DEV_READ)
	{
		mq_prev= NULL;
		for (mq= tty->t_rd_queue; mq; mq_prev= mq, mq= mq->mq_next)
		{
			if (mq->mq_mess.NDEV_PROC != m->NDEV_PROC ||
				mq->mq_mess.NDEV_REF != m->NDEV_REF)
			{
				continue;
			}

			/* Found one. */
			if (!mq_prev)
			{	/* Head of the queue. */
				status= tty->t_rd_offset;
				if (status == 0)
					status= EINTR;
				tty->t_rd_queue= mq->mq_next;
				if (tty->t_rd_queue != NULL)
					setup_read(tty);
				mq_free(mq);
				return status;
			}
			/* Somewhere in the middle of the queue. */
			mq_prev->mq_next= mq->mq_next;
			if (mq_prev->mq_next == NULL)
				tty->t_rd_queue_tail= mq_prev;
			mq_free(mq);
			return EINTR;
		}

		/* Apperently no read was to be canceled. */
	}

	if (operation == CANCEL_ANY || operation == DEV_WRITE)
	{
		mq_prev= NULL;
		for (mq= tty->t_wr_queue; mq; mq_prev= mq, mq= mq->mq_next)
		{
			if (mq->mq_mess.NDEV_PROC != m->NDEV_PROC ||
				mq->mq_mess.NDEV_REF != m->NDEV_REF)
				continue;
			/* Found one. */
			if (!mq_prev)
			{	/* Head of the queue. */
				tty->t_wr_queue= mq->mq_next;
				status= tty->t_wr_offset;
				setup_write(tty);
				if (status == 0)
					status= EINTR;
				mq_free(mq);
				return status;
			}
			/* Somewhere in the middle of the queue. */
			mq_prev->mq_next= mq->mq_next;
			if (mq_prev->mq_next == NULL)
				tty->t_wr_queue_tail= mq_prev;
			mq_free(mq);
			return EINTR;
		}
		/* Apperently no write was to be canceled. */
	}

	if (operation == CANCEL_ANY || operation == DEV_IOCTL3)
	{
		mq_prev= NULL;
		for (mq= tty->t_ioc_queue; mq; mq_prev= mq, mq= mq->mq_next)
		{
			if (mq->mq_mess.NDEV_PROC != m->NDEV_PROC ||
				mq->mq_mess.NDEV_REF != m->NDEV_REF)
			{
				continue;
			}
			/* Found one. */
			if (!mq_prev)
			{	/* Head of the queue. */
				tty->t_ioc_queue= mq->mq_next;
				restart_ioctl(tty);
				status= EINTR;
				mq_free(mq);
				return status;
			}
			/* Somewhere in the middle of the queue. */
			mq_prev->mq_next= mq->mq_next;
			if (mq_prev->mq_next == NULL)
				tty->t_ioc_queue_tail= mq_prev;
			mq_free(mq);
			return EINTR;
		}
		/* Apperently no ioctl was to be canceled. */
	}

	panic("Unable to find request to be canceled", NO_NUM);
	/*NOTREACHED*/
}


/*===========================================================================*
 *				setattr					     *
 *===========================================================================*/
PRIVATE void setattr(tty, termios)
tty_t *tty;
struct termios *termios;
{
	if (!(termios->c_lflag & ICANON) && termios->c_cc[VTIME] != 0 &&
		(is_canon(tty) ||
		termios->c_cc[VTIME] != tty->t_termios.c_cc[VTIME] ||
		termios->c_cc[VMIN] != tty->t_termios.c_cc[VMIN]))
	{
		/* Assume that the last request and last character times
		 * are not up to date. This is too conservative in some
		 * cases.
		 */
		tty->t_rd_req_time= tty->t_rd_char_time= get_uptime();
		set_rd_timer(tty);
	}
	if (!(termios->c_cflag & CREAD))
		tty->t_putf= put_discard;
	else if (termios->c_lflag & ICANON)
	{
		tty->t_putf= put_canon;
		if (!is_canon(tty))
			tty->t_rd_lnext= 0;
	}
	else if (termios->c_iflag & (ICRNL | IGNCR | INLCR | IXON))
	{
		/* Note that IGNBRK, BRKINT, IGNPAR, PARMRK, INCPK,
		 * and ISTRIP are handled by the device drivers.
		 */
		/* IXOFF is ignored, XANY is not relevant. */
		tty->t_putf= put_noncanon;
	}
	else if (termios->c_lflag & (ECHO | ECHONL | IEXTEN |
		ISIG))
	{
		/* ECHOK and ECHOE are only used in canonical mode. */
		/* Note that NOFLUSH and TOSTOP are not needed in input
		 * processing.
		 */
		tty->t_putf= put_noncanon;
	}
	else if (termios->c_cc[VTIME] != 0 || termios->c_cc[VMIN] != 1)
	{
		/* Note that c_oflag flags and c_cflag flags are not important
		 * to put_* functions.
		 */
		tty->t_putf= put_timed_raw;
	}
	else
	{
		tty->t_putf= put_raw;
	}
	tty->t_termios= *termios;

	if (cfgetospeed(termios) == B0)
	{
		if (tty->t_sesldr)
			cause_sig(tty->t_sesldr, SIGHUP);
	}

	/* now set physical characteristics */
	(*tty->t_setattr)(tty->t_line, termios);
}


/*===========================================================================*
 *				read_reply				     *
 *===========================================================================*/
PRIVATE void read_reply(tty, result)
tty_t *tty;
int result;
{
	mq_t *mq;

	mq= tty->t_rd_queue;
	if ((tty->t_rd_queue= mq->mq_next) != NULL)
		setup_read(tty);
	reply(tty, mq, result, TRUE);
}


/*===========================================================================*
 *				write_reply				     *
 *===========================================================================*/
PRIVATE void write_reply(tty, result)
tty_t *tty;
int result;
{
	mq_t *mq;

	mq= tty->t_wr_queue;
	if ((tty->t_wr_queue= mq->mq_next) != NULL)
		setup_write(tty);
	reply(tty, mq, result, TRUE);
}


/*===========================================================================*
 *				reply					     *
 *===========================================================================*/
PRIVATE void reply(tty, mq, result, can_enqueue)
tty_t *tty;
mq_t *mq;
int result;
int can_enqueue;
{
	int proc, ref, status, operation;
	message m, *mp;

	proc= mq->mq_mess.NDEV_PROC;
	ref= mq->mq_mess.NDEV_REF;
	operation= mq->mq_mess.m_type;

	if (can_enqueue)
		mp= &mq->mq_mess;
	else
		mp= &m;

	mp->m_type= DEVICE_REPLY;
	mp->REP_PROC_NR= proc;
	mp->REP_STATUS= result;
	mp->REP_REF= ref;
	mp->REP_OPERATION= operation;

	status= send(mq->mq_mess.m_source, mp);
	if (status == ELOCKED && can_enqueue)
	{
		(*tty->t_reply)(tty->t_line, mq);
		return;
	}
	if (status != OK)
		panic("send failed", result);
	if (can_enqueue)
		mq_free(mq);
}


/*===========================================================================*
 *				tty_get					     *
 *===========================================================================*/
PUBLIC void tty_get(tty)
tty_t *tty;
{
/* Try to give the device some data. */

	unsigned size, result;
	int done_something;
	int r;
	u8_t *cp;

	tty->t_flags &= ~TF_WRITE_INITIATIVE;
	if (tty->t_flags & TF_XON_OUTPUT_OFF)
	{
		/* Output is suspended with a STOP character. */
		return;
	}

	done_something= 0;
	for(;;)
	{
		r= ob_get_head(tty, &cp);

		if (r == 0)
		{
			ob_flush(tty);
			restart_write(tty);
			r= ob_get_head(tty, &cp);
			if (r == 0)
			{
				/* Tell the device that we are done. */
				if (done_something)
					(*tty->t_put)(tty->t_line, NULL, 0);
				tty->t_flags |= TF_WRITE_INITIATIVE;
				return;
			}
		}

		size= r;
		assert(size > 0);
		result= (*tty->t_put)(tty->t_line, (char *)cp, size);
		assert(result <= size);
		ob_delete(tty, result);

		if (result != size)
		{
			/* device didn't accept everything. */
			break;
		}
		done_something= 1;
	}
	restart_write(tty);			/* Try to prefill the buffer. */
}


/*===========================================================================*
 *				tty_put					     *
 *===========================================================================*/
PUBLIC int tty_put(tty, buf, siz)
tty_t *tty;
char *buf;
int siz;
{
	int reqsize;
	mq_t *mq;
	int chr;
	int icanon, lnext, isig, xon, xany;
	int result;

	return (*tty->t_putf)(tty, buf, siz);
}


/*===========================================================================*
 *				tty_hup					     *
 *===========================================================================*/
PUBLIC void tty_hup(tty)
tty_t *tty;
{
/* The device tells us to quit. */
	unsigned offset;
	mq_t *mq, *mq_tmp;

	tty->t_flags |= TF_DEV_CLOSED;

	while(tty->t_rd_queue != NULL)
	{
		read_reply(tty, tty->t_rd_offset);
	}
	while(tty->t_wr_queue != NULL)
	{
		offset= tty->t_rd_offset;
		if (offset == 0)
			offset= EIO;
		write_reply(tty, offset);
	}
	while(tty->t_ioc_queue != NULL)
	{
		mq= tty->t_ioc_queue;
		tty->t_ioc_queue= mq->mq_next;
		reply(tty, mq, EIO, TRUE);
	}
	if (tty->t_flags & TF_TTY_CLOSED)
		(*tty->t_hup)(tty->t_line);
	else 
	{
		if (tty->t_sesldr)
			cause_sig(tty->t_sesldr, SIGHUP);
	}
}


/*===========================================================================*
 *				tty_init				     *
 *===========================================================================*/
PUBLIC void tty_init(tty, dev_ref, readbuf, readbuf_siz, writebuf, 
						writebuf_siz, task, cntxt)
struct tty *tty;
int dev_ref;
char *readbuf;
int readbuf_siz;
char *writebuf;
int writebuf_siz;
int task;
tmrs_context_ut *cntxt;
{
	int i;

	assert(_POSIX_VDISABLE == (cc_t)'\377');	/* For PARMRK */

	default_termios.c_iflag= TINPUT_DEF;
	default_termios.c_oflag= TOUTPUT_DEF;
	default_termios.c_cflag= TCTRL_DEF;
	default_termios.c_lflag= TLOCAL_DEF & ~(ECHO|ECHOE|ECHOK|ECHONL);
	cfsetispeed(&default_termios, TSPEED_DEF);
	cfsetospeed(&default_termios, TSPEED_DEF);
	for (i= 0; i<NCCS; i++)
		default_termios.c_cc[i]= _POSIX_VDISABLE;
	default_termios.c_cc[VEOF]= TEOF_DEF;
	default_termios.c_cc[VEOL]= TEOL_DEF;
	default_termios.c_cc[VERASE]= TERASE_DEF;
	default_termios.c_cc[VINTR]= TINTR_DEF;
	default_termios.c_cc[VKILL]= TKILL_DEF;
	default_termios.c_cc[VMIN]= TMIN_DEF;
	default_termios.c_cc[VQUIT]= TQUIT_DEF;
	default_termios.c_cc[VTIME]= TTIME_DEF;
	default_termios.c_cc[VSUSP]= TSUSP_DEF;
	default_termios.c_cc[VSTART]= TSTART_DEF;
	default_termios.c_cc[VSTOP]= TSTOP_DEF;
	default_termios.c_cc[VREPRINT]= TREPRINT_DEF;
	default_termios.c_cc[VLNEXT]= TLNEXT_DEF;
	default_termios.c_cc[VDISCARD]= TDISCARD_DEF;

	default_winsize.ws_row= 0;
	default_winsize.ws_col= 0;
	default_winsize.ws_xpixel= 0;
	default_winsize.ws_ypixel= 0;

	tty->t_get= def_get;
	tty->t_put= def_put;
	tty->t_hup= def_hup;
	tty->t_reply= def_reply;
	tty->t_setattr= def_setattr;
	tty->t_break= def_break;

	tty->t_flags= TF_DEV_CLOSED | TF_TTY_CLOSED | TF_READ_INITIATIVE |
		TF_WRITE_INITIATIVE;
	tty->t_line= dev_ref;
	tty->t_tmr_cntxt= cntxt;
	tty->t_rd_queue= NULL;
	tty->t_wr_queue= NULL;
	tty->t_ioc_queue= NULL;
	tty->t_sesldr= 0;
	tty->t_opencnt= 0;

	/* check if the input buffer is big enough. */
	if (readbuf_siz < T_RD_BUF_SIZ_MIN)
		panic("tty_init: input buffer too small: ", readbuf_siz);

	tty->t_rd_buf= (u8_t *)readbuf;
	tty->t_rd_buf_siz= readbuf_siz;
	ib_flush(tty);
	tty->t_putf= put_canon;
	tty->t_rd_timer_set= 0;
	tty->t_rd_req_time= 0;
	tty->t_rd_char_time= 0;
	tmrs_inittimer(&tty->t_rd_timer);

	/* check if the output buffer is big enough. */
	if (writebuf_siz < T_WR_BUF_SIZ_MIN)
		panic("tty_init: output buffer too small: ", writebuf_siz);

	tty->t_wr_buf= (u8_t *)writebuf;
	tty->t_wr_buf_siz= writebuf_siz;
	tty->t_wr_begin_n= 0;
	tty->t_wr_end_n= 0;
	tty->t_wr_position= 0;

	tty->t_task= task;
	tty->t_termios= default_termios;

	/* It doesn't matter if the char_size_table is initialized once more. */
	for (i= 0; i< 256; i++)
		char_size_table[i]= 1;	/* Default is a normal character. */
	char_size_table['\n']= 0;	/* Linefeed has no size. */
	char_size_table['\r']= -100;	/* Carriage return. */
	char_size_table['\10']= -1;	/* Backspace */
	char_size_table['\t']= 8;	/* Tab */
	char_size_table['\4']= 100;	/* EOT */
}


/*===========================================================================*
 *				tty_break_int				     *
 *===========================================================================*/
PUBLIC void tty_break_int(tty)
struct tty *tty;
{
	if (!(tty->t_termios.c_cflag & CREAD))
		return;
	ib_flush(tty);
	ob_flush(tty);
	if (tty->t_sesldr)
		cause_sig(tty->t_sesldr, SIGINT);
}


/*===========================================================================*
 *				def_get					     *
 *===========================================================================*/
PRIVATE void def_get(dev_ref)
int dev_ref;
{
/* The default get does nothing. */
}


/*===========================================================================*
 *				def_put					     *
 *===========================================================================*/
PRIVATE int def_put(dev_ref, buf, siz)
int dev_ref;
char *buf;
int siz;
{
/* The default put does nothing. */
	return 0;
}


/*===========================================================================*
 *				def_hup					     *
 *===========================================================================*/
PRIVATE void def_hup(dev_ref)
int dev_ref;
{
/* The default hup does nothing. */
}


/*===========================================================================*
 *				def_reply				     *
 *===========================================================================*/
PRIVATE void def_reply(dev_ref, mq)
int dev_ref;
mq_t *mq;
{
/* The default reply should not be called. */
	panic("def_reply called", NO_NUM);
}


/*===========================================================================*
 *				def_setattr				     *
 *===========================================================================*/
PRIVATE void def_setattr(dev_ref, tmiop)
int dev_ref;
struct termios *tmiop;
{
/* The default setattr does nothing. */
}


/*===========================================================================*
 *				def_break				     *
 *===========================================================================*/
PRIVATE void def_break(dev_ref, dur)
int dev_ref;
clock_t dur;
{
/* The default break does nothing. */
}


/*===========================================================================*
 *				setup_read				     *
 *===========================================================================*/
PRIVATE void setup_read(tty_t *tty)
{
	mq_t *mq;
	int size;
	phys_bytes phys;

	tty->t_rd_offset= 0;
	while (tty->t_rd_queue)
	{
		mq= tty->t_rd_queue;
		size= mq->mq_mess.NDEV_COUNT;
		tty->t_rd_size= size;
		phys= numap(mq->mq_mess.NDEV_PROC,
			(vir_bytes)mq->mq_mess.NDEV_BUFFER, size);
		if (phys != 0)
		{
			tty->t_rd_phys= phys;
			if (!is_canon(tty) &&
				tty->t_termios.c_cc[VTIME] != 0 &&
				tty->t_termios.c_cc[VMIN] == 0)
			{
				tty->t_rd_req_time= get_uptime();
			}
			return;
		}
		tty->t_rd_queue= mq->mq_next;
		reply(tty, mq, EFAULT, TRUE);
	}
}


/*===========================================================================*
 *				setup_write				     *
 *===========================================================================*/
PRIVATE void setup_write(tty_t *tty)
{
	mq_t *mq;
	int size;
	phys_bytes phys;

	tty->t_wr_offset= 0;
	while (tty->t_wr_queue)
	{
		mq= tty->t_wr_queue;
		size= mq->mq_mess.NDEV_COUNT;
		tty->t_wr_size= size;
		phys= numap(mq->mq_mess.NDEV_PROC,
			(vir_bytes)mq->mq_mess.NDEV_BUFFER, size);
		if (phys != 0)
		{
			tty->t_wr_phys= phys;
			return;
		}
		tty->t_wr_queue= mq->mq_next;
		reply(tty, mq, EFAULT, TRUE);
	}
}


/*===========================================================================*
 *				restart_read_canon			     *
 *===========================================================================*/
PRIVATE void restart_read_canon(tty)
tty_t *tty;
{
/* Try to deliver data to the user. */
	int r;
	u8_t *cp1, *cp2;
	int sz1, sz2;
	int cnt, size, offset, result;
	phys_bytes phys_user, phys_local;
	mq_t *mq;

	/* Canonical input mode, return at most one line. */
	while(tty->t_rd_queue && ib_compl_ln(tty))
	{
		assert(tty->t_rd_offset == 0);

		r= ib_get_head(tty, &cp1, &sz1, &cp2, &sz2);
		assert(r >= 0);

		size= tty->t_rd_size;
		offset= 0;
		cnt= size;
		if (cnt > sz1)
			cnt= sz1;
		if (cnt)
		{
			phys_user= tty->t_rd_phys;
			phys_local= vir2phys(cp1);
			phys_copy(phys_local, phys_user, (phys_bytes)cnt);
			offset += cnt;
		}
		if (offset < size && sz2 != 0)
		{
			cnt= size-offset;
			if (cnt > sz2)
				cnt= sz2;
			phys_user= tty->t_rd_phys + offset;
			phys_local= vir2phys(cp2);
			phys_copy(phys_local, phys_user, (phys_bytes)cnt);
			offset += cnt;
		}
		ib_delete(tty, offset);
		read_reply(tty, offset);
	}
	assert(ib_check(tty));
}


/*===========================================================================*
 *				restart_read_noncanon			     *
 *===========================================================================*/
PRIVATE void restart_read_noncanon(tty)
tty_t *tty;
{
/* Try to deliver data to the user. */
	int r;
	u8_t *cp1, *cp2;
	int sz1, sz2;
	int cnt, min, size, offset, result;
	phys_bytes phys_user, phys_local;
	mq_t *mq;

	/* Non Canonical mode. */

	/* make sure that VMIN is satisfied. */
	min= tty->t_termios.c_cc[VMIN];
	if (min == 0 && tty->t_termios.c_cc[VTIME] != 0)
		min= 1;

	while(tty->t_rd_queue)
	{
		r= ib_get_head(tty, &cp1, &sz1, &cp2, &sz2);
		if (r == -1)
		{
			if (tty->t_rd_offset != 0 || min == 0)
			{
				read_reply(tty, tty->t_rd_offset);
				continue;
			}
			return;
		}
		if (r < min)
		{
			if (!ib_compl_ln(tty))
				return;
			ib_pack(tty);
			r= ib_get_head(tty, &cp1, &sz1, &cp2, &sz2);
			if (r < min)
				return;
		}

		if (r == 0)
		{
			/* Left over from canonical mode. */
			ib_delete(tty, 0);
			continue;
		}

		size= tty->t_rd_size;
		offset= tty->t_rd_offset;
		cnt= size-offset;
		if (cnt > sz1)
			cnt= sz1;
		phys_user= tty->t_rd_phys + offset;
		phys_local= vir2phys(cp1);
 
		phys_copy(phys_local, phys_user, (phys_bytes)cnt);
		ib_delete(tty, cnt);
		offset += cnt;
		if (offset < size && sz2 != 0)
		{
			cnt= size-offset;
			if (cnt > sz2)
				cnt= sz2;
			phys_user= tty->t_rd_phys + offset;
			phys_local= vir2phys(cp2);
 
			phys_copy(phys_local, phys_user, (phys_bytes)cnt);
			ib_delete(tty, cnt);
			offset += cnt;
		}
		if (offset == size)
			read_reply(tty, offset);
		else
			tty->t_rd_offset= offset;
	}
}


/*===========================================================================*
 *				restart_write				     *
 *===========================================================================*/
PRIVATE void restart_write(tty)
struct tty *tty;
{
/* If the output is empty or the entire write request fits in the buffer, we
 * fetch the data from user space, and translate it if OPOST is set.
 */
	int wr_size, wr_offset, size, result;
	phys_bytes phys_user, phys_local;
	mq_t *mq;
	int do_crlf, do_tabs, do_noeot;
	int g, i, j, chr, chr_siz, tab_size, position;
	u8_t *cp, *cp1, *cp2;
	int r, sz1, sz2;
	int flusho;

	flusho= !!(tty->t_termios.c_lflag & LFLUSHO);

	if (tty->t_flags & TF_DRAIN)
	{
		if (!ob_empty(tty))
			return;
		tty->t_flags &= ~TF_DRAIN;
		restart_ioctl(tty);
		assert(!(tty->t_flags & TF_DRAIN));
	}

	if (flusho)
	{
		while(tty->t_wr_queue)
			write_reply(tty, tty->t_wr_size);
		return;
	}

	if (!(tty->t_termios.c_oflag & OPOST))
	{
		r= ob_space(tty, &cp1, &sz1, &cp2, &sz2);

		/* No post processing, just copy the data into the buffer. */
		while(tty->t_wr_queue)
		{
			if (r == 0)
			{
				if (tty->t_flags & TF_WRITE_INITIATIVE)
				{
					tty_get(tty);
					r= ob_space(tty, &cp1, &sz1,
								&cp2, &sz2);
					continue;
				}
				break;
			}
			if (sz1 == 0)
			{
				sz1= sz2;
				cp1= cp2;
			}

			wr_size= tty->t_wr_size;
			wr_offset= tty->t_wr_offset;
			assert(wr_size > wr_offset);
			size= wr_size;
			size -= wr_offset;
			if (size > sz1)
			{
				if (!ob_empty(tty))
				{
					return;	/* Only fetch large blocks */
						/* of data. */
				}
				size= sz1;
			}
			phys_user= tty->t_wr_phys + wr_offset;
			phys_local= vir2phys(cp1);
			phys_copy(phys_user, phys_local, (phys_bytes)size);
			wr_offset += size;
			tty->t_wr_offset= wr_offset;
			if (wr_offset ==  wr_size)
				write_reply(tty, wr_offset);
			ob_grow(tty, size);
			r -= size;
			cp1 += size;
			sz1 -= size;
		}
		/* indicate that the cursor position is unknown and that
		 * the echo of input is garbled. */
		tty->t_wr_position= 0;
		return;
	}

	/* Post processing, keep track of the position, map CR on CR-LF,
	 * expand tabs, discard EOF caracters (if the relevant option is
	 * enabled).
	 */
	do_crlf= tty->t_termios.c_oflag & ONLCR;
	do_tabs= tty->t_termios.c_oflag & XTABS;
	do_noeot= tty->t_termios.c_oflag & ONOEOT;

	while(tty->t_wr_queue)
	{
		r= ob_space(tty, &cp1, &sz1, &cp2, &sz2);
		g= 0;
		if (r < 8)
		{
			if (tty->t_flags & TF_WRITE_INITIATIVE)
			{
				tty_get(tty);
				continue;
			}
			break;
		}

		wr_size= tty->t_wr_size;
		wr_offset= tty->t_wr_offset;
		assert(wr_size > wr_offset);
		size= wr_size;
		size -= wr_offset;
		if (size > sizeof(buffer))
			size= sizeof(buffer);
		assert(size != 0);

		phys_user= tty->t_wr_phys + wr_offset;
		phys_local= vir2phys(buffer);
		phys_copy(phys_user, phys_local, (phys_bytes)size);

		position= tty->t_wr_position;
		for (i= 0, cp= buffer; i<size && r>0; i++, cp++)
		{
			if (sz1 == 0)
			{
				sz1= sz2;
				cp1= cp2;
			}

			chr= *cp;
			assert(chr >= 0 && chr < 256);
			chr_siz= char_size_table[chr];
			if (chr_siz == 100)
			{
				if (do_noeot)
					continue;
				chr_siz= 1;
			}
			if (chr_siz == 1 || chr_siz == -1)
			{
				/* Normal characters, and ^H. */
				*cp1++= chr;
				sz1--;
				r--;
				g++;
				position += chr_siz;
				continue;
			}
			if (chr_siz == 8)
			{
				/* Tab */
				tab_size= 8 - (position & 7);
				if (!do_tabs)
				{
					*cp1++= chr;
					sz1--;
					r--;
					g++;
					position += tab_size;
					continue;
				}
				if (r < tab_size)
					break;
				for (j= 0; j<tab_size; j++)
				{
					*cp1++= ' ';
					sz1--;

					if (sz1 == 0)
					{
						sz1= sz2;
						cp1= cp2;
					}
				}
				r -= tab_size;
				g += tab_size;
				position += tab_size;
				continue;
			}
			if (chr_siz == 0)
			{
				/* LF */
				if (!do_crlf)
				{
					*cp1++= chr;
					sz1--;
					r--;
					g++;
					continue;
				}
				if (r < 2)
					break;
				*cp1++= '\r';
				sz1--;
				if (sz1 == 0)
				{
					sz1= sz2;
					cp1= cp2;
				}
				*cp1++= chr;
				sz1--;
				r -= 2;
				g += 2;
				position= 0;
				continue;
			}
			if (chr_siz == -100)
			{
				/* CR */
				*cp1++= chr;
				sz1--;
				r--;
				g++;
				position= 0;
				continue;
			}
			panic("strange value in char_size_table", chr_siz);
		}
		assert(sz1 >= 0);
		tty->t_wr_position= position;
		result= i;
		assert(result >= 0 && result <= size);
		wr_offset += result;
		if (wr_offset ==  wr_size)
			write_reply(tty, wr_offset);
		else
			tty->t_wr_offset= wr_offset;
		if (g != 0)
			ob_grow(tty, g);
	}
}


/*===========================================================================*
 *				restart_ioctl				     *
 *===========================================================================*/
PRIVATE void restart_ioctl(tty)
struct tty *tty;
{
	mq_t *ioc_q, *mq;

	ioc_q= tty->t_ioc_queue;
	if (ioc_q == NULL)
		return;
	tty->t_ioc_queue= NULL;
	while(ioc_q)
	{
		mq= ioc_q;
		ioc_q= ioc_q->mq_next;
		(void)do_ioctl(tty, mq);
	}
}


/*===========================================================================*
 *				echonl					     *
 *===========================================================================*/
PRIVATE void echonl(tty)
tty_t *tty;
{
	if (tty->t_termios.c_lflag & (ECHO | ECHONL))
		output(tty, '\n');
}


/*===========================================================================*
 *				echo_eof				     *
 *===========================================================================*/
PRIVATE void echo_eof(tty)
tty_t *tty;
{
	int pos, npos;

	if (!(tty->t_termios.c_lflag & ECHO))
		return;
	pos= tty->t_wr_position;
	echo(tty, tty->t_termios.c_cc[VEOF]);
	npos= tty->t_wr_position;
	if (npos <= pos || npos > pos+8)
		return;
	for (; pos<npos; pos++)
		output(tty, '\10');
}


/*===========================================================================*
 *				output					     *
 *===========================================================================*/
PRIVATE void output(tty, chr)
tty_t *tty;
int chr;
{
	int opost, do_crlf, do_tabs, do_noeot, chr_size, tab_size, i, r;
	u8_t *cp1, *cp2;
	int sz1, sz2;

	if (tty->t_termios.c_lflag & LFLUSHO)
		return;

	chr= (u8_t)chr;
	opost= tty->t_termios.c_oflag & OPOST;
	do_crlf= opost && (tty->t_termios.c_oflag & ONLCR);
	do_tabs= opost && (tty->t_termios.c_oflag & XTABS);
	do_noeot= opost && (tty->t_termios.c_oflag & ONOEOT);

	r= ob_space(tty, &cp1, &sz1, &cp2, &sz2);
	if (r == 0)
		return;

	assert(chr >= 0 && chr < 256);
	chr_size= char_size_table[chr];
	if (chr_size == 100)
	{
		if (do_noeot)
			return;
		chr_size= 1;
	}
	if (chr_size == 1 || chr_size == -1)
	{
		/* Normal characters, and ^H. */
		*cp1= chr;
		ob_grow(tty, 1);
		tty->t_wr_position += chr_size;
		return;
	}
	if (chr_size == 8)
	{
		/* Tab */
		tab_size= 8 - (tty->t_wr_position & 7);
		if (!do_tabs)
		{
			*cp1= chr;
			ob_grow(tty, 1);
			tty->t_wr_position += tab_size;
			return;
		}
		if (r < tab_size)
			return;
		for (i= 0; i<tab_size; i++)
		{
			*cp1++= ' ';
			sz1--;
			if (sz1 == 0)
			{
				sz1= sz2;
				cp1= cp2;
			}
		}
		ob_grow(tty, tab_size);
		tty->t_wr_position += tab_size;
		return;
	}
	if (chr_size == 0)
	{
		/* LF */
		if (!do_crlf)
		{
			*cp1= chr;
			ob_grow(tty, 1);
			return;
		}
		if (r < 2)
			return;
		*cp1++= '\r';
		sz1--;
		if (sz1 == 0)
		{
			sz1= sz2;
			cp1= cp2;
		}
		*cp1= chr;
		ob_grow(tty, 2);
		tty->t_wr_position= 0;
		return;
	}
	if (chr_size == -100)
	{
		*cp1= chr;
		ob_grow(tty, 1);
		tty->t_wr_position= 0;
		return;
	}
	panic("strange value in char_size_table", chr_size);
}


/*===========================================================================*
 *				echo					     *
 *===========================================================================*/
PRIVATE void echo(tty, chr)
tty_t *tty;
int chr;
{
	if (!(tty->t_termios.c_lflag & ECHO))
		return;
	chr= (unsigned char)chr;
	if (chr < 32 && chr != '\t')
	{
		output(tty, '^');
		chr += '@';
	}
	else if (chr == 127)
	{
		output(tty, '^');
		chr= '?';
	}
	output(tty, chr);
}


/*===========================================================================*
 *				input_canon_zero_space			     *
 *===========================================================================*/
PRIVATE int input_canon_zero_space(tty, chr)
tty_t *tty;
int chr;
{
	int r, sz1, sz2;
	u8_t *cp1, *cp2; 
	int del_chr, pos, delta_pos, old_pos, i;
	int wr_pos;

	if (chr == tty->t_termios.c_cc[VREPRINT] && 
		(tty->t_termios.c_lflag & IEXTEN))
	{
		do_reprint(tty);
		return 1;
	}
	if (chr == tty->t_termios.c_cc[VERASE])
	{
		r= ib_get_tail(tty, &cp1, &sz1, &cp2, &sz2);
		if (r <= 0)
			return 1;		/* Nothing to do */
		if (sz2 != 0)
		{
			del_chr= cp2[sz2-1];
			sz2--;
		}
		else
		{
			del_chr= cp1[sz1-1];
			sz1--;
		}
		ib_shrink(tty, 1);
		if (!(tty->t_termios.c_lflag & ECHO))
			return 1;		/* Nothing to echo. */
		if (!(tty->t_termios.c_lflag & ECHOE))
		{
			/* No visual erase. */
			echo(tty, tty->t_termios.c_cc[VERASE]);
			return 1;
		}
		/* calculate the new line position. */
		pos= tty->t_rd_lineoffs;
		for (i= 0; i<sz1; i++)
			pos= output_pos(tty, pos, cp1[i]);
		for (i= 0; i<sz2; i++)
			pos= output_pos(tty, pos, cp2[i]);
		old_pos= output_pos(tty, pos, del_chr);
		delta_pos= old_pos - pos;

		if (old_pos == tty->t_wr_position && delta_pos >= 0 && 
			delta_pos <= 8)
		{
			for (i= 0; i<delta_pos; i++)
			{
				output(tty, '\10');
				output(tty, ' ');
				output(tty, '\10');
			}
		}
		else
			do_reprint(tty);
		return 1;
	}
	if (chr == tty->t_termios.c_cc[VKILL])
	{
		r= ib_get_tail(tty, &cp1, &sz1, &cp2, &sz2);
		if (r <= 0)
			return 1;		/* Nothing to do */
		ib_shrink(tty, r);
		if (!(tty->t_termios.c_lflag & ECHO))
		{
			return 1;		/* Nothing to echo. */
		}
		if (!(tty->t_termios.c_lflag & ECHOE))
		{
			/* No visual erase. */
			echo(tty, tty->t_termios.c_cc[VKILL]);
			if (tty->t_termios.c_lflag & ECHOK)
				output(tty, '\n');
			return 1;
		}
		/* calculate the new line position. */
		pos= tty->t_rd_lineoffs;
		old_pos= pos;
		for (i= 0; i<sz1; i++)
			pos= output_pos(tty, pos, cp1[i]);
		for (i= 0; i<sz2; i++)
			pos= output_pos(tty, pos, cp2[i]);
		delta_pos= pos - old_pos;
		wr_pos= tty->t_wr_position;
		if (pos == tty->t_wr_position && delta_pos >= 0)
		{
			for (i= 0; i<delta_pos; i++)
			{
				output(tty, '\10');
				output(tty, ' ');
				output(tty, '\10');
			}
		}
		else
		{
			do_reprint(tty);
		}
		return 1;
	}
	return 0;
}


/*===========================================================================*
 *				do_reprint				     *
 *===========================================================================*/
PRIVATE void do_reprint(tty)
tty_t *tty;
{
	int i, r, sz1, sz2;
	u8_t *cp1, *cp2;

	if (!(tty->t_termios.c_lflag & ECHO))
		return;
	if (tty->t_termios.c_cc[VREPRINT] != _POSIX_VDISABLE && 
		tty->t_termios.c_lflag & IEXTEN)
	{
		echo(tty, tty->t_termios.c_cc[VREPRINT]);
	}
	output(tty, '\n');
	tty->t_rd_lineoffs= tty->t_wr_position;
	r= ib_get_tail(tty, &cp1, &sz1, &cp2, &sz2);
	if (r <= 0)
		return;
	for (i= 0; i<sz1; i++)
		echo(tty, cp1[i]);
	for (i= 0; i<sz2; i++)
		echo(tty, cp2[i]);
}


/*===========================================================================*
 *				output_pos				     *
 *===========================================================================*/
PRIVATE int output_pos(tty, pos, chr)
tty_t *tty;
int pos;
int chr;
{
	int opost, do_crlf, do_tabs, do_noeot, chr_size, tab_size;

	chr= (u8_t)chr;
	if (chr < 32 && chr != '\t')
	{
		pos= output_pos(tty, pos, '^');
		chr += '@';
	}
	else if (chr == 127)
	{
		pos= output_pos(tty, pos, '^');
		chr= '?';
	}
	opost= tty->t_termios.c_oflag & OPOST;
	do_crlf= opost && (tty->t_termios.c_oflag & ONLCR);
	do_tabs= opost && (tty->t_termios.c_oflag & XTABS);
	do_noeot= opost && (tty->t_termios.c_oflag & ONOEOT);

	assert(chr >= 0 && chr < 256);
	chr_size= char_size_table[chr];
	if (chr_size == 100)
	{
		if (do_noeot)
			return pos;
		chr_size= 1;
	}
	if (chr_size == 1 || chr_size == -1)
	{
		/* Normal characters, and ^H. */
		return pos + chr_size;
	}
	if (chr_size == 8)
	{
		/* Tab */
		tab_size= 8 - (pos & 7);
		return pos + tab_size;
	}
	if (chr_size == 0)
	{
		/* LF */
		if (!do_crlf)
		{
			return pos;
		}
		return 0;
	}
	if (chr_size == -100)
	{
		return 0;
	}
	panic("strange value in char_size_table", chr_size);
	/*NOTREACHED*/
}


/*===========================================================================*
 *				set_rd_timer				     *
 *===========================================================================*/
PRIVATE void set_rd_timer(tty)
tty_t *tty;
{
	int r, sz1, sz2;
	u8_t *cp1, *cp2;
	clock_t to;
	tmr_arg_ut arg;

	if (tty->t_rd_queue == NULL)
		return;
	to= tty->t_termios.c_cc[VTIME];
	if (to == 0)
		return;
	if (tty->t_termios.c_cc[VMIN] != 0)
	{
		r= ib_get_head(tty, &cp1, &sz1, &cp2, &sz2);
		if (r == -1)
			return;
		if (r == 0 && ib_compl_ln(tty))
		{
			ib_pack(tty);
			r= ib_get_head(tty, &cp1, &sz1, &cp2, &sz2);
			if (r <= 0)
				return;
		}
		to= tty->t_rd_char_time + to*HZ/10;
	}
	else
		to= tty->t_rd_req_time + to*HZ/10;

	tty->t_rd_timer_set= 1;
	arg.ta_ptr= tty;
	tmrs_settimer(tty->t_tmr_cntxt, &tty->t_rd_timer, to, 
		rd_timer_to, arg);
#if DEBUG
	printW(); printf("set timer\n");
#endif
}


/*===========================================================================*
 *				rd_timer_to				     *
 *===========================================================================*/
PRIVATE void rd_timer_to(tp, arg)
struct tmrs *tp;
tmr_arg_ut arg;
{
	tty_t *tty;
	int min, r;
	u8_t *cp1, *cp2;
	int sz1, sz2;
	clock_t now, to;

#if DEBUG
	printW(); printf("rd_timer_to\n");
#endif

	tty= arg.ta_ptr;
	assert(tp == &tty->t_rd_timer);

	tty->t_rd_timer_set= 0;
	if (tty->t_rd_queue == NULL)
		return;
	if (is_canon(tty))
		return;
	to= tty->t_termios.c_cc[VTIME];
	min= tty->t_termios.c_cc[VMIN];
	if (to == 0)
		return;
	if (min != 0)
	{
		r= ib_get_head(tty, &cp1, &sz1, &cp2, &sz2);
		if (r == -1)
			return;
		if (r == 0 && ib_compl_ln(tty))
		{
			ib_pack(tty);
			r= ib_get_head(tty, &cp1, &sz1, &cp2, &sz2);
			if (r <= 0)
				return;
		}
		to= tty->t_rd_req_time + to*HZ/10;
	}
	else
		to= tty->t_rd_char_time + to*HZ/10;
	now= get_uptime();
	if (to > now)
	{
		set_rd_timer(tty);
		return;
	}
	if (min == 0)
	{
		read_reply(tty, 0);
		if (tty->t_rd_queue != NULL)
			set_rd_timer(tty);
	}
	else
	{
		/* The data to be transferd */
		tty->t_termios.c_cc[VMIN]= 1;
		restart_read_noncanon(tty);
		tty->t_termios.c_cc[VMIN]= min;
	}
}


/*===========================================================================*
 *				input_canon				     *
 *===========================================================================*/
PRIVATE int input_canon(tty, chr_p)
tty_t *tty;
int *chr_p;
/* Interpret EOF, \n, and LNEXT characters.
 * Returns:	 0 ... use character
 *		 1 ... discard character
 *		-1 ... buffer overflow?
 */
{
	int r, linsz;
	u8_t *cp;

	if (*chr_p == tty->t_termios.c_cc[VEOF])
	{
		r= ib_newline(tty);
		if (r == -1)
			return -1;

		echo_eof(tty);
		return 1;
	}
	if (*chr_p == tty->t_termios.c_cc[VEOL])
	{
		r= ib_get_space(tty, &cp, &linsz);
		if (r == -1)
			return -1;
		assert(r != 0);
		*cp= *chr_p;
		ib_grow(tty, 1);
		r= ib_newline(tty);
		if (r == -1)
		{
			ib_shrink(tty, 1);
			return -1;
		}
		echo(tty, *chr_p);
		return 1;
	}

	if (*chr_p == '\n')
	{
		r= ib_get_space(tty, &cp, &linsz);
		if (r == -1)
			return -1;
		assert(r != 0);
		*cp= *chr_p;
		ib_grow(tty, 1);
		r= ib_newline(tty);
		if (r == -1)
		{
			ib_shrink(tty, 1);
			return -1;
		}

		echonl(tty);
		return 1;
	}
	if (*chr_p == tty->t_termios.c_cc[VLNEXT] &&
		(tty->t_termios.c_lflag & IEXTEN))
	{
		r= ib_get_space(tty, &cp, &linsz);
		if (r == -1)
			return -1;
		assert(r != 0);
		tty->t_rd_lnext= 1;
		output(tty, '^');
		output(tty, '\10');
		return 1;
	}
	return 0;
}


/*===========================================================================*
 *				do_sig					     *
 *===========================================================================*/
PRIVATE void do_sig(tty, sig)
tty_t *tty;
int sig;
{
	if (tty->t_sesldr)
		cause_sig(tty->t_sesldr, sig);
	if (!(tty->t_termios.c_lflag & NOFLSH))
	{
		tty->t_flags &= ~TF_XON_OUTPUT_OFF;
		ib_flush(tty);
		ob_flush(tty);
		tty_get(tty);
	}
}


/*===========================================================================*
 *				put_canon				     *
 *===========================================================================*/
PRIVATE int put_canon(tty, buf, siz)
tty_t *tty;
char *buf;
int siz;
{
/* We got some data from the device. */
	mq_t *mq;
	int chr;
	int isig, xon, xany;
	int offset, r, linsz;
	u8_t *cp;
	
	tty->t_flags &= ~TF_READ_INITIATIVE;

	if (siz == 0)
	{
		/* This is a flush command. */
		restart_read_canon(tty);
		return 0;
	}
		
	assert(is_canon(tty));
	isig= tty->t_termios.c_lflag & ISIG;
	xon= tty->t_termios.c_iflag & IXON;
	xany= tty->t_termios.c_iflag & IXANY;
	offset= 0;
	while(offset < siz)
	{
		chr= buf[offset];

		r= ib_get_space(tty, &cp, &linsz);
		if (r == -1)
		{
			restart_read_canon(tty);
			r= ib_get_space(tty, &cp, &linsz);
			if (r == -1)
				break;		/* Input queue is full. */
		}
		
		/* Keep track of the offset of input characters from
		 * the left margin to be able to backspace over tabs, 
		 * and redraw the current line if it has been messed up
		 * with output.
		 */
		if (r > 0 && linsz == 0)
			tty->t_rd_lineoffs= tty->t_wr_position;

		/* LNEXT overides any special processing. Also process
		 * chr if chr == _POSIX_VDISABLE. */
		if (tty->t_rd_lnext || chr == _POSIX_VDISABLE)
		{
			if (r == 0)
			{
				output(tty, '\a');	/* ^G ring bell. */
				offset++;
				continue;
			}
			*cp= chr;
			ib_grow(tty, 1);
			if (chr == '\377' && (tty->t_termios.c_iflag &
				PARMRK))
			{
				r= ib_get_space(tty, &cp, &linsz);
				if (r == -1)
				{
					restart_read_canon(tty);
					r= ib_get_space(tty, &cp, &linsz);
					if (r == -1)
					{
						ib_shrink(tty, 1);
						break;
					}
				}
				if (r == 0)
				{
					output(tty, '\a');
					offset++;
					ib_shrink(tty, 1);
					continue;
				}
				*cp= chr;
				ib_grow(tty, 1);
				echo(tty, chr);
			}
			echo(tty, chr);
			tty->t_rd_lnext= 0;
			offset++;
			continue;
		}

		/* Check the START/STOP characters in case IXANY
		 * is enabled (in that case any character acts like START 
		 * character).
		 */
		if (xon)
		{
			if (chr == tty->t_termios.c_cc[VSTOP])
			{
				tty->t_flags |= TF_XON_OUTPUT_OFF;
				offset++;
				continue;
			}
			if (chr == tty->t_termios.c_cc[VSTART] ||
				(tty->t_flags & TF_XON_OUTPUT_OFF) && xany)
			{
				tty->t_flags &= ~TF_XON_OUTPUT_OFF;
				/* Pretend the device needs input. */
				tty_get(tty);
				if (chr == tty->t_termios.c_cc[VSTART]) {
					offset++;
					continue;
				}
			}
		}

		if (isig)
		{
			if (chr == tty->t_termios.c_cc[VINTR])
			{
				do_sig(tty, SIGINT);
				echo(tty, chr);
				offset++;
				continue;
			}
			if (chr == tty->t_termios.c_cc[VQUIT])
			{
				do_sig(tty, SIGQUIT);
				echo(tty, chr);
				offset++;
				continue;
			}
		}
		if (chr == tty->t_termios.c_cc[VDISCARD] &&
			(tty->t_termios.c_lflag & IEXTEN))
		{
			echo(tty, chr);
			if (tty->t_termios.c_lflag & LFLUSHO)
				tty->t_termios.c_lflag &= ~LFLUSHO;
			else
				tty->t_termios.c_lflag |= LFLUSHO;
			offset++;
			continue;
		}
		r= input_canon_zero_space(tty, chr);
		if (r == 1)
		{
			offset++;
			continue;
		}
		assert(r == 0);
		
		if (linsz == 255)
		{
			output(tty, '\a');		/* ^G ring bell. */
			offset++;
			continue;
		}

		if (chr == '\r') 
		{
			if (tty->t_termios.c_iflag & IGNCR) {
				offset++;
				continue;
			}
			if (tty->t_termios.c_iflag & ICRNL)
				chr = '\n';
		}
		else if (chr == '\n')
		{
			if (tty->t_termios.c_iflag & INLCR)
				chr = '\r';
		}
		r= input_canon(tty, &chr);
		if (r == 1)
		{
			offset++;
			continue;
		}
		if (r == -1)
			break;
		assert(r == 0);

		echo(tty, chr);
		*cp= chr;
		ib_grow(tty, 1);
		offset++;
		continue;
	}
	if (offset != siz)
		tty->t_flags |= TF_READ_INITIATIVE;
	if (!ob_empty(tty) && tty->t_flags & TF_WRITE_INITIATIVE)
	{
		tty_get(tty);			/* Flush output by echos. */
	}
	return offset;
}


/*===========================================================================*
 *				put_noncanon				     *
 *===========================================================================*/
PRIVATE int put_noncanon(tty, buf, siz)
tty_t *tty;
char *buf;
int siz;
{
/* We got some data from the device. */
	mq_t *mq;
	int chr;
	int isig, xon, xany;
	int offset, r, linsz;
	u8_t *cp;

	tty->t_flags &= ~TF_READ_INITIATIVE;

	if (siz == 0)
	{
		/* Flush */
		if (tty->t_rd_queue != NULL && tty->t_rd_offset != 0)
			read_reply(tty, tty->t_rd_offset);
		if (tty->t_rd_queue != NULL)
			restart_read_noncanon(tty);

		/* Set timers, if VTIME !=0. If VMIN == 0, read requests
		 * are timed. If VMIN != 0 we need to update timing information
		 * if at least one character is present in the buffer.
		 * If a timer is set, we let the timer expire early.
		 */
		if (tty->t_termios.c_cc[VTIME] != 0 &&
			tty->t_termios.c_cc[VMIN] != 0)
		{
			tty->t_rd_char_time= get_uptime();
			set_rd_timer(tty);
		}
		return 0;
	}
		
	offset= 0;
	assert(!is_canon(tty));
	isig= tty->t_termios.c_lflag & ISIG;
	xon= tty->t_termios.c_iflag & IXON;
	xany= tty->t_termios.c_iflag & IXANY;
	while(offset < siz)
	{
		chr= buf[offset];

		r= ib_get_space(tty, &cp, &linsz);
		if (r == -1)
		{
			restart_read_noncanon(tty);
			r= ib_get_space(tty, &cp, &linsz);
			if (r == -1)
				break;		/* Input queue is full. */
		}
		if (r == 0)
		{
			r= ib_newline(tty);
			if (r == -1)
			{
				restart_read_noncanon(tty);
				r= ib_newline(tty);
				if (r == -1)
					break;
			}
			continue;
		}

		/* LNEXT overides any special processing. Also process
		 * chr if chr == _POSIX_VDISABLE. */
		if (tty->t_rd_lnext || chr == _POSIX_VDISABLE)
		{
			*cp= chr;
			ib_grow(tty, 1);

			if (chr == '\377' && (tty->t_termios.c_iflag &
				PARMRK))
			{
				r= ib_get_space(tty, &cp, &linsz);
				if (r == -1)
				{
					restart_read_noncanon(tty);
					r= ib_get_space(tty, &cp, &linsz);
					if (r == -1)
					{
						ib_shrink(tty, 1);
						break;
					}
				}
				if (r == 0)
				{
					r= ib_newline(tty);
					if (r == -1)
					{
						restart_read_noncanon(tty);
						r= ib_newline(tty);
						if (r == -1)
						{
							ib_shrink(tty, 1);
							break;
						}
					}
				}
				*cp= chr;
				ib_grow(tty, 1);
				echo(tty, chr);
			}
			echo(tty, chr);
			tty->t_rd_lnext= 0;
			offset++;
			continue;
		}

		/* Check the START/STOP characters in case IXANY
		 * is enabled (in that case any character acts like START 
		 * character).
		 */
		if (xon)
		{
			if (chr == tty->t_termios.c_cc[VSTOP])
			{
				tty->t_flags |= TF_XON_OUTPUT_OFF;
				offset++;
				continue;
			}
			if (chr == tty->t_termios.c_cc[VSTART] ||
				(tty->t_flags & TF_XON_OUTPUT_OFF) && xany)
			{
				tty->t_flags &= ~TF_XON_OUTPUT_OFF;
				offset++;
				/* Pretent the device needs input. */
				tty_get(tty);
				continue;
			}
		}

		if (chr == tty->t_termios.c_cc[VDISCARD] &&
			(tty->t_termios.c_lflag & IEXTEN))
		{
			echo(tty, chr);
			if (tty->t_termios.c_lflag & LFLUSHO)
				tty->t_termios.c_lflag &= ~LFLUSHO;
			else
				tty->t_termios.c_lflag |= LFLUSHO;
			offset++;
			continue;
		}

		if (isig)
		{
			if (chr == tty->t_termios.c_cc[VINTR])
			{
				do_sig(tty, SIGINT);
				echo(tty, chr);
				offset++;
				continue;
			}
			if (chr == tty->t_termios.c_cc[VQUIT])
			{
				do_sig(tty, SIGQUIT);
				echo(tty, chr);
				offset++;
				continue;
			}
		}

		if (chr == tty->t_termios.c_cc[VLNEXT] &&
			(tty->t_termios.c_lflag & IEXTEN))
		{
			tty->t_rd_lnext= 1;
			offset++;
			if (tty->t_termios.c_lflag & ECHO)
			{
				output(tty, '^');
				output(tty, '\10');
			}
			continue;
		}

		if (chr == '\r') 
		{
			if (tty->t_termios.c_iflag & IGNCR) {
				offset++;
				continue;
			}
			if (tty->t_termios.c_iflag & ICRNL)
				chr = '\n';
		}
		else if (chr == '\n')
		{
			if (tty->t_termios.c_iflag & INLCR)
				chr = '\r';
		}
		if (chr == '\n')
			echonl(tty);
		else
			echo(tty, chr);

		*cp= chr;
		ib_grow(tty, 1);

		offset++;
		continue;
	}
	if (offset != siz)
		tty->t_flags |= TF_READ_INITIATIVE;
	if (!ob_empty(tty) && tty->t_flags & TF_WRITE_INITIATIVE)
	{
		tty_get(tty);			/* Flush output by echos. */
	}
	return offset;
}


/*===========================================================================*
 *				put_timed_raw				     *
 *===========================================================================*/
PRIVATE int put_timed_raw(tty, buf, siz)
tty_t *tty;
char *buf;
int siz;
/* Raw input, no need for processing. Input is directly copied to user space
 * if read requests are present, and if enough data is present to satisfy
 * VMIN. Timers are set when necessary,
 */
{
	int cnt, offset;
	int rd_offset, rd_size;
	int r, r0, min, linsz, min_ok;
	u8_t *cp, *cp1, *cp2;
	int sz, sz1, sz2;
	phys_bytes phys_local, phys_user;

	tty->t_flags &= ~TF_READ_INITIATIVE;

	if (siz == 0)
	{
		/* Flush */
		if (tty->t_rd_queue != NULL && tty->t_rd_offset != 0)
			read_reply(tty, tty->t_rd_offset);

		/* Set timers, if VTIME !=0. If VMIN == 0, read requests
		 * are timed. If VMIN != 0 we need to update timing information
		 * if at least one character is present in the buffer.
		 * If a timer is set, we let the timer expire early.
		 */
		if (tty->t_termios.c_cc[VTIME] != 0 &&
			tty->t_termios.c_cc[VMIN] != 0)
		{
			tty->t_rd_char_time= get_uptime();
			set_rd_timer(tty);
		}
		return 0;
	}
	offset= 0;
	if (tty->t_rd_queue != NULL)
	{
		min= tty->t_termios.c_cc[VMIN];
		if (min == 0 && tty->t_termios.c_cc[VTIME] != 0)
			min= 1;

		r= ib_get_head(tty, &cp1, &sz1, &cp2, &sz2);
		r0= (r<0 ? 0 : r);
		min_ok= 0;
		if (r0 + siz >= min)
			min_ok= 1;
		else if (ib_compl_ln(tty))
		{
			ib_pack(tty);
			r= ib_get_head(tty, &cp1, &sz1, &cp2, &sz2);
			r0= (r<0 ? 0 : r);
			if (r0 + siz >= min)
				min_ok= 1;
		}

		if (!min_ok)
			goto not_min_ok;

		/* First copy data from the buffer */
		while(tty->t_rd_queue)
		{
			if (r == -1)
				break;

			if (r == 0)
			{
				/* Left over from canonical mode. */
				ib_delete(tty, 0);
				r= ib_get_head(tty, &cp1, &sz1, &cp2, &sz2);
				continue;
			}

			rd_size= tty->t_rd_size;
			rd_offset= tty->t_rd_offset;
			cnt= rd_size-rd_offset;
			if (cnt > sz1)
				cnt= sz1;
			phys_user= tty->t_rd_phys + rd_offset;
			phys_local= vir2phys(cp1);
			phys_copy(phys_local, phys_user, (phys_bytes)rd_size);
			ib_delete(tty, cnt);
			rd_offset += cnt;
			if (rd_offset < rd_size && sz2 != 0)
			{
				cnt= rd_size-rd_offset;
				if (cnt > sz2)
					cnt= sz2;
				phys_user= tty->t_rd_phys + rd_offset;
				phys_local= vir2phys(cp2);
				phys_copy(phys_local, phys_user,
					(phys_bytes)rd_size);
				ib_delete(tty, cnt);
				rd_offset += cnt;
			}
			if (rd_offset == rd_size)
				read_reply(tty, rd_offset);
			else
				tty->t_rd_offset= rd_offset;

			r= ib_get_head(tty, &cp1, &sz1, &cp2, &sz2);
		}

		/* Next, try to copy the new data directly to the user.  */
		phys_local= vir2phys(buf);
		while (tty->t_rd_queue != NULL && offset < siz)
		{
			rd_offset= tty->t_rd_offset;
			rd_size= tty->t_rd_size;
			cnt= rd_size-rd_offset;
			if (cnt > siz-offset)
				cnt= siz-offset;
			phys_copy(phys_local + offset,
				tty->t_rd_phys+rd_offset, cnt);
			offset += cnt;
			rd_offset += cnt;
			if (rd_offset == rd_size)
				read_reply(tty, rd_offset);
			else
				tty->t_rd_offset= rd_offset;
		}
	}
not_min_ok:
	/* Buffer the remaining data */
	while (offset < siz)
	{
		r= ib_get_space(tty, &cp, &linsz);
		if (r == 0)
		{
			r= ib_newline(tty);
			if (r == -1)
				break;		/* Input buffer is full */
			r= ib_get_space(tty, &cp, &linsz);
		}
		if (r == -1)
			break;			/* Input buffer is full. */
		cnt= siz-offset;
		if (cnt > r)
			cnt= r;
		memcpy(cp, buf+offset, cnt);
		offset += cnt;
		ib_grow(tty, cnt);
	}
	if (offset != siz)
		tty->t_flags |= TF_READ_INITIATIVE;
	return offset;
}


/*===========================================================================*
 *				put_raw					     *
 *===========================================================================*/
PRIVATE int put_raw(tty, buf, siz)
tty_t *tty;
char *buf;
int siz;
/* Raw input, no need for processing. Input is directly copied to user space
 * if read requests are present.
 */
{
	int cnt, offset;
	int rd_offset, rd_size;
	int r, linsz;
	u8_t *cp;
	phys_bytes phys_local;

	tty->t_flags &= ~TF_READ_INITIATIVE;

	if (siz == 0)
	{
		/* Flush */
		if (tty->t_rd_queue != NULL && tty->t_rd_offset != 0)
			read_reply(tty, tty->t_rd_offset);
		return 0;
	}
	offset= 0;
	if (tty->t_rd_queue != NULL)
	{
		/* Try to copy the data directly to the user */
		phys_local= vir2phys(buf);
		while (tty->t_rd_queue != NULL && offset < siz)
		{
			rd_offset= tty->t_rd_offset;
			rd_size= tty->t_rd_size;
			cnt= rd_size-rd_offset;
			if (cnt > siz-offset)
				cnt= siz-offset;
			phys_copy(phys_local + offset,
				tty->t_rd_phys+rd_offset, cnt);
			offset += cnt;
			rd_offset += cnt;
			if (rd_offset == rd_size)
				read_reply(tty, rd_offset);
			else
				tty->t_rd_offset= rd_offset;
		}
	}
	while (offset < siz)
	{
		r= ib_get_space(tty, &cp, &linsz);
		if (r == 0)
		{
			r= ib_newline(tty);
			if (r == -1)
				break;		/* Input buffer is full */
			r= ib_get_space(tty, &cp, &linsz);
		}
		if (r == -1)
			break;			/* Input buffer is full. */
		cnt= siz-offset;
		if (cnt > r)
			cnt= r;
		memcpy(cp, buf+offset, cnt);
		offset += cnt;
		ib_grow(tty, cnt);
	}
	if (offset != siz)
		tty->t_flags |= TF_READ_INITIATIVE;
	return offset;
}


/*===========================================================================*
 *				put_discard				     *
 *===========================================================================*/
PRIVATE int put_discard(tty, buf, siz)
tty_t *tty;
char *buf;
int siz;
/* This routine is silly. It is used to simulate disabling the receiver if
 * CREAD is not set.
 */
{
	tty->t_flags &= ~TF_READ_INITIATIVE;

	return siz;
}


/*
 * Input buffer routines
 */

/*===========================================================================*
 *				ib_flush				     *
 *===========================================================================*/
/* Flush the input buffer */
PRIVATE void ib_flush(tty)
tty_t *tty;
{
	tty->t_rd_begin_n= 0;
	tty->t_rd_curr_n= 0;
	tty->t_rd_end_n= 0;
	tty->t_rd_lnext= 0;			/* No LNEXT char seen yet. */
	if (tty->t_flags & TF_READ_INITIATIVE)
	{
		tty->t_flags &= ~TF_READ_INITIATIVE;
		(*tty->t_get)(tty->t_line);
	}	
}


/*===========================================================================*
 *				ib_compl_ln				     *
 *===========================================================================*/
PRIVATE int ib_compl_ln(tty)
tty_t *tty;
/* Return non-zero if a completed line is available. By definition, this is
 * the case if begin != curr.
 */
{
	return tty->t_rd_begin_n != tty->t_rd_curr_n;
}


/*===========================================================================*
 *				ib_get_head				     *
 *===========================================================================*/
PRIVATE int ib_get_head(tty, cp1, sz1, cp2, sz2)
tty_t *tty;
u8_t **cp1;
int *sz1;
u8_t **cp2;
int *sz2;
/* Return the number of bytes in the first line, and fill in pointers and
 * sizes describing the location of the data. Return -1 if the buffer is 
 * empty.
 */
{
	int begin, bufsize, r;

	begin= tty->t_rd_begin_n;
	bufsize= tty->t_rd_buf_siz;

	if (begin == tty->t_rd_end_n)
		return -1;
	r= tty->t_rd_buf[begin];
	begin++;
	if (begin >= bufsize)
		begin -= bufsize;
	*cp1= &tty->t_rd_buf[begin];
	if (begin + r >= bufsize)
	{
		*sz1= bufsize - begin;
		*cp2= tty->t_rd_buf;
		*sz2= begin + r - bufsize;
	}
	else
	{
		*sz1= r;
		*cp2= NULL;
		*sz2= 0;
	}
	return r;
}


/*===========================================================================*
 *				ib_get_tail				     *
 *===========================================================================*/
PRIVATE int ib_get_tail(tty, cp1, sz1, cp2, sz2)
tty_t *tty;
u8_t **cp1;
int *sz1;
u8_t **cp2;
int *sz2;
/* Return the number of bytes in the current (last) line, and fill in pointers
 * and sizes describing the location of the data. Return -1 if the buffer is 
 * empty.
 */
{
	int curr, bufsize, r;

	curr= tty->t_rd_curr_n;
	bufsize= tty->t_rd_buf_siz;

	if (curr == tty->t_rd_end_n)
		return -1;
	r= tty->t_rd_buf[curr];
	curr++;
	if (curr >= bufsize)
		curr -= bufsize;
	*cp1= &tty->t_rd_buf[curr];
	if (curr + r >= bufsize)
	{
		*sz1= bufsize - curr;
		*cp2= tty->t_rd_buf;
		*sz2= r - (bufsize-curr);
	}
	else
	{
		*sz1= r;
		*cp2= NULL;
		*sz2= 0;
	}
	return r;
}


/*===========================================================================*
 *				ib_get_space				     *
 *===========================================================================*/
PRIVATE int ib_get_space(tty, cp, linsz)
tty_t *tty;
u8_t **cp;
int *linsz;
/* Return the number of bytes that can be added to the current line. -1 if
 * the buffer is full. Otherwise the amount is limited by: the amount of
 * free space in the buffer, the maximum line length, and by buffer
 * bounderies.
 */
{
	int begin, curr, end, bufsize, size;

	begin= tty->t_rd_begin_n;
	curr= tty->t_rd_curr_n;
	end= tty->t_rd_end_n;
	bufsize= tty->t_rd_buf_siz;

	if (curr == end)
	{
		/* Try to create a new line. */
		end++;
		if (end >= bufsize)
			end -= bufsize;
		if (end == begin)
			return -1;	/* Buffer is full */
		tty->t_rd_end_n= end;
		tty->t_rd_buf[curr]= 0;
	}

	/* If begin>end then the amount of new data is limited by the space
	 * between end and begin, else the amount is limited by the end of
	 * the buffer.
	 */
	if (begin > end)
		size= begin-end-1;
	else
	{
		size= bufsize-end;
		if (begin == 0)
		{
			size--;
			assert(size >= 0);
		}
	}
	if (size == 0)
		return -1;
	*cp= &tty->t_rd_buf[end];
	*linsz= tty->t_rd_buf[curr];
	if (size > 255-*linsz)
		size= 255-*linsz;
	return size;
}


/*===========================================================================*
 *				ib_delete				     *
 *===========================================================================*/
PRIVATE int ib_delete(tty, size)
tty_t *tty;
int size;
{
	int begin, curr, bufsize, lnsize, obegin;

	begin= tty->t_rd_begin_n;
	curr= tty->t_rd_curr_n;
	bufsize= tty->t_rd_buf_siz;
	assert(begin != tty->t_rd_end_n);
	lnsize= tty->t_rd_buf[begin];

	assert(size <= lnsize ||
 (printW(), printf("size= %d, lnsize= %d\n", size, lnsize), 0));

	obegin= begin;
	if (size == lnsize)
	{
		begin += size+1;
		if (begin >= bufsize)
			begin -= bufsize;
		if (obegin == curr)
		{
			curr= begin;
			tty->t_rd_curr_n= curr;
		}
	}
	else
	{
		lnsize -= size;
		begin += size;
		if (begin >= bufsize)
			begin -= bufsize;
		if (obegin == curr)
		{
			curr= begin;
			tty->t_rd_curr_n= curr;
		}
		tty->t_rd_buf[begin]= lnsize;
	}
	tty->t_rd_begin_n= begin;
}


/*===========================================================================*
 *				ib_grow					     *
 *===========================================================================*/
PRIVATE int ib_grow(tty, size)
tty_t *tty;
int size;
{
	int curr, end, bufsize;

	curr= tty->t_rd_curr_n;
	end= tty->t_rd_end_n;
	bufsize= tty->t_rd_buf_siz;
	tty->t_rd_buf[curr] += size;
	end += size;
	if (end >= bufsize)
		end -= bufsize;
	tty->t_rd_end_n= end;
}


/*===========================================================================*
 *				ib_shrink				     *
 *===========================================================================*/
PRIVATE int ib_shrink(tty, size)
tty_t *tty;
int size;
{
	int curr, end, bufsize;

	curr= tty->t_rd_curr_n;
	end= tty->t_rd_end_n;
	bufsize= tty->t_rd_buf_siz;
	tty->t_rd_buf[curr] -= size;
	end -= size;
	if (end < 0)
		end += bufsize;
	tty->t_rd_end_n= end;
}


/*===========================================================================*
 *				ib_newline				     *
 *===========================================================================*/
PRIVATE int ib_newline(tty)
tty_t *tty;
{
	int curr, end, bufsize;

	end= tty->t_rd_end_n;
	bufsize= tty->t_rd_buf_siz;
	curr= end;
	end++;
	if (end >= bufsize)
		end -= bufsize;

	if (end == tty->t_rd_begin_n)
		return -1;
	tty->t_rd_curr_n= curr;
	tty->t_rd_buf[curr]= 0;
	tty->t_rd_end_n= end;

	return 0;
}


/*===========================================================================*
 *				ib_pack					     *
 *===========================================================================*/
PRIVATE void ib_pack(tty)
tty_t *tty;
{
	int begin, end, src, src_l, src_p, dst, dst_l, dst_p;
	int bufsize, cnt;

	begin= tty->t_rd_begin_n;
	end= tty->t_rd_end_n;
	bufsize= tty->t_rd_buf_siz;
	if (begin == tty->t_rd_curr_n)
		return;				/* Nothing to pack */
	if (tty->t_rd_buf[begin] == 255)
		return;				/* No reason to pack */
	dst= begin;
	src= begin + 1 + tty->t_rd_buf[begin];
	if (src >= bufsize)
		src -= bufsize;
	while (src != end)
	{
		dst_l= tty->t_rd_buf[dst];
		if (dst_l == 255)
		{
			dst += 1 + 255;
			if (dst >= bufsize)
				dst -= bufsize;
			tty->t_rd_buf[dst]= 0;
			continue;
		}
		src_l= tty->t_rd_buf[src];
		cnt= src_l;
		if (cnt > 255-dst_l)
			cnt= 255-dst_l;
		if (cnt != 0)
		{
			dst_p= dst+1+dst_l;
			if (dst_p >= bufsize)
				dst_p -= bufsize;
			if (bufsize - dst_p < cnt)
				cnt= bufsize - dst_p;
			src_p= src+1;
			if (src_p >= bufsize)
				src_p -= bufsize;
			if (bufsize - src_p < cnt)
				cnt= bufsize - src_p;
			memmove(&tty->t_rd_buf[dst_p],
				&tty->t_rd_buf[src_p], cnt);
			tty->t_rd_buf[dst] += cnt;
		}
		if (cnt == src_l)
			src += cnt+1;
		else
			src += cnt;
		if (src >= bufsize)
			src -= bufsize;
	}
	tty->t_rd_curr_n= dst;
	dst += 1 + tty->t_rd_buf[dst];
	if (dst >= bufsize)
		dst -= bufsize;
	tty->t_rd_end_n= dst;
}


/*===========================================================================*
 *				ib_check				     *
 *===========================================================================*/
PRIVATE int ib_check(tty)
tty_t *tty;
{
	int begin, curr, end;
	int max, ls, chr;
	int curr_found;

	begin= tty->t_rd_begin_n;
	if (begin < 0 || begin >= tty->t_rd_buf_siz)
	{
		printf("t_rd_begin too small or too large: ! 0 <= %x < %x\n", 
			begin, tty->t_rd_buf_siz);
		return 0;
	}
	curr= tty->t_rd_curr_n;
	if (curr < 0 || curr >= tty->t_rd_buf_siz)
	{
		printf("t_rd_curr too small or too large: ! 0 <= %x < %x\n", 
			curr, tty->t_rd_buf_siz);
		return 0;
	}
	end= tty->t_rd_end_n;
	if (end < 0 || end >= tty->t_rd_buf_siz)
	{
		printf("t_rd_end too small or too large: ! 0 <= %x < %x\n", 
			end, tty->t_rd_buf_siz);
		return 0;
	}
	max= tty->t_rd_buf_siz;
	curr_found= 0;
	while (max >= 0)
	{
		if (begin == curr)
			curr_found= 1;
		if (begin == end)
			break;
		ls= tty->t_rd_buf[begin];
		if (ls+1 > max)
		{
			printf("line at %d too large: %d, %d\n",
				begin, ls, max);
			return 0;
		}
		max -= ls+1;
		begin += ls+1;
		if (begin >= tty->t_rd_buf_siz)
			begin -= tty->t_rd_buf_siz;
	}
	if (!curr_found)
	{
		printf("current not found\n");
		return 0;
	}
	if (begin != end)
	{
		printf("end not found\n");
		return 0;
	}
	return 1;
}


/*===========================================================================*
 *				ib_print				     *
 *===========================================================================*/
PRIVATE void ib_print(tty)
tty_t *tty;
{
	int begin, curr, end;

	begin= tty->t_rd_begin_n;
	curr= tty->t_rd_curr_n;
	end= tty->t_rd_end_n;

	printf("b/c/e: %d", begin);
	if (begin != curr)
		printf("(%d)", tty->t_rd_buf[begin]);
	printf("/%d", curr);
	if (curr != end)
		printf("(%d)", tty->t_rd_buf[curr]);
	printf("/%d", end);
}


/*
 * Output buffer routines.
 */

/*===========================================================================*
 *				ob_empty				     *
 *===========================================================================*/
PRIVATE int ob_empty(tty)
tty_t *tty;
/* Return true iff the output buffer is empty. */
{
	return tty->t_wr_begin_n == tty->t_wr_end_n;
}


/*===========================================================================*
 *				ob_flush				     *
 *===========================================================================*/
PRIVATE void ob_flush(tty)
tty_t *tty;
{
	tty->t_wr_begin_n= 0;
	tty->t_wr_end_n= 0;
	if (tty->t_flags & TF_WRITE_INITIATIVE)
	{
		tty_get(tty);	/* This function is normally called */
				/* by the device to get some data. */
	}
}


/*===========================================================================*
 *				ob_space				     *
 *===========================================================================*/
PRIVATE int ob_space(tty, cp1, sz1, cp2, sz2)
tty_t *tty;
u8_t **cp1;
int *sz1;
u8_t **cp2;
int *sz2;
{
	int r, begin, end, bufsize;

	begin= tty->t_wr_begin_n;
	end= tty->t_wr_end_n;
	bufsize= tty->t_wr_buf_siz;

	if (end < begin)
	{
		r= begin-end-1;
		*cp1= &tty->t_wr_buf[end];
		*sz1= r;
		*cp2= NULL;
		*sz2= 0;
		return r;
	}
	if (begin == 0)
	{
		r= bufsize-end-1;
		*cp1= &tty->t_wr_buf[end];
		*sz1= r;
		*cp2= NULL;
		*sz2= 0;
		return r;
	}
	r= bufsize-end;
	*cp1= &tty->t_wr_buf[end];
	*sz1= r;
	*cp2= &tty->t_wr_buf[0];
	*sz2= begin-1;
	r += begin-1;
	return r;
}


/*===========================================================================*
 *				ob_get_head				     *
 *===========================================================================*/
PRIVATE int ob_get_head(tty, cp)
tty_t *tty;
u8_t **cp;
{
	int r, begin, end, bufsize;

	begin= tty->t_wr_begin_n;
	end= tty->t_wr_end_n;
	bufsize= tty->t_wr_buf_siz;

	*cp= &tty->t_wr_buf[begin];
	if (begin <= end)
		r= end-begin;
	else
		r= bufsize-begin;
	return r;
}


/*===========================================================================*
 *				ob_grow					     *
 *===========================================================================*/
PRIVATE void ob_grow(tty, n)
tty_t *tty;
int n;
{
	int end, bufsize;

	end= tty->t_wr_end_n;
	bufsize= tty->t_wr_buf_siz;

	end += n;
	if (end >= bufsize)
		end -= bufsize;
	assert(end >= 0 && end < bufsize);
	tty->t_wr_end_n= end;
}


/*===========================================================================*
 *				ob_delete				     *
 *===========================================================================*/
PRIVATE void ob_delete(tty, n)
tty_t *tty;
int n;
{
	int begin, bufsize;

	begin= tty->t_wr_begin_n;
	bufsize= tty->t_wr_buf_siz;

	begin += n;
	if (begin >= bufsize)
		begin -= bufsize;
	assert(begin >= 0 && begin < bufsize);
	tty->t_wr_begin_n= begin;
}

/*
 * $PchId: tty.c,v 1.5 1996/01/19 22:12:21 philip Exp $
 */
