/*
kernel/timer.c

Module for synchronous and asynchronous timers.

Created:	Jan 19, 1993 by Philip Homburg
*/


#include "kernel.h"
#include "assert.h"
INIT_ASSERT
#include "timer.h"

PRIVATE tmra_ut *tac_chain;	/* = NULL */
PUBLIC clock_t tac_exp_time;	/* = 0 */

FORWARD void s_expire _ARGS(( tmra_ut *tp, tmr_arg_ut arg ));

PUBLIC void tmra_inittimer(tp)
tmra_ut *tp;
{
	tp->tmra_flags= 0;
	tp->tmra_next= NULL;
}

PUBLIC void tmra_settimer(tp, exp_time, fp, arg)
tmra_ut *tp;
clock_t exp_time;
tmra_func_ut fp;
tmr_arg_ut arg;
{
	tmra_ut *prev, *curr;

	if (tp->tmra_flags & TAF_SET)
		tmra_clrtimer(tp);
	tp->tmra_flags |= TAF_SET;
	if (exp_time == 0)
		exp_time= 1;
	tp->tmra_exp_time= exp_time;
	tp->tmra_func= fp;
	tp->tmra_arg= arg;

	for (prev= NULL, curr= tac_chain; curr != NULL; prev= curr,
							curr= curr->tmra_next)
	{
		if (exp_time < curr->tmra_exp_time)
			break;

		/* Continue */
	}
	tp->tmra_next= curr;
	if (prev == NULL)
	{
		tac_chain= tp;
		tac_exp_time= exp_time;
	}
	else
	{
		prev->tmra_next= tp;
	}
}

PUBLIC void tmra_clrtimer(tp)
tmra_ut *tp;
{
	tmra_ut *prev, *curr;

	if (!(tp->tmra_flags & TAF_SET))
		return;

	for (prev= NULL, curr= tac_chain; curr; prev= curr,
							curr= curr->tmra_next)
	{
		if (curr == tp)
			break;

		/* Continue */
	}
	if (curr == NULL)
		panic("tmra_clrtimer unable to find timer", NO_NUM);
	if (prev == NULL)
		tac_chain= curr->tmra_next;
	else
		prev->tmra_next= curr->tmra_next;
	tp->tmra_flags &= ~TAF_SET;
}

PUBLIC void tmra_exptimers()
{
	clock_t curr_time;
	tmra_ut *tp;

	tac_exp_time= 0;
	curr_time= get_uptime();
	while (tac_chain)
	{
		if (tac_chain->tmra_exp_time > curr_time)
			break;
		tp= tac_chain;
		tac_chain= tp->tmra_next;
		tp->tmra_flags &= ~TAF_SET;
		(*tp->tmra_func)(tp, tp->tmra_arg);
	}
	if (tac_chain)
		tac_exp_time= tac_chain->tmra_exp_time;
}

PUBLIC void tmrs_initcontext(cp, sig_p, task)
tmrs_context_ut *cp;
int *sig_p;
int task;
{
	cp->tsc_chain= NULL;
	cp->tsc_exp_time= 0;
	tmra_inittimer(&cp->tsc_timer);
	cp->tsc_sig_p= sig_p;
	*sig_p= 0;
	cp->tsc_task= task;
}

PUBLIC void tmrs_inittimer(tp)
tmrs_ut *tp;
{
	tp->tmrs_flags= 0;
	tp->tmrs_next= NULL;
}

PUBLIC void tmrs_settimer(cp, tp, exp_time, fp, arg)
tmrs_context_ut *cp;
tmrs_ut *tp;
clock_t exp_time;
tmrs_func_ut fp;
tmr_arg_ut arg;
{
	tmrs_ut *prev, *curr;
	tmr_arg_ut a_arg;

	if (tp->tmrs_flags & TSF_SET)
		tmrs_clrtimer(tp);
	tp->tmrs_flags |= TSF_SET;
	if (exp_time == 0)
		exp_time= 1;
	tp->tmrs_context= cp;
	tp->tmrs_exp_time= exp_time;
	tp->tmrs_func= fp;
	tp->tmrs_arg= arg;

	for (prev= NULL, curr= cp->tsc_chain; curr != NULL; prev= curr,
							curr= curr->tmrs_next)
	{
		if (exp_time < curr->tmrs_exp_time)
			break;

		/* Continue */
	}
	tp->tmrs_next= curr;
	if (prev == NULL)
	{
		cp->tsc_chain= tp;
		if (cp->tsc_exp_time == 0 || exp_time < cp->tsc_exp_time)
		{
			cp->tsc_exp_time= exp_time;
			a_arg.ta_ptr= cp;
			tmra_settimer(&cp->tsc_timer, exp_time, s_expire, 
									a_arg);
		}
	}
	else
	{
		prev->tmrs_next= tp;
	}
}

PUBLIC void tmrs_clrtimer(tp)
tmrs_ut *tp;
{
	tmrs_context_ut *cp;
	tmrs_ut *prev, *curr;

	if (!(tp->tmrs_flags & TSF_SET))
		return;

	cp= tp->tmrs_context;
	for (prev= NULL, curr= cp->tsc_chain; curr; prev= curr,
							curr= curr->tmrs_next)
	{
		if (curr == tp)
			break;

		/* Continue */
	}
	if (curr == NULL)
		panic("tmrs_clrtimer unable to find timer", NO_NUM);
	if (prev == NULL)
		cp->tsc_chain= curr->tmrs_next;
	else
		prev->tmrs_next= curr->tmrs_next;
	tp->tmrs_flags &= ~TSF_SET;
}

PUBLIC void tmrs_exptimers(cp)
tmrs_context_ut *cp;
{
	clock_t curr_time;
	tmrs_ut *tp;
	tmr_arg_ut a_arg;

	*cp->tsc_sig_p= 0;
	curr_time= get_uptime();
	while (cp->tsc_chain)
	{
		if (cp->tsc_chain->tmrs_exp_time > curr_time)
			break;
		tp= cp->tsc_chain;
		cp->tsc_chain= tp->tmrs_next;
		tp->tmrs_flags &= ~TSF_SET;
		(*tp->tmrs_func)(tp, tp->tmrs_arg);
	}
	if (cp->tsc_chain)
	{
		a_arg.ta_ptr= cp;
		tmra_settimer(&cp->tsc_timer, cp->tsc_chain->tmrs_exp_time,
							s_expire, a_arg);
	}
}

PRIVATE void s_expire(tp, arg)
tmra_ut *tp;
tmr_arg_ut arg;
{
	tmrs_context_ut *cp;
	clock_t curr_time, exp_time;
	tmr_arg_ut a_arg;

	cp= arg.ta_ptr;
	assert(&cp->tsc_timer == tp);

	cp->tsc_exp_time= 0;
	if (*cp->tsc_sig_p != 0)
		return;
	if (cp->tsc_chain == NULL)
		return;
	exp_time= cp->tsc_chain->tmrs_exp_time;
	curr_time= get_uptime();
	if (exp_time > curr_time)
	{
		cp->tsc_exp_time= exp_time;
		a_arg.ta_ptr= cp;
		tmra_settimer(&cp->tsc_timer, exp_time, s_expire, a_arg);
		return;
	}

	*cp->tsc_sig_p= 1;
	interrupt(cp->tsc_task);
}

/*
 * $PchId: timer.c,v 1.6 1996/01/19 22:18:12 philip Exp $
 */
