/*
pty.c

This file contains an implementation of pseudo ttys.

Created:	before Dec 28, 1992 by Philip Homburg
 */

#include "kernel.h"
#include <minix/callnr.h>
#include <minix/com.h>
#include <termios.h>
#include "assert.h"
INIT_ASSERT
#include "mq.h"
#include "proc.h"
#include "timer.h"
#include "tty.h"
#include "config.h"

/*
 * Requests:
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_COUNT NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_OPEN    |minor dev  | proc nr   |  fd       |           |           |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_CLOSE   |minor dev  | proc nr   |  fd       |           |           |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_READ    |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_WRITE   |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_IOCTL NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_IOCTL3  |minor dev  | proc nr   |  fd       |  command  | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_OPERATION
 * -------------------------------------------------------------------|
 * | DEV_CANCEL  |minor dev  | proc nr   |  fd       | which operation|
 * |-------------+-----------+-----------+-----------+----------------|
 *
 * Replies:
 *
 *    m_type        REP_PROC_NR   REP_STATUS   REP_REF    REP_OPERATION
 * ----------------------------------------------------------------------|
 * | DEVICE_REPLY |   proc nr   |  status    |  fd     | which operation |
 * |--------------+-------------+------------+---------+-----------------|
 */

#define NPTY	    NR_PTYS	/* Number of ptys (from config.h) */
#define TTY_RB_SIZ	260	/* Should be at least T_RD_BUF_SIZ_MIN */
#define TTY_WB_SIZ	256	/* Should be at least T_WR_BUF_SIZ_MIN */
#define PTY_BUF_SIZ	256

typedef struct pty
{
	int		p_flags;
	mq_t *		p_rd_queue;
	mq_t *		p_rd_queue_tail;
	unsigned	p_rd_offset;
	mq_t *		p_wr_queue;
	mq_t *		p_wr_queue_tail;
	unsigned	p_wr_offset;
	char		p_tty_readbuf[TTY_RB_SIZ];
	char		p_tty_writebuf[TTY_WB_SIZ];
	struct tty	p_tty;
} pty_t;

#define PF_PTY_CLOSED		 0x1
#define PF_TTY_CLOSED		 0x2
#define PF_READ_INITIATIVE	 0x4
#define PF_WRITE_INITIATIVE	 0x8
#define PF_ISTRIP		0x10

PRIVATE pty_t pty_table[NPTY];
PRIVATE mq_t *repl_queue, *repl_queue_tail;
PRIVATE tmrs_context_ut pty_cntxt;
PRIVATE int exp_timers= 0;
PRIVATE char write_buffer[PTY_BUF_SIZ];
PRIVATE int pty_tasknr= ANY;

FORWARD _PROTOTYPE( void pty_init, (void)				);
FORWARD _PROTOTYPE( int reply_queue, (int proc, int ref, int operation)	);
FORWARD _PROTOTYPE( int do_open, (pty_t *pty, message *m )		);
FORWARD _PROTOTYPE( void do_close, (pty_t *pty, message *m )		);
FORWARD _PROTOTYPE( int do_read, (pty_t *pty, mq_t *mq )		);
FORWARD _PROTOTYPE( int do_write, (pty_t *pty, mq_t *mq )		);
FORWARD _PROTOTYPE( int do_ioctl, (pty_t *pty, mq_t *mq )		);
FORWARD _PROTOTYPE( int do_cancel, (pty_t *pty, message *m )		);
FORWARD _PROTOTYPE( void reply, (mq_t *mq, int result, int can_enqueue ));
FORWARD _PROTOTYPE( void pty_get, (int pty_line)			);
FORWARD _PROTOTYPE( int pty_put, (int pty_line, char *buf, int siz)	);
FORWARD _PROTOTYPE( void pty_reply, (int pty_line, mq_t *mq)		);
FORWARD _PROTOTYPE( void pty_hup, (int pty_line)			);
FORWARD _PROTOTYPE( void pty_setattr, (int pty_line,
					struct termios *termios)	);

/*===========================================================================*
 *				pty_task				     *
 *===========================================================================*/
PUBLIC void pty_task()
{
	mq_t *mq;
	int result, send_reply, free_mess;
	int minor, proc;
	pty_t *pty;

	pty_tasknr= proc_number(proc_ptr);

	pty_init();

	while(TRUE)
	{
		mq= mq_get();
		if (!mq)
			panic("out of messages", NO_NUM);
		
		result= receive(ANY, &mq->mq_mess);
		if (result < 0)
			panic("unable to receive: ", result);

		if (mq->mq_mess.m_source == HARDWARE)
		{
			assert(mq->mq_mess.m_type == HARD_INT);
			if (exp_timers)
				tmrs_exptimers(&pty_cntxt);
			mq_free(mq);
			continue;
		}
		if (mq->mq_mess.m_type == DEV_IOCTL)
		{
			/* Compatibility code for old (sgtty) ioctls */
			minor= mq->mq_mess.m2_i1;
			if (minor >= 0 && minor < NPTY)
			{
				/* This packet is handled by the tty library. */
				pty= &pty_table[minor];
				tty_receive(&pty->p_tty, mq);
				continue;
			}
			/* The pty task doesn't implement old ioctls */
			proc= mq->mq_mess.m2_i2;	/* PROC_NR */
			mq->mq_mess.m_type= TASK_REPLY;
			mq->mq_mess.m2_i1= proc;	/* REP_PROC_NR */
			mq->mq_mess.m2_i2= ENOTTY;
			result= send(mq->mq_mess.m_source, &mq->mq_mess);
			assert(result == OK);
			mq_free(mq);
			continue;
		}
		if (repl_queue && mq->mq_mess.m_type != DEV_IOCTL)
		{
			if (mq->mq_mess.m_type == DEV_CANCEL)
			{
				result= reply_queue(mq->mq_mess.NDEV_PROC,
					mq->mq_mess.NDEV_REF, 
					mq->mq_mess.NDEV_OPERATION);
				if (result)
				{
					/* The reply for the canceled request
					 * was already in the queue.
					 */
					mq_free(mq);
					continue;
				}
			}
			else
				reply_queue(ANY, 0, 0);
		}

		minor= mq->mq_mess.NDEV_MINOR;
		if (minor >= 0 && minor < NPTY)
		{
			/* This packet is handled by the tty library. */
			pty= &pty_table[minor];
			tty_receive(&pty->p_tty, mq);
			continue;
		}
		if (!(minor >= 128 && minor < 128 + NPTY))
		{
			reply(mq, ENXIO, FALSE);
			mq_free(mq);
			continue;
		}
		pty= &pty_table[minor - 128];
		switch (mq->mq_mess.m_type)
		{
		case DEV_OPEN:
			result= do_open(pty, &mq->mq_mess);
			send_reply= 1;
			free_mess= 1;
			break;
		case DEV_CLOSE:
			do_close(pty, &mq->mq_mess);
			result= OK;
			send_reply= 1;
			free_mess= 1;
			break;
		case DEV_READ:
			result= do_read(pty, mq);
			assert(result == OK || result == SUSPEND);
			send_reply= (result == SUSPEND);
			free_mess= 0;
			break;
		case DEV_WRITE:
			result= do_write(pty, mq);
			assert(result == OK || result == SUSPEND);
			send_reply= (result == SUSPEND);
			free_mess= 0;
			break;
		case DEV_IOCTL3:
			result= do_ioctl(pty, mq);
			assert(result == OK || result == SUSPEND);
			send_reply= (result == SUSPEND);
			free_mess= 0;
			break;
		case DEV_CANCEL:
			result= do_cancel(pty, &mq->mq_mess);
			send_reply= 1;
			free_mess= 1;
			mq->mq_mess.m_type= mq->mq_mess.NDEV_OPERATION;
			break;
		default:
			panic("got unknown message, type= ", 
				mq->mq_mess.m_type);
		}
		if (send_reply)
		{
			reply(mq, result, FALSE);
		}
		if (free_mess)
			mq_free(mq);
	}
}


/*===========================================================================*
 *				do_open					     *
 *===========================================================================*/
PRIVATE int do_open(pty, m)
pty_t *pty;
message *m;
{
	if ((pty->p_flags & (PF_PTY_CLOSED|PF_TTY_CLOSED)) !=
		(PF_PTY_CLOSED|PF_TTY_CLOSED))
	{
		return EIO;
	}
	pty->p_flags &= ~(PF_PTY_CLOSED|PF_TTY_CLOSED);
	pty->p_flags |= PF_READ_INITIATIVE | PF_WRITE_INITIATIVE;
	return OK;
}


/*===========================================================================*
 *				do_read					     *
 *===========================================================================*/
PRIVATE int do_read(pty, mq)
pty_t *pty;
mq_t *mq;
{
	assert(!(pty->p_flags & PF_PTY_CLOSED));
	if (pty->p_flags & PF_TTY_CLOSED)
	{
		reply(mq, 0, TRUE);
		return OK;
	}
	if (pty->p_rd_queue)
	{
		pty->p_rd_queue_tail->mq_next= mq;
		pty->p_rd_queue_tail= mq;
		assert (!(pty->p_flags & PF_READ_INITIATIVE));
		return SUSPEND;
	}
	pty->p_rd_queue= mq;
	pty->p_rd_queue_tail= mq;
	pty->p_rd_offset= 0;
	if (pty->p_flags & PF_READ_INITIATIVE)
	{
		tty_get(&pty->p_tty);
	}
	if (pty->p_rd_queue)
	{
		return SUSPEND;
	}
	return OK;
}


/*===========================================================================*
 *				do_write				     *
 *===========================================================================*/
PRIVATE int do_write(pty, mq)
pty_t *pty;
mq_t *mq;
{
	assert(!(pty->p_flags & PF_PTY_CLOSED));
	if (pty->p_flags & PF_TTY_CLOSED)
	{
		reply(mq, 0, TRUE);
		return OK;
	}
	if (pty->p_wr_queue)
	{
		pty->p_wr_queue_tail->mq_next= mq;
		pty->p_wr_queue_tail= mq;
		assert (!(pty->p_flags & PF_WRITE_INITIATIVE));
		return SUSPEND;
	}
	pty->p_wr_queue= mq;
	pty->p_wr_queue_tail= mq;
	pty->p_wr_offset= 0;
	if (pty->p_flags & PF_WRITE_INITIATIVE)
	{
		pty_get(pty-pty_table);	/* This function is normallty called
					 * by the tty to get some data.
					 */
	}
	if (pty->p_wr_queue)
	{
		return SUSPEND;
	}
	return OK;
}


/*===========================================================================*
 *				do_ioctl				     *
 *===========================================================================*/
PRIVATE int do_ioctl(pty, mq)
pty_t *pty;
mq_t *mq;
{
	assert(!(pty->p_flags & PF_PTY_CLOSED));
	tty_receive(&pty->p_tty, mq);
	return OK;
}


/*===========================================================================*
 *				do_close				     *
 *===========================================================================*/
PRIVATE void do_close(pty, m)
pty_t *pty;
message *m;
{
	assert(!(pty->p_flags & PF_PTY_CLOSED));
	assert(!pty->p_rd_queue);
	assert(!pty->p_wr_queue);
	if (!(pty->p_flags & PF_TTY_CLOSED))
		tty_hup(&pty->p_tty);
	pty->p_flags |= PF_PTY_CLOSED;
}


/*===========================================================================*
 *				do_cancel				     *
 *===========================================================================*/
PRIVATE int do_cancel(pty, m)
pty_t *pty;
message *m;
{
	int operation;
	int status;
	mq_t *mq, *mq_prev;

	operation= m->NDEV_OPERATION;

	if (operation == CANCEL_ANY || operation == DEV_READ)
	{
		mq_prev= NULL;
		for (mq= pty->p_rd_queue; mq; mq_prev= mq, mq= mq->mq_next)
		{
			if (mq->mq_mess.NDEV_PROC != m->NDEV_PROC ||
				mq->mq_mess.NDEV_REF != m->NDEV_REF)
				continue;
			/* Found one. */
			if (!mq_prev)
			{	/* Head of the queue. */
				pty->p_rd_queue= mq->mq_next;
				status= pty->p_rd_offset;
				pty->p_rd_offset= 0;
				if (status == 0)
					status= EINTR;
				mq_free(mq);
				return status;
			}
			/* Somewhere in the middle of the queue. */
			mq_prev->mq_next= mq->mq_next;
			if (mq_prev->mq_next == NULL)
				pty->p_rd_queue_tail= mq_prev;
			mq_free(mq);
			return EINTR;
		}
		/* Apperently no read was to be canceled. */
	}

	if (operation == CANCEL_ANY || operation == DEV_WRITE)
	{
		mq_prev= NULL;
		for (mq= pty->p_wr_queue; mq; mq_prev= mq, mq= mq->mq_next)
		{
			if (mq->mq_mess.NDEV_PROC != m->NDEV_PROC ||
				mq->mq_mess.NDEV_REF != m->NDEV_REF)
				continue;
			/* Found one. */
			if (!mq_prev)
			{	/* Head of the queue. */
				pty->p_wr_queue= mq->mq_next;
				status= pty->p_wr_offset;
				pty->p_wr_offset= 0;
				if (status == 0)
					status= EINTR;
				mq_free(mq);
				return status;
			}
			/* Somewhere in the middle of the queue. */
			mq_prev->mq_next= mq->mq_next;
			if (mq_prev->mq_next == NULL)
				pty->p_wr_queue_tail= mq_prev;
			mq_free(mq);
			return EINTR;
		}
		/* Apperently no write was to be canceled. */
	}
	/* Ioctls don't block. */

	panic("Unable to find request to be canceled", NO_NUM);
	/*NOTREACHED*/
}


/*===========================================================================*
 *				reply					     *
 *===========================================================================*/
PRIVATE void reply(mq, result, can_enqueue)
mq_t *mq;
int result;
int can_enqueue;
{
	int proc, ref, status, operation;
	message m, *mp;

	proc= mq->mq_mess.NDEV_PROC;
	ref= mq->mq_mess.NDEV_REF;
	operation= mq->mq_mess.m_type;

	if (can_enqueue)
		mp= &mq->mq_mess;
	else
		mp= &m;

	mp->m_type= DEVICE_REPLY;
	mp->REP_PROC_NR= proc;
	mp->REP_STATUS= result;
	mp->REP_REF= ref;
	mp->REP_OPERATION= operation;

	status= send(mq->mq_mess.m_source, mp);
	if (status == ELOCKED && can_enqueue)
	{
		if (repl_queue)
			repl_queue_tail->mq_next= mq;
		else
			repl_queue= mq;
		repl_queue_tail= mq;
		return;
	}
	if (status != OK)
		panic("send failed", result);
	if (can_enqueue)
		mq_free(mq);
}


/*===========================================================================*
 *				reply_queue				     *
 *===========================================================================*/
PRIVATE int reply_queue(proc, ref, operation)
int proc;
int ref;
int operation;
{
	mq_t *mq, *cancel_reply, *tmp_m;
	int result;

	cancel_reply= NULL;
	for (mq= repl_queue; mq;)
	{
		if (cancel_reply == NULL &&
			mq->mq_mess.REP_PROC_NR == proc && 
			mq->mq_mess.REP_REF &&
			(operation == CANCEL_ANY || operation == 
			mq->mq_mess.REP_OPERATION))
		{	/* Found a reply to the canceled request. */
			cancel_reply= mq;
			mq= mq->mq_next;
			continue;
		}
		result= send(mq->mq_mess.m_source, &mq->mq_mess);
		if (result != OK)
			panic("send failed", result);
		tmp_m= mq;
		mq= mq->mq_next;
		mq_free(tmp_m);
	}
	if (cancel_reply)
	{
		result= send(cancel_reply->mq_mess.m_source, 
			&cancel_reply->mq_mess);
		if (result != OK)
			panic("send failed", result);
		mq_free(cancel_reply);
	}
	repl_queue= NULL;
	if (cancel_reply)
		return 1;
	else
		return 0;
}


/*===========================================================================*
 *				pty_init				     *
 *===========================================================================*/
PRIVATE void pty_init()
{
	int i;
	pty_t *pty;

	repl_queue= NULL;

	tmrs_initcontext(&pty_cntxt, &exp_timers, pty_tasknr);
	for (i=0, pty= pty_table; i<NPTY; i++, pty++)
	{
		pty->p_flags= PF_PTY_CLOSED | PF_TTY_CLOSED | 
			PF_READ_INITIATIVE | PF_WRITE_INITIATIVE;
		pty->p_rd_queue= NULL;
		pty->p_wr_queue= NULL;
		tty_init(&pty->p_tty, i, pty->p_tty_readbuf, TTY_RB_SIZ,
			pty->p_tty_writebuf, TTY_WB_SIZ, pty_tasknr,
			&pty_cntxt);
		pty->p_tty.t_get= pty_get;
		pty->p_tty.t_put= pty_put;
		pty->p_tty.t_reply= pty_reply;
		pty->p_tty.t_hup= pty_hup;
		pty->p_tty.t_setattr= pty_setattr;
	}
}


/*===========================================================================*
 *				pty_get					     *
 *===========================================================================*/
PRIVATE void pty_get(pty_line)
int pty_line;
{
/* Try to give the tty some data. */
	pty_t *pty;
	unsigned size, offset;
	vir_bytes buf;
	mq_t *mq;
	phys_bytes phys_user;
	int i, result;

	assert (pty_line >= 0 || pty_line < NPTY);
	pty= &pty_table[pty_line];

	pty->p_flags |= PF_WRITE_INITIATIVE;

	while(pty->p_wr_queue)
	{
		size= pty->p_wr_queue->mq_mess.NDEV_COUNT;
		offset= pty->p_wr_offset;
		buf= (vir_bytes)pty->p_wr_queue->mq_mess.NDEV_BUFFER;
		assert(size > offset);
		size -= offset;
		buf += offset;
		if (size > PTY_BUF_SIZ)
			size= PTY_BUF_SIZ;
		phys_user= numap(pty->p_wr_queue->mq_mess.NDEV_PROC,
			buf, size);
		if (!phys_user)
		{
			mq= pty->p_wr_queue;
			pty->p_wr_queue= pty->p_wr_queue->mq_next;
			pty->p_wr_offset= 0;
			result= offset;
			if (!result)
				result= EFAULT;
			reply(mq, result, TRUE);
			continue;
		}
		phys_copy(phys_user, vir2phys(write_buffer), (phys_bytes)size);
		if (pty->p_flags & PF_ISTRIP)
		{
			for (i= 0; i<size; i++)
				write_buffer[i] &= 0x7f;
		}
		result= tty_put(&pty->p_tty, write_buffer, size);
		assert(result >= 0 && result <= size);
		offset += result;
		if (offset ==  pty->p_wr_queue->mq_mess.NDEV_COUNT)
		{
			mq= pty->p_wr_queue;
			pty->p_wr_queue= pty->p_wr_queue->mq_next;
			pty->p_wr_offset= 0;
			reply(mq, offset, TRUE);
			continue;
		}
		pty->p_wr_offset= offset;
		if (result != size)
		{
			/* tty didn't accept everything. */
			pty->p_flags &= ~PF_WRITE_INITIATIVE;
			return;
		}
	}
	if (!pty->p_wr_queue)
	{
		/* Tell the tty that we are done. */
		tty_put(&pty->p_tty, NULL, 0);
	}
}


/*===========================================================================*
 *				pty_put					     *
 *===========================================================================*/
PRIVATE int pty_put(pty_line, buf, siz)
int pty_line;
char *buf;
int siz;
{
/* We got some data from tty. */
	pty_t *pty;
	mq_t *mq;
	unsigned total, size, offset;
	phys_bytes phys_user;
	vir_bytes user_buf;
	int result;

	assert (pty_line >= 0 || pty_line < NPTY);
	pty= &pty_table[pty_line];

	pty->p_flags |= PF_READ_INITIATIVE;

	if (siz == 0)
	{
		/* This is a flush command. */
		if (pty->p_rd_queue == NULL || pty->p_rd_offset == 0)
			return 0;
		mq= pty->p_rd_queue;
		pty->p_rd_queue= pty->p_rd_queue->mq_next;
		reply(mq, pty->p_rd_offset, TRUE);
		pty->p_rd_offset= 0;
		return 0;
	}
		
	total= 0;
	assert(siz > 0);
	while(siz && pty->p_rd_queue)
	{
		size= pty->p_rd_queue->mq_mess.NDEV_COUNT;
		offset= pty->p_rd_offset;
		user_buf= (vir_bytes)pty->p_rd_queue->mq_mess.NDEV_BUFFER;
		assert(size > offset);
		size -= offset;
		user_buf += offset;
		if (size > siz)
			size= siz;
		phys_user= numap(pty->p_rd_queue->mq_mess.NDEV_PROC,
			user_buf, size);
		if (!phys_user)
		{
			mq= pty->p_rd_queue;
			pty->p_rd_queue= pty->p_rd_queue->mq_next;
			pty->p_rd_offset= 0;
			result= offset;
			if (!result)
				result= EFAULT;
			reply(mq, result, TRUE);
			continue;
		}
		phys_copy(vir2phys(buf), phys_user, (phys_bytes)size);
		total += size;
		siz -= size;
		offset += size;
		if (offset ==  pty->p_rd_queue->mq_mess.NDEV_COUNT)
		{
			mq= pty->p_rd_queue;
			pty->p_rd_queue= pty->p_rd_queue->mq_next;
			pty->p_rd_offset= 0;
			reply(mq, offset, TRUE);
			continue;
		}
		pty->p_rd_offset= offset;
	}
	if (!siz)
	{
		pty->p_flags &= ~PF_READ_INITIATIVE;
	}
	return total;
}


/*===========================================================================*
 *				pty_reply				     *
 *===========================================================================*/
PRIVATE void pty_reply(pty_line, mq)
int pty_line;
mq_t *mq;
{
/* Tty wants to send a reply but the send is block because FS is sending to
 * this task. Enqueue to reply.
 */
	pty_t *pty;

	assert (pty_line >= 0 || pty_line < NPTY);
	pty= &pty_table[pty_line];

#if DEBUG
printf("in pty_reply\n");
#endif
	mq->mq_next= NULL;
	if (repl_queue)
		repl_queue_tail->mq_next= mq;
	else
		repl_queue= mq;
	repl_queue_tail= mq;
}


/*===========================================================================*
 *				pty_hup					     *
 *===========================================================================*/
PRIVATE void pty_hup(pty_line)
int pty_line;
{
/* The tty tells us to quit. */
	pty_t *pty;
	unsigned offset;
	mq_t *mq, *mq_tmp;

	assert (pty_line >= 0 || pty_line < NPTY);
	pty= &pty_table[pty_line];

	pty->p_flags |= PF_TTY_CLOSED;

	offset= pty->p_rd_offset;
	mq= pty->p_rd_queue;
	pty->p_rd_queue= NULL;
	while (mq)
	{
		mq_tmp= mq;
		mq= mq->mq_next;
		reply(mq_tmp, offset, TRUE);
		offset= 0;
	}

	offset= pty->p_wr_offset;
	mq= pty->p_wr_queue;
	pty->p_wr_queue= NULL;
	while (mq)
	{
		mq_tmp= mq;
		mq= mq->mq_next;
		reply(mq_tmp, offset, TRUE);
		offset= 0;
	}
}

/*===========================================================================*
 *				pty_setattr				     *
 *===========================================================================*/
PRIVATE void pty_setattr(pty_line, termios)
int pty_line;
struct termios *termios;
{
	pty_t *pty;

	assert (pty_line >= 0 || pty_line < NPTY);
	pty= &pty_table[pty_line];

	pty->p_flags &= ~PF_ISTRIP;
	if (termios->c_iflag & ISTRIP)
		pty->p_flags |= PF_ISTRIP;
}

/*
 * $PchId: pty.c,v 1.6 1996/01/19 23:27:01 philip Exp $
 */
