/* This file contains the device dependent part of the drivers for the
 * following special files:
 *     /dev/null	- null device (data sink)
 *     /dev/mem		- absolute memory
 *     /dev/kmem	- kernel virtual memory
 *     /dev/ram		- RAM disk
 *     /dev/zero	- null byte feed
 *
 * The file contains one entry point:
 *
 *   mem_task:	main entry when system is brought up
 *
 *  Changes:
 *	20 Apr  1992 by Kees J. Bot: device dependent/independent split
 */

#include "kernel.h"
#include <sys/ioctl.h>
#include "assert.h"
INIT_ASSERT
#include "driver.h"
#include "proc.h"
#if (CHIP == INTEL)
#include "i386/protect.h"
#endif

#define NR_RAMS            6	/* number of RAM-type devices */

PRIVATE struct device m_geom[NR_RAMS];	/* Base and size of each RAM disk */
PRIVATE int m_device;		/* current device */
FORWARD _PROTOTYPE( struct device *m_prepare, (int device) );
FORWARD _PROTOTYPE( int m_transfer, (int proc_nr, int opcode, u64_t position,
					iovec_t *iov, unsigned nr_req) );
FORWARD _PROTOTYPE( int m_do_open, (struct driver *dp, message *m_ptr) );
FORWARD _PROTOTYPE( void m_init, (void) );
FORWARD _PROTOTYPE( int m_ioctl, (struct driver *dp, message *m_ptr) );
FORWARD _PROTOTYPE( void checked_copy, (phys_bytes source, 
			phys_bytes dest, phys_bytes count) );
FORWARD _PROTOTYPE( void m_geometry, (struct partition *entry) );


/* Entry points to this driver. */
PRIVATE struct driver m_dtab = {
  no_name,	/* current device's name */
  m_do_open,	/* open or mount */
  do_nop,	/* nothing on a close */
  m_ioctl,	/* specify ram disk geometry */
  m_prepare,	/* prepare for I/O on a given minor device */
  m_transfer,	/* do the I/O */
  nop_cleanup,	/* no need to clean up */
  m_geometry,	/* memory device "geometry" */
};


/*===========================================================================*
 *				mem_task				     *
 *===========================================================================*/
PUBLIC void mem_task()
{
  m_init();
  driver_task(&m_dtab);
}


/*===========================================================================*
 *				m_prepare				     *
 *===========================================================================*/
PRIVATE struct device *m_prepare(device)
int device;
{
/* Prepare for I/O on a device. */

  if (device < 0 || device >= NR_RAMS) return(NIL_DEV);
  m_device = device;

  return(&m_geom[device]);
}


/*===========================================================================*
 *				m_transfer				     *
 *===========================================================================*/
PRIVATE int m_transfer(proc_nr, opcode, position, iov, nr_req)
int proc_nr;			/* process doing the request */
int opcode;			/* DEV_GATHER or DEV_SCATTER */
u64_t position;			/* offset on device to read or write */
iovec_t *iov;			/* pointer to read or write request vector */
unsigned nr_req;		/* length of request vector */
{
/* Read or write /dev/null, /dev/mem, /dev/kmem, /dev/ram, or /dev/zero. */

  int device;
  phys_bytes mem_phys, user_phys;
  unsigned count;
  struct device *dv;
  phys_bytes user_base = proc_vir2phys(proc_addr(proc_nr), 0);

  /* Get minor device number and check for /dev/null & /dev/zero. */
  device = m_device;
  dv = &m_geom[device];

  while (nr_req > 0) {
	user_phys = user_base + iov->iov_addr;
	count = iov->iov_size;

	switch (device) {
	case NULL_DEV:
		if (opcode == DEV_GATHER) return(OK);	/* Always at EOF. */
		break;

	case ZERO_DEV:
		if (opcode == DEV_GATHER) {
			/* Zero user memory. */
			phys_zero(user_phys, (phys_bytes) count);
		}
		break;

	default:
		/* /dev/mem, /dev/kmem, /dev/ram: Check for EOF */
		if (cmp64(position, dv->dv_size) >= 0) return(OK);
		if (cmp64(add64u(position, count), dv->dv_size) > 0)
			count = diff64(dv->dv_size, position);
		mem_phys = cv64ul(add64(dv->dv_base, position));

		/* Copy the data. */
		if (opcode == DEV_GATHER)
		{
#if VIRT_MEM
			/* Copy with care, some pages are missing... */
			checked_copy(mem_phys, user_phys, (phys_bytes) count);
#else /* !VIRT_MEM */
			phys_copy(mem_phys, user_phys, (phys_bytes) count);
#endif /* !VIRT_MEM */
		} else {
			phys_copy(user_phys, mem_phys, (phys_bytes) count);
		}
	}

	/* Book the number of bytes transferred. */
	position = add64ul(position, count);
	iov->iov_addr += count;
  	if ((iov->iov_size -= count) == 0) { iov++; nr_req--; }
  }
  return(OK);
}


/*============================================================================*
 *				m_do_open				      *
 *============================================================================*/
PRIVATE int m_do_open(dp, m_ptr)
struct driver *dp;
message *m_ptr;
{
/* Check device number on open.  Give I/O privileges to a process opening
 * /dev/mem or /dev/kmem.
 */

  if (m_prepare(m_ptr->DEVICE) == NIL_DEV) return(ENXIO);

#if (CHIP == INTEL)
  if (m_device == MEM_DEV || m_device == KMEM_DEV)
	enable_iop(proc_addr(m_ptr->PROC_NR));
#endif

  return(OK);
}


/*===========================================================================*
 *				m_init					     *
 *===========================================================================*/
PRIVATE void m_init()
{
  /* Initialize this task. */
  extern int end;

  m_geom[KMEM_DEV].dv_base = cvul64(vir2phys(0));
  m_geom[KMEM_DEV].dv_size = cvul64(vir2phys(&end));

#if (CHIP == INTEL)
  if (!protected_mode) {
	m_geom[MEM_DEV].dv_size =   cvul64(0x100000); /* 1M for 8086 systems */
  } else {
#if WORD_SIZE == 2
	m_geom[MEM_DEV].dv_size =  cvul64(0x1000000); /* 16M for 286 systems */
#else
	m_geom[MEM_DEV].dv_size =       make64(0, 1); /* 4G for 386 systems */
#endif
  }
#else /* !(CHIP == INTEL) */
#if (CHIP == M68000)
  m_geom[MEM_DEV].dv_size = cvul64(MEM_BYTES);
#else /* !(CHIP == M68000) */
#error /* memory limit not set up */
#endif /* !(CHIP == M68000) */
#endif /* !(CHIP == INTEL) */
}


/*===========================================================================*
 *				m_ioctl					     *
 *===========================================================================*/
PRIVATE int m_ioctl(dp, m_ptr)
struct driver *dp;
message *m_ptr;			/* pointer to read or write message */
{
/* Set parameters for one of the RAM disks. */

  struct device *dv;
  block_t req_bytes;
  u32_t ram_base, ram_blocks;
  phys_bytes user_phys;

  /* Check parameters. */
  if (m_ptr->REQUEST != MIOCRAMSIZE) return(do_diocntl(&m_dtab, m_ptr));
  if (m_ptr->PROC_NR >= LOW_USER) return(EPERM);
  if ((dv = m_prepare(m_ptr->DEVICE)) == NIL_DEV) return(ENXIO);
  if (m_device != RAM_DEV) return(ENOTTY);

  /* Fetch the parameters. */
  user_phys = numap(m_ptr->PROC_NR, (vir_bytes) m_ptr->ADDRESS, sizeof(u32_t));
  if (user_phys == 0) return(EFAULT);
  phys_copy(user_phys, vir2phys(&ram_blocks), sizeof(u32_t));

  /* FS can't check the size of the ramdisk, vm might allocate the ramdisk
   * on the swap device...
   */

  req_bytes = (phys_bytes) ram_blocks * BLOCK_SIZE;
  if (req_bytes > 0) {
	ram_base= vm_alloc_space(proc_ptr, req_bytes);
	vm_make_clear(proc_ptr, ram_base, req_bytes);
	dv->dv_base = cvul64(ram_base);
	dv->dv_size = cvul64(req_bytes);
  }
  return(OK);
}

/*===========================================================================*
 *				checked_copy				     *
 *===========================================================================*/
PRIVATE void checked_copy(source, dest, count)
phys_bytes source;
phys_bytes dest;
phys_bytes count;
{
/* Do a phys_copy but only if the source is part of an address space, e.g.
 * we do a reverse umap before this phys_copy. This umap is done by the
 * virtual memory system.
 */
	struct proc *pp;
	phys_bytes next_page, bytes;

	/* Suspend the system task, since all important changes to
	 * the address space of a user process go through the system task.
	 * This clearly shows the lack of synchronization primitives in
	 * Minix because we can't guarantee that no other task modifies
	 * the address space of a user process.
	 */
	pp= cproc_addr(SYSTASK);
	assert(!(pp->p_flags & P_STOP));
	pp->p_flags |= P_STOP;
	lock_unready(pp);

	vm_map_zp(TRUE);	/* Zero page may be interesting too. */

	while (count > 0)
	{
		bytes= count;

		/* Assume CLICK_SIZE is a multiple of the page size. */
		next_page= (source + CLICK_SIZE) & ~(CLICK_SIZE-1);
		if (next_page - source < bytes)
			bytes= next_page - source;
		assert(bytes > 0 && bytes <= count);

		if (vm_ismapped((u32_t)source))
		{
			/* Everything is OK, we just call phys_copy. We
			 * assume that the destination was umapped...
			 */
			phys_copy(source, dest, bytes);
		}
		else
		{
			phys_zero(dest, bytes);
		}
		source += bytes;
		dest += bytes;
		count -= bytes;
	}
	vm_map_zp(FALSE);

	/* Now we can safely unlock the system task. */
	assert(pp->p_flags & P_STOP);
	pp->p_flags &= ~P_STOP;
	lock_ready(pp);
}


/*============================================================================*
 *				m_geometry				      *
 *============================================================================*/
PRIVATE void m_geometry(entry)
struct partition *entry;
{
  /* Memory devices don't have a geometry, but the outside world insists. */
  entry->cylinders = div64u(m_geom[m_device].dv_size, SECTOR_SIZE) / (64 * 32);
  entry->heads = 64;
  entry->sectors = 32;
}

/*
 * $PchId: memory.c,v 1.6 1996/01/19 22:35:19 philip Exp $
 */
