/*
vga.c

Implement direct access to the vga card by user processes.

Created:	Sep 26, 1992 by Philip Homburg
*/

/*
 * Requests:
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_COUNT NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_OPEN    |minor dev  | proc nr   |  fd       |           |           |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_CLOSE   |minor dev  | proc nr   |  fd       |           |           |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_READ    |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_WRITE   |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_IOCTL NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_IOCTL3  |minor dev  | proc nr   |  fd       |  command  | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_OPERATION
 * -------------------------------------------------------------------|
 * | DEV_CANCEL  |minor dev  | proc nr   |  fd       | which operation|
 * |-------------+-----------+-----------+-----------+----------------|
 *
 * Replies:
 *
 *    m_type        REP_PROC_NR   REP_STATUS   REP_REF    REP_OPERATION
 * ----------------------------------------------------------------------|
 * | DEVICE_REPLY |   proc nr   |  status    |  fd     | which operation |
 * |--------------+-------------+------------+---------+-----------------|
 */

#include "kernel.h"
#include <sys/ioctl.h>
#include <sys/memio.h>
#include <minix/com.h>
#include "assert.h"
INIT_ASSERT
#include "mq.h"
#include "proc.h"
#include "i386/vm386.h"

#define VIDMEM_SIZE	0x20000			/* Max 128K video memory */
#define VIDMEM_BASE	0xA0000

PRIVATE reply_func_t vga_reply;
PRIVATE int open_cnt;

FORWARD _PROTOTYPE( int do_ioctl, (mq_t *mq)				);
FORWARD _PROTOTYPE( int do_map, (int proc_nr, vir_bytes addr)		);

/*===========================================================================*
 *				vga_init				     *
 *===========================================================================*/
PUBLIC void vga_init(reply)
reply_func_t reply;
{
	vga_reply= reply;
	vm_lock_hardware(VIDMEM_BASE, VIDMEM_SIZE);
}


/*===========================================================================*
 *				vga_mess				     *
 *===========================================================================*/
PUBLIC void vga_mess(mq)
mq_t *mq;
{
	int r;

	switch(mq->mq_mess.m_type)
	{
	case DEV_OPEN:
		open_cnt++;
		(*vga_reply)(mq, OK, FALSE);
		mq_free(mq);
		break;
	case DEV_CLOSE:
		assert(open_cnt > 0);
		open_cnt--;
		if (open_cnt == 0)
		{
			disable_cons_output= log_cons_output= FALSE;
			enable_video(TRUE);
		}
		(*vga_reply)(mq, OK, FALSE);
		mq_free(mq);
		break;
	case DEV_READ:
	case DEV_WRITE:
		(*vga_reply)(mq, EIO, FALSE);
		mq_free(mq);
		break;
	case DEV_IOCTL3:
		r= do_ioctl(mq);
		(*vga_reply)(mq, r, FALSE);
		mq_free(mq);
		break;
	default:
		panic("vga_mess: illegal type", mq->mq_mess.m_type);
	}
}


/*===========================================================================*
 *				do_ioctl				     *
 *===========================================================================*/
PRIVATE int do_ioctl(mq)
mq_t *mq;
{
	int r;

	switch(mq->mq_mess.NDEV_IOCTL)
	{
	case MIOCMAP:
		enable_video(TRUE);
		disable_cons_output= log_cons_output= TRUE;
		r= do_map(mq->mq_mess.NDEV_PROC, 
					(vir_bytes)mq->mq_mess.NDEV_BUFFER);
		return r;
	default:
		return ENOTTY;
	}
}


/*===========================================================================*
 *				do_map					     *
 *===========================================================================*/
PRIVATE int do_map(proc_nr, addr)
int proc_nr;
vir_bytes addr;
{
	struct proc *pp;
	mio_map_t mio_map;
	phys_bytes phys_src;
	vir_bytes base, size;
	vir_clicks base_click, top_click;
	phys_bytes phys_base, phys_size;
	int r;

	pp= proc_addr(proc_nr);
	phys_src= umap(pp, SEG_D, addr, sizeof(mio_map));
	if (phys_src == 0)
		return EFAULT;
	phys_copy(phys_src, vir2phys(&mio_map), sizeof(mio_map));

	/* check if the range is within the users address space, if
	 * the range is page aliged, and if the user doesn't ask too much. */
	base= mio_map.mm_base;
	size= mio_map.mm_size;
	if (base & VM_PAGEMASK)
	{
		/* Base is not page aligned */
		return EINVAL;
	}
	if (size & VM_PAGEMASK)
	{
		/* Size is not a multiple of the page size. */
		return EINVAL;
	}
	if (size > VIDMEM_SIZE)
		return EINVAL;
	base_click= base >> CLICK_SHIFT;
	top_click= (base+size+ CLICK_SIZE-1) >> CLICK_SHIFT;
	if (top_click < base_click)
		return EINVAL;
	r= EINVAL;
	if (base_click >= pp->p_map[SEG_D].mem_vir && top_click <= 
		pp->p_map[SEG_D].mem_vir+pp->p_map[SEG_D].mem_len)
	{
		r= OK;
	}
	else if (base_click >= pp->p_map[SEG_S].mem_vir && top_click <=
		pp->p_map[SEG_S].mem_vir+pp->p_map[SEG_S].mem_len)
	{
		r= OK;
	}
	if (r != OK)
		return EINVAL;

	/* OK, after we passed all checks, perform the unmap, and map */
	phys_base= (pp->p_map[SEG_D].mem_phys << CLICK_SHIFT) + base;
	phys_size= size;
	vm_unmap_pages(phys_base, phys_size);
	vm_map_hardware(VIDMEM_BASE, phys_base, phys_size);

	/* Also allow access to the registers of the vga card. At the 
	 * moment we just give the task access to all i/o ports!
	 */
	enable_iop(pp);

	return OK;
}

/*
 * $PchId: vga.c,v 1.5 1996/01/19 22:51:59 philip Exp $
 */
