/* Keyboard driver for PC's and AT's.
 *
 * Changed by Marcus Hampel	(04/02/1994)
 *  - Loadable keymaps
 *
 * Philip Homburg
 * - Almost complete rewrite	(March 1996)
 */

#include "kernel.h"
#include <signal.h>
#include <unistd.h>
#include <time.h>	/* For <sys/kbdio.h> */
#include <sys/kbdio.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include <minix/keymap.h>
#include "assert.h"
INIT_ASSERT

/* Keymap and special key strings. */
extern struct keymap keymap[KEYMAP_ENTRIES];
extern u8_t keymap_str[KEYMAP_STRINGS][KM_STRLEN];

#define SET1MODE	1	/* Set this for junk keyboards (i.e. always) */

/* Keyboard controller ports. */
#define KBC_DATA	0x60	/* I/O port for keyboard data */

#define KBC_CMD		0x64	/* I/O port for commands */
#define	    KBC_CMD_RD_RAM	0x20	/* Read controller RAM base */
#define	    KBC_CMD_WR_RAM	0x60	/* Write controller RAM base */
#define	      KBC_CCB		     0	/* Command control byte */
#define		KBC_CCB_EN_KBD_INT	0x01	/* Enable kbd int */
#define		KBC_CCB_EN_AUX_INT	0x02	/* Enable aux int */
#define		KBC_CCB_DI_KBD		0x10	/* Disable kbd */
#define		KBC_CCB_DI_AUX		0x20	/* Disable aux */
#define		KBC_CCB_TRANSLATE	0x40	/* Translate scan codes */
#define	    KBC_CMD_DI_AUX	0xA7	/* Disable AUX device */
#define	    KBC_CMD_EN_AUX	0xA8	/* Enable AUX device */
#define	    KBC_CMD_AUX_TEST	0xA9	/* Test presence of aux device */
#define	    KBC_CMD_DI_KBD	0xAD	/* Disable keyboard */
#define	    KBC_CMD_EN_KBD	0xAE	/* Enable keyboard */
#define	    KBC_CMD_WRITE_AUX	0xD4	/* Send byte to AUX device */
#define     KBC_CMD_PLS_OUT	0xF0	/* base for cmds to pulse out port */
#define		KBC_PLS_OUT_RESET	0x01	/* bit in output port to
						 * reset CPU */
#define		KBC_PLS_OUT_A20		0x02	/* bit in output port to
						 * enable A20 line */
#define	    KBC_RES_TRANS_TO	0xFE	/* Send timeout, iff KBC_ST_TIMEOUT */


#define KBC_STATUS	0x64	/* I/O port for status */
#define KBC_ST_OUT_FULL		0x01	/* status bit set when DATA port
					 * ready for an in.
					 */
#define KBC_ST_IN_FULL		0x02	/* status bit set when DATA port not
					 * ready for an out.
					 */
#define KBC_ST_AUX		0x20	/* Aux byte ready. */
#define KBC_ST_TIMEOUT		0x40	/* General Time-Out */

#define KBC_IN_DELAY		   7	/* wait 7 micorseconds when polling
					 * the keyboard controller.
					 */
/* Keyboard commands */
#define KBD_CMD_SETLEDS		0xED	/* command to keyboard to set LEDs */
#define     KBD_SETLEDS_SCR_LCK		0x1	/* Scroll lock */
#define     KBD_SETLEDS_NUM_LCK		0x2	/* Num lock */
#define     KBD_SETLEDS_CAPS_LCK	0x4	/* Caps lock */
#define KBD_CMD_ECHO		0xEE	/* Echo command */
#define KBD_CMD_SELECT_SCAN	0xF0	/* Select Scan Codes */
#define	    KBD_SEL_SCAN_REPORT	0x0	/* Report current scan set */
#define	    KBD_SEL_SCAN_1		0x1	/* Scan code set 1 */
#define	    KBD_SEL_SCAN_2		0x2	/* Scan code set 2 */
#define	    KBD_SEL_SCAN_3		0x3	/* Scan code set 3 */
#define KBD_CMD_READ_ID		0xF2	/* Read ID */
#define KBD_CMD_SETRATE		0xF3	/* set typematic rate delay command */
#define     KBD_RATE_DFLT		0x24	/* delay 0.5s, rate 20 cps. */
#define KBD_CMD_ENABLE		0xF4	/* Enable command */
#define KBD_CMD_DISABLE		0xF5	/* Default Disable command */
#define KBD_CMD_ALL_TMB		0xFA	/* Set all keys Typematics/Make/Brk */

/* Result codes sent by the keyboard */
#define KBD_RES_ID0		0xAB	/* first ID byte */
#define KBD_RES_ID1_84		0x84	/* 84/85-key keyboard */
#define KBD_RES_ID1_101		0x83	/* 101/102-key keyboard */
#define KBD_RES_ID1_122		0x86	/* 122-key keyboard */
#define KBD_RES_BREAK		0xF0	/* A key is released */
#define KBD_RES_ACK		0xFA	/* Acknowledge */
#define KBD_RES_RESEND		0xFE	/* Resend */

#define KBD_ACK_DELAY		20000	/* 20 milliseconds */
#define KBD_RETRIES		42

/* Aux device */
#define AUX_ACK_DELAY		20000	/* 20 milliseconds */

/* PS/2 mouse commands */
#define PMS_CMD_ENABLE		0xf4	/* Enable mouse */
#define PMS_CMD_DISABLE		0xf5	/* Disable mouse */
#define PMS_CMD_RESET		0xff	/* Reset mouse */

/* Aux results */
#define PMS_RES_ACK		0xFA	/* ACK */
#define PMS_RES_BAT_OK		0xAA	/* Reset OK */

#define PMS_RESET_DELAY		500000	/* 500 milliseconds */

/* Miscellaneous. */
#define MEMCHECK_ADR   0x472	/* address to stop memory check after reboot */
#define MEMCHECK_MAG  0x1234	/* magic number to stop memory check */

/* Pseudo scancode for modifiers: KM_modifiers|KM_{shift,ctrl,...} */
#define KM_modifiers	0xF0
#define KM_shift	 0x1
#define KM_ctrl		 0x2
#define KM_alt		 0x4
#define KM_alt_gr	 0x8

#define KB_IBUFSIZE	  32	/* size of keyboard input buffer */

#if SET1MODE
PRIVATE u8_t map_set1_to_keynum[]=
{
/* 00 */   0, 110,   2,   3,   4,   5,   6,   7,
/* 08 */   8,   9,  10,  11,  12,  13,  15,  16,
/* 10 */  17,  18,  19,  20,  21,  22,  23,  24,
/* 18 */  25,  26,  27,  28,  43,  58,  31,  32,
/* 20 */  33,  34,  35,  36,  37,  38,  39,  40,
/* 28 */  41,   1,  44,  29,  46,  47,  48,  49,
/* 30 */  50,  51,  52,  53,  54,  55,  57, 100,
/* 38 */  60,  61,  30, 112, 113, 114, 115, 116,
/* 40 */ 117, 118, 119, 120, 121,  90, 125,  91,
/* 48 */  96, 101, 105,  92,  97, 102, 106,  93,
/* 50 */  98, 103,  99, 104, 124,   0,  45, 122,
/* 58 */ 123,   0,   0,   0,   0,   0,   0,   0,
/* 60 */   0, 255,   0,   0,   0,   0,   0,   0,
/* 68 */ 255,   0,   0,   0,   0,   0,   0,   0,
/* 70 */   0,   0,   0,   0,   0,   0,   0,   0,
/* 78 */   0,   0,   0,   0,   0,   0,   0,   0,
};

PRIVATE u8_t map_set1esc_to_keynum[]=
{
/* 00 */   0, 110,   2,   3,   4,   5,   6,   7,
/* 08 */   8,   9,  10,  11,  12,  13,  15,  16,
/* 10 */  17,  18,  19,  20,  21,  22,  23,  24,
/* 18 */  25,  26,  27,  28, 108,  64,  31,  32,
/* 20 */  33,  34,  35,  36,  37,  38,  39,  40,
/* 28 */  41,   1,  44,  29,  46,  47,  48,  49,
/* 30 */  50,  51,  52,  53,  54,  95,  57, 124,
/* 38 */  62,  61,  30, 112, 113, 114, 115, 116,
/* 40 */ 117, 118, 119, 120, 121,  90, 125,  80,
/* 48 */  83,  85, 105,  79,  97,  89, 106,  81,
/* 50 */  84,  86,  75,  76, 124,   0,  45, 122,
/* 58 */ 123,   0,   0,   0,   0,   0,   0,   0,
/* 60 */   0, 255,   0,   0,   0,   0,   0,   0,
/* 68 */ 255,   0,   0,   0,   0,   0,   0,   0,
/* 70 */   0,   0,   0,   0,   0,   0,   0,   0,
/* 78 */   0,   0,   0,   0,   0,   0,   0,   0,
};
#endif

PRIVATE u8_t map_set3_to_keynum[KEYMAP_ENTRIES]=
{
/* 00 */   0,   0,   0,   0,   0,   0,   0, 112,
/* 08 */ 110,   0,   0,   0,   0,  16,   1, 113,
/* 10 */   0,  58,  44,  45,  30,  17,   2, 114,
/* 18 */   0,  60,  46,  32,  31,  18,   3, 115,
/* 20 */   0,  48,  47,  33,  19,   5,   4, 116,
/* 28 */   0,  61,  49,  34,  21,  20,   6, 117,
/* 30 */   0,  51,  50,  36,  35,  22,   7, 118,
/* 38 */   0,  62,  52,  37,  23,   8,   9, 119,
/* 40 */   0,  53,  38,  24,  25,  11,  10, 120,
/* 48 */   0,  54,  55,  39,  40,  26,  12, 121,
/* 50 */   0,   0,  41,  42,  27,  13, 122, 124,
/* 58 */  64,  57,  43,  28,  29,   0, 123, 125,
/* 60 */  84,  79, 126,  83,  76,  81,  15,  75,
/* 68 */   0,  93,  89,  92,  91,  86,  80,  85,
/* 70 */  99, 104,  98,  97, 102,  96,  90,  95,
/* 78 */   0, 108, 103,   0, 106, 101, 100,   0,
/* 80 */   0,   0,   0,   0, 105,   0,   0,   0,
/* 88 */   0,   0,   0,   0,   0,   0,   0,   0,
};

PRIVATE port_t kbc_data;	/* keyboard controller data port. */
PRIVATE port_t kbc_cmd;		/* keyboard controller command port. */
PRIVATE port_t kbc_status;	/* keyboard controller status port. */

PRIVATE int kbdi_shift_l;	/* left shift key state */
PRIVATE int kbdi_shift_r;	/* right shift key state */
PRIVATE int kbd_shift;		/* combined shift key state */
PRIVATE int kbdi_ctrl_l;	/* left control key state */
PRIVATE int kbdi_ctrl_r;	/* right control key state */
PRIVATE int kbd_ctrl;		/* combined control key state */
PRIVATE int kbdi_alt_l;		/* left alt key state */
PRIVATE int kbdi_alt_r;		/* right alt key state */
PRIVATE int kbd_alt;		/* combined alt key state */
PRIVATE int kbdi_alt_gr;	/* right alt key state */
PRIVATE int kbd_alt_gr;		/* right alt key state */
PRIVATE int kbd_capslock;	/* caps lock key state */
PRIVATE int kbd_numlock;	/* number lock key state */
PRIVATE int kbd_scrlock;	/* scroll lock key state */
PRIVATE int kbd_ttytask= ANY;	/* ANY to prevent accidents */

PRIVATE int kbd_expect_ack;	/* The interrupt handle should expect an ACK */

/* Keyboard structure. */
struct kb_s {
  int minor;			/* minor number of this line (base 0) */

  /* Configuration */
  int present;
  int keys;
  int oldmode;
  int newmode;

  /* Buffers */
  unsigned char *ibuf;		/* start of input buffer */
  unsigned char *ibufend;	/* end of input buffer */
  unsigned char *iptr;		/* next free spot in input buffer */

  unsigned char ibuf1[KB_IBUFSIZE + 1];	/* 1st input buffer, guard at end */
  unsigned char ibuf2[KB_IBUFSIZE + 1];	/* 2nd input buffer (for swapping) */
};

PRIVATE struct kb_s kb_line;			/* Only one. */

FORWARD _PROTOTYPE( void kbc_wait, (void) );
FORWARD _PROTOTYPE( void kbc_cmd0, (int cmd) );
FORWARD _PROTOTYPE( void kbc_cmd1, (int cmd, int arg) );
FORWARD _PROTOTYPE( int kbc_read, (void) );
FORWARD _PROTOTYPE( void kbd_cmd, (int cmd) );
FORWARD _PROTOTYPE( int kbd_cmd_ack, (int cmd) );
FORWARD _PROTOTYPE( int kbd_ack, (void) );
FORWARD _PROTOTYPE( void aux_cmd, (int cmd) );
FORWARD _PROTOTYPE( int aux_read, (unsigned long delay) );
FORWARD _PROTOTYPE( int kbd_hw_int, (int irq) );
FORWARD _PROTOTYPE( int aux_hw_int, (int irq) );
FORWARD _PROTOTYPE( void keyboard_hotkey, (char *key) );
FORWARD _PROTOTYPE( void set_leds, (void) );
FORWARD _PROTOTYPE( int check_map, (struct kio_map *maphdr) );
FORWARD _PROTOTYPE( void patch_set2_84, (void) );
FORWARD _PROTOTYPE( int map_set2, (int code, int *brk) );


/*===========================================================================*
 *				kb_init					     *
 *===========================================================================*/
PUBLIC void kb_init(minor, task_nr)
int minor;
int task_nr;
{
/* Initialize the keyboard driver. */

	struct kb_s *kb;
	int r;
	int ccb, n_ccb;
	int b0, b1;

	/* Set kbc_data, kbc_cmd and kbc_status. */
	kbc_data= KBC_DATA;
	kbc_cmd= KBC_CMD;
	kbc_status= KBC_STATUS;

	kb = &kb_line;

	/* Record minor number. */
	kb->minor = minor;

	/* Configuration */
	kb->present= 0;
	kb->keys= -1;
	kb->oldmode= -1;
	kb->newmode= -1;

	/* Set up input queue. */
	kb->iptr = kb->ibuf = kb->ibuf1;
	kb->ibufend = kb->ibuf1 + KB_IBUFSIZE;
	kb->iptr = kb->ibuf1;

	/* Disable the keyboard and the aux device on the controller. */
	kbc_cmd0(KBC_CMD_DI_KBD);
	kbc_cmd0(KBC_CMD_DI_AUX);

	/* Save the current controller command byte. */
	kbc_cmd0(KBC_CMD_RD_RAM + KBC_CCB);
	ccb= kbc_read();
	if (ccb == -1)
		panic("keyboard controller failed", NO_NUM);

#if SET1MODE
	n_ccb= (ccb & ~(KBC_CCB_EN_KBD_INT|KBC_CCB_EN_AUX_INT));
#else
	n_ccb= (ccb & ~(KBC_CCB_EN_KBD_INT|KBC_CCB_EN_AUX_INT|
		KBC_CCB_TRANSLATE));
#endif
	kbc_cmd1(KBC_CMD_WR_RAM + KBC_CCB, n_ccb);

	/* Reenable the keyboard on the controller. */
	kbc_cmd0(KBC_CMD_EN_KBD);

	/* See if the keyboard reponds */
	kbd_cmd(KBD_CMD_ECHO);
	r= kbc_read();
	if (r == -1)
	{
		/* No keyboard detected */
	}
	else if (r != KBD_CMD_ECHO)
	{
		printf("kb_init: keyboard ECHO command failed, got: ", r);
	}
	else
	{
		if (debug)
			printf("kb_init: keyboard responded to ECHO\n");
		kb->present= 1;
	}

	if (kb->present)
	{
		/* And tell the keyboard itself to remain silent. */
		r= kbd_cmd_ack(KBD_CMD_DISABLE);
		if (r == -1)
		{
			printf(
			"kb_init: timeout waiting for disable command\n");
			kb->present= 0;
		}
		else
		{
			if (debug)
			{
				printf(
				"kb_init: successfull keyboard disable.\n");
			}
		}
	}

#if !SET1MODE
	if (kb->present)
	{
		/* Ask the keyboard for its scan mode. */
		kbd_cmd(KBD_CMD_SELECT_SCAN);
		r= kbc_read();
		if (r != KBD_RES_ACK)
		{
			printf("kb_init: scan command failed: 0x%x\n", r);
		}
		else if (kbd_cmd(KBD_SEL_SCAN_REPORT),
			r= kbc_read(),
			r != KBD_RES_ACK)
		{
			printf(
		"kb_init: scan command did not accept argument: 0x%x\n",
				r);
			return;
		}
		else
		{
			r= kbc_read();
			kb->oldmode= r;
		}
	}

	if (kb->present)
	{
		/* Ask the keyboard for it's size. */
		r= kbd_cmd_ack(KBD_CMD_READ_ID);
		if (r == -1)
		{
			if (debug)
			{
				printf(
				"kb_init: keyboard READ ID command failed\n");
			}
		}
		else
		{
			b0= kbc_read();
			if (b0 != -1)
				b1= kbc_read();
			else
				b1= -1;
			if (b0 != KBD_RES_ID0)
			{
				if (debug)
				{
					printf(
		"kb_init: first byte of kbd id doesn't match, got: 0x%x\n",
					b0);
				}
			}
			else
			{
				switch(b1)
				{
				case KBD_RES_ID1_84:
					kb->keys= 84;
					break;
				case KBD_RES_ID1_101:
					kb->keys= 101;
					break;
				case KBD_RES_ID1_122:
					kb->keys= 122;
					break;
				default:
					printf(
				"kb_init: unknown keyboard type: 0x%x\n",
						b1);
					break;
				}
			}
		}
	}

	if (debug)
	{
		if (!kb->present)
			printf("kb_init: no keyboard detected\n");
		if (kb->oldmode != -1)
		{
			printf("kb_init: current scan mode is %d\n",
				kb->oldmode);
		}
		if (kb->keys != -1)
			printf("kb_init: got a %d-key keyboard\n", kb->keys);
	}

	if (kb->present)
	{
		/* Try to switch to scan mode 3. */
		r= kbd_cmd_ack(KBD_CMD_SELECT_SCAN);
		if (r == -1)
		{
			if (debug)
				printf("kb_init: scan command failed\n");
		}
		else if (r= kbd_cmd_ack(KBD_SEL_SCAN_3),
			(r == -1))
		{
			if (debug)
			{
				printf(
			"kb_init: scan command did not accept argument\n");
			}
		}
		else
			kb->newmode= 3;
	}

	if (kb->newmode == -1)
	{
		kb->newmode= 2;
		if (kb->keys == 84 || kb->keys == -1)
			patch_set2_84();
	}
#endif

	/* Reenable interrupts */
#if SET1MODE
	n_ccb= (ccb & ~(KBC_CCB_EN_AUX_INT));
#else
	n_ccb= (ccb & ~(KBC_CCB_EN_AUX_INT|KBC_CCB_TRANSLATE));
#endif
	kbc_cmd1(KBC_CMD_WR_RAM + KBC_CCB, n_ccb);

	/* Reenable the keyboard on the controller */
	kbc_cmd0(KBC_CMD_EN_KBD);

	if (kb->present)
	{
		/* Tell the keyboard to start */
		kbd_cmd(KBD_CMD_ENABLE);
		r= kbc_read();
		if (r != KBD_RES_ACK)
		{
			printf(
			"kb_init: keyboard did not accept ENABLE command\n");
		}

		set_leds();			/* turn off numlock led */
	}

#if !SET1MODE
	if (kb->present && kb->newmode == 3)
	{
		if (debug)
			printf("kbd: setting key types\n");

		/* Set all keys typematics/make/break. */
		kbd_cmd(KBD_CMD_ALL_TMB);
		r= kbc_read();
		if (r != KBD_RES_ACK)
		{
			printf(
"kb_init: keyboard did not accept Set All Typematic/Make/Break command\n");
		}
	}
#endif

	kbd_ttytask= task_nr;

	if (pc_at)
	{
		kbd_cmd(KBD_CMD_SETRATE);
		if (kbd_ack() == -1)
			printf("keyboard: setrate failed\n");
		else
		{
			kbd_cmd(KBD_RATE_DFLT);
			kbd_ack();
		}
	}

	put_irq_handler(KEYBOARD_IRQ, kbd_hw_int);
	enable_irq(KEYBOARD_IRQ);
}


/*===========================================================================*
 *				kbc_wait				     *
 *===========================================================================*/
PRIVATE void kbc_wait()
{
	int st, byte;

	/* Wait at most 1 millisecond for the controller to become ready. */
	struct timeval tv;

	micro_init(&tv);

	while(micro_elapsed(&tv) < 1000)
	{
		st= in_byte(kbc_status);
		if (!(st & (KBC_ST_OUT_FULL|KBC_ST_IN_FULL)))
			break;
		if (st & KBC_ST_OUT_FULL)
		{
			micro_delay(KBC_IN_DELAY);
			st= in_byte(kbc_status);
			if (st & KBC_ST_OUT_FULL)
			{
				byte= in_byte(kbc_data);
				printf(
				"keyboard`kbc_wait: ignoring %s byte 0x%x\n",
				(st & KBC_ST_AUX) ? "aux" : "kbd", byte);
			}
		}
	}
}


/*===========================================================================*
 *				kbc_cmd0				     *
 *===========================================================================*/
PRIVATE void kbc_cmd0(cmd)
int cmd;
{
	kbc_wait();
	out_byte(kbc_cmd, cmd);
}


/*===========================================================================*
 *				kbc_cmd1				     *
 *===========================================================================*/
PRIVATE void kbc_cmd1(cmd, data)
int cmd;
int data;
{
	kbc_wait();
	out_byte(kbc_cmd, cmd);
	kbc_wait();
	out_byte(kbc_data, data);
}


/*===========================================================================*
 *				kbc_read				     *
 *===========================================================================*/
PRIVATE int kbc_read()
{
	struct timeval tv;
	int st;
	int byte;

	/* Wait at most 1 second for a byte from the keyboard or
	 * the kbd controller, return -1 on a timeout.
	 */
	micro_init(&tv);
	while (micro_elapsed(&tv) < 1000000)
	{
		st= in_byte(kbc_status);
		if (st & KBC_ST_OUT_FULL)
		{
			micro_delay(KBC_IN_DELAY);
			byte= in_byte(kbc_data);
			if (st & KBC_ST_AUX)
			{
				printf(
		"keyboard`kbc_read: ignoring byte (0x%x) from aux device.\n",
					byte);
				continue;
			}
			return byte;
		}
	}
	return -1;
}

/*===========================================================================*
 *				kbd_cmd					     *
 *===========================================================================*/
PRIVATE void kbd_cmd(cmd)
int cmd;
{
	kbc_wait();
	out_byte(kbc_data, cmd);
}


/*===========================================================================*
 *				kbd_cmd_ack				     *
 *===========================================================================*/
PRIVATE int kbd_cmd_ack(cmd)
int cmd;
{
	int retry;
	struct timeval tv;
	int st, byte;

	for (retry= 0; retry < KBD_RETRIES; retry++)
	{
		kbc_wait();
		out_byte(kbc_data, cmd);

		micro_init(&tv);
		while (micro_elapsed(&tv) < KBD_ACK_DELAY)
		{
			st= in_byte(kbc_status);
			if (!(st & KBC_ST_OUT_FULL))
				continue;
			micro_delay(KBC_IN_DELAY);
			st= in_byte(kbc_status);
			if (!(st & KBC_ST_OUT_FULL))
				continue;
			if (st & KBC_ST_AUX)
			{
				byte= in_byte(kbc_data);
				printf(
		"keyboard`kbd_ack: discarding byte from aux: 0x%x\n",
					byte);
				continue;
			}
			break;
		}
		byte= in_byte(kbc_data);
		if (byte == KBC_RES_TRANS_TO &&
			(st & KBC_ST_TIMEOUT))
		{
			return -1;
		}

		if (byte == KBD_RES_ACK)
		{
			return 0;
		}
		if (byte == KBD_RES_RESEND)
			continue;

		break;
	}
	printf("keyboard`kbd_cmd_ack: ignoring byte 0x%x\n", byte);
	return -1;
}


/*==========================================================================*
 *				kbd_ack					    *
 *==========================================================================*/
PRIVATE int kbd_ack()
{
/* Wait until kbd acknowledges last command; return -1 if this times out.
 * This becomes complicated when interrupts are enabled: the interrupt
 * handler will eat the ACK. We solve this problem by setting a flag
 * asking the interrupt handler to consume one ACK and clear the flag.
 */
	struct timeval tv;
	int st, byte;

	kbd_expect_ack =1;
	micro_init(&tv);
	while (micro_elapsed(&tv) < KBD_ACK_DELAY)
	{
		if (!kbd_expect_ack)
			return 0;

		st= in_byte(kbc_status);
		if (!(st & KBC_ST_OUT_FULL))
			continue;
		micro_delay(KBC_IN_DELAY);
		st= in_byte(kbc_status);
		if (st & KBC_ST_OUT_FULL)
		{
			byte= in_byte(kbc_data);
			if (st & KBC_ST_AUX)
			{
				printf(
		"keyboard`kbd_ack: discarding byte from aux: 0x%x\n",
					byte);
				continue;
			}
			if (byte == KBD_RES_ACK)
			{
				kbd_expect_ack= 0;
				return 0;
			}
			printf("keyboard`kbd_ack: ignoring byte %d\n", byte);
		}
	}
	kbd_expect_ack= 0;
	return -1;
}


/*==========================================================================*
 *				aux_cmd					    *
 *==========================================================================*/
PRIVATE void aux_cmd(int cmd)
{
	/* Send a byte to the auxiliary device. */

	kbc_cmd1(KBC_CMD_WRITE_AUX, cmd);
}


/*==========================================================================*
 *				aux_read				    *
 *==========================================================================*/
PRIVATE int aux_read(delay)
unsigned long delay;
{
 /* Wait until a byte arrived from the aux device arrives. */
	struct timeval tv;
	int st, byte;

	micro_init(&tv);
	while (micro_elapsed(&tv) < delay)
	{
		st= in_byte(kbc_status);
		if (!(st & KBC_ST_OUT_FULL))
			continue;
		micro_delay(KBC_IN_DELAY);
		st= in_byte(kbc_status);
		if (st & KBC_ST_OUT_FULL)
		{
			byte= in_byte(kbc_data);
			if (!(st & KBC_ST_AUX))
			{
				printf(
		"keyboard`aux_read: might be keyboard byte: 0x%x\n",
					byte);
			}
			if (byte != KBC_RES_TRANS_TO)
				return byte;
			if (st & KBC_ST_TIMEOUT)
				return -1;
			return byte;
		}
	}
	return -1;
}


/*===========================================================================*
 *				kbd_hw_int				     *
 *===========================================================================*/
PRIVATE int kbd_hw_int(irq)
int irq;
{
	/* A keyboard interrupt has occurred.  Process it. */
	static int release;
#if SET1MODE
	static int next_set1esc;
	int set1esc;
#endif
	u8_t code;
	char c;
	unsigned char *charp;
	int key, brk, r, st, type, send_modifiers;
	int special;
	register struct kb_s *kb;

	kb = &kb_line;

	assert(irq == KEYBOARD_IRQ);

	/* Fetch the character from the keyboard. */
	st= in_byte(kbc_status);
	if (!(st & KBC_ST_OUT_FULL))
	{
		printf("kbd: nothing available\n");
		return 1;
	}
	if (st & KBC_ST_AUX)
	{
		printf("Strange: got aux character.\n");
		code= in_byte(kbc_data);
		kbdaux_put((char *)&code, 1);
		return 1;
	}
	code = in_byte(kbc_data);

	if (code == KBD_RES_ACK && kbd_expect_ack)
	{
		kbd_expect_ack= 0;
		return 1;
	}

	/* The keyboard interrupts twice per key, once when depressed,
	 * once when released.  Filter out the latter.
	 * Shift-type keys are processed here.
	 */
#if SET1MODE
	if (code == 0xE0)
	{
		next_set1esc= 1;	/* the next key has an escape code */
		return 1;
	}
	set1esc= next_set1esc;
	next_set1esc= 0;
	key= (set1esc ? map_set1esc_to_keynum : map_set1_to_keynum)[code&0x7F];
	if (key == 0)
	{
		if (debug)
			printf("kbd: strange scancode: %s%02X\n",
				set1esc ? "0xE0 " : "", code);
		return 1;
	}
	if (key == 255)
		return 1;		/* key code 255 is to be ignored */
	if (code & 0x80)
		release= 1;
#else
	if (kb->newmode == 3)
	{
		if (code == KBD_RES_BREAK)
		{
			release= 1;
			if (rawkbd_enabled)
				rawkbd_put((char *)&code, 1);
			return 1;
		}

		if (code < 0 || code >= KEYMAP_ENTRIES)
		{
			printf("kbd_kw_int: scancode out of range (0x%x)\n",
				code);
			return 1;
		}
		code= map_set3_to_keynum[code];
	}
	else
	{
		r= map_set2(code, &brk);

		if (brk)
		{
			release= 1;
			if (rawkbd_enabled)
			{
				c= KBD_RES_BREAK;
				rawkbd_put(&c, 1);
			}
		}

		if (r == -1)
			return 1;
		code= r;
	}
#endif
	/* The console gets the keys while scroll lock is active.  The SysRq
	 * key is an alternate Scroll Lock that works even when the console
	 * is disabled.
	 */
	special= kbd_scrlock;

	if (key < 0 || key >= KEYMAP_ENTRIES)
	{
		/* Strange scan code. */
		if (debug)
			printf("kbd_hw_int: strange keynum: 0x%x\n", key);
		if (!rawkbd_enabled)
		{
			release= 0;
			return 1;
		}
	}
	else if (keymap[key].fl & KM_SPECIAL)
	{
		type= keymap[key].n;
		send_modifiers= 1;
		switch(type)
		{
		case KM_SHIFT_L: kbdi_shift_l= !release; break;
		case KM_SHIFT_R: kbdi_shift_r= !release; break;
		case KM_CTRL_L:  kbdi_ctrl_l= !release; break;
		case KM_CTRL_R:  kbdi_ctrl_r= !release; break;
		case KM_ALT_L:   kbdi_alt_l= !release; break;
		case KM_ALT_R:   kbdi_alt_r= !release; break;
		case KM_ALT_GR:  kbdi_alt_gr= !release; break;
		default:
			send_modifiers= 0;
			if (type == KM_SYSREQ)
				special= 1;
		}
#if SET1MODE
		if (release) {
			/* Release of escaped modifiers often goes wrong. */
			switch (type)
			{
			case KM_CTRL_L:
			case KM_CTRL_R:
				kbdi_ctrl_l= kbdi_ctrl_r= 0;
				break;
			case KM_ALT_L:
			case KM_ALT_R:
			case KM_ALT_GR:
				kbdi_alt_l= kbdi_alt_r= kbdi_alt_gr= 0;
				break;
			}
		}
#endif

		if ((!rawkbd_enabled || special) && send_modifiers)
		{
			key= KM_modifiers;
			if (kbdi_shift_l || kbdi_shift_r)
				key |= KM_shift;
			if (kbdi_ctrl_l || kbdi_ctrl_r)
				key |= KM_ctrl;
			if (kbdi_alt_l || kbdi_alt_r)
				key |= KM_alt;
			if (kbdi_alt_gr)
				key |= KM_alt_gr;
			release= 0;
		}
	}

	if (rawkbd_enabled && !special)
	{
#if SET1MODE
		if (set1esc) {
			c= 0xE0;
			rawkbd_put(&c, 1);
		}
#endif
		c= code;				/* key or code? */
		rawkbd_put(&c, 1);

		release= 0;
		return 1;	/* Reenable keyboard interrupt */
	}

	if (release)
	{
		/* Ignore the release of a key. */
		release= 0;
		return 1;
	}

	/* Store the character in memory so the task can get at it later. */
	if (kb->iptr < kb->ibufend) {
		*kb->iptr = key;
		++kb->iptr;
		interrupt(kbd_ttytask);
		kbd_int_pending= TRUE;
	}
	/* Else it doesn't fit - discard it. */
	return 1;	/* Reenable keyboard interrupt */
}


/*===========================================================================*
 *				aux_hw_int				     *
 *===========================================================================*/
PRIVATE int aux_hw_int(irq)
int irq;
{
	/* An aux interrupt has occurred.  Process it. */
	u8_t code;
	char c;
	int st;

	assert(irq == KBD_AUX_IRQ);

	/* Fetch the character. */
	st= in_byte(kbc_status);
	if (!(st & KBC_ST_OUT_FULL))
	{
		printf("kbd: nothing aux available\n");
		return 1;
	}
	code= in_byte(kbc_data);
	kbdaux_put((char *)&code, 1);
	return 1;
}


/*==========================================================================*
 *				kb_read					    *
 *==========================================================================*/
PUBLIC int kb_read(bufindirect)
unsigned char **bufindirect;
{
/* Swap the keyboard input buffers, giving the old one to TTY. */

	unsigned char *ibuf;
	register struct kb_s *kb;
	int nread;

	enable_video(TRUE);
	kbd_int_pending= FALSE;
	kb = &kb_line;
	if (kb->iptr == (ibuf = kb->ibuf))
		return 0;
	*bufindirect = ibuf;

	lock();
	nread = kb->iptr - ibuf;
	if (ibuf == kb->ibuf1)
		ibuf = kb->ibuf2;
	else
		ibuf = kb->ibuf1;
	kb->ibufend = ibuf + KB_IBUFSIZE;
	kb->iptr = ibuf;
	unlock();

	kb->ibuf = ibuf;
	return nread;
}


/*===========================================================================*
 *				kb_translate				     *
 *===========================================================================*/
PUBLIC int kb_translate(code, bufp)
unsigned char code;	/* scan code of key just struck or released */
unsigned char **bufp;
{
	static int reboot_count= 0;
	static unsigned char ch;

	int c, fl, ch1, ch2, index;

	*bufp= &ch;				/* Normal case */

	if (code >= KM_modifiers)
	{
		kbd_shift= !!(code & KM_shift);
		kbd_ctrl= !!(code & KM_ctrl);
		kbd_alt= !!(code & KM_alt);
		kbd_alt_gr= !!(code & KM_alt_gr);
		return 0;
	}

	assert(code >= 0 && code < KEYMAP_ENTRIES);

	fl= keymap[code].fl;
	if (fl & KM_SPECIAL)
	{
		switch (keymap[code].n)
		{
		case KM_CAPSLOCK:
			kbd_capslock= !kbd_capslock;
			set_leds();
			return 0;
		case KM_NUMLOCK:
			kbd_numlock= !kbd_numlock;
			set_leds();
			return 0;
		case KM_SYSREQ:
		case KM_SCRLOCK:
			kbd_scrlock= !kbd_scrlock;
			set_leds();
			return 0;
		default:
			printf("unknown special code: %d\n", keymap[code].n);
			return 0;
		}
	}

#if 0
	if (kbd_ctrl && (kbd_alt|kbd_alt_gr))
	{
		/* Hmm, ctrl and alt. Check for DEL to reboot. */
		ch1= keymap[code].n & ~(KM_CAPSLOCK|KM_NUMLOCK);
		if (ch1 == 0x7F)
		{
			cause_sig(INIT_PROC_NR, SIGABRT);
			return 0;
		}
		printf("CTRL and ALT, ignoring code\n");
		return 0;
	}
#endif
	if (kbd_ctrl)
	{
		/* Select ctrl, shift-ctrl, and ctrl-alt. */
		ch1= keymap[code].c;
		ch2= keymap[code].sc;
		if (kbd_alt|kbd_alt_gr)
			ch1= ch2= keymap[code].ca;
	}
	else if (kbd_alt)
	{
		/* Select alt and shift-alt */
		ch1= keymap[code].a;
		ch2= keymap[code].sa;
	}
	else if (kbd_alt_gr)
	{
		/* Select alt-graphics and shift-alt */
		ch1= keymap[code].ag;
		ch2= keymap[code].sa;
	}
	else
	{
		/* Select normal and shifted */
		ch1= keymap[code].n;
		ch2= keymap[code].s;
	}

	if (kbd_shift)
	{
		/* Select the shifted state by swapping ch1 and ch2. */
		c= ch1; ch1= ch2; ch2= c;
	}
	if (kbd_capslock && (ch1 & KM_CAPSLOCK))
	{
		/* Caps lock, negate the effect of shift. */
		c= ch1; ch1= ch2; ch2= c;
	}
	if (kbd_numlock && (ch1 & KM_NUMLOCK))
	{
		/* Num lock, negate the effect of shift. */
		c= ch1; ch1= ch2; ch2= c;
	}

	if (ch1 & KM_STRING)
	{
		index= ch1 & (KM_STRING-1);
		*bufp= &keymap_str[0][0]+index;
		if (ch1 & KM_HOTKEY)
		{
			keyboard_hotkey((char *) *bufp);
			if (!rawkbd_enabled) console_hotkey((char *) *bufp);
			return 0;
		}
		return strlen((char *)*bufp);
	}

	if (ch1 != 0)
	{
		ch= ch1 & 0xff;
		return 1;
	}

#if DEBUG
	if (keymap[code].l == NULL)
	{
		printf("kbd: Got key without label (0x%x)\n", code);
	}
	else
	{
		printf("Got key: '%s%s%s%s%s%s'\n",
			kbd_shift ? "SHIFT-" : "",
			kbd_ctrl ? "CTRL-" : "",
			kbd_alt ? "ALT-" : "",
			kbd_alt_gr ? "ALTGR-" : "",
			keymap[code].l,
			kbd_capslock ? " (CAPS)" : "",
			kbd_numlock ? " (NUM)" : "");
	}
#endif
	return 0;
}


/*===========================================================================*
 *				keyboard_hotkey				     *
 *===========================================================================*/
PRIVATE void keyboard_hotkey(key)
char *key;			/* key string */
{
/* This procedure performs hot-key actions for system control or debugging. */
  int line;

  if (strcmp(key, "reboot") == 0)
  {
	/* Immediately halt, or tell init to halt or reboot. */
	if (kbd_shift)
		wreboot(RBT_HALT);
	cause_sig(INIT_PROC_NR, SIGABRT);
  }
  if (strcmp(key, "pdmp") == 0) p_dmp();	/* print process table */
  if (strcmp(key, "mdmp") == 0) map_dmp();	/* print memory map */
  if (strcmp(key, "scroll") == 0) toggle_scroll();/* hard vs. soft scrolling */
  if (strcmp(key, "serdbug") == 0) {
	line = ser_nextline();
	if (line < 0) {
		printf("debug output to serial line is disabled\n");
	} else {
		printf("debug output is sent to serial line %d\n", line);
	}
  }
#if ENABLE_NETWORKING
  if (strcmp(key, "netstat") == 0) dp_dump();	/* network statistics */
#endif
  if (strcmp(key, "prio") == 0) sched_dmp();	/* scheduler table */
#if LOADAV
  if (strcmp(key, "loadav") == 0) loadav_dmp();	/* load averages */
#endif
  if (strcmp(key, "vmdmp") == 0) vm_dump();	/* memory sizes and stats */
  if (strcmp(key, "fpemu") == 0) {
  	extern int fpu_usremu;

  	fpu_usremu= !fpu_usremu;
	printf("fpu user emu o%s\n", fpu_usremu ? "n" : "ff");
  }
  if (strcmp(key, "debug") == 0) {
	debug = !debug;
	printf("debugging o%s\n", debug ? "n" : "ff");
  }
}


/*===========================================================================*
 *				set_leds				     *
 *===========================================================================*/
PRIVATE void set_leds()
{
/* Set the LEDs on the caps lock and num lock keys */

  unsigned leds;

  leds= 0;
  if (kbd_numlock)
  	leds |= KBD_LEDS_NUM;
  if (kbd_capslock)
  	leds |= KBD_LEDS_CAPS;
  if (kbd_scrlock)
  	leds |= KBD_LEDS_SCROLL;
  kbd_set_leds(leds);
}


/*===========================================================================*
 *				kbd_set_leds				     *
 *===========================================================================*/
PUBLIC void kbd_set_leds(leds)
unsigned leds;
{
/* Set the LEDs on the caps lock, num lock, and scroll lock keys */

  int kbd_leds, data_port, status_port;

  kbd_leds= 0;
  if (leds & KBD_LEDS_SCROLL)
  	kbd_leds |= KBD_SETLEDS_SCR_LCK;
  if (leds & KBD_LEDS_NUM)
  	kbd_leds |= KBD_SETLEDS_NUM_LCK;
  if (leds & KBD_LEDS_CAPS)
  	kbd_leds |= KBD_SETLEDS_CAPS_LCK;

  kbd_cmd(KBD_CMD_SETLEDS);
  if (kbd_ack() == -1)
	return;
  kbd_cmd(kbd_leds);
  if (kbd_ack() == -1)
	return;
}


/*==========================================================================*
 *				kbd_aux					    *
 *==========================================================================*/
PUBLIC int kbd_aux(enable)
int enable;
{
	int ccb;
	int r, result;

	if (!enable)
	{
		/* Disable the AUX device. */
		aux_cmd(PMS_CMD_DISABLE);
		kbc_cmd0(KBC_CMD_DI_AUX);

		return OK;
	}

	/* Set the interrupt handler for the aux device. */
	put_irq_handler(KBD_AUX_IRQ, aux_hw_int);

	/* Disable the keyboard and aux all together. */
	kbc_cmd0(KBC_CMD_DI_KBD);
	kbc_cmd0(KBC_CMD_DI_AUX);

	/* Disable both keyboard and the aux interrupt. */
	disable_irq(KBD_AUX_IRQ);
	disable_irq(KEYBOARD_IRQ);

	/* Get the current configuration byte */
	kbc_cmd0(KBC_CMD_RD_RAM + KBC_CCB);
	ccb= kbc_read();

	/* Enable both interrupts. */
	kbc_cmd1(KBC_CMD_WR_RAM + KBC_CCB, ccb | 3);

	/* Enable the aux device. */
	kbc_cmd0(KBC_CMD_EN_AUX);

	aux_cmd(PMS_CMD_RESET);
	r= aux_read(AUX_ACK_DELAY);
	if (r == -1)
	{
		printf(
		"kbd_aux: timeout waiting for ACK after reset\n");
		result= ENXIO;
		goto done;
	}
	else if (r != PMS_RES_ACK)
		printf("kbd_aux: no ACK after reset: 0x%x\n", r);
	r= aux_read(PMS_RESET_DELAY);
	if (r == -1)
	{
		printf(
		"kbd_aux: timeout waiting for reset completion\n");
	}
	else if (r != PMS_RES_BAT_OK)
		printf("kbd_aux: no OK after reset: 0x%x\n", r);

	kbc_cmd0(KBC_CMD_AUX_TEST);

	aux_cmd(PMS_CMD_ENABLE);
	enable_irq(KBD_AUX_IRQ);

	result= OK;
done:
	/* Re-enable the keyboard device. */
	kbc_cmd0(KBC_CMD_EN_KBD);

	enable_irq(KEYBOARD_IRQ);
	kbd_hw_int(KEYBOARD_IRQ);

	return result;
}


/*==========================================================================*
 *				wreboot					    *
 *==========================================================================*/
PUBLIC void wreboot(how)
int how;		/* 0 = halt, 1 = reboot, 2 = panic!, ... */
{
/* Wait for keystrokes for printing debugging info and reboot. */

  long delay = 5;	/* by default 5 minutes before an automatic reboot */
  int st, n, len, i, key;
  unsigned char *keyp, *charp;
  static u16_t magic = MEMCHECK_MAG;
  struct tasktab *ttp;

  /* Mask all interrupts. */
  out_byte(INT_CTLMASK, ~0);

  /* Disable any aux device */
  kbc_cmd0(KBC_CMD_DI_AUX);

  /* Disable raw keyboard access */
  rawkbd_enabled= 0;

  /* Tell all tasks to stop. */
  for (ttp = tasktab; ttp->tt_name != NULL; ttp++) {
	if (ttp->tt_stop != NULL) (*ttp->tt_stop)();
  }

  if (how == RBT_HALT) {
	printf("System Halted\n");
	if (!mon_return) how = RBT_PANIC;
  }

  if (how == RBT_PANIC) {
	/* A panic! */
	if (env_parse("reboot_delay", "d", 0, &delay, 0L, LONG_MAX / 60000)
								== EP_OFF) {
		delay = -1;
	} else {
		printf("Automatic reboot in %ld minutes.\n", delay);
		delay *= 60000;
	}
	printf("Hit ESC to reboot, F-keys for debug dumps\n");

	while (delay != 0)
	{
		micro_delay(1000L);	/* pause for a millisecond */
		if (delay > 0) delay--;

		/* See if the keyboard has something for us. */
		st= in_byte(kbc_status);
		if (!(st & KBC_ST_OUT_FULL))
			continue;			/* nothing */

		micro_delay(KBC_IN_DELAY);
		st= in_byte(kbc_status);
		if (!(st & KBC_ST_OUT_FULL))
			continue;			/* strange */

		/* Call the interrupt handler to deal with the scan code. */
		kbd_hw_int(KEYBOARD_IRQ);

		/* Ask the interrupt handler for any new scan code. */
		n= kb_read(&keyp);
		delay = -1;			/* stop countdown */

		for (i= 0; i< n; i++)
		{
			key= keyp[i];
			len= kb_translate(key, &charp);	/* map keycode */

			if (len == 1 && charp[0] == 27)	/* ESC */
			{
				delay= 0;
				break;
			}
		}
	}
	cons_stop();		/* again tell console to stop (hotkeys) */
	how = RBT_REBOOT;
  }

  if (how == RBT_REBOOT) printf("Rebooting\n");

  if (mon_return && how != RBT_RESET) {
	/* Reinitialize the interrupt controllers to the BIOS defaults. */
	intr_init(0);
	out_byte(INT_CTLMASK, 0);
	out_byte(INT2_CTLMASK, 0);

	/* Return to the boot monitor. */
	if (how == RBT_HALT) {
		reboot_code = vir2phys("");
	} else
	if (how == RBT_REBOOT) {
		reboot_code = vir2phys("delay;boot");
	}
	level0(monitor);
  }

  /* Stop BIOS memory test. */
  vm_map_zp(TRUE);
  phys_copy(vir2phys(&magic), (phys_bytes) MEMCHECK_ADR,
  						(phys_bytes) sizeof(magic));

  if (protected_mode) {
	/* Use the AT keyboard controller to reset the processor.
	 * The A20 line is kept enabled in case this code is ever
	 * run from extended memory, and because some machines
	 * appear to drive the fake A20 high instead of low just
	 * after reset, leading to an illegal opode trap.  This bug
	 * is more of a problem if the fake A20 is in use, as it
	 * would be if the keyboard reset were used for real mode.
	 */
	kbc_wait();
	kbc_cmd0(KBC_CMD_PLS_OUT |
		(0x0F & ~(KBC_PLS_OUT_A20 | KBC_PLS_OUT_RESET)));
	micro_delay(10000L);

	/* If the nice method fails then do a reset.  In protected
	 * mode this means a processor shutdown.
	 */
	printf("Hard reset...\n");
	micro_delay(250000L);
  }
  /* In real mode, jumping to the reset address is good enough. */
  level0(reset);
}


/*===========================================================================*
 *				kbd_loadmap				     *
 *===========================================================================*/
PUBLIC int kbd_loadmap(maphdr, proc)
struct kio_map *maphdr;
int proc;
{
	phys_bytes u_km, u_str, k_phys;
	int r;

	/* Check the sizes */
	if (maphdr->km_entries > KEYMAP_ENTRIES)
	{
		printf("kbd_loadmap: too many keymap entries\n");
		return EINVAL;
	}
	if (maphdr->km_strsize > KEYMAP_STRINGS * KM_STRLEN)
	{
		printf("kbd_loadmap: too much for strings\n");
		return EINVAL;
	}

	/* Umap both */
	u_km= numap(proc, (vir_bytes)maphdr->km_keymap,
		maphdr->km_entries * sizeof(struct keymap));
	if (u_km == 0)
		return EFAULT;
	u_str= numap(proc, (vir_bytes)maphdr->km_strtab, maphdr->km_strsize);
	if (u_str == 0)
		return EFAULT;

	/* Copy in */
	k_phys= umap(proc_ptr, SEG_D,
		(vir_bytes)keymap,
		maphdr->km_entries * sizeof(struct keymap));
	assert(k_phys != 0);
	phys_copy(u_km, k_phys,
		maphdr->km_entries * sizeof(struct keymap));
	k_phys= umap(proc_ptr, SEG_D,
		(vir_bytes)keymap_str, maphdr->km_strsize);
	assert(k_phys != 0);
	phys_copy(u_str, k_phys, maphdr->km_strsize);

	/* Check and patch the new map. */
	r= check_map(maphdr);
	return r;
}

/*===========================================================================*
 *				check_map				     *
 *===========================================================================*/
PRIVATE int check_map(maphdr)
struct kio_map *maphdr;
{
	u16_t *ptrs[7];
	size_t offset;
	int i, j, r;

	r= OK;

	/* Terminating NUL in keymap_str */
	keymap_str[KEYMAP_STRINGS-1][KM_STRLEN-1]= '\0';

	/* Check all references to keymap_str and patch keyname
	 * pointers.
	 */
	for (i= 0; i<maphdr->km_entries; i++)
	{
		offset= (size_t)keymap[i].l;
		if (offset >= maphdr->km_strsize)
		{
			printf(
			"kbd(KIOCSMAP): label ptr out of range, key %d: %d\n",
				i, offset);
			keymap[i].l= NULL;
			r= EINVAL;
		}
		else if (offset > 0)
			keymap[i].l= (char *)&keymap_str[0][0]+offset;

		/* Should check flags here. */

		/* Check columns */
		ptrs[0]= &keymap[i].n;
		ptrs[1]= &keymap[i].s;
		ptrs[2]= &keymap[i].c;
		ptrs[3]= &keymap[i].a;
		ptrs[4]= &keymap[i].ag;
		ptrs[5]= &keymap[i].sa;
		ptrs[6]= &keymap[i].sc;
		for (j= 0; i<7; i++)
		{
			/* Should check modifiers, etc. */

			if (!(*ptrs[j] & KM_STRING))
				continue;
			offset= (*ptrs[j] & (KM_STRING-1));
			if (offset >= maphdr->km_strsize)
			{
				printf(
"kbd(KIOCSMAP): string offset out of range for key %d, column %d: %d\n",
					i, j, offset);
				*ptrs[j]= 0;
				r= EINVAL;
			}
		}
	}
	return r;
}


/* Support for scancode set 2 */

PRIVATE u8_t map_set2_to_keynum[KEYMAP_ENTRIES]=
{
/* 00 */   0, 120,   0, 116, 114, 112, 113, 123,
/* 08 */   0, 121, 119, 117, 115,  16,   1,   0,
/* 10 */   0,  60,  44,   0,  58,  17,   2,   0,
/* 18 */   0,   0,  46,  32,  31,  18,   3,   0,
/* 20 */   0,  48,  47,  33,  19,   5,   4,   0,
/* 28 */   0,  61,  49,  34,  21,  20,   6,   0,
/* 30 */   0,  51,  50,  36,  35,  22,   7,   0,
/* 38 */   0,   0,  52,  37,  23,   8,   9,   0,
/* 40 */   0,  53,  38,  24,  25,  11,  10,   0,
/* 48 */   0,  54,  55,  39,  40,  26,  12,   0,
/* 50 */   0,   0,  41,   0,  27,  13,   0,   0,
/* 58 */  30,  57,  43,  28,   0,  29,   0,   0,
/* 60 */   0,  45,   0,   0,   0,   0,  15,   0,
/* 68 */   0,  93,   0,  92,  91,   0,   0,   0,
/* 70 */  99, 104,  98,  97, 102,  96, 110,  90,
/* 78 */ 122, 106, 103, 105, 100, 101, 125,   0,
/* 80 */   0,   0,   0, 118,   0,   0,   0,   0,
/* 88 */   0,   0,   0,   0,   0,   0,   0,   0,
};

PRIVATE struct ext_keymap
{
	u8_t code;
	u8_t key;
} map_E0_set2_to_keynum[]=
{
    {	0x11,  62,	},
    {	0x14,  64,	},
    {	0x70,  75,	},
    {	0x71,  76,	},
    {	0x6B,  79,	},
    {	0x6c,  80,	},
    {	0x69,  81,	},
    {	0x75,  83,	},
    {	0x72,  84,	},
    {	0x7D,  85,	},
    {	0x7A,  86,	},
    {	0x74,  89,	},
    {	0x4A,  95,	},
    {	0x5A, 108,	},
    {	0x7C, 124,	},
    {	0x00,   0,	},
};


/*==========================================================================*
 *				patch_set2_84				    *
 *==========================================================================*/
PRIVATE void patch_set2_84()
{
	/* Patch the the scan code to keey number mapping for a 84 key
	 * keyboard.
	 */
	u8_t *map;
	struct ext_keymap *extmap;

	if (debug)
		printf("keyboard: patching set2 for a 84-key keyboard\n");

	map= map_set2_to_keynum;
	extmap= map_E0_set2_to_keynum;

	map[0x02]= 118;	/* F7 */
	map[0x7F]= 124;	/* SysReq */
}

/*==========================================================================*
 *				map_set2				    *
 *==========================================================================*/
PRIVATE int map_set2(code, brk)
int code;
int *brk;
{
	static enum { S_BASIC, S_ESCAPE, S_ESCAPE_BREAK, S_E1 } state;

	int k, i;
	u8_t *map;
	struct ext_keymap *extmap;

	map= map_set2_to_keynum;
	extmap= map_E0_set2_to_keynum;
	k= -1;
	*brk =0;

#if 0
	if (debug)
		printf("map_set2(0x%x), state= %d\n", code, state);
#endif

	switch(state)
	{
	case S_BASIC:
		if (code == KBD_RES_BREAK)
		{
			*brk= 1;
			return -1;
		}
		if (code == 0xE0)
		{
			state= S_ESCAPE;
			break;
		}
		if (code == 0xE1)
		{
			state= S_E1;
			break;
		}
		if (code < 0 || code >= KEYMAP_ENTRIES)
		{
			printf("map_set2: scancode out of range (0x%x)\n", code);
			break;
		}
		k= map_set2_to_keynum[code];
		break;
	case S_ESCAPE:
		if (code == KBD_RES_BREAK)
		{
			state= S_ESCAPE_BREAK;
			break;
		}
		if (code == 0x12 || code == 0x59)
		{
			/* This indicate a shift state, ignore */
			state= S_BASIC;
			break;
		}

		for (i= 0; extmap[i].code; i++)
		{
			if (extmap[i].code == code)
				break;
		}
		if (extmap[i].code == 0)
		{
			printf(
			"map_set2: extended combination unknown: E0 %02X\n",
				code);
			state= S_BASIC;
			break;
		}
		k= extmap[i].key;
		state= S_BASIC;
		break;
	case S_ESCAPE_BREAK:
		if (code == 0x12 || code == 0x59)
		{
			/* This indicate a shift state, ignore */
			state= S_BASIC;
			break;
		}

		for (i= 0; extmap[i].code; i++)
		{
			if (extmap[i].code == code)
				break;
		}
		if (extmap[i].code == 0)
		{
			printf(
			"map_set2: extended combination unknown: E0 F0 %02X\n",
				code);
			state= S_BASIC;
			break;
		}
		k= extmap[i].key;
		*brk= 1;
		state= S_BASIC;
		break;
	case S_E1:
		if (code == 0xE1)
		{
			state= S_BASIC;
			break;
		}
		if (code == 0x14)
			break;	/* ignore */
		if (code == 0x77)
		{
			k= 126;
			break;
		}
		printf("map_set2: strange sequence: E1 .. %02X\n",
			code);
		state= S_BASIC;
		break;
	default:
		panic("map_set2: strange state", state);
	}

	return k;
}


/*
 * $PchId: keyboard.c,v 1.9 1996/03/14 08:21:32 philip Exp $
 */
