/* This file contains routines for initializing the 8259 interrupt controller:
 *	get_irq_handler: address of handler for a given interrupt
 *	put_irq_handler: register an interrupt handler
 *	intr_init:	initialize the interrupt controller(s)
 */

#include "kernel.h"

#define ICW1_AT         0x11	/* edge triggered, cascade, need ICW4 */
#define ICW1_PC         0x13	/* edge triggered, no cascade, need ICW4 */
#define ICW1_PS         0x19	/* level triggered, cascade, need ICW4 */
#define ICW4_AT         0x01	/* not SFNM, not buffered, normal EOI, 8086 */
#define ICW4_PC         0x09	/* not SFNM, buffered, normal EOI, 8086 */

FORWARD _PROTOTYPE( int spurious_irq, (int irq) );


/*=========================================================================*
 *				get_irq_handler				   *
 *=========================================================================*/
PUBLIC irq_handler_t get_irq_handler(irq)
int irq;
{
/* Return the handler registered for interrupt irq, null if no handler. */

  if (irq < 0 || irq >= NR_IRQ_VECTORS)
	panic("invalid call to get_irq_handler", irq);

  if (irq_table[irq] == spurious_irq)
	return(0);	/* Not a real handler */

  return irq_table[irq];
}


/*=========================================================================*
*				put_irq_handler				   *
*=========================================================================*/
PUBLIC void put_irq_handler(irq, handler)
int irq;
irq_handler_t handler;
{
/* Register an interrupt handler. */

  if (irq < 0 || irq >= NR_IRQ_VECTORS)
	panic("invalid call to put_irq_handler", irq);

  if (irq_table[irq] == handler)
	return;		/* extra initialization */

  if (handler == 0)
	handler= spurious_irq;
  else
  if (irq_table[irq] != spurious_irq)
	panic("attempt to register second irq handler for irq", irq);

  disable_irq(irq);
  irq_table[irq]= handler;
  irq_use |= 1 << irq;
  irq_admin[irq].irq_spur_cnt= 0;
  irq_admin[irq].irq_spur_nxt= 1;
}


/*=========================================================================*
*				spurious_irq				   *
*=========================================================================*/
PRIVATE int spurious_irq(irq)
int irq;
{
/* Default interrupt handler.  It complains a lot. */

  if (irq < 0 || irq >= NR_IRQ_VECTORS)
	panic("invalid call to spurious_irq", irq);

  irq_admin[irq].irq_spur_cnt++;
  if (irq_admin[irq].irq_spur_cnt == irq_admin[irq].irq_spur_nxt) {
	printf("# of spurious irqs for irq %d is now %lu\n", irq,
		irq_admin[irq].irq_spur_cnt);
	irq_admin[irq].irq_spur_nxt *= 10;
  }
  return 1;	/* Reenable interrupt */
}


/*==========================================================================*
 *				intr_init				    *
 *==========================================================================*/
PUBLIC void intr_init(mine)
int mine;
{
/* Initialize the 8259s, finishing with all interrupts disabled.  The flag
 * "mine" is set if the 8259s are to be programmed for Minix, or to be reset
 * to what the BIOS expects.
 */
  int i;

  lock();

  /* Two interrupt controllers, one master, one slaved at IRQ 2. */
  out_byte(INT_CTL, ps_mca ? ICW1_PS : ICW1_AT);
  out_byte(INT_CTLMASK, mine ? IRQ0_VECTOR : BIOS_IRQ0_VEC);
  						/* ICW2 for master */
  out_byte(INT_CTLMASK, (1 << CASCADE_IRQ));	/* ICW3 tells slaves */
  out_byte(INT_CTLMASK, ICW4_AT);
  out_byte(INT_CTLMASK, ~(1 << CASCADE_IRQ));	/* IRQ 0-7 mask */
  out_byte(INT2_CTL, ps_mca ? ICW1_PS : ICW1_AT);
  out_byte(INT2_CTLMASK, mine ? IRQ8_VECTOR : BIOS_IRQ8_VEC);
  						/* ICW2 for slave */
  out_byte(INT2_CTLMASK, CASCADE_IRQ);		/* ICW3 is slave nr */
  out_byte(INT2_CTLMASK, ICW4_AT);
  out_byte(INT2_CTLMASK, ~0);			/* IRQ 8-15 mask */

  /* Initialize the table of interrupt handlers. */
  for (i = 0; i< NR_IRQ_VECTORS; i++) {
	irq_table[i]= spurious_irq;
	irq_admin[i].irq_spur_cnt= 0;
	irq_admin[i].irq_spur_nxt= 1;
  }
}

/*
 * $PchId: i8259.c,v 1.5 1996/01/19 23:30:34 philip Exp $
 */
