/* Code and data for the IBM console driver.
 *
 * The 6845 video controller used by the IBM PC shares its video memory with
 * the CPU somewhere in the 0xB0000 memory bank.  To the 6845 this memory
 * consists of 16-bit words.  Each word has a character code in the low byte
 * and a so-called attribute byte in the high byte.  The CPU directly modifies
 * video memory to display characters, and sets two registers on the 6845 that
 * specify the video origin and the cursor position.  The video origin is the
 * place in video memory where the first character (upper left corner) can
 * be found.  Moving the origin is a fast way to scroll the screen.  Some
 * video adapters wrap around the top of video memory, so the origin can
 * move without bounds.  For other adapters screen memory must sometimes be
 * moved to reset the origin.  All computations on video memory use character
 * (word) addresses for simplicity and assume there is no wrapping.  The
 * assembly support functions translate the word addresses to byte addresses
 * and the scrolling function worries about wrapping.
 *
 * This console driver implements the same escape sequences as the X11 xterm
 * program.  (Or at least ignores them well enough.)  As terminal type 'vs100'
 * is used, one of the alternate names of 'xterm'.  When necessary one can
 * also use 'xterm', 'minix', 'vt100', or 'ansi'.  (The driver implements a
 * superset of 'minix' and 'ansi', and should do 'vt100' well enough for
 * most circumstances.)
 */

#include "kernel.h"

#include <termios.h>
#include <sys/ioctl.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include "assert.h"
INIT_ASSERT
#include "mq.h"
#include "proc.h"
#include "i386/protect.h"
#include "timer.h"
#include "timerreg.h"
#include "tty.h"
#include "config.h"

/*
 * Requests:
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_COUNT NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_OPEN    |minor dev  | proc nr   |  fd       |           |           |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_CLOSE   |minor dev  | proc nr   |  fd       |           |           |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_READ    |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_WRITE   |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_IOCTL NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_IOCTL3  |minor dev  | proc nr   |  fd       |  command  | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_OPERATION
 * -------------------------------------------------------------------|
 * | DEV_CANCEL  |minor dev  | proc nr   |  fd       | which operation|
 * |-------------+-----------+-----------+-----------+----------------|
 *
 * Replies:
 *
 *    m_type        REP_PROC_NR   REP_STATUS   REP_REF    REP_OPERATION
 * ----------------------------------------------------------------------|
 * | DEVICE_REPLY |   proc nr   |  status    |  fd     | which operation |
 * |--------------+-------------+------------+---------+-----------------|
 */

#define NCONSOLE   NR_CONSOLES	/* Number of consoles (from config.h) */
#define TTY_RB_SIZ	260	/* Should be at least T_RD_BUF_SIZ_MIN */
#define TTY_WB_SIZ	256	/* Should be at least T_WR_BUF_SIZ_MIN */
#define CONS_RAM_WORDS  132	/* ram buffer size */
#define MAX_ESC_PARMS     2	/* number of escape sequence params allowed */

#if PROFILING
#define PROFILE_MINOR	 16
#endif /* PROFILING */

#if _PKG_LOG
#define LOG_MINOR	 17
#endif /* _PKG_LOG */

#if MAPPED_VGA
#define VGA_MINOR	128
#endif /* MAPPED_VGA */

#if _PKG_RAWKBD
#define RAWKBD_MINOR	129
#define RAWKBD_AUX_MINOR	130
#endif /* _PKG_RAWKBD */

typedef struct console
{
	int		c_flags;
	char		c_tty_readbuf[TTY_RB_SIZ];
	char		c_tty_writebuf[TTY_WB_SIZ];
	struct tty	c_tty;
	int		c_esc_state;
	int		c_column;
	int		c_row;
	int		c_rwords;
	u16_t		c_ramqueue[CONS_RAM_WORDS];
	unsigned	c_start;
	unsigned	c_limit;
	unsigned	c_org;
	unsigned	c_usr_width;
	int		c_esc_parmv[MAX_ESC_PARMS];
						/* list of escape parameters */
	int *		c_esc_parmp;
				/* pointer to current escape parameter */
	unsigned	c_cur;	/* current position of cursor in video RAM */
	unsigned	c_attr;		/* character attribute */
	unsigned	c_blank;	/* blank attribute */
} console_t;

#define CF_TTY_CLOSED		1
#define CF_READ_INITIATIVE	4

#define ESC		033
#define ES(c1,c2)  ((((c1)&0xFF)<<8)|(c2))  /* escape state after ESC c1 c2 */

PRIVATE int nconsole= 1;
PRIVATE int current;
PRIVATE console_t cons_table[NCONSOLE];
PRIVATE console_t *curcons;
PRIVATE mq_t *repl_queue, *repl_queue_tail;
PRIVATE tmrs_context_ut cons_cntxt;
PRIVATE tmra_ut blank_timer;
PRIVATE int blank_time;
PRIVATE int exp_timers= 0;

/* Definitions used by the console driver. */
#define MONO_BASE    0xB0000L	/* base of mono video memory */
#define COLOR_BASE   0xB8000L	/* base of color video memory */
#define MONO_SIZE     0x1000	/* 4K mono video memory */
#define COLOR_SIZE    0x4000	/* 16K color video memory */
#define EGA_SIZE      0x8000	/* EGA & VGA have at least 32K */
#define BLANK_COLOR   0x0700	/* determines cursor color on blank screen */
#define STATUS_COLOR  0x7000	/* reverse video status line */
#define SCROLL_UP          0	/* scroll forward */
#define SCROLL_DOWN        1	/* scroll backward */
#define STATUS_DISABLE	   0	/* disable status line */
#define STATUS_ENABLE	   1	/* enable status line */
#define BLANK_MEM ((u16_t *) 0)	/* tells mem_vid_copy() to blank the screen */
#define BLANK_TIME        10	/* default #min to blank a VGA screen */

/* Constants relating to the controller chips. */
#define M_6845         0x3B4	/* base port for 6845 mono */
#define C_6845         0x3D4	/* base port for 6845 color */
#define INDEX              0	/* 6845's index register */
#define DATA               1	/* 6845's data register */
#define VID_ORG           12	/* 6845's origin register */
#define CURSOR            14	/* 6845's cursor register */

/* Beeper. */
#define BEEP_FREQ	 896	/* beep frequency in HZ */
#define B_TIME		   3	/* length of CTRL-G beep is ticks */

#define TAB_SIZE           8	/* distance between tab stops */
#define TAB_MASK           7	/* mask to compute a tab stop position */

/* Global variables used by the console driver and assembly support. */
PUBLIC unsigned vid_seg;	/* video ram selector (0xB0000 or 0xB8000) */
PUBLIC unsigned vid_size;	/* 0x2000 for color or 0x0800 for mono */
PUBLIC unsigned vid_mask;	/* 0x1FFF for color or 0x07FF for mono */
PUBLIC unsigned blank_color;	/* display code for blank */

/* Private variables used by the console driver. */
PRIVATE int vid_port;		/* I/O port for accessing 6845 */
PRIVATE int wrap;		/* hardware can wrap? */
PRIVATE int softscroll;		/* 1 = software scrolling, 0 = hardware */
PRIVATE int beeping;		/* speaker is beeping? */
PRIVATE int cons_tasknr= ANY;	/* ANY to prevent accidents */
PRIVATE unsigned font_lines;	/* font lines per character */
PRIVATE unsigned scr_width;	/* # characters on a line */
PRIVATE unsigned scr_lines;	/* # lines on the screen */
PRIVATE unsigned scr_size;	/* # characters on the screen */
/*	 cons->c_usr_width;	 * # user accessable characters on a line */
PRIVATE unsigned usr_lines;	/* # user accessable lines on the screen */
PRIVATE unsigned usr_size;	/* # user accessable characters on the screen */

/* Color if using a color controller. */
#define color	(vid_port == C_6845)

/* Map from ANSI colors to the attributes used by the PC */
PRIVATE int ansi_colors[8] = {0, 4, 2, 6, 1, 5, 3, 7};

/* Map ISO Latin-1 codes 128-255 onto the IBM character set if possible.
 * Letters that can't be translated lose their accents, other glyphs become
 * code NIH.
 */
#define NIH	254		/* Not Latin-1 to IBM translatable */

PRIVATE u8_t map_lat1_ibm[128] = {
 32,   4, 177, NIH, NIH, NIH, NIH, 248, 241, NIH, NIH, 217, 191, 218, 192, 197,
NIH, NIH, 196, NIH, NIH, 195, 180, 193, 194, 179, 243, 242, 227, NIH, 156, 250,
 32, 173, 155, 156, NIH, 157, 124,  21, NIH, NIH, 166, 174, 170,  45, NIH, NIH,
248, 241, 253, NIH,  39, 230,  20, 250, NIH, NIH, 167, 175, 172, 171, NIH, 168,
'A', 'A', 'A', 'A', 142, 143, 146, 128, 'E', 144, 'E', 'E', 'I', 'I', 'I', 'I',
'D', 165, 'O', 'O', 'O', 'O', 153, NIH, 'O', 'U', 'U', 'U', 154, 'Y', NIH, 225,
133, 160, 131, 'a', 132, 134, 145, 135, 138, 130, 136, 137, 141, 161, 140, 139,
'o', 164, 149, 162, 147, 'o', 148, 246, 237, 151, 163, 150, 129, 'y', NIH, 152,
};

FORWARD _PROTOTYPE( void cons_init, (void)				);
FORWARD _PROTOTYPE( int reply_queue, (int proc, int ref, int operation)	);
FORWARD _PROTOTYPE( void reply, (mq_t *mq, int result, int can_enqueue)	);
FORWARD _PROTOTYPE( void cons_get, (int cons_line)			);
FORWARD _PROTOTYPE( int cons_put, (int cons_line, char *buf, int siz)	);
FORWARD _PROTOTYPE( void cons_reply, (int cons_line, mq_t *mq)		);
FORWARD _PROTOTYPE( void flush, (console_t *cons)			);
FORWARD _PROTOTYPE( void cons_org0, (void)				);
FORWARD _PROTOTYPE( void select_console, (int cons_line)		);
FORWARD _PROTOTYPE( void out_char, (console_t *cons, int c)		);
FORWARD _PROTOTYPE( void set_6845, (int reg, unsigned val)		);
FORWARD _PROTOTYPE( void parse_escape, (console_t *cons, int c)		);
FORWARD _PROTOTYPE( void beep, (void)					);
FORWARD _PROTOTYPE( void scroll_screen, (console_t *cons, int dir)	);
FORWARD _PROTOTYPE( void status_state, (int enable)			);
FORWARD _PROTOTYPE( void delimit_line, (console_t *cons, unsigned offset,
						unsigned nlines)	);
FORWARD _PROTOTYPE( void stop_beep, (void)				);
FORWARD _PROTOTYPE( void blank_video, (struct tmra *tp, tmr_arg_ut arg)	);
FORWARD _PROTOTYPE( void read_kb, (void)				);
FORWARD _PROTOTYPE( int con_copyfont, (int proc_nr, ioreq_t request,
						vir_bytes font_vir)	);


/*===========================================================================*
 *				cons_task				     *
 *===========================================================================*/
PUBLIC void cons_task()
{
	mq_t *mq;
	int result;
	int minor, proc;
	console_t *cons;

	cons_tasknr= proc_number(proc_ptr);
	clck_tasknr= findproc(CLOCK_NAME);

	cons_init();
#if PROFILING
	profile_init(reply, cons_tasknr);
#endif /* PROFILING */
#if _PKG_LOG
	log_init();
	log_setreply(reply, cons_tasknr);
#endif /* _PKG_LOG */
#if MAPPED_VGA
	vga_init(reply);
#endif /* MAPPED_VGA */
#if _PKG_RAWKBD
	rawkbd_init(reply, cons_tasknr);
#endif /* _PKG_RAWKBD */

	while(TRUE)
	{
		mq= mq_get();
		if (!mq)
			panic("out of messages", NO_NUM);

		result= receive(ANY, &mq->mq_mess);
		if (result < 0)
			panic("unable to receive: ", result);

		if (mq->mq_mess.m_source == HARDWARE)
		{
			assert(mq->mq_mess.m_type == HARD_INT);
			if (kbd_int_pending)
				read_kb();

			if (exp_timers)
				tmrs_exptimers(&cons_cntxt);

#if PROFILING
			if (profile_int_pending)
				profile_int();
#endif /* PROFILING */
#if _PKG_LOG
			if (log_int_pending)
				log_int();
#endif /* _PKG_LOG */
#if _PKG_RAWKBD
			if (rawkbd_int_pending)
				rawkbd_int();
#endif /* _PKG_RAWKBD */

			mq_free(mq);
			continue;
		}

		if (mq->mq_mess.m_type == DEV_IOCTL)
		{
			/* Compatibility code for old (sgtty) ioctls */
			minor= mq->mq_mess.m2_i1;
			if (minor >= 0 && minor < NCONSOLE)
			{
				/* This packet is handled by the tty library. */
				cons= &cons_table[minor];
				tty_receive(&cons->c_tty, mq);
				continue;
			}
			/* The console task doesn't implement old ioctls */
			proc= mq->mq_mess.m2_i2;	/* PROC_NR */
			mq->mq_mess.m_type= TASK_REPLY;
			mq->mq_mess.m2_i1= proc;	/* REP_PROC_NR */
			mq->mq_mess.m2_i2= ENOTTY;
			result= send(mq->mq_mess.m_source, &mq->mq_mess);
			assert(result == OK);
			mq_free(mq);
			continue;
		}

		assert(mq->mq_mess.m_type != DEV_IOCTL);
		if (repl_queue)
		{
			if (mq->mq_mess.m_type == DEV_CANCEL)
			{
				result= reply_queue(mq->mq_mess.NDEV_PROC,
					mq->mq_mess.NDEV_REF,
					mq->mq_mess.NDEV_OPERATION);
				if (result)
				{
					/* The reply for the canceled request
					 * was already in the queue.
					 */
					mq_free(mq);
					continue;
				}
			}
			else
				reply_queue(ANY, 0, 0);
		}

		minor= mq->mq_mess.NDEV_MINOR;
		if (minor >= 0 && minor < nconsole)
		{
			/* Font change? */
			if (mq->mq_mess.m_type == DEV_IOCTL3
				&& (mq->mq_mess.NDEV_IOCTL == TIOCSFON
				 || mq->mq_mess.NDEV_IOCTL == TIOCGFON))
			{
				result= con_copyfont(mq->mq_mess.NDEV_PROC,
					mq->mq_mess.NDEV_IOCTL,
					(vir_bytes) mq->mq_mess.NDEV_BUFFER);
				reply(mq, result, FALSE);
				mq_free(mq);
				continue;
			}
			/* This packet is handled by the tty library. */
			cons= &cons_table[minor];
			cons->c_tty.t_winsize.ws_row= usr_lines;
			cons->c_tty.t_winsize.ws_col= cons->c_usr_width;
			cons->c_tty.t_winsize.ws_xpixel= cons->c_usr_width * 8;
			cons->c_tty.t_winsize.ws_ypixel= usr_lines * font_lines;
			tty_receive(&cons->c_tty, mq);
			continue;
		}
#if PROFILING
		if (minor == PROFILE_MINOR)
		{
			profile_mess(mq);
			continue;
		}
#endif /* PROFILING */

#if _PKG_LOG
		if (minor == LOG_MINOR)
		{
			log_mess(mq);
			continue;
		}
#endif /* _PKG_LOG */
#if _PKG_RAWKBD
		if (minor == RAWKBD_MINOR)
		{
			rawkbd_mess(mq);
			continue;
		}
		if (minor == RAWKBD_AUX_MINOR)
		{
			kbdaux_mess(mq);
			continue;
		}
#endif /* _PKG_RAWKBD */

#if MAPPED_VGA
		if (minor == VGA_MINOR)
		{
			vga_mess(mq);
			continue;
		}
#endif /* MAPPED_VGA */

		if (minor < 0 || minor >= 0x100)
			panic("console.c: illegal minor", minor);
		reply(mq, ENXIO, FALSE);
		mq_free(mq);
		continue;
	}
}


/*===========================================================================*
 *				reply					     *
 *===========================================================================*/
PRIVATE void reply(mq, result, can_enqueue)
mq_t *mq;
int result;
int can_enqueue;
{
	int proc, ref, status, operation;
	message m, *mp;

	assert(mq->mq_allocated);

	proc= mq->mq_mess.NDEV_PROC;
	ref= mq->mq_mess.NDEV_REF;
	operation= mq->mq_mess.m_type;

	if (can_enqueue)
		mp= &mq->mq_mess;
	else
		mp= &m;

	mp->m_type= DEVICE_REPLY;
	mp->REP_PROC_NR= proc;
	mp->REP_STATUS= result;
	mp->REP_REF= ref;
	mp->REP_OPERATION= operation;

	status= send(mq->mq_mess.m_source, mp);
	if (status == ELOCKED && can_enqueue)
	{
		mq->mq_next= NULL;
		if (repl_queue)
			repl_queue_tail->mq_next= mq;
		else
			repl_queue= mq;
		repl_queue_tail= mq;
		return;
	}
	if (status != OK)
		panic("send failed", result);
	if (can_enqueue)
		mq_free(mq);
}


/*===========================================================================*
 *				reply_queue				     *
 *===========================================================================*/
PRIVATE int reply_queue(proc, ref, operation)
int proc;
int ref;
int operation;
{
	mq_t *mq, *cancel_reply, *tmp_m;
	int result;

	cancel_reply= NULL;
	for (mq= repl_queue; mq;)
	{
		if (cancel_reply == NULL &&
			mq->mq_mess.REP_PROC_NR == proc &&
			mq->mq_mess.REP_REF == ref &&
			(operation == CANCEL_ANY || operation ==
			mq->mq_mess.REP_OPERATION))
		{	/* Found a reply to the canceled request. */
			cancel_reply= mq;
			mq= mq->mq_next;
			continue;
		}
		result= send(mq->mq_mess.m_source, &mq->mq_mess);
		if (result != OK)
			panic("send failed", result);
		tmp_m= mq;
		mq= mq->mq_next;
		mq_free(tmp_m);
	}
	if (cancel_reply)
	{
		result= send(cancel_reply->mq_mess.m_source,
			&cancel_reply->mq_mess);
		if (result != OK)
			panic("send failed", result);
		mq_free(cancel_reply);
	}
	repl_queue= NULL;
	if (cancel_reply)
		return 1;
	else
		return 0;
}


/*===========================================================================*
 *				cons_init				     *
 *===========================================================================*/
PRIVATE void cons_init()
{
	int i;
	console_t *cons;

	repl_queue= NULL;

	tmrs_initcontext(&cons_cntxt, &exp_timers, cons_tasknr);
	for (i = 0; i < nconsole; i++)
	{
		cons = &cons_table[i];
		cons->c_flags= CF_READ_INITIATIVE;
		tty_init(&cons->c_tty, i, cons->c_tty_readbuf, TTY_RB_SIZ,
			cons->c_tty_writebuf, TTY_WB_SIZ, cons_tasknr,
			&cons_cntxt);
		cons->c_tty.t_get= cons_get;
		cons->c_tty.t_put= cons_put;
		cons->c_tty.t_reply= cons_reply;
		scr_init(i);
	}
	tmra_inittimer(&blank_timer);
	enable_video(TRUE);
	kb_init(0, cons_tasknr);
}


/*===========================================================================*
 *				cons_stop				     *
 *===========================================================================*/
PUBLIC void cons_stop()
{
	/* Prepare for halt or reboot. */
	log_stop();
#if PROFILING
	prof_stop();
#endif
	cons_org0();
	softscroll = 1;
	select_console(0);
	cons_table[0].c_attr = cons_table[0].c_blank = BLANK_COLOR;
	status_state(STATUS_DISABLE);
	enable_video(TRUE);
	tmra_clrtimer(&blank_timer);
}


/*===========================================================================*
 *				cons_get				     *
 *===========================================================================*/
PRIVATE void cons_get(cons_line)
int cons_line;
{
/* The tty wants some data, but we don't have any. */

	console_t *cons;

	assert (cons_line >= 0 || cons_line < NCONSOLE);
	cons= &cons_table[cons_line];

	tty_put(&cons->c_tty, NULL, 0);
}


/*===========================================================================*
 *				cons_put				     *
 *===========================================================================*/
PRIVATE int cons_put(cons_line, buf, siz)
int cons_line;
char *buf;
int siz;
{
/* We got some data from tty. */
	console_t *cons;
	int total;
	int c;

	if (log_cons_output)
		log_put(buf, siz, proc_number(proc_ptr));
	if (disable_cons_output)
		return siz;

	assert (cons_line >= 0 || cons_line < NCONSOLE);
	cons= &cons_table[cons_line];

	if (siz == 0)
	{
		/* This is a flush command. */
		flush(cons);
		return 0;
	}

	total= siz;				/* We always do every thing. */
	assert(siz > 0);

	do {
		c = *buf++ & BYTE;
		if (c < ' ' || cons->c_esc_state != 0
			|| cons->c_column >= cons->c_usr_width - 1
			|| cons->c_rwords >= CONS_RAM_WORDS)
		{
			out_char(cons, c);
		} else {
			if (c >= 0x80) c = map_lat1_ibm[c - 0x80];
			cons->c_ramqueue[cons->c_rwords++] = cons->c_attr | c;
			cons->c_column++;
		}
	}
	while (--siz != 0);

	return total;
}


/*===========================================================================*
 *				cons_reply				     *
 *===========================================================================*/
PRIVATE void cons_reply(cons_line, mq)
int cons_line;
mq_t *mq;
{
/* Tty wants to send a reply but the send is block because FS is sending to
 * this task. Enqueue to reply.
 */
	console_t *cons;

	assert (cons_line >= 0 || cons_line < NCONSOLE);
	cons= &cons_table[cons_line];

printf("in cons_reply\n");
	mq->mq_next= NULL;
	if (repl_queue)
		repl_queue_tail->mq_next= mq;
	else
		repl_queue= mq;
	repl_queue_tail= mq;
}


/*===========================================================================*
 *				scr_init				     *
 *===========================================================================*/
PUBLIC void scr_init(minor)
int minor;
{
/* Initialize the screen driver. */
  console_t *cons;
  phys_bytes vid_base;
  unsigned page_size;
  long v;
  u16_t bios_columns, bios_crtbase, bios_fontlines;
  u8_t bios_rows;
  static int initialized0 = FALSE;	/* "screen 0" is initialized early */
  static char var[] = "console";
  static char fmt[] = "x,d:d:d";

  if (minor == 0) {
	if (initialized0) return;
	initialized0 = TRUE;
  }
  cons= &cons_table[minor];

  /* Get the BIOS parameters that describe the VDU. */
  vm_map_zp(TRUE);
  phys_copy(0x44AL, vir2phys(&bios_columns), 2L);
  phys_copy(0x463L, vir2phys(&bios_crtbase), 2L);
  phys_copy(0x484L, vir2phys(&bios_rows), 1L);
  phys_copy(0x485L, vir2phys(&bios_fontlines), 2L);
  vm_map_zp(FALSE);

  vid_port = bios_crtbase;
  scr_width = bios_columns;
  font_lines = bios_fontlines;
  cons->c_usr_width = scr_width;		/* initially no line limit */
  scr_lines = ega ? bios_rows+1 : 25;
  usr_lines = scr_lines;			/* initially no status line */

  if (color) {
	vid_base = COLOR_BASE;
	vid_size = COLOR_SIZE;
  } else {
	vid_base = MONO_BASE;
	vid_size = MONO_SIZE;
  }
  if (ega) vid_size = EGA_SIZE;
  wrap = !ega;

  vid_seg = protected_mode ? VIDEO_SELECTOR : physb_to_hclick(vid_base);
  init_dataseg(&gdt[VIDEO_INDEX], vid_base, (phys_bytes) vid_size,
							TASK_PRIVILEGE);
  vid_size >>= 1;		/* word count */
  vid_mask = vid_size - 1;

  /* Size of the screen (number of displayed characters.) */
  scr_size = scr_lines * scr_width;
  usr_size = scr_size;

  /* There can be as many consoles as video memory allows. */
  nconsole = vid_size / scr_size;
  if (nconsole > NCONSOLE) nconsole = NCONSOLE;
  if (nconsole > 1) wrap = 0;
  page_size = vid_size / nconsole;
  cons->c_start = minor * page_size;
  cons->c_limit = cons->c_start + page_size;
  cons->c_org = cons->c_start;
  cons->c_attr = cons->c_blank = BLANK_COLOR;

  /* Clear the screen. */
  blank_color = BLANK_COLOR;
  mem_vid_copy(BLANK_MEM, cons->c_start, scr_size);
  select_console(0);

  /* Line limit? */
  v = scr_width;
  (void) env_parse(var, fmt, 1, &v, 8L, (long) scr_width);
  cons->c_usr_width = v;
  delimit_line(cons, cons->c_start, scr_lines);

  /* Software scrolling? */
  v = FALSE;
  (void) env_parse(var, fmt, 2, &v, 0L, 1L);
  softscroll = v;

  /* Screenblank timeout? */
  v = BLANK_TIME;
  (void) env_parse(var, fmt, 3, &v, 0L, 60L);
  blank_time = v;
}


/*===========================================================================*
 *				select_console				     *
 *===========================================================================*/
PRIVATE void select_console(int cons_line)
{
/* Set the current console to console number 'cons_line'. */

  if (cons_line < 0 || cons_line >= nconsole) return;
  if (current != cons_line) status_state(STATUS_ENABLE);
  current = cons_line;
  curcons = &cons_table[cons_line];
  set_6845(VID_ORG, curcons->c_org);
  set_6845(CURSOR, curcons->c_cur);
}


/*===========================================================================*
 *				out_char				     *
 *===========================================================================*/
PRIVATE void out_char(cons, c)
register console_t *cons;	/* pointer to console struct */
int c;				/* character to be output */
{
/* Output a character on a console.  Check for escape sequences first. */
  if (cons->c_esc_state != 0) {
	parse_escape(cons, c);
	return;
  }

  switch (c) {
	case 000:		/* null is typically used for padding */
		return;		/* better not do anything */

	case 007:		/* ring the bell */
		flush(cons);	/* print any chars queued for output */
		beep();
		return;

	case '\b':		/* backspace */
		if (--cons->c_column < 0) {
			if (--cons->c_row >= 0)
				cons->c_column += cons->c_usr_width;
		}
		flush(cons);
		return;

	case '\n':		/* line feed */
	case 013:		/* CTRL-K */
	case 014:		/* CTRL-L */
		if (cons->c_row == usr_lines-1) {
			scroll_screen(cons, SCROLL_UP);
		} else {
			cons->c_row++;
			flush(cons);
		}
		return;

	case '\r':		/* carriage return */
		cons->c_column = 0;
		flush(cons);
		return;

	case '\t':		/* tab */
		cons->c_column = (cons->c_column + TAB_SIZE) & ~TAB_MASK;
		if (cons->c_column > cons->c_usr_width) {
			cons->c_column -= cons->c_usr_width;
			if (cons->c_row == usr_lines-1) {
				scroll_screen(cons, SCROLL_UP);
			} else {
				cons->c_row++;
			}
		}
		flush(cons);
		return;

	case 016:		/* CTRL-N Shift Out (ignored) */
	case 017:		/* CTRL-O Shift In (ignored) */
		return;

	case ESC:		/* ESC - start of an escape sequence */
		parse_escape(cons, ESC);
		return;

	default:		/* printable chars are stored in ramqueue */
		if (cons->c_column >= cons->c_usr_width) {
			if (cons->c_row == usr_lines-1) {
				scroll_screen(cons, SCROLL_UP);
			} else {
				cons->c_row++;
			}
			cons->c_column = 0;
			flush(cons);
		}
		if (cons->c_rwords == CONS_RAM_WORDS) flush(cons);
		if (c >= 0x80) c = map_lat1_ibm[c - 0x80];
		cons->c_ramqueue[cons->c_rwords++]= cons->c_attr | c;
		cons->c_column++;	/* next column */
		return;
  }
}


/*===========================================================================*
 *				scroll_screen				     *
 *===========================================================================*/
PRIVATE void scroll_screen(cons, dir)
register console_t *cons;	/* pointer to console struct */
int dir;			/* SCROLL_UP or SCROLL_DOWN */
{
  unsigned new_line, new_org, chars;

  flush(cons);
  chars = usr_size - scr_width;		/* one screen minus one line */

  /* Scrolling the screen is a real nuisance due to the various incompatible
   * video cards.  This driver supports software scrolling (Hercules?),
   * hardware scrolling (mono and CGA cards) and hardware scrolling without
   * wrapping (EGA cards).  In the latter case we must make sure that
   *		c_start <= c_org && c_org + scr_size <= c_limit
   * holds, because EGA doesn't wrap around the end of video memory.
   */
  if (dir == SCROLL_UP) {
	/* Scroll one line up in 3 ways: soft, avoid wrap, use origin. */
	if (softscroll) {
		vid_vid_copy(cons->c_start + scr_width, cons->c_start, chars);
	} else
	if (!wrap && cons->c_org + scr_size + scr_width >= cons->c_limit) {
		vid_vid_copy(cons->c_org + scr_width, cons->c_start, chars);
		vid_vid_copy(cons->c_org + usr_size, cons->c_start + usr_size,
								scr_width);
		cons->c_org = cons->c_start;
	} else {
		vid_vid_copy(cons->c_org + usr_size, cons->c_org + scr_size,
								scr_width);
		cons->c_org = (cons->c_org + scr_width) & vid_mask;
	}
	new_line = (cons->c_org + chars) & vid_mask;
  } else {
	/* Scroll one line down in 3 ways: soft, avoid wrap, use origin. */
	if (softscroll) {
		vid_vid_copy(cons->c_start, cons->c_start + scr_width, chars);
	} else
	if (!wrap && cons->c_org < cons->c_start + scr_width) {
		new_org = cons->c_limit - scr_size;
		vid_vid_copy(cons->c_org + usr_size, new_org + usr_size,
								scr_width);
		vid_vid_copy(cons->c_org, new_org + scr_width, chars);
		cons->c_org = new_org;
	} else {
		cons->c_org = (cons->c_org - scr_width) & vid_mask;
		vid_vid_copy(cons->c_org + scr_size, cons->c_org + usr_size,
								scr_width);
	}
	new_line = cons->c_org;
  }
  /* Blank the new line at top or bottom. */
  blank_color = cons->c_blank;
  mem_vid_copy(BLANK_MEM, new_line, scr_width);
  delimit_line(cons, new_line, 1);

  /* Set the new video origin. */
  if (cons == curcons) set_6845(VID_ORG, cons->c_org);
  flush(cons);
}


/*===========================================================================*
 *				status_state				     *
 *===========================================================================*/
PRIVATE void status_state(enable)
int enable;
{
/* Either enable or disable the status line.  To enable the status line the
 * screen is scrolled one line up if the cursor is on the bottom line, and
 * the bottom line is turned into a status line.  To disable the status line
 * it is blanked and the number of usable lines increased by one.
 */
  int cons_line;
  console_t *cons;
  u16_t ident;

  if (enable == (scr_lines - usr_lines)) return;   /* already en/disabled */

  for (cons_line = 0; cons_line < nconsole; cons_line++) {
	cons = &cons_table[cons_line];

	if (enable) {
		if (cons->c_row == usr_lines - 1) {
			cons->c_row--;
			scroll_screen(cons, SCROLL_UP);
		}
		blank_color = STATUS_COLOR;
		mem_vid_copy(BLANK_MEM, cons->c_org + scr_size - scr_width,
								scr_width);
		ident = (STATUS_COLOR | '0') + cons_line;
		mem_vid_copy(&ident, cons->c_org + scr_size - scr_width + 1, 1);
	} else {
		blank_color = cons->c_blank;
		mem_vid_copy(BLANK_MEM, cons->c_org + usr_size, scr_width);
		delimit_line(cons, cons->c_org + usr_size, 1);
	}
	if (cons->c_tty.t_sesldr)	/* signal change of screen size */
		cause_sig(cons->c_tty.t_sesldr, SIGWINCH);
  }
  usr_lines = enable ? scr_lines-1 : scr_lines;	/* give or take one line */
  usr_size = usr_lines * scr_width;
}


/*===========================================================================*
 *				delimit_line				     *
 *===========================================================================*/
PRIVATE void delimit_line(cons, offset, nlines)
register console_t *cons;	/* pointer to console struct */
unsigned offset, nlines;	/* video ram offset and # lines to delimit */
{
/* Add a vertical bar to delimit a line if lines are shorter than the screen
 * allows.
 */

  if (cons->c_usr_width < scr_width) {
	u16_t delimiter = cons->c_blank | '|';

	while (nlines > 0) {
		mem_vid_copy(&delimiter, offset + cons->c_usr_width, 1);
		offset += scr_width;
		nlines--;
	}
  }
}


/*===========================================================================*
 *				beep					     *
 *===========================================================================*/
PRIVATE void beep()
{
/* Making a beeping sound on the speaker (output for CRTL-G).
 */

  enable_video(TRUE);
  kbd_bell(BEEP_FREQ, B_TIME);
}


/*===========================================================================*
 *				kbd_bell				     *
 *===========================================================================*/
PUBLIC void kbd_bell(pitch, duration)
unsigned pitch;
unsigned duration;
{
/* Making a beeping sound on the speaker (output for CRTL-G).
 * This routine works by turning on the bits 0 and 1 in port B of the 8255
 * chip that drive the speaker.
 */

  message mess;
  unsigned beep_freq;

  if (beeping) return;
  if (pitch == 0 || pitch > 10000)
	return;				/* Strange pitch */
  if (duration == 0 || duration > 10*HZ)	/* Strange duration */
	return;
  if (cons_tasknr == ANY)
	return;				/* Not initialized yet */
  beep_freq= TIMER_DIV(pitch);

  /* set up timer channel 2 (square wave) */
  out_byte(TIMER_CTRL, TIMER_SEL2 | TIMER_SQWAVE | TIMER_16BIT);
  out_byte(TIMER2, beep_freq & BYTE);	/* load low-order bits of frequency */
  out_byte(TIMER2, (beep_freq >> 8) & BYTE);	/* now high-order bits */
  lock();			/* guard PORT_B from keyboard intr handler */
  out_byte(PORT_B, in_byte(PORT_B) | 3);	/* turn on beep bits */
  unlock();
  beeping = TRUE;

  mess.m_type = SET_ALARM;
  mess.CLOCK_PROC_NR = cons_tasknr;
  mess.DELTA_TICKS = duration;
  mess.FUNC_TO_CALL = (sighandler_t) stop_beep;
  sendrec(clck_tasknr, &mess);
}


/*===========================================================================*
 *				parse_escape				     *
 *===========================================================================*/
PRIVATE void parse_escape(cons, c)
register console_t *cons;	/* pointer to console struct */
int c;				/* next character in escape sequence */
{
/* The following XTERM escape sequences are currently supported.  The rest are
 * conveniently ignored.  Extra are the ANSI colors.  If n, m, or s are
 * omitted, they default to 1:
 *
 *   ESC E - Next Line (same effect as ^M^J)
 *   ESC D - Index (same effect as ^J)
 *   ESC M - Reverse Index (like ^J but moving up)
 *   ESC c - Full Reset
 *   ESC [nA - Cursor Up n Lines
 *   ESC [nB - Cursor Down n Lines
 *   ESC [nC - Cursor Forward n Columns
 *   ESC [nD - Cursor Backward n Columns
 *   ESC [m;nH" - Cursor Position to (m,n)
 *   ESC [sJ - Erase in Display
 *   ESC [sK - Erase in Line
 *   ESC [nL - Insert n Lines
 *   ESC [nM - Delete n Lines
 *   ESC [n@ - Insert n Characters
 *   ESC [nP - Delete n Characters
 *   ESC [nm - Character Attributes (0=normal, 1=bold, 4=underline, 5=blinking,
 *             7=reverse, 30=black, 31=red, 32=green, 33=orange, 34=blue,
 *             35=magenta, 36=light blue, 37=white, 40-47=background color)
 *   ESC [?3h - 132 column mode
 *   ESC [?3l - 80 column mode
 *   ESC ]2;t^G - Write t to the Status Line
 */
  int n;
  unsigned src, dst, count;

  switch (cons->c_esc_state) {
    case 0:			/* ESC ... */
	flush(cons);	/* print any chars queued for output */
	cons->c_esc_parmp = cons->c_esc_parmv + MAX_ESC_PARMS;
	do {
		*--cons->c_esc_parmp = 0;
	} while (cons->c_esc_parmp > cons->c_esc_parmv);
	cons->c_esc_state = ESC;	/* mark ESC as seen */
	return;

    case ESC:			/* ESC 'c' */
	switch (c) {
	    case '[':	/* ESC [ ... - Control Sequence Introducer */
	    case ']':	/* ESC ] ... - Set Text Parameters */
		cons->c_esc_state = ES(ESC, c);
		return;

	    case 'E':	/* ESC E - Next Line */
		cons->c_column = 0;
		/*FALL THROUGH*/

	    case 'D':	/* ESC D - Index */
		if (cons->c_row == usr_lines-1) {
			scroll_screen(cons, SCROLL_UP);
		} else {
			cons->c_row++;
		}
		flush(cons);
		break;

	    case 'M':	/* ESC M - Reverse Index */
		if (cons->c_row == 0) {
			scroll_screen(cons, SCROLL_DOWN);
		} else {
			cons->c_row--;
		}
		flush(cons);
		break;

	    case '#':	/* ESC # 8 - Screen Alignment Test (ignored) */
	    case '(':	/* ESC ( C - Designate G0 Character Set (ignored) */
	    case ')':	/* ESC ) C - Designate G1 Character Set (ignored) */
	    case '*':	/* ESC * C - Designate G1 Character Set (ignored) */
	    case '+':	/* ESC + C - Designate G1 Character Set (ignored) */
		cons->c_esc_state = ES(ESC, '#');
		return;

	    case 'P':	/* ESC P ... - Device Control String (ignored) */
	    case '^':	/* ESC ^ ... - Privacy Message (ignored) */
	    case '_':	/* ESC _ ... - Application Program Command (ignored) */
		cons->c_esc_state = ES(ESC, 'P');
		return;

	    case 'c':	/* ESC c - Full Reset */
		blank_color = cons->c_attr = cons->c_blank = BLANK_COLOR;
		mem_vid_copy(BLANK_MEM, cons->c_org, usr_size);
		delimit_line(cons, cons->c_org, usr_lines);
		cons->c_row = cons->c_column = 0;
		flush(cons);
		break;
	}
	break;

    case ES(ESC, '['):		/* ESC [ ... */
	if ('0' <= c && c <= '9') {
		if (cons->c_esc_parmp < cons->c_esc_parmv + MAX_ESC_PARMS)
			*cons->c_esc_parmp = *cons->c_esc_parmp * 10 + (c-'0');
	} else
	if (c == ';') {
		if (cons->c_esc_parmp < cons->c_esc_parmv + MAX_ESC_PARMS)
			cons->c_esc_parmp++;
	} else
	if (c == '?') {
		/* ESC [ ? ... - Private Mode Set/Reset. */
		cons->c_esc_state = ES('[', '?');
		return;
	} else {
		/* End of parameter list, enter a new state to execute the
		 * function denoted by 'c'.
		 */
		cons->c_esc_state = ES('[', c);
		parse_escape(cons, 0);
	}
	return;

    case ES('[', 'A'):		/* ESC [nA - Cursor Up n Lines */
	n = cons->c_esc_parmv[0];
	if (n == 0) n = 1;
	n = (cons->c_esc_parmv[0] == 0 ? 1 : cons->c_esc_parmv[0]);
	cons->c_row -= n;
	flush(cons);
	break;

    case ES('[', 'B'):		/* ESC [nB - Cursor Down n Lines */
	n = cons->c_esc_parmv[0];
	if (n == 0) n = 1;
	n = (cons->c_esc_parmv[0] == 0 ? 1 : cons->c_esc_parmv[0]);
	cons->c_row += n;
	flush(cons);
	break;

    case ES('[', 'C'):		/* ESC [nC - Cursor Forward n Columns */
	n = cons->c_esc_parmv[0];
	if (n == 0) n = 1;
	n = (cons->c_esc_parmv[0] == 0 ? 1 : cons->c_esc_parmv[0]);
	cons->c_column += n;
	flush(cons);
	break;

    case ES('[', 'D'):		/* ESC [nD - Cursor Backward n Columns */
	n = cons->c_esc_parmv[0];
	if (n == 0) n = 1;
	n = (cons->c_esc_parmv[0] == 0 ? 1 : cons->c_esc_parmv[0]);
	cons->c_column -= n;
	flush(cons);
	break;

    case ES('[', 'H'):		/* ESC [m;nH" - Cursor Position to (m,n) */
	cons->c_row = cons->c_esc_parmv[0] - 1;
	cons->c_column = cons->c_esc_parmv[1] - 1;
	flush(cons);
	break;

    case ES('[', 'J'):		/* ESC [sJ - Erase in Display */
	switch (cons->c_esc_parmv[0]) {
	    case 0:	/* Clear from cursor to end of screen */
		count = usr_size - (cons->c_cur - cons->c_org);
		dst = cons->c_cur;
		break;
	    case 1:	/* Clear from start of screen to cursor */
		count = cons->c_cur - cons->c_org;
		dst = cons->c_org;
		break;
	    case 2:	/* Clear entire screen */
		count = usr_size;
		dst = cons->c_org;
		break;
	    default:	/* Do nothing */
		count = 0;
		dst = cons->c_org;
	}
	blank_color = cons->c_blank;
	mem_vid_copy(BLANK_MEM, dst, count);
	delimit_line(cons, cons->c_org, usr_lines);
	break;

    case ES('[', 'K'):		/* ESC [sK - Erase in Line */
	switch (cons->c_esc_parmv[0]) {
	    case 0:	/* Clear from cursor to end of line */
		count = scr_width - cons->c_column;
		dst = cons->c_cur;
		break;
	    case 1:	/* Clear from beginning of line to cursor */
		count = cons->c_column;
		dst = cons->c_cur - cons->c_column;
		break;
	    case 2:	/* Clear entire line */
		count = scr_width;
		dst = cons->c_cur - cons->c_column;
		break;
	    default:	/* Do nothing */
		count = 0;
		dst = cons->c_cur;
	}
	blank_color = cons->c_blank;
	mem_vid_copy(BLANK_MEM, dst, count);
	delimit_line(cons, cons->c_cur - cons->c_column, 1);
	break;

    case ES('[', 'L'):		/* ESC [nL - Insert n Lines */
	n = cons->c_esc_parmv[0];
	if (n < 1) n = 1;
	if (n > (usr_lines - cons->c_row)) n = usr_lines - cons->c_row;

	src = cons->c_org + cons->c_row * scr_width;
	dst = src + n * scr_width;
	count = (usr_lines - cons->c_row - n) * scr_width;
	vid_vid_copy(src, dst, count);
	blank_color = cons->c_blank;
	mem_vid_copy(BLANK_MEM, src, n * scr_width);
	delimit_line(cons, src, n);
	break;

    case ES('[', 'M'):		/* ESC [nM - Delete n Lines */
	n = cons->c_esc_parmv[0];
	if (n < 1) n = 1;
	if (n > (usr_lines - cons->c_row)) n = usr_lines - cons->c_row;

	dst = cons->c_org + cons->c_row * scr_width;
	src = dst + n * scr_width;
	count = (usr_lines - cons->c_row - n) * scr_width;
	vid_vid_copy(src, dst, count);
	blank_color = cons->c_blank;
	mem_vid_copy(BLANK_MEM, dst + count, n * scr_width);
	delimit_line(cons, dst + count, n);
	break;

    case ES('[', '@'):		/* ESC [n@ - Insert n Characters */
	n = cons->c_esc_parmv[0];
	if (n < 1) n = 1;
	if (n > (cons->c_usr_width - cons->c_column))
		n = cons->c_usr_width - cons->c_column;

	src = cons->c_cur;
	dst = src + n;
	count = cons->c_usr_width - cons->c_column - n;
	vid_vid_copy(src, dst, count);
	blank_color = cons->c_blank;
	mem_vid_copy(BLANK_MEM, src, n);
	break;

    case ES('[', 'P'):		/* ESC [nP - Delete n Characters */
	n = cons->c_esc_parmv[0];
	if (n < 1) n = 1;
	if (n > (cons->c_usr_width - cons->c_column))
		n = cons->c_usr_width - cons->c_column;

	dst = cons->c_cur;
	src = dst + n;
	count = cons->c_usr_width - cons->c_column - n;
	vid_vid_copy(src, dst, count);
	blank_color = cons->c_blank;
	mem_vid_copy(BLANK_MEM, dst + count, n);
	break;

    case ES('[', 'm'):		/* ESC [nm - Character Attributes */
	switch (cons->c_esc_parmv[0]) {
	    case 0:	/* NORMAL */
		cons->c_attr = cons->c_blank;
		break;

	    case 1:	/* BOLD */
		if (color) {
			/* Can't do bold, so use yellow */
			cons->c_attr = (cons->c_attr & 0xf0ff) | 0x0E00;
		} else {
			/* Set intensity bit */
			cons->c_attr |= 0x0800;
		}
		break;

	    case 4:	/* UNDERLINE */
		if (color) {
			/* Use light green */
			cons->c_attr = (cons->c_attr & 0xf0ff) | 0x0A00;
		} else {
			cons->c_attr = (cons->c_attr & 0x8900);
		}
		break;

	    case 5:	/* BLINKING */
		if (color) {
			/* Use magenta */
			cons->c_attr = (cons->c_attr & 0xf0ff) | 0x0500;
		} else {
			/* Set the blink bit */
			cons->c_attr |= 0x8000;
		}
		break;

	    case 7:	/* REVERSE */
		if (color) {
			/* Swap fg and bg colors */
			cons->c_attr =	((cons->c_attr & 0xf000) >> 4) |
					((cons->c_attr & 0x0f00) << 4);
		} else
		if ((cons->c_attr & 0x7000) == 0) {
			cons->c_attr = (cons->c_attr & 0x8800) | 0x7000;
		} else {
			cons->c_attr = (cons->c_attr & 0x8800) | 0x0700;
		}
		break;

	    default:	/* COLOR */
		n = cons->c_esc_parmv[0];
		if (30 <= n && n <= 37) {
			/* Foreground color. */
			cons->c_attr =	(cons->c_attr & 0xf0ff) |
					(ansi_colors[(n - 30)] << 8);
			cons->c_blank =	(cons->c_blank & 0xf0ff) |
					(ansi_colors[(n - 30)] << 8);
		} else
		if (40 <= n && n <= 47) {
			/* Background color. */
			cons->c_attr =	(cons->c_attr & 0x0fff) |
					(ansi_colors[(n - 40)] << 12);
			cons->c_blank =	(cons->c_blank & 0x0fff) |
					(ansi_colors[(n - 40)] << 12);
		}
		break;
	}
	break;

    case ES(ESC, ']'):		/* ESC ] ... */
	if ('0' <= c && c <= '9') {
		cons->c_esc_parmv[0] = cons->c_esc_parmv[0] * 10 + (c-'0');
	} else {
		if (cons->c_esc_parmv[0] != 2 || c != ';') break;

				/* ESC ]2; ... - Set Window Title */
		status_state(STATUS_ENABLE);
		cons->c_ramqueue[0] = STATUS_COLOR | ' ';
		cons->c_ramqueue[1] = (STATUS_COLOR | '0') + (cons-cons_table);
		cons->c_ramqueue[2] = STATUS_COLOR | ':';
		cons->c_ramqueue[3] = STATUS_COLOR | ' ';
		cons->c_rwords = 4;
		cons->c_esc_state = ES(']', ';');
	}
	return;

    case ES(']', ';'):		/* ESC ]2; - writing the status line */
	n = MIN(CONS_RAM_WORDS, scr_width);

	if (c >= ' ') {
		if (cons->c_rwords < n) {
			cons->c_ramqueue[cons->c_rwords++] = STATUS_COLOR | c;
		}
		return;
	}
	/* End of status line, display result. */
	while (cons->c_rwords < n)
		cons->c_ramqueue[cons->c_rwords++] = STATUS_COLOR | ' ';
	cons->c_cur = cons->c_org + usr_size;
	cons->c_esc_state = 0;
	flush(cons);
	break;

    case ES('[', '?'):		/* ESC [ ? ... Private Mode Set/Reset */
	if ('0' <= c && c <= '9') {
		cons->c_esc_parmv[0] = cons->c_esc_parmv[0] * 10 + (c-'0');
		return;
	} else
	if ((c == 'l' || c == 'h') && cons->c_esc_parmv[0] == 3) {
		/* Set 80 or 132 columns mode or something. */
		if (scr_width > 80) cons->c_usr_width = 80;
		if (c == 'h') cons->c_usr_width = scr_width;
		cons->c_esc_state = ESC;
		parse_escape(cons, 'c');	/* full reset */
		if (cons->c_tty.t_sesldr)	/* screen size changed */
			cause_sig(cons->c_tty.t_sesldr, SIGWINCH);
	}
	break;

    case ES(ESC, 'P'):		/* ESC P ... ESC \ (ignored) */
	if (c == ESC) cons->c_esc_state = ES('P', ESC);
	return;
  }
  cons->c_esc_state = 0;	/* ESC sequence finished or ignored */
}


/*===========================================================================*
 *				stop_beep				     *
 *===========================================================================*/
PRIVATE void stop_beep()
{
/* Turn off the beeper by turning off bits 0 and 1 in PORT_B. */

  lock();			/* guard PORT_B from keyboard intr handler */
  out_byte(PORT_B, in_byte(PORT_B) & ~3);
  beeping = FALSE;
  unlock();
}


/*===========================================================================*
 *				flush					     *
 *===========================================================================*/
PRIVATE void flush(cons)
register console_t *cons;	/* pointer to console struct */
{
/* Send characters buffered in 'ramqueue' to screen memory, check the new
 * cursor position, compute the new hardware cursor position and set it.
 */
  unsigned cur;

  if (cons->c_esc_state == ES(']', ';')) return;   /* on the status line? */

  /* Have the characters in 'ramqueue' transferred to the screen. */
  if (cons->c_rwords > 0) {
	mem_vid_copy(cons->c_ramqueue, cons->c_cur, cons->c_rwords);
	cons->c_rwords = 0;
  }

  /* Check and update the cursor position. */
  if (cons->c_column < 0) cons->c_column = 0;
  if (cons->c_column > cons->c_usr_width) cons->c_column = cons->c_usr_width;
  if (cons->c_row < 0) cons->c_row = 0;
  if (cons->c_row >= usr_lines) cons->c_row = usr_lines - 1;
  cur = cons->c_org + cons->c_row * scr_width + cons->c_column;
  if (cur != cons->c_cur) {
	if (cons == curcons) set_6845(CURSOR, cur);
	cons->c_cur = cur;
  }
}


/*===========================================================================*
 *				set_6845				     *
 *===========================================================================*/
PRIVATE void set_6845(reg, val)
int reg;			/* which register pair to set */
unsigned val;			/* 16-bit value to set it to */
{
/* Set a register pair inside the 6845.
 * Registers 12-13 tell the 6845 where in video ram to start
 * Registers 14-15 tell the 6845 where to put the cursor
 */
  lock();			/* try to stop h/w loading in-between value */
  out_byte(vid_port + INDEX, reg);		/* set the index register */
  out_byte(vid_port + DATA, (val>>8) & BYTE);	/* output high byte */
  out_byte(vid_port + INDEX, reg + 1);		/* again */
  out_byte(vid_port + DATA, val & BYTE);	/* output low byte */
  unlock();
}


/*===========================================================================*
 *				toggle_scroll				     *
 *===========================================================================*/
PUBLIC void toggle_scroll()
{
/* Toggle between hardware and software scroll. */

  cons_org0();
  softscroll = !softscroll;
  printf("%sware scrolling enabled.\n", softscroll ? "Soft" : "Hard");
}


/*===========================================================================*
 *				cons_org0				     *
 *===========================================================================*/
PRIVATE void cons_org0()
{
/* Scroll video memory back to put the origin at 0. */
  int cons_line;
  console_t *cons;
  unsigned n;

  for (cons_line = 0; cons_line < nconsole; cons_line++) {
	cons = &cons_table[cons_line];
	while (cons->c_org > cons->c_start) {
		n = vid_size - scr_size;	/* amount of unused memory */
		if (n > cons->c_org - cons->c_start)
			n = cons->c_org - cons->c_start;
		vid_vid_copy(cons->c_org, cons->c_org - n, scr_size);
		cons->c_org -= n;
	}
	flush(cons);
  }
  select_console(current);
}


/*===========================================================================*
 *				enable_video				     *
 *===========================================================================*/
PUBLIC void enable_video(on)
int on;
{
/*
 * Turn video adapter on or off by toggling the screen off bit for VGA
 * adapters.
 */
#define VGA_SEQREG		0x3C4	/* sequencer registers */
#define VGA_SEQ_CLOCKINGMODE	1	/* clocking mode index */
  int clockmode;
  tmr_arg_ut dummy;
  static int current_state;

  /* Only do an enabled VGA console. */
  if (!vga || disable_cons_output) return;

  if (current_state != on) {
	/* Toggle the screen off bit in the clocking mode register. */
	lock();
	out_byte(VGA_SEQREG, VGA_SEQ_CLOCKINGMODE);
	clockmode = in_byte(VGA_SEQREG+1);
	clockmode = on ? clockmode & ~0x20 : clockmode | 0x20;
	out_word(VGA_SEQREG, 0x0100);
	out_word(VGA_SEQREG, (clockmode << 8) | 0x01);
	out_word(VGA_SEQREG, 0x0300);
	unlock();
	current_state = on;
  }

  if (on && blank_time != 0) {
	/* Schedule a "blank video" timer. */
	tmra_settimer(&blank_timer, get_uptime() + blank_time * 60L * HZ,
							blank_video, dummy);
  }
}


/*===========================================================================*
 *				blank_video				     *
 *===========================================================================*/
PRIVATE void blank_video(tp, arg)
struct tmra *tp;
tmr_arg_ut arg;
{
/* Called by the clock task to blank the screen to prevent burn-in. */
  enable_video(FALSE);
}


/*===========================================================================*
 *				read_kb					     *
 *===========================================================================*/
PRIVATE void read_kb()
{
/* Hand keyboard input to tty. */
  int i, n, r, code, flush, len;
  unsigned char *keyp, *charp;

  n= kb_read(&keyp);

  flush= 0;
  for (i= 0; i< n; i++)
  {
	code= keyp[i];
	len = kb_translate(code, &charp);	/* map scancode to ASCII */
	if (len == 0) continue;			/* not for us */

	r= tty_put(&curcons->c_tty, (char *)charp, len);
	if (r != len)
		printf("keyboard buffer overflow\7\n");
	flush= 1;
  }
  if (flush)
	  tty_put(&curcons->c_tty, NULL, 0);	/* Flush */
}


/*===========================================================================*
 *				cons_print				     *
 *===========================================================================*/
PUBLIC void cons_print(buf, size)
char *buf;
unsigned size;
{
	int old_lco, s;
	char *ptr, *nl;

	if (size == 0)	/* Flush */
	{
		cons_put(0, NULL, 0);
		return;
	}

	old_lco= log_cons_output;
	log_cons_output= FALSE;

	for (ptr= buf; size;)
	{
		nl= memchr(ptr, '\n', size);
		if (nl == NULL)
			s= size;
		else
			s= nl+1-ptr;

		cons_put(0, ptr, s);
		if (nl)
			cons_put(0, "\r", 1);
		ptr += s;
		size -= s;
	}
	log_cons_output= old_lco;
}


/*===========================================================================*
 *				console_hotkey				     *
 *===========================================================================*/
PUBLIC void console_hotkey(key)
char *key;
{
  /* Choose current console, or en/disable status line. */
  if (strcmp(key, "vc-") == 0) select_console(current - 1);
  if (strcmp(key, "vc+") == 0) select_console(current + 1);
  if (strcmp(key, "vc0") >= 0 && strcmp(key, "vc7") <= 0)
	select_console(key[2] - '0');
  if (strcmp(key, "st+") == 0) status_state(STATUS_ENABLE);
  if (strcmp(key, "st-") == 0) status_state(STATUS_DISABLE);
}


/*===========================================================================*
 *				con_copyfont				     *
 *===========================================================================*/

#define GA_SEQUENCER_INDEX	0x3C4
#define GA_SEQUENCER_DATA	0x3C5
#define GA_GRAPHICS_INDEX	0x3CE
#define GA_GRAPHICS_DATA	0x3CF
#define GA_VIDEO_ADDRESS	0xA0000L
#define GA_FONT_SIZE		8192L

struct sequence {
	unsigned short index;
	unsigned char port;
	unsigned char value;
};

FORWARD _PROTOTYPE( void ga_program, (struct sequence *seq) );

PRIVATE void ga_program(seq)
struct sequence *seq;
{
  int len= 7;
  do {
	out_byte(seq->index, seq->port);
	out_byte(seq->index+1, seq->value);
	seq++;
  } while (--len > 0);
}

PRIVATE int con_copyfont(proc_nr, request, font_vir)
int proc_nr;
ioreq_t request;
vir_bytes font_vir;
{
/* Load a font into the EGA or VGA adapter. */
  phys_bytes user_phys;
  static struct sequence seq1[7] = {
	{ GA_SEQUENCER_INDEX, 0x00, 0x01 },
	{ GA_SEQUENCER_INDEX, 0x02, 0x04 },
	{ GA_SEQUENCER_INDEX, 0x04, 0x07 },
	{ GA_SEQUENCER_INDEX, 0x00, 0x03 },
	{ GA_GRAPHICS_INDEX, 0x04, 0x02 },
	{ GA_GRAPHICS_INDEX, 0x05, 0x00 },
	{ GA_GRAPHICS_INDEX, 0x06, 0x00 },
  };
  static struct sequence seq2[7] = {
	{ GA_SEQUENCER_INDEX, 0x00, 0x01 },
	{ GA_SEQUENCER_INDEX, 0x02, 0x03 },
	{ GA_SEQUENCER_INDEX, 0x04, 0x03 },
	{ GA_SEQUENCER_INDEX, 0x00, 0x03 },
	{ GA_GRAPHICS_INDEX, 0x04, 0x00 },
	{ GA_GRAPHICS_INDEX, 0x05, 0x10 },
	{ GA_GRAPHICS_INDEX, 0x06,    0 },
  };
  int c;

  seq2[6].value= color ? 0x0E : 0x0A;

  user_phys = numap(proc_nr, font_vir, GA_FONT_SIZE);
  if (user_phys == 0) return(EFAULT);
  if (!ega || disable_cons_output) return(ENOTTY);

  lock();
  ga_program(seq1);	/* bring font memory into view */

  if (request == TIOCSFON) {
	/* Load font into video memory. */
	phys_copy(user_phys, GA_VIDEO_ADDRESS, GA_FONT_SIZE);

	/* Disable the Latin-1 to IBM character mapping kludge. */
	for (c = 0x80; c < 0x100; c++) map_lat1_ibm[c - 0x80] = c;
  } else {
	/* Copy font to user space. */
	phys_copy(GA_VIDEO_ADDRESS, user_phys, GA_FONT_SIZE);
  }

  ga_program(seq2);	/* restore */
  unlock();

  return(OK);
}


#ifdef SHOW_EVENT
/*===========================================================================*
 *				show_event				     *
 *===========================================================================*/
PUBLIC void show_event(ev, code)
int ev;
char code;
{
	u16_t w;

	status_state(STATUS_ENABLE);
	w = STATUS_COLOR | code;
	mem_vid_copy(&w, cons_table[0].c_org + scr_size - ev - 1, 1);
}
#endif

/*
 * $PchId: console.c,v 1.5 1996/01/19 22:57:15 philip Exp $
 */
