/* This file contains the device dependent part of a driver for the IBM-AT
 * winchester controller.
 * It was written by Adri Koppes.
 *
 * The file contains one entry point:
 *
 *   at_winchester_task:	main entry when system is brought up
 *
 *
 * Changes:
 *	13 Apr 1992 by Kees J. Bot: device dependent/independent split.
 *	10 Dec 1995 by Kees J. Bot: d-d/i rewrite.
 */

#include "kernel.h"
#include "driver.h"
#include "drvlib.h"
#if ENABLE_AT_WINI
#include <time.h>

#define AT_DEBUG	    0	/* Set to 1 at debugging time. */


/* I/O Ports used by winchester disk controllers. */

/* Read and write registers */
#define REG_BASE0	0x1F0	/* base register of controller 0 */
#define REG_BASE1	0x170	/* base register of controller 1 */
#define REG_DATA	    0	/* data register (offset from the base reg.) */
#define REG_PRECOMP	    1	/* start of write precompensation */
#define REG_COUNT	    2	/* sectors to transfer */
#define REG_SECTOR	    3	/* sector number */
#define REG_CYL_LO	    4	/* low byte of cylinder number */
#define REG_CYL_HI	    5	/* high byte of cylinder number */
#define REG_LDH		    6	/* lba, drive and head */
#define   LDH_DEFAULT		0xA0	/* ECC enable, 512 bytes per sector */
#define   LDH_LBA		0x40	/* Use LBA addressing */
#define   ldh_init(drive)	(LDH_DEFAULT | ((drive) << 4))

/* Read only registers */
#define REG_STATUS	    7	/* status */
#define   STATUS_BSY		0x80	/* controller busy */
#define	  STATUS_RDY		0x40	/* drive ready */
#define	  STATUS_WF		0x20	/* write fault */
#define	  STATUS_SC		0x10	/* seek complete (obsolete) */
#define	  STATUS_DRQ		0x08	/* data transfer request */
#define	  STATUS_CRD		0x04	/* corrected data */
#define	  STATUS_IDX		0x02	/* index pulse */
#define	  STATUS_ERR		0x01	/* error */
#define REG_ERROR	    1	/* error code */
#define	  ERROR_BB		0x80	/* bad block */
#define	  ERROR_ECC		0x40	/* bad ecc bytes */
#define	  ERROR_ID		0x10	/* id not found */
#define	  ERROR_AC		0x04	/* aborted command */
#define	  ERROR_TK		0x02	/* track zero error */
#define	  ERROR_DM		0x01	/* no data address mark */

/* Write only registers */
#define REG_COMMAND	    7	/* command */
#define   CMD_IDLE		0x00	/* for w_command: drive idle */
#define   CMD_RECALIBRATE	0x10	/* recalibrate drive */
#define   CMD_READ		0x20	/* read data */
#define   CMD_WRITE		0x30	/* write data */
#define   CMD_READVERIFY	0x40	/* read verify */
#define   CMD_FORMAT		0x50	/* format track */
#define   CMD_SEEK		0x70	/* seek cylinder */
#define   CMD_DIAG		0x90	/* execute device diagnostics */
#define   CMD_SPECIFY		0x91	/* specify parameters */
#define   ATA_READMULTIPLE	0xC4	/* read multiple sectors */
#define   ATA_WRITEMULTIPLE	0xC5	/* write multiple sectors */
#define   ATA_SETMULTIPLE	0xC6	/* set multiple mode */
#define   ATA_IDENTIFY		0xEC	/* identify drive */
#define   ATA_STANDBYIMM	0xE0	/* go standby immediately */
#define REG_CTL		0x206	/* control register */
#define   CTL_NORETRY		0x80	/* disable access retry */
#define   CTL_NOECC		0x40	/* disable ecc retry */
#define   CTL_EIGHTHEADS	0x08	/* more than eight heads */
#define   CTL_RESET		0x04	/* reset controller */
#define   CTL_INTDISABLE	0x02	/* disable interrupts */

/* Interrupt request lines. */
#define AT_IRQ0		14	/* interrupt number for controller 0 */
#define AT_IRQ1		15	/* interrupt number for controller 1 */

/* Common command block */
struct command {
  u8_t	precomp;	/* REG_PRECOMP, etc. */
  u8_t	count;
  u8_t	sector;
  u8_t	cyl_lo;
  u8_t	cyl_hi;
  u8_t	ldh;
  u8_t	command;
};


/* Error codes */
#define ERR		 (-1)	/* general error */
#define ERR_BAD_SECTOR	 (-2)	/* block marked bad detected */

/* Some controllers don't interrupt, the clock will wake us up. */
#define WAKEUP		(32*HZ)	/* drive may be out for 31 seconds max */

/* Miscellaneous. */
#define MAX_DRIVES         4	/* this driver supports 4 drives (hd0 - hd19) */
#if INTEL_32BITS
#define MAX_SECS	 256	/* controller can transfer this many sectors */
#else
#define MAX_SECS	 127	/* but not to a 16 bit process */
#endif
#define MAX_ERRORS         4	/* how often to try rd/wt before quitting */
#define NR_DEVICES      (MAX_DRIVES * DEV_PER_DRIVE)
#define SUB_PER_DRIVE	(NR_PARTITIONS * NR_PARTITIONS)
#define NR_SUBDEVS	(MAX_DRIVES * SUB_PER_DRIVE)
#define TIMEOUT     32000000	/* controller timeout in us */
#define RECOVERYTIME  500000	/* controller recovery time in us */
#define SLEEP_TIME	  15	/* put disks to sleep after 15 idle minutes */
#define INITIALIZED	0x01	/* drive is initialized */
#define DEAF		0x02	/* controller must be reset */
#define SMART		0x04	/* drive supports ATA commands */


/* Variables. */
PRIVATE struct wini {		/* main drive struct, one entry per drive */
  unsigned state;		/* drive state: deaf, initialized, dead */
  unsigned base;		/* base register of the register file */
  unsigned irq;			/* interrupt request line */
  unsigned lcylinders;		/* logical number of cylinders (BIOS) */
  unsigned lheads;		/* logical number of heads */
  unsigned lsectors;		/* logical number of sectors per track */
  unsigned pcylinders;		/* physical number of cylinders (translated) */
  unsigned pheads;		/* physical number of heads */
  unsigned psectors;		/* physical number of sectors per track */
  unsigned ldhpref;		/* top four bytes of the LDH (head) register */
  unsigned precomp;		/* write precompensation cylinder / 4 */
  unsigned max_count;		/* max request for this drive */
  unsigned multiple;		/* maximum multiple sector block */
  unsigned sleeptime;		/* time idle before turning the drive off */
  unsigned open_ct;		/* in-use count */
  tmrs_ut tmrs_sleep;		/* go to sleep timer */
  struct device part[DEV_PER_DRIVE];    /* primary partitions: hd[0-4] */
  struct device subpart[SUB_PER_DRIVE]; /* subpartitions: hd[1-4][a-d] */
} wini[MAX_DRIVES], *w_wn;

PRIVATE int win_tasknr = ANY;
PRIVATE int w_command;			/* current command in execution */
PRIVATE int w_status;			/* status after interrupt */
PRIVATE int w_drive;			/* selected drive */
PRIVATE struct device *w_dv;		/* device's base and size */
PRIVATE tmra_ut w_tmra_timeout;		/* timer for w_timeout */

FORWARD _PROTOTYPE( void init_params, (void) );
FORWARD _PROTOTYPE( int w_do_open, (struct driver *dp, message *m_ptr) );
FORWARD _PROTOTYPE( struct device *w_prepare, (int device) );
FORWARD _PROTOTYPE( int w_identify, (void) );
FORWARD _PROTOTYPE( char *w_name, (void) );
FORWARD _PROTOTYPE( int w_specify, (void) );
FORWARD _PROTOTYPE( int w_transfer, (int proc_nr, int opcode, u64_t position,
					iovec_t *iov, unsigned nr_req) );
FORWARD _PROTOTYPE( void w_vec_portio, (int opcode, phys_bytes user_base,
					iovec_t *iop, unsigned count) );
FORWARD _PROTOTYPE( int com_out, (struct command *cmd) );
FORWARD _PROTOTYPE( void w_need_reset, (void) );
FORWARD _PROTOTYPE( void w_cleanup, (void) );
FORWARD _PROTOTYPE( void w_sleep, (struct tmrs *tp, tmr_arg_ut drive) );
FORWARD _PROTOTYPE( int w_do_close, (struct driver *dp, message *m_ptr) );
FORWARD _PROTOTYPE( int com_simple, (struct command *cmd) );
FORWARD _PROTOTYPE( void w_timeout, (struct tmra *tp, tmr_arg_ut arg) );
FORWARD _PROTOTYPE( int w_reset, (void) );
FORWARD _PROTOTYPE( int w_intr_wait, (void) );
FORWARD _PROTOTYPE( int w_waitfor, (int mask, int value) );
FORWARD _PROTOTYPE( int w_handler, (int irq) );
FORWARD _PROTOTYPE( void w_geometry, (struct partition *entry) );

/* w_waitfor loop unrolled once for speed. */
#define waitfor(mask, value)	\
	((in_byte(w_wn->base + REG_STATUS) & mask) == value \
		|| w_waitfor(mask, value))


/* Entry points to this driver. */
PRIVATE struct driver w_dtab = {
  w_name,		/* current device's name */
  w_do_open,		/* open or mount request, initialize device */
  w_do_close,		/* release device */
  do_diocntl,		/* get or set a partition's geometry */
  w_prepare,		/* prepare for I/O on a given minor device */
  w_transfer,		/* do the I/O */
  w_cleanup,		/* start a timer to do a spin down */
  w_geometry,		/* tell the geometry of the disk */
};

#if ENABLE_ATAPI
#include "atapi.c"	/* extra code for ATAPI CD-ROM */
#endif


/*===========================================================================*
 *				at_winchester_task			     *
 *===========================================================================*/
PUBLIC void at_winchester_task()
{
/* Set special disk parameters then call the generic main loop. */
  int i;
  struct wini *wn;
  long v;
  char *name;
  static char fmt[] = "d:d:d";

  win_tasknr = proc_number(proc_ptr);

  init_params();

  for (i = 0; i < MAX_DRIVES; i++) {
	(void) w_prepare(i * DEV_PER_DRIVE);
	wn = w_wn;

	/* Look into the environment for special parameters. */
	name = w_name();

	v = SLEEP_TIME;
	(void) env_parse(name, fmt, 0, &v, 0L, 60L);
	wn->sleeptime = v * 60 * HZ;

	v = MAX_SECS;
	(void) env_parse(name, fmt, 1, &v, 1L, (long) MAX_SECS);
	wn->max_count = (unsigned) v << SECTOR_SHIFT;

	v = 1;
	(void) env_parse(name, fmt, 2, &v, 0L, 255L);
	wn->multiple = v;

	if (i < 2) {
		/* Controller 0. */
		wn->base = REG_BASE0;
		wn->irq = AT_IRQ0;
	} else {
		/* Controller 1. */
		wn->base = REG_BASE1;
		wn->irq = AT_IRQ1;
	}

	tmrs_inittimer(&wn->tmrs_sleep);
  }
  tmra_inittimer(&w_tmra_timeout);
  driver_task(&w_dtab);
}


/*============================================================================*
 *				init_params				      *
 *============================================================================*/
PRIVATE void init_params()
{
/* This routine is called at startup to initialize the drive parameters. */

  u16_t parv[2];
  unsigned int vector;
  int drive, nr_drives;
  struct wini *wn;
  u8_t params[16];
  phys_bytes param_phys = vir2phys(params);

  /* Get the number of drives from the BIOS data area */
  vm_map_zp(TRUE);
  phys_copy(0x475L, param_phys, 1L);
  vm_map_zp(FALSE);
  if ((nr_drives = params[0]) > 2) nr_drives = 2;

  for (drive = 0, wn = wini; drive < MAX_DRIVES; drive++, wn++) {
	if (drive < nr_drives) {
		/* Copy the BIOS parameter vector */
		vm_map_zp(TRUE);
		vector = drive == 0 ? WINI_0_PARM_VEC : WINI_1_PARM_VEC;
		phys_copy(vector * 4L, vir2phys(parv), 4L);

		/* Calculate the address of the parameters and copy them */
		phys_copy(hclick_to_physb(parv[1]) + parv[0], param_phys, 16L);
		vm_map_zp(FALSE);

		/* Copy the parameters to the structures of the drive */
		wn->lcylinders = bp_cylinders(params);
		wn->lheads = bp_heads(params);
		wn->lsectors = bp_sectors(params);
		wn->precomp = bp_precomp(params) >> 2;
	}
	wn->ldhpref = ldh_init(drive);
  }
}


/*============================================================================*
 *				w_do_open				      *
 *============================================================================*/
PRIVATE int w_do_open(dp, m_ptr)
struct driver *dp;
message *m_ptr;
{
/* Device open: Initialize the controller and read the partition table. */

  struct wini *wn;

  if (w_prepare(m_ptr->DEVICE) == NIL_DEV) return(ENXIO);
  wn = w_wn;

  if (wn->state == 0) {
	/* Try to identify the device. */
	if (w_identify() != OK) {
		printf("%s: probe failed\n", w_name());
		if (wn->state & DEAF) w_reset();
		wn->state = 0;
		return(ENXIO);
	}
  }
  if (wn->open_ct++ == 0) {
	/* Partition the disk. */
	partition(&w_dtab, w_drive * DEV_PER_DRIVE, P_PRIMARY);
  }
  return(OK);
}


/*===========================================================================*
 *				w_prepare				     *
 *===========================================================================*/
PRIVATE struct device *w_prepare(device)
int device;
{
/* Prepare for I/O on a device. */

  if (device < NR_DEVICES) {			/* hd0, hd1, ... */
	w_drive = device / DEV_PER_DRIVE;	/* save drive number */
	w_wn = &wini[w_drive];
	w_dv = &w_wn->part[device % DEV_PER_DRIVE];
  } else
  if ((unsigned) (device -= MINOR_hd1a) < NR_SUBDEVS) {	/* hd1a, hd1b, ... */
	w_drive = device / SUB_PER_DRIVE;
	w_wn = &wini[w_drive];
	w_dv = &w_wn->subpart[device % SUB_PER_DRIVE];
  } else {
	return(NIL_DEV);
  }
  return(w_dv);
}


/*===========================================================================*
 *				w_identify				     *
 *===========================================================================*/
PRIVATE int w_identify()
{
/* Find out if a device exists, if it is an old AT disk, or a newer ATA
 * drive, a removable media device, etc.
 */

  struct wini *wn = w_wn;
  struct command cmd;
  char id_string[40];
  int i, r;
  unsigned multiple;
  unsigned long size;
#define id_byte(n)	(&tmp_buf[2 * (n)])
#define id_word(n)	(((u16_t) id_byte(n)[0] <<  0) \
			|((u16_t) id_byte(n)[1] <<  8))
#define id_longword(n)	(((u32_t) id_byte(n)[0] <<  0) \
			|((u32_t) id_byte(n)[1] <<  8) \
			|((u32_t) id_byte(n)[2] << 16) \
			|((u32_t) id_byte(n)[3] << 24))

  /* Check if the one of the registers exists. */
  r = in_byte(wn->base + REG_CYL_LO);
  out_byte(wn->base + REG_CYL_LO, ~r);
  if (in_byte(wn->base + REG_CYL_LO) == r) return(ERR);

  /* Looks OK; register IRQ and try an ATA identify command. */
  put_irq_handler(wn->irq, w_handler);
  enable_irq(wn->irq);

  cmd.ldh     = wn->ldhpref;
  cmd.command = ATA_IDENTIFY;
  if (wn->multiple != 0 && com_simple(&cmd) == OK) {
	/* This is an ATA device. */
	wn->state |= SMART;

	/* Device information. */
	port_read(wn->base + REG_DATA, tmp_phys, SECTOR_SIZE);

	/* Why are the strings byte swapped??? */
	for (i = 0; i < 40; i++) id_string[i] = id_byte(27)[i^1];

	/* Preferred CHS translation mode. */
	wn->pcylinders = id_word(1);
	wn->pheads = id_word(3);
	wn->psectors = id_word(6);
	size = (u32_t) wn->pcylinders * wn->pheads * wn->psectors;

	if ((id_byte(49)[1] & 0x02) && size > 512L*1024*2) {
		/* Drive is LBA capable and is big enough to trust it to
		 * not make a mess of it.
		 */
		wn->ldhpref |= LDH_LBA;
		size = id_longword(60);
	}

	if (wn->lcylinders == 0) {
		/* No BIOS parameters?  Then make some up. */
		wn->lcylinders = wn->pcylinders;
		wn->lheads = wn->pheads;
		wn->lsectors = wn->psectors;
		while (wn->lcylinders > 1024) {
			wn->lheads *= 2;
			wn->lcylinders /= 2;
		}
	}

	/* How large can a multiple sector block be? */
	multiple = id_byte(47)[0];

	if (debug) {
#if !AT_DEBUG
		printf(
	"%s: logical=%ux%ux%u, physical=%ux%ux%u, max multiple=%u, lba=%d\n",
			w_name(),
			wn->lcylinders, wn->lheads, wn->lsectors,
			wn->pcylinders, wn->pheads, wn->psectors,
			multiple, (wn->ldhpref & LDH_LBA) ? 1 : 0);
#else /* AT_DEBUG */
		printf("%s: identify data:\n", w_name());
		printf("config = %04x (RMB=%d, !RMD=%d)\n",
			id_word(0), (id_word(0) >> 7)&1,
			(id_word(0) >> 6)&1);
		printf("default cylinders = %u\n", id_word(1));
		printf("default heads = %u\n", id_word(3));
		printf("default sectors = %u\n", id_word(6));
		if (id_word(10) != 0) {
			printf("serial = ");
			for (i= 0; i < 20; i++) printf("%c", id_byte(10)[i^1]);
			printf("\n");
		}
		if (id_word(23) != 0) {
			printf("revision = ");
			for (i= 0; i < 8; i++) printf("%c", id_byte(23)[i^1]);
			printf("\n");
		}
		if (id_word(27) != 0) {
			printf("model = ");
			for (i= 0; i < 40; i++) printf("%c", id_byte(27)[i^1]);
			printf("\n");
		}
		printf("max multiple = %u\n", id_byte(47)[0]);
		printf("capabilities = %04x (stdSTANDBY=%d, suppIORDY=%d, ",
			id_word(49), (id_word(49) >> 13)&1,
			(id_word(49) >> 11)&1);
		printf("disIORDY=%d, LBA=%d, DMA=%d)\n",
			(id_word(49) >> 10)&1, (id_word(49) >> 9)&1,
			(id_word(49) >> 8)&1);
		printf("LBA sectors = %lu\n", id_longword(60));
#endif /* AT_DEBUG */
	}
  } else {
	/* Not an ATA device; no translations, no special features.  Don't
	 * touch it unless the BIOS knows about it.
	 */
	if (AT_DEBUG && debug) {
		printf("status=%02x, error=%02x\n",
			w_status, in_byte(wn->base + REG_ERROR));
	}
	if (wn->lcylinders == 0) return(ERR);	/* no BIOS parameters */
	wn->pcylinders = wn->lcylinders;
	wn->pheads = wn->lheads;
	wn->psectors = wn->lsectors;
	size = (u32_t) wn->pcylinders * wn->pheads * wn->psectors;
	wn->sleeptime = 0;
	multiple = 0;
  }

  /* Size of the whole drive */
  wn->part[0].dv_size = mul64u(size, SECTOR_SIZE);

  /* Multiple sector block size? */
  if (wn->multiple > multiple) wn->multiple = multiple;
  if (wn->multiple == 0) wn->multiple = 1;

  if (w_specify() != OK && w_specify() != OK) return(ERR);

  printf("%s: ", w_name());
  if (wn->state & SMART) {
	printf("%.40s\n", id_string);
  } else {
	printf("%ux%ux%u\n", wn->pcylinders, wn->pheads, wn->psectors);
  }
  return(OK);
}


/*===========================================================================*
 *				w_name					     *
 *===========================================================================*/
PRIVATE char *w_name()
{
/* Return a name for the current device. */
  static char name[] = "at-hd15";
  unsigned device = w_drive * DEV_PER_DRIVE;

  if (device < 10) {
	name[5] = '0' + device;
	name[6] = 0;
  } else {
	name[5] = '0' + device / 10;
	name[6] = '0' + device % 10;
  }
  return name;
}


/*===========================================================================*
 *				w_specify				     *
 *===========================================================================*/
PRIVATE int w_specify()
{
/* Routine to initialize the drive after boot or when a reset is needed. */

  struct wini *wn = w_wn;
  struct command cmd;

  if ((wn->state & DEAF) && w_reset() != OK) return(ERR);

  /* Specify parameters: precompensation, number of heads and sectors. */
  cmd.precomp = wn->precomp;
  cmd.count   = wn->psectors;
  cmd.ldh     = w_wn->ldhpref | (wn->pheads - 1);
  cmd.command = CMD_SPECIFY;		/* Specify some parameters */

  /* Output command block and see if controller accepts the parameters. */
  if (com_simple(&cmd) != OK) return(ERR);

  if (wn->state & SMART) {
	if (wn->multiple > 1) {
		/* Set multiple sector block size. */
		cmd.count   = wn->multiple;
		cmd.ldh     = w_wn->ldhpref;
		cmd.command = ATA_SETMULTIPLE;

		if (com_simple(&cmd) != OK) wn->multiple = 1;
	}
  } else {
	/* Calibrate an old disk. */
	cmd.sector  = 0;
	cmd.cyl_lo  = 0;
	cmd.cyl_hi  = 0;
	cmd.ldh     = w_wn->ldhpref;
	cmd.command = CMD_RECALIBRATE;

	if (com_simple(&cmd) != OK) return(ERR);
  }

  wn->state |= INITIALIZED;
  return(OK);
}


/*===========================================================================*
 *				w_transfer				     *
 *===========================================================================*/
PRIVATE int w_transfer(proc_nr, opcode, position, iov, nr_req)
int proc_nr;			/* process doing the request */
int opcode;			/* DEV_GATHER or DEV_SCATTER */
u64_t position;			/* offset on device to read or write */
iovec_t *iov;			/* pointer to read or write request vector */
unsigned nr_req;		/* length of request vector */
{
  struct wini *wn = w_wn;
  iovec_t *iop, *iov_end = iov + nr_req;
  int r, errors;
  unsigned long block;
  struct command cmd;
  unsigned multiple = wn->multiple;
  unsigned cylinder, head, sector, nbytes, count, chunk;
  unsigned secspcyl = wn->pheads * wn->psectors;
  phys_bytes user_base = proc_vir2phys(proc_addr(proc_nr), 0);

  /* Check disk address. */
  if (rem64u(position, SECTOR_SIZE) != 0) return(EINVAL);

  errors = 0;

  while (nr_req > 0) {
	/* How many bytes to transfer? */
	nbytes = 0;
	for (iop = iov; iop < iov_end; iop++) nbytes += iop->iov_size;
	if ((nbytes & SECTOR_MASK) != 0) return(EINVAL);

	/* Which block on disk and how close to EOF? */
	if (cmp64(position, w_dv->dv_size) >= 0) return(OK);	/* At EOF */
	if (cmp64(add64u(position, nbytes), w_dv->dv_size) > 0)
		nbytes = diff64(w_dv->dv_size, position);
	block = div64u(add64(w_dv->dv_base, position), SECTOR_SIZE);

	if (nbytes >= wn->max_count) {
		/* The drive can't do more then max_count at once. */
		nbytes = wn->max_count;
	}

	/* First check to see if a reinitialization is needed. */
	if (!(wn->state & INITIALIZED) && w_specify() != OK) return(EIO);

	/* Tell the controller to transfer nbytes bytes. */
	cmd.precomp = wn->precomp;
	cmd.count   = (nbytes >> SECTOR_SHIFT) & BYTE;
	if (wn->ldhpref & LDH_LBA) {
		cmd.sector  = (block >>  0) & 0xFF;
		cmd.cyl_lo  = (block >>  8) & 0xFF;
		cmd.cyl_hi  = (block >> 16) & 0xFF;
		cmd.ldh     = wn->ldhpref | ((block >> 24) & 0xF);
	} else {
		cylinder = block / secspcyl;
		head = (block % secspcyl) / wn->psectors;
		sector = block % wn->psectors;
		cmd.sector  = sector + 1;
		cmd.cyl_lo  = cylinder & BYTE;
		cmd.cyl_hi  = (cylinder >> 8) & BYTE;
		cmd.ldh     = wn->ldhpref | head;
	}
	if (multiple > 1) {
		/* Multiple sector block transfer. */
		if (opcode == DEV_SCATTER) {
			cmd.command = ATA_WRITEMULTIPLE;
		} else {
			cmd.command = ATA_READMULTIPLE;
		}
	} else {
		/* Normal per sector transfer. */
		if (opcode == DEV_SCATTER) {
			cmd.command = CMD_WRITE;
		} else {
			cmd.command = CMD_READ;
		}
	}

	r = com_out(&cmd);

	count = 0;
	while (r == OK && nbytes > 0) {
		/* For each sector multiple, wait for an interrupt and fetch
		 * the data (read), or supply data to the controller and wait
		 * for an interrupt (write).
		 */

		count = multiple << SECTOR_SHIFT;
		if (count > nbytes) count = nbytes;

		if (opcode == DEV_GATHER) {
			/* First an interrupt, then data. */
			if ((r = w_intr_wait()) != OK) {
				/* An error, send data to the bit bucket. */
				while ((w_status & STATUS_DRQ) && count > 0) {
					(void) in_byte(wn->base + REG_DATA);
					count--;
				}
				break;
			}
		}

		/* Wait for data transfer requested. */
		if (!waitfor(STATUS_DRQ, STATUS_DRQ)) { r = ERR; break; }

		/* Copy bytes to or from the device's buffer. */
		w_vec_portio(opcode, user_base, iov, count);

		if (opcode == DEV_SCATTER) {
			/* Data sent, wait for an interrupt. */
			if ((r = w_intr_wait()) != OK) break;
		}

		/* Book the bytes successfully transferred. */
		nbytes -= count;
		position = add64u(position, count);
		do {
			chunk = iov->iov_size;
			if (chunk > count) chunk = count;

			iov->iov_addr += chunk;
			if ((iov->iov_size -= chunk) == 0) { iov++; nr_req--; }
			count -= chunk;
		} while (count > 0);
	}

	/* Any errors? */
	if (r != OK) {
		/* Can't pinpoint error in multiple sector block mode. */
		if (++errors == 1) {
			multiple = 1;
		} else
		/* Don't retry if sector marked bad or too many errors. */
		if (r == ERR_BAD_SECTOR || errors == MAX_ERRORS) {
			w_command = CMD_IDLE;
			return(EIO);
		}
	}
  }

  w_command = CMD_IDLE;
  return(OK);
}


/*===========================================================================*
 *				w_vec_portio				     *
 *===========================================================================*/
PRIVATE void w_vec_portio(opcode, user_base, iop, count)
int opcode;
phys_bytes user_base;
iovec_t *iop;
unsigned count;
{
/* Copy count bytes to/from the device's buffer from/to an I/O vector. */
  unsigned n;

  while (count > 0) {
	n = iop->iov_size;
	if (n > count) n = count;
	if (opcode == DEV_GATHER) {
		port_read(w_wn->base + REG_DATA, user_base + iop->iov_addr, n);
	} else {
		port_write(w_wn->base + REG_DATA, user_base + iop->iov_addr, n);
	}
	iop++;
	count -= n;
  }
}


/*============================================================================*
 *				com_out					      *
 *============================================================================*/
PRIVATE int com_out(cmd)
struct command *cmd;		/* Command block */
{
/* Output the command block to the winchester controller and return status */

  tmr_arg_ut dummy;
  struct wini *wn = w_wn;
  unsigned base = wn->base;

  if (!waitfor(STATUS_BSY, 0)) {
	printf("%s: controller not ready\n", w_name());
	return(ERR);
  }

  /* Select drive. */
  out_byte(base + REG_LDH, cmd->ldh);

  if (!waitfor(STATUS_BSY, 0)) {
	printf("%s: drive not ready\n", w_name());
	return(ERR);
  }

  /* Schedule a wakeup call, some controllers are flaky. */
  tmra_settimer(&w_tmra_timeout, get_uptime() + WAKEUP, w_timeout, dummy);

  out_byte(base + REG_CTL, wn->pheads >= 8 ? CTL_EIGHTHEADS : 0);
  out_byte(base + REG_PRECOMP, cmd->precomp);
  out_byte(base + REG_COUNT, cmd->count);
  out_byte(base + REG_SECTOR, cmd->sector);
  out_byte(base + REG_CYL_LO, cmd->cyl_lo);
  out_byte(base + REG_CYL_HI, cmd->cyl_hi);
  lock();
  out_byte(base + REG_COMMAND, cmd->command);
  w_command = cmd->command;
  w_status = STATUS_BSY;
  unlock();
  return(OK);
}


/*===========================================================================*
 *				w_need_reset				     *
 *===========================================================================*/
PRIVATE void w_need_reset()
{
/* The controller needs to be reset. */
  struct wini *wn;

  for (wn = wini; wn < &wini[MAX_DRIVES]; wn++) {
	wn->state |= DEAF;
	wn->state &= ~INITIALIZED;
  }
}


/*===========================================================================*
 *				w_cleanup				     *
 *===========================================================================*/
PRIVATE void w_cleanup()
{
  tmr_arg_ut drive;

  /* Start a timer that will put the disk to sleep when it expires. */
  if (w_wn->sleeptime > 0) {
	drive.ta_int = w_drive;

	tmrs_settimer(&w_dtab.dr_tmrs_context, &w_wn->tmrs_sleep,
		get_uptime() + w_wn->sleeptime,
		w_sleep, drive);
  }
}


/*===========================================================================*
 *				w_sleep					     *
 *===========================================================================*/
PRIVATE void w_sleep(tp, drive)
struct tmrs *tp;
tmr_arg_ut drive;
{
/* Send a disk to sleep. */

  struct command cmd;

  (void) w_prepare(drive.ta_int * DEV_PER_DRIVE);

  if (w_wn->state & INITIALIZED) {
	/* Tell the disk to go standby. */
	cmd.ldh     = w_wn->ldhpref;
	cmd.command = ATA_STANDBYIMM;

	if (com_simple(&cmd) != OK) w_wn->sleeptime = 0;
  }
}


/*============================================================================*
 *				w_do_close				      *
 *============================================================================*/
PRIVATE int w_do_close(dp, m_ptr)
struct driver *dp;
message *m_ptr;
{
/* Device close: Release a device. */

  if (w_prepare(m_ptr->DEVICE) == NIL_DEV) return(ENXIO);
  w_wn->open_ct--;
  return(OK);
}


/*============================================================================*
 *				com_simple				      *
 *============================================================================*/
PRIVATE int com_simple(cmd)
struct command *cmd;		/* Command block */
{
/* A simple controller command, only one interrupt and no data-out phase. */
  int r;

  if ((r = com_out(cmd)) == OK) r = w_intr_wait();
  w_command = CMD_IDLE;
  return(r);
}


/*===========================================================================*
 *				w_timeout				     *
 *===========================================================================*/
PRIVATE void w_timeout(tp, arg)
struct tmra *tp;
tmr_arg_ut arg;
{
  struct wini *wn = w_wn;

  switch (w_command) {
  case CMD_IDLE:
	break;		/* fine */
  case CMD_READ:
  case CMD_WRITE:
  case ATA_READMULTIPLE:
  case ATA_WRITEMULTIPLE:
	/* Impossible, but not on PC's:  The controller does not respond. */

	/* Limiting multisector I/O seems to help. */
	if (wn->multiple > 1) {
		wn->multiple = 1;
	} else
	if (wn->max_count > 8 * SECTOR_SIZE) {
		wn->max_count = 8 * SECTOR_SIZE;
	} else {
		wn->max_count = SECTOR_SIZE;
	}
	/*FALL THROUGH*/
  default:
	/* Some other command. */
	printf("%s: timeout on command %02x\n", w_name(), w_command);
	w_need_reset();
	w_status = 0;
	interrupt(win_tasknr);
  }
}


/*===========================================================================*
 *				w_reset					     *
 *===========================================================================*/
PRIVATE int w_reset()
{
/* Issue a reset to the controller.  This is done after any catastrophe,
 * like the controller refusing to respond.
 */

  struct wini *wn;

  /* Wait for any internal drive recovery. */
  micro_delay(RECOVERYTIME);

  /* Strobe reset bit */
  out_byte(w_wn->base + REG_CTL, CTL_RESET);
  micro_delay(1000L);
  out_byte(w_wn->base + REG_CTL, 0);
  micro_delay(1000L);

  /* Wait for controller ready */
  if (!w_waitfor(STATUS_BSY | STATUS_RDY, STATUS_RDY)) {
	printf("%s: reset failed, drive busy\n", w_name());
	return(ERR);
  }

  /* The error register should be checked now, but some drives mess it up. */

  for (wn = wini; wn < &wini[MAX_DRIVES]; wn++) {
	if (wn->base == w_wn->base) wn->state &= ~DEAF;
  }
  return(OK);
}


/*============================================================================*
 *				w_intr_wait				      *
 *============================================================================*/
PRIVATE int w_intr_wait()
{
/* Wait for a task completion interrupt and return results. */

  message mess;
  int r;

  /* Wait for an interrupt that sets w_status to "not busy". */
  while (w_status & STATUS_BSY) receive(HARDWARE, &mess);

  /* Check status. */
  lock();
  if ((w_status & (STATUS_BSY | STATUS_RDY | STATUS_WF | STATUS_ERR))
							== STATUS_RDY) {
	r = OK;
	w_status |= STATUS_BSY;	/* assume still busy with I/O */
  } else
  if ((w_status & STATUS_ERR) && (in_byte(w_wn->base + REG_ERROR) & ERROR_BB)) {
  	r = ERR_BAD_SECTOR;	/* sector marked bad, retries won't help */
  } else {
  	r = ERR;		/* any other error */
  }
  unlock();
  return(r);
}


/*==========================================================================*
 *				w_waitfor				    *
 *==========================================================================*/
PRIVATE int w_waitfor(mask, value)
int mask;			/* status mask */
int value;			/* required status */
{
/* Wait until controller is in the required state.  Return zero on timeout. */

  struct timeval tv;

  micro_init(&tv);
  do {
       if ((in_byte(w_wn->base + REG_STATUS) & mask) == value) return 1;
  } while (micro_elapsed(&tv) < TIMEOUT);

  w_need_reset();	/* Controller gone deaf. */
  return(0);
}


/*==========================================================================*
 *				w_handler				    *
 *==========================================================================*/
PRIVATE int w_handler(irq)
int irq;
{
/* Disk interrupt, send message to winchester task and reenable interrupts. */

  w_status = in_byte(w_wn->base + REG_STATUS);	/* acknowledge interrupt */
  interrupt(win_tasknr);
  return 1;
}


/*============================================================================*
 *				w_geometry				      *
 *============================================================================*/
PRIVATE void w_geometry(entry)
struct partition *entry;
{
  entry->cylinders = w_wn->lcylinders;
  entry->heads = w_wn->lheads;
  entry->sectors = w_wn->lsectors;
}
#endif /* ENABLE_AT_WINI */

/*
 * $PchId: at_wini.c,v 1.6 1996/01/19 23:30:34 philip Exp $
 */
