/* This file contains the main program of MINIX.  The routine main()
 * initializes the system and starts the ball rolling by setting up the proc
 * table, interrupt vectors, and scheduling each task to run to initialize
 * itself.
 *
 * The entries into this file are:
 *   main:		MINIX main program
 *   panic:		abort MINIX due to a fatal error
 */

#include "kernel.h"
#include <a.out.h>
#include <signal.h>
#include <unistd.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include <minix/minlib.h>
#include "assert.h"
INIT_ASSERT
#include "mq.h"
#include "proc.h"
#include "protect.h"
#include "../version/version.h"	/* Tells the O.S. release and version. */

#define MAP_NR		  16	/* Assume no more than MAP_NR different 
				 * executables in an image.
				 */
#define IMAGE_HDRS	0x600L	/* The struct exec header of the parts of the
				 * image are stored at absolute address
				 * IMAGE_HDRS.
				 */

struct image_map
{
	struct mem_map im_map[NR_SEGS];
	phys_clicks im_bss_clicks;
	phys_clicks im_stack_clicks;
	phys_bytes im_initial_pc;
	int im_valid;
};

struct image_map image_maps[MAP_NR];

FORWARD _PROTOTYPE( void calc_maps, (void) );
FORWARD _PROTOTYPE( void remap_image, (struct proc *pp, int index) );
FORWARD _PROTOTYPE( void create_env, (struct proc *pp)			);
#if DEBUG
FORWARD _PROTOTYPE( void k_printenv, (void) );
#endif
FORWARD _PROTOTYPE( void init_task, (int t, task_t *initial_pc,
	int stack_size, char *name, u32_t msw, reg_t *task_stackp)	);


/*===========================================================================*
 *                                   main                                    *
 *===========================================================================*/
PUBLIC void main()
{
/* Start the ball rolling. */

  register struct proc *rp;
  register int t;
  int imageindex;
  reg_t ktsb;			/* kernel task stack base */
  reg_t msw;
  struct tasktab *ttp;

  /* Initialize the interrupt controller. */
  intr_init(1);

  scr_init(0);		/* We initialize the console before the tty driver 
  			 * to be able to generate console output now.
  			 */
  log_init();

  /* Display the Minix startup banner. */
  printf("Minix-vmd %s.%s  Copyright 1996 Prentice-Hall, Inc.\n\n",
						OS_RELEASE, OS_VERSION);

  /* Set the debug variable. */
  {	long ldebug = 0;
	if (env_parse("debug", "x", 0, &ldebug, 0x0000L, 0xFFFFL) == EP_ON) {
		debug = 1;
	} else {
		debug = ldebug;
	}
  }

  mem_init();		/* Make list of available memory chunks */

  /* Calculate where the different parts of the image are loaded. The results
   * are stored in image_maps[]. Later this information can be remapped in
   * a virtual address space, before it is stored in the process structure.
   * calc_maps also calculates which part of the memory should not be used for
   * user processes.
   */
  calc_maps();		

  vm_init();	/* Allocate all remaining physical memory for virtual memory 
  		 * and put the new address space in the table */
  assert(vm_hard_check_stats() ? 1: (vm_dump(), 0));

  mq_init();
  fpu_init();

  /* Clear the process table.
   * Set up mappings for proc_addr() and proc_number() macros.
   */
  for (rp = BEG_PROC_ADDR, t = -NR_TASKS; rp < END_PROC_ADDR; ++rp, ++t) {
	rp->p_flags = P_SLOT_FREE;
	rp->p_status = 0;
	rp->p_nr = t;		/* proc number from ptr */
	(pproc_addr + NR_TASKS)[t] = rp;	/* proc ptr from number */
	rp->p_name[0]= 0;
	rp->p_flags |= NO_MAP;
  }

  /* Set up proc table entries for tasks and servers.  The stacks of the
   * kernel tasks are initialized to an array in data space.  The bss and
   * stack areas of the servers will be allocated in virtual memory.  The
   * kernel is in low memory, the rest if loaded in extended memory.
   */

  /* Task stacks. */
  ktsb = (reg_t) t_stack;

  /* Setup a dummy proc_ptr. */
  proc_ptr= cproc_addr(IDLE);
  proc_ptr->p_priority= PPRI_IDLE;

  /* Fetch the msw. This value of msw will be stored in process structure and 
   * reloaded during a context switch, so make sure that all changes to msw
   * after this point are done on the msw field in the stackframe
   * (pp->p_reg.msw). Note, msw is actually cr0 on a 386 and above.
   */
  msw= get_msw();
  
  /* Initialize the tasks. */
  for (ttp= tasktab; ttp->tt_name != NULL; ttp++)
  {
  	t= ttp->tt_proc_no;	/* fixed process slot?  (well known task) */
  	if (t == 0)
  	{
		for (t= -NR_TASKS; t < 0; t++)
		{ 
			if (proc_addr(t)->p_flags & P_SLOT_FREE)
				break;
		}
		if (t >= 0)
		{
			printf("no space for task '%s'\n", ttp->tt_name);
			continue;
		}
	}
  	assert(proc_addr(t)->p_flags & P_SLOT_FREE);

	init_task(t, ttp->tt_initial_pc, ttp->tt_stksize, ttp->tt_name,
		msw, &ktsb);
  }

  /* Initialize the servers and processes 1 (init), 2 (inet?), ... */
  for (imageindex= 1; imageindex < MAP_NR && image_maps[imageindex].im_valid;
  	imageindex++)
  {
  	t= imageindex - 1;
  	rp= proc_addr(t);
  	assert(isokprocn(t) && rp->p_flags & P_SLOT_FREE);

	rp->p_reg.sf_psw = INIT_PSW;
	rp->p_reg.sf_msw = msw;
	if (processor >= 486) {
		/* Enable the 486 alignment check. */
		rp->p_reg.sf_msw |= CR0_AM;
	}
	rp->p_flags = 0;
	rp->p_priority = t < LOW_USER ? PPRI_SERVER : PPRI_USER;

	/* Remap this part of the image into a virtual
	 * address space.
	 */
	remap_image(rp, imageindex);

	/* Let's setup the segment map */
	rp->p_map[SEG_T]= image_maps[imageindex].im_map[SEG_T];
	rp->p_map[SEG_D]= image_maps[imageindex].im_map[SEG_D];
	rp->p_map[SEG_S]= image_maps[imageindex].im_map[SEG_S];

	rp->p_reg.sf_pc = (reg_t) image_maps[imageindex].im_initial_pc;
	rp->p_reg.sf_sp = (rp->p_map[SEG_S].mem_vir + 
				rp->p_map[SEG_S].mem_len) << CLICK_SHIFT;
	rp->p_reg.sf_sp -= sizeof(reg_t);

	/* Set inital schedule quantum for init, etc. */
	rp->p_schedquant = DEF_SCHEDQUANT;

	/* Set the name of MM, and FS.  Set the rest to #1, #2, ... */
	switch (t) {
	case MM_PROC_NR:
		strcpy(rp->p_name, "MM");
		break;
	case FS_PROC_NR:
		strcpy(rp->p_name, "FS");
		break;
	default:
		rp->p_name[0]= '#';
		strcpy(rp->p_name+1, itoa((t - INIT_PROC_NR) + 1));
	}
	create_env(rp);

	lock_ready(rp);

	alloc_segments(rp);
  }

  assert(vm_hard_check_stats());

  lock_pick_proc();

  /* Unmap page zero. */
  vm_map_zp(FALSE);

  /* Now go to the assembly code to start running the current process. */
  restart();
}


/*===========================================================================*
 *                                   init_task				     *
 *===========================================================================*/
PRIVATE void init_task(t, initial_pc, stack_size, name, msw, task_stackp)
int t;
task_t *initial_pc;
int stack_size;
char *name;
u32_t msw;
reg_t *task_stackp;
{
	struct proc *rp;

	rp= proc_addr(t);

	assert(rp == cproc_addr(HARDWARE) || stack_size >= 4096);
	if (stack_size > 0) {
		rp->p_stguard = (reg_t *) *task_stackp;
		*rp->p_stguard = STACK_GUARD;
	}
	*task_stackp += stack_size;
	rp->p_reg.sf_sp = *task_stackp;
	
	rp->p_reg.sf_pc = (reg_t) initial_pc;
	rp->p_reg.sf_psw = INIT_TASK_PSW;
	rp->p_reg.sf_msw = msw;
	rp->p_flags = 0;
	
	if (rp != cproc_addr(IDLE))
		rp->p_priority= PPRI_TASK;
	rp->p_status |= P_ST_VM_INCORE;

	/* Setup the kernel segment map */
	if (!image_maps[0].im_valid)
		panic("invalid map for task ", t);

	/* Text and data maps are already setup */
	rp->p_map[SEG_T]= image_maps[0].im_map[SEG_T];
	rp->p_map[SEG_D]= image_maps[0].im_map[SEG_D];

	/* We don't have a stack, the stack is in the data segment.
	 * we also don't have to map any bss since that is part of the
	 * image.
	 */
	assert(image_maps[0].im_bss_clicks == 0);
	assert(image_maps[0].im_stack_clicks == 0);

	rp->p_map[SEG_S]= rp->p_map[SEG_D];
	rp->p_map[SEG_S].mem_vir += rp->p_map[SEG_S].mem_len;
	rp->p_map[SEG_S].mem_len= 0;

#if VMDEXT_KERNEL_LINMAP
	if (rp->p_map[SEG_D].mem_phys == 0)
	{
		/* We got a linearly mapped kernel data segment. */
		rp->p_map[SEG_S].mem_vir= 0xFFFFFFFF >> CLICK_SHIFT;
	}
#endif /* VMDEXT_KERNEL_LINMAP */

	/* IDLE, HARDWARE neveready */
	if (!isidlehardware(t)) lock_ready(rp);

	alloc_segments(rp);
	strcpy(rp->p_name, name);
}


/*===========================================================================*
 *                                   panic				     *
 *===========================================================================*/
PUBLIC void panic(s,n)
_CONST char *s;
int n;
{
/* The system has run aground of a fatal error.  Terminate execution.
 * If the panic originated in MM or FS, the string will be empty and the
 * file system already syncked.  If the panic originates in the kernel, we are
 * kind of stuck.
 */
  stacktrace(0);
  if (s != NULL && *s != 0) {
	printf("\nKernel panic: %s",s);
	if (n != NO_NUM) printf(" %d", n);
	printf("\n");
  }
  wreboot(RBT_PANIC);
}

#if DEBUG
PRIVATE void k_printenv()
{
  register char *namep;
  register char *envp;
  extern char k_environ[];

  for (envp = k_environ; *envp != 0;) {
	printf("'%s'\n", envp);
	while (*envp++ != 0)
		;
  }
}
#endif /* DEBUG */


/*===========================================================================*
 *				calc_maps				     * 
 *===========================================================================*/
PRIVATE void calc_maps()
{
/* Calculate the maps for different kernel parts. */
	int i;
	phys_clicks image_base_clicks, text_base_clicks, data_base_clicks;
	struct exec e_hdr;
	phys_clicks text_clicks, data_clicks, bss_clicks, stack_clicks;
	phys_bytes a_text, a_data, a_bss, a_total, a_entry, a_stack, a_tmp;
	int a_flags;
	struct mem_map *mapp;

	for (i= 0; i<MAP_NR; i++)
		image_maps[i].im_valid= FALSE;

	for (i= 0; i<MAP_NR; i++)
	{
		/* Image headers are placed at IMAGE_HDRS, (normally 0x600)
		 * by the bootstrap loader.
		 */
		phys_copy(IMAGE_HDRS + i * A_MINHDR, vir2phys(&e_hdr),
							(phys_bytes) A_MINHDR);
		if (BADMAG(e_hdr))
			break;			/* We are through. */
		a_flags= e_hdr.a_flags;
		/* We ignore the cpu field. */
		a_text= e_hdr.a_text;
		a_data= e_hdr.a_data;
		a_bss= e_hdr.a_bss;
		a_entry= e_hdr.a_entry;
		a_total= e_hdr.a_total;
		/* a_syms is ignored! */

		/* Where does the text start?  The first (kernel) segment 
		 * starts at kernel_code_base, this includes UZP. Later parts
		 * start at image_base.
		 */
		if (i == 0)
		{
			assert((kernel_code_base & (CLICK_SIZE-1)) == 0);
			image_base_clicks= kernel_code_base >> CLICK_SHIFT;
			if (a_flags & A_UZP)
			{
				/* kernel_code_base is the start of the segment,
				 * we are looking for the start of the actual
				 * image.
				 */
				image_base_clicks++;
			}
		}

		a_stack = a_total - a_data - a_bss;
		if (!(a_flags & A_SEP))
		{
			a_stack -= a_text;
		}

		/* A page aligned program has a header at the front. */
		if (a_flags & A_PAL)
		{
			a_text += e_hdr.a_hdrlen;
		}

		/* Treat a common I&D program as having no text. */
		if (!(a_flags & A_SEP))
		{
			a_data += a_text;
			a_text= 0;
		}

		/* Let's fixup a_text, a_data, ... by rounding them to
		 * click boundaries.  Subtract what is added to a_data
		 * from a_bss, etc.
		 */
		a_text= (a_text + CLICK_SIZE-1) & ~(CLICK_SIZE-1);
		a_tmp= a_data;
		a_data= (a_data + CLICK_SIZE-1) & ~(CLICK_SIZE-1);
		a_bss -= (a_data - a_tmp);
		a_tmp= a_bss;
		a_bss= (a_bss + CLICK_SIZE-1) & ~(CLICK_SIZE-1);
		a_stack -= (a_bss - a_tmp);
		a_stack= (a_stack + CLICK_SIZE-1) & ~(CLICK_SIZE-1);

		/* Now that we got some reasonable sizes, we can allocate
		 * some memory to this part of the image.
		 */
		text_clicks= (a_text >> CLICK_SHIFT);
		data_clicks= (a_data >> CLICK_SHIFT);
		bss_clicks= (a_bss >> CLICK_SHIFT);
		stack_clicks= (a_stack >> CLICK_SHIFT);

		/* Maybe we should check if the total number of clicks does
		 * not overflow.
		 */
		text_base_clicks= image_base_clicks;
		data_base_clicks= text_base_clicks + text_clicks;

		mapp= image_maps[i].im_map;
		mapp[SEG_T].mem_phys= text_base_clicks;
		mapp[SEG_T].mem_vir= 0;
		mapp[SEG_T].mem_len= text_clicks;
		mapp[SEG_D].mem_phys= data_base_clicks;
		mapp[SEG_D].mem_vir= 0;
		mapp[SEG_D].mem_len= data_clicks;

		if (i == 0)
		{
			/* The kernel bss and stack are not virtual, but are
			 * created by the bootstrap loader.  Add them to the
			 * data segment.
			 */
			mapp[SEG_D].mem_len += bss_clicks + stack_clicks;
			bss_clicks= stack_clicks= 0;
		}
		image_maps[i].im_bss_clicks= bss_clicks;
		image_maps[i].im_stack_clicks= stack_clicks;
		image_base_clicks= mapp[SEG_D].mem_phys + mapp[SEG_D].mem_vir +
			mapp[SEG_D].mem_len;

		/* Delete the memory for the process from the list of memory
		 * chunks.  Also remove memory in front of the kernel.
		 */
		if (i == 0) chunk_del(0, text_base_clicks);
		chunk_del(text_base_clicks, image_base_clicks-text_base_clicks);

		/* Adjust for UZP. */
		if (a_flags & A_UZP)
		{
			/* Assume CLICK_SIZE == page size */
			mapp[SEG_T].mem_phys--;
			mapp[SEG_T].mem_vir++;
			mapp[SEG_D].mem_phys--;
			mapp[SEG_D].mem_vir++;
		}

		image_maps[i].im_initial_pc= a_entry;
		image_maps[i].im_valid= TRUE;

#if DEBUG
		printf(
"map[%d]: (p,v,l) (0x%x, 0x%x, 0x%x) (0x%x, 0x%x, 0x%x) b=0x%x, s=0x%x\n",
			i,
			mapp[SEG_T].mem_phys, mapp[SEG_T].mem_vir,
							mapp[SEG_T].mem_len,
			mapp[SEG_D].mem_phys, mapp[SEG_D].mem_vir,
							mapp[SEG_D].mem_len,
			image_maps[i].im_bss_clicks,
			image_maps[i].im_stack_clicks);
#endif

#if WORD_SIZE == 4
		if (i == 0) {
			/* MM, FS, etc. are loaded in extended memory. */
			image_base_clicks= 0x100000 >> CLICK_SHIFT;
		}
#endif
	}
	if (i == MAP_NR)
		printf("calc_maps: warning to many image parts (>%d)\n", i-1);
}


/*===========================================================================*
 *				remap_image				     * 
 *===========================================================================*/
PRIVATE void remap_image(pp, index)
struct proc *pp;
int index;
{
	phys_clicks t_phys, t_vir, t_len, d_phys, d_vir, d_len;
	phys_clicks bss_clicks, stack_clicks, total_clicks;
	u32_t total;
	u32_t text_base, data_base, vm_base;
	int sep, uzp;

	t_phys= image_maps[index].im_map[SEG_T].mem_phys;
	t_vir= image_maps[index].im_map[SEG_T].mem_vir;
	t_len= image_maps[index].im_map[SEG_T].mem_len;
	d_phys= image_maps[index].im_map[SEG_D].mem_phys;
	d_vir= image_maps[index].im_map[SEG_D].mem_vir;
	d_len= image_maps[index].im_map[SEG_D].mem_len;

	bss_clicks= image_maps[index].im_bss_clicks;
	stack_clicks= image_maps[index].im_stack_clicks;
	if (stack_clicks < 512)
		stack_clicks= 512;

	sep= (t_phys != d_phys);
	assert(!sep || d_phys == t_phys + t_len);	/* ! */
	uzp= (t_vir != 0);
	assert(!uzp || (t_vir == 1 && d_vir == 1));
	
	/* Calculate how much space we need. */
	total_clicks= 0;
	if (sep)
	{
		total_clicks += t_vir + t_len;
	}
	total_clicks += d_vir + d_len + bss_clicks + stack_clicks;
	total= total_clicks << CLICK_SHIFT;
	vm_base= vm_alloc_space(pp, total);

	/* Now map the text segment (if sep I&D), and fill in the segment
	 * map.
	 */
	if (sep)
	{
		text_base= vm_base;
		if (uzp)
			text_base += t_vir << CLICK_SHIFT;
		vm_map_image(pp, text_base, (t_phys+t_vir) << CLICK_SHIFT,
							t_len << CLICK_SHIFT);
	}
	image_maps[index].im_map[SEG_T].mem_phys= vm_base >> CLICK_SHIFT;
	image_maps[index].im_map[SEG_T].mem_vir= t_vir;
	image_maps[index].im_map[SEG_T].mem_len= t_len;

	/* Next map the data segment, and fill in the map. */
	if (sep)
	{
		vm_base= (image_maps[index].im_map[SEG_T].mem_phys +
			image_maps[index].im_map[SEG_T].mem_vir +
			image_maps[index].im_map[SEG_T].mem_len) << CLICK_SHIFT;
	}
	data_base= vm_base;
	if (uzp)
		data_base += d_vir << CLICK_SHIFT;
	vm_map_image(pp, data_base, (d_phys+d_vir) << CLICK_SHIFT,
							d_len << CLICK_SHIFT);
	image_maps[index].im_map[SEG_D].mem_phys= vm_base >> CLICK_SHIFT;
	image_maps[index].im_map[SEG_D].mem_vir= d_vir;
	image_maps[index].im_map[SEG_D].mem_len= d_len;

	/* And we have preallocate the bss. */
	vm_base= (image_maps[index].im_map[SEG_D].mem_phys +
		image_maps[index].im_map[SEG_D].mem_vir +
		image_maps[index].im_map[SEG_D].mem_len) << CLICK_SHIFT;
	vm_make_clear(pp, vm_base, bss_clicks << CLICK_SHIFT);
	image_maps[index].im_map[SEG_D].mem_len += bss_clicks;

	image_maps[index].im_map[SEG_S].mem_phys= 
				image_maps[index].im_map[SEG_D].mem_phys;
	image_maps[index].im_map[SEG_S].mem_vir= 
		image_maps[index].im_map[SEG_D].mem_vir +
		image_maps[index].im_map[SEG_D].mem_len + stack_clicks;
	image_maps[index].im_map[SEG_S].mem_len= 0;

	assert(vm_hard_check_stats());
}


/*===========================================================================*
 *				create_env				     * 
 *===========================================================================*/
PRIVATE void create_env(pp)
struct proc *pp;
{
	vir_bytes envs, envv;
	vir_bytes args, argv;
	int len, argc, envc, args_size, envs_size;
	char *p;
	char *name;
	char *ptr;
	vir_bytes sp;
	phys_bytes phys_env;
	u32_t base, top;
	vir_clicks clicks;

	/* Program name. */
	name= pp->p_name;

	/* First, calculate the position of the argument and environment
	 * vectors.
	 */
	len= strlen(name)+1;		/* Program name */
	args_size= len;
	argc= 1;

	envs_size= 0;
	envc= 0;
	p= k_environ;
	while (p[0] != '\0')
	{
		len= strlen(p)+1;
		envs_size += len;
		envc++;
		p += len;
	}

	/* Make space for strings, vectors and argument count. */
	sp= pp->p_reg.sf_sp;
	sp -= envs_size + args_size;
	sp -= (envc + 1 + argc + 1) * sizeof(char *);
	sp -= sizeof(char *);
	sp &= ~(sizeof(char *) -1);

	/* Map stack pages */
	top= (pp->p_map[SEG_S].mem_phys + pp->p_map[SEG_S].mem_vir + 
				pp->p_map[SEG_S].mem_len) << CLICK_SHIFT;
	base= top - (pp->p_reg.sf_sp - sp);
	base &= ~(CLICK_SIZE-1);
	clicks= (top-base) >> CLICK_SHIFT;
	pp->p_map[SEG_S].mem_vir -= clicks;
	pp->p_map[SEG_S].mem_len += clicks;
	vm_make_clear(pp, base, top-base);

	/* Addresses of vectors and strings. */
	argv= sp + sizeof(char *);
	envv= argv + (argc+1) * sizeof(char *);
	args= envv + (envc+1) * sizeof(char *);
	envs= args + args_size;

	/* Set argc. */
	ptr= (char *) argc;
	phys_env= umap(pp, SEG_D, sp, sizeof(char *));
	assert(phys_env != 0);
	phys_copy(vir2phys(&ptr), phys_env, sizeof(char *));

	/* Copy the arguments (only the program name at the moment). */
	ptr= (char *) args;
	phys_env= umap(pp, SEG_D, argv, sizeof(char *));
	assert(phys_env != 0);
	phys_copy(vir2phys(&ptr), phys_env, sizeof(char *));
	argv += sizeof(char *);

	len= strlen(name)+1;
	phys_env= umap(pp, SEG_D, args, sizeof(char *));
	assert(phys_env != 0);
	phys_copy(vir2phys(name), phys_env, len);
	args += len;

	/* Put the terminating NULL */
	ptr= NULL;
	phys_env= umap(pp, SEG_D, argv, sizeof(char *));
	assert(phys_env != 0);
	phys_copy(vir2phys(&ptr), phys_env, sizeof(char *));
	argv += sizeof(char *);

	assert(args == envs);
	assert(argv == envv);

	/* Copy the environment */
	p= k_environ;
	while (p[0] != '\0')
	{
		len= strlen(p)+1;

		/* Set envv[n]. */
		ptr= (char *) envs;
		phys_env= umap(pp, SEG_D, envv, sizeof(char *));
		assert(phys_env != 0);
		phys_copy(vir2phys(&ptr), phys_env, sizeof(char *));
		envv += sizeof(char *);

		/* Copy the string */
		phys_env= umap(pp, SEG_D, envs, len);
		assert(phys_env != 0);
		phys_copy(vir2phys(p), phys_env, len);
		envs += len;

		p += len;
	}

	/* Add a terminating NULL for the environment pointer array */
	ptr= NULL;
	phys_env= umap(pp, SEG_D, envv, sizeof(char *));
	assert(phys_env != 0);
	phys_copy(vir2phys(&ptr), phys_env, sizeof(char *));
	envv += sizeof(char *);

	assert(envv == args - args_size);
	assert((pp->p_reg.sf_sp - envs) < sizeof(char *));

	/* Commit the new stack. */
	pp->p_reg.sf_sp= sp;

	assert(vm_hard_check_stats());
}

/*
 * $PchId: main.c,v 1.5 1996/01/19 23:02:26 philip Exp $
 */
