/* This file contains device independent device driver interface.
 *							Author: Kees J. Bot.
 *
 * The drivers support the following operations (using message format m2):
 *
 *    m_type      DEVICE    PROC_NR     COUNT    POSITION  ADRRESS
 * ----------------------------------------------------------------
 * |  DEV_OPEN  | device  | proc nr |         |         |         |
 * |------------+---------+---------+---------+---------+---------|
 * |  DEV_CLOSE | device  | proc nr |         |         |         |
 * |------------+---------+---------+---------+---------+---------|
 * |  DEV_READ  | device  | proc nr |  bytes  |  offset | buf ptr |
 * |------------+---------+---------+---------+---------+---------|
 * |  DEV_WRITE | device  | proc nr |  bytes  |  offset | buf ptr |
 * |------------+---------+---------+---------+---------+---------|
 * | DEV_GATHER | device  | proc nr | iov len |  offset | iov ptr |
 * |------------+---------+---------+---------+---------+---------|
 * | DEV_SCATTER| device  | proc nr | iov len |  offset | iov ptr |
 * |------------+---------+---------+---------+---------+---------|
 * |  DEV_IOCTL | device  | proc nr |func code|         | buf ptr |
 * ----------------------------------------------------------------
 *
 * The file contains one entry point:
 *
 *   driver_task:	called by the device dependent task entry
 *
 *
 * Changes:
 *       2 Apr 1992 by Kees J. Bot: created from the old AT and floppy driver.
 *	13 Apr 1992 by Kees J. Bot: device dependent/independent split.
 *	10 Dec 1995 by Kees J. Bot: d-d/i rewrite.
 */

#include "kernel.h"
#include <sys/diskio.h>
#include <sys/ioctl.h>
#include "driver.h"
#include "proc.h"

#if (CHIP == INTEL)
#if ENABLE_SCSI && DMA_BUF_SIZE < 8192
/* A bit extra scratch for the SCSI driver. */
#define BUF_EXTRA	(8192 - DMA_BUF_SIZE)
#else
#define BUF_EXTRA	0
#endif

/* Claim space for variables. */
PRIVATE u8_t buffer[(unsigned) 2 * DMA_BUF_SIZE + BUF_EXTRA];
u8_t *tmp_buf;			/* the DMA buffer eventually */
phys_bytes tmp_phys;		/* phys address of DMA buffer */

#else /* CHIP != INTEL */

/* Claim space for variables. */
u8_t tmp_buf[DMA_BUF_SIZE];	/* the DMA buffer */
phys_bytes tmp_phys;		/* phys address of DMA buffer */

#endif /* CHIP != INTEL */

FORWARD _PROTOTYPE( void init_buffer, (void) );
FORWARD _PROTOTYPE( int do_ioctl, (struct driver *dp, message *mp) );
#if PAGING_VM
FORWARD _PROTOTYPE( int do_swapon, (struct driver *dp, message *mp) );
FORWARD _PROTOTYPE( int do_swapinfo, (struct driver *dp, message *mp) );
#endif /* PAGING_VM */


/*===========================================================================*
 *				driver_task				     *
 *===========================================================================*/
PUBLIC void driver_task(dp)
struct driver *dp;	/* Device dependent entry points. */
{
/* Main program of any device driver task. */

  int r, exp, caller, proc_nr;
  message mess;

  init_buffer();	/* Get a DMA buffer. */

  tmrs_initcontext(&dp->dr_tmrs_context, &exp, proc_number(proc_ptr));


  /* Here is the main loop of the disk task.  It waits for a message, carries
   * it out, and sends a reply.
   */

  while (TRUE) {
	/* Check if a timer expired. */
	if (exp) tmrs_exptimers(&dp->dr_tmrs_context);

	/* Wait for a request to read or write a disk block. */
	receive(ANY, &mess);

	caller = mess.m_source;
	proc_nr = mess.PROC_NR;

	/* Check if legitimate caller: FS or a task. */
	if (caller != FS_PROC_NR && caller >= 0) {
		printf("%s: got message from %d\n", (*dp->dr_name)(), caller);
		continue;
	}

	/* Now carry out the work. */
	switch(mess.m_type) {
	case DEV_OPEN:		r = (*dp->dr_open)(dp, &mess);	break;
	case DEV_CLOSE:		r = (*dp->dr_close)(dp, &mess);	break;
	case DEV_IOCTL:		r = do_ioctl(dp, &mess);	break;

	case DEV_READ:
	case DEV_WRITE:		r = do_rdwt(dp, &mess);		break;

	case DEV_GATHER:
	case DEV_SCATTER:	r = do_vrdwt(dp, &mess);	break;

	case HARD_INT:		/* Leftover interrupt or expired timer. */
				continue;

	default:		r = EINVAL;			break;
	}

	/* Clean up leftover state. */
	(*dp->dr_cleanup)();

	/* Finally, prepare and send the reply message. */
	mess.m_type = TASK_REPLY;
	mess.REP_PROC_NR = proc_nr;

	mess.REP_STATUS = r;	/* # of bytes transferred or error code */
	send(caller, &mess);	/* send reply to caller */
  }
}


/*===========================================================================*
 *				init_buffer				     *
 *===========================================================================*/
PRIVATE void init_buffer()
{
/* Select a buffer that can safely be used for dma transfers.  It may also
 * be used to read partition tables and such.  Its absolute address is
 * 'tmp_phys', the normal address is 'tmp_buf'.
 */

#if (CHIP == INTEL)
  unsigned left;

  tmp_buf = buffer;
  tmp_phys = vir2phys(buffer);

  if ((left = dma_bytes_left(tmp_phys)) < DMA_BUF_SIZE) {
	/* First half of buffer crosses a 64K boundary, can't DMA into that */
	tmp_buf += left;
	tmp_phys += left;
  }
#else /* CHIP != INTEL */
  tmp_phys = vir2phys(tmp_buf);
#endif /* CHIP != INTEL */
}


/*===========================================================================*
 *				do_rdwt					     *
 *===========================================================================*/
PUBLIC int do_rdwt(dp, m_ptr)
struct driver *dp;		/* device dependent entry points */
message *m_ptr;			/* pointer to read or write message */
{
/* Carry out a single read or write request. */
  iovec_t iovec1;
  vir_bytes address;
  int r, opcode, nbytes;
  struct proc *pp;
  phys_bytes phys_addr;
  u64_t position;

  /* Disk address?  Address and length of the user buffer? */
  position = make64(m_ptr->POSITION, m_ptr->HIGHPOS);
  address = (vir_bytes) m_ptr->ADDRESS;
  if ((nbytes = m_ptr->COUNT) < 0) return(EINVAL);

  /* Check the user buffer. */
  pp = proc_addr(m_ptr->PROC_NR);
  if ((phys_addr = umap(pp, SEG_D, address, nbytes)) == 0) return(EFAULT);

  /* Prepare for I/O. */
  if ((*dp->dr_prepare)(m_ptr->DEVICE) == NIL_DEV) return(ENXIO);

#if PAGING_VM
  /* Check if we need locking and if so lock the user buffer into memory. */
  if (!(pp->p_status & P_ST_VM_INCORE)) {
	if ((r = vm_lock(phys_addr, nbytes, FALSE)) != OK) return(r);
  }
#endif /* PAGING_VM */

  /* Create a one element scatter/gather vector for the buffer. */
  opcode = m_ptr->m_type == DEV_READ ? DEV_GATHER : DEV_SCATTER;
  iovec1.iov_addr = address;
  iovec1.iov_size = nbytes;

  /* Transfer bytes from/to the device. */
  r = (*dp->dr_transfer)(m_ptr->PROC_NR, opcode, position, &iovec1, 1);

#if PAGING_VM
  /* Unlock the user buffer. */
  if (!(pp->p_status & P_ST_VM_INCORE)) vm_unlock(phys_addr, nbytes);
#endif

  /* Return the number of bytes transferred or an error code. */
  return(r == OK ? (nbytes - iovec1.iov_size) : r);
}


/*==========================================================================*
 *				do_vrdwt				    *
 *==========================================================================*/
PUBLIC int do_vrdwt(dp, m_ptr)
struct driver *dp;	/* device dependent entry points */
message *m_ptr;		/* pointer to read or write message */
{
/* Carry out an device read or write to/from a vector of user addresses.
 * The "user addresses" are assumed to be safe, i.e. FS transferring to/from
 * its own buffers, so they are not checked.
 */
  iovec_t iovec[NR_IOREQS];
  iovec_t *iov;
  phys_bytes iovec_phys, user_iovec_phys;
  size_t iovec_size;
  unsigned nr_req;
  int r;
  u64_t position;

  /* Disk address?  Length of I/O vector? */
  position = make64(m_ptr->POSITION, m_ptr->HIGHPOS);
  nr_req = m_ptr->COUNT;

  if (m_ptr->m_source < 0) {
	/* Called by a task, no need to copy vector. */
	iov = (iovec_t *) m_ptr->ADDRESS;
  } else {
	/* Copy the vector from the caller to kernel space. */
	if (nr_req > NR_IOREQS) nr_req = NR_IOREQS;
	iovec_size = nr_req * sizeof(iovec[0]);
	user_iovec_phys = umap(proc_addr(m_ptr->m_source), SEG_D,
				(vir_bytes) m_ptr->ADDRESS, iovec_size);
	if (user_iovec_phys == 0) panic("bad I/O vector by", m_ptr->m_source);

	iovec_phys = vir2phys(iovec);
	phys_copy(user_iovec_phys, iovec_phys, (phys_bytes) iovec_size);
	iov = iovec;
  }

  /* Prepare for I/O. */
  if ((*dp->dr_prepare)(m_ptr->DEVICE) == NIL_DEV) return(ENXIO);

  /* Transfer bytes from/to the device. */
  r = (*dp->dr_transfer)(m_ptr->PROC_NR, m_ptr->m_type, position, iov, nr_req);

  /* Copy the I/O vector back to the caller. */
  if (m_ptr->m_source >= 0) {
	  phys_copy(iovec_phys, user_iovec_phys, (phys_bytes) iovec_size);
  }
  return(r);
}


/*===========================================================================*
 *				no_name					     *
 *===========================================================================*/
PUBLIC char *no_name()
{
/* If no specific name for the device. */

  return(proc_ptr->p_name);
}


/*============================================================================*
 *				do_nop					      *
 *============================================================================*/
PUBLIC int do_nop(dp, m_ptr)
struct driver *dp;
message *m_ptr;
{
/* Nothing there, or nothing to do. */

  switch (m_ptr->m_type) {
  case DEV_OPEN:	return(ENODEV);
  case DEV_CLOSE:	return(OK);
  case DEV_IOCTL:	return(ENOTTY);
  default:		return(EIO);
  }
}


/*===========================================================================*
 *				nop_cleanup				     *
 *===========================================================================*/
PUBLIC void nop_cleanup()
{
/* Nothing to clean up. */
}


/*============================================================================*
 *				do_diocntl				      *
 *============================================================================*/
PUBLIC int do_diocntl(dp, m_ptr)
struct driver *dp;
message *m_ptr;			/* pointer to ioctl request */
{
/* Carry out a partition setting/getting request. */
  struct device *dv;
  phys_bytes user_phys, entry_phys;
  struct partition entry;
  struct partition32 entry32;
  size_t entry_size;

  switch (m_ptr->REQUEST) {
  case DIOCSETP:
  case DIOCGETP:
	entry_phys = vir2phys(&entry);
	entry_size = sizeof(entry);
	break;
  case DIOCSETP32:
  case DIOCGETP32:
	entry_phys = vir2phys(&entry32);
	entry_size = sizeof(entry32);
	break;
  default:
	return(ENOTTY);
  }

  /* Decode the message parameters. */
  if ((dv = (*dp->dr_prepare)(m_ptr->DEVICE)) == NIL_DEV) return(ENXIO);

  user_phys = numap(m_ptr->PROC_NR, (vir_bytes) m_ptr->ADDRESS, entry_size);
  if (user_phys == 0) return(EFAULT);

  switch (m_ptr->REQUEST) {
  case DIOCSETP:
	/* Set the base and size of a partition. */
	phys_copy(user_phys, entry_phys, (phys_bytes) entry_size);
	dv->dv_base = entry.base;
	dv->dv_size = entry.size;
	break;
  case DIOCGETP:
	/* Return a partition table entry and the geometry of the drive. */
	entry.base = dv->dv_base;
	entry.size = dv->dv_size;
	(*dp->dr_geometry)(&entry);
	phys_copy(entry_phys, user_phys, (phys_bytes) entry_size);
	break;
  case DIOCSETP32:
	/* Standard Minix compatibility: Set the base and size. */
	phys_copy(user_phys, entry_phys, (phys_bytes) entry_size);
	dv->dv_base = cvul64(entry32.base);
	dv->dv_size = cvul64(entry32.size);
	break;
  case DIOCGETP32:
	/* Standard Minix compatibility: Return partition info. */
	entry32.base = cv64ul(dv->dv_base);
	entry32.size = cv64ul(dv->dv_size);
	(*dp->dr_geometry)(&entry);
	entry32.cylinders = entry.cylinders;
	entry32.heads = entry.heads;
	entry32.sectors = entry.sectors;
	phys_copy(entry_phys, user_phys, (phys_bytes) entry_size);
	break;
  }
  return(OK);
}


/*===========================================================================*
 *				do_ioctl				     *
 *===========================================================================*/
PRIVATE int do_ioctl(dp, mp)
struct driver *dp;
message *mp;
{
  switch(mp->REQUEST) {
#if PAGING_VM
  case DIOCSWAPON:	return(do_swapon(dp, mp));
  case DIOCSWAPINFO:	return(do_swapinfo(dp, mp));
#endif /* PAGING_VM */
  default:		return((*dp->dr_ioctl)(dp, mp));
  }
}


#if PAGING_VM
/*===========================================================================*
 *				do_swapon				     *
 *===========================================================================*/
PRIVATE int do_swapon(dp, mp)
struct driver *dp;
message *mp;
{
  dio_swapon_t dio_swapon;
  int who;
  vir_bytes addr;
  phys_bytes addr_phys;

  who = mp->PROC_NR;
  addr = (vir_bytes)(mp->ADDRESS);

  /* Fetch the parameters. */
  addr_phys = numap(who, addr, sizeof(dio_swapon));
  if (addr_phys == 0) {
	/* Illegal address. */
	return(EFAULT);
  }
  phys_copy(addr_phys, vir2phys(&dio_swapon), sizeof(dio_swapon));
  return(vm_swapon(proc_number(proc_ptr), mp->DEVICE,
				dio_swapon.ds_offset, dio_swapon.ds_size,
				dio_swapon.ds_priority));
}


/*===========================================================================*
 *				do_swapinfo				     *
 *===========================================================================*/
PRIVATE int do_swapinfo(dp, mp)
struct driver *dp;
message *mp;
{
  dio_swapinfo_t dio_swapinfo;
  int who;
  vir_bytes addr;
  phys_bytes addr_phys;
  int r;

  r = vm_swapinfo(proc_number(proc_ptr), mp->DEVICE, &dio_swapinfo.dsi_index);
  if (r != OK) return(r);

  who = mp->PROC_NR;
  addr = (vir_bytes)(mp->ADDRESS);

  addr_phys = numap(who, addr, sizeof(dio_swapinfo));
  if (addr_phys == 0) {
	/* Illegal address. */
	return(EFAULT);
  }

  /* Copy the struct to the user process. */
  phys_copy(vir2phys(&dio_swapinfo), addr_phys, sizeof(dio_swapinfo));
#if DEBUG
  printf("task %d, got a swapinfo request for minor %d, got %ld\n",
					proc_number(proc_ptr), mp->DEVICE,
					(long)dio_swapinfo.dsi_index);
#endif
  return(OK);
}
#endif /* PAGING_VM */

/*
 * $PchId: driver.c,v 1.6 1996/01/19 23:27:01 philip Exp $
 */
