/* This file contains a few general purpose utility routines.
 *
 * The entry points into this file are
 *   clock_time:  ask the clock task for the real time
 *   copy:	  copy a block of data
 *   fetch_name:  go get a path name from user space
 *   no_sys:      reject a system call that FS does not handle
 *   panic:       something awful has occurred;  MINIX cannot continue
 *   swap2:	  do byte swapping on a 16-bit int
 *   swap4:	  do byte swapping on a 32-bit long
 */

#include "fs.h"
#include <unistd.h>
#include <minix/com.h>
#if (CHIP == INTEL && WORD_SIZE == 4)
#include <ibm/reg.h>
#endif /* CHIP == INTEL && WORD_SIZE == 4 */
#include "buf.h"
#include "file.h"
#include "fproc.h"
#include "inode.h"
#include "param.h"

PRIVATE int panicking;		/* inhibits recursive panics during sync */
PRIVATE message clock_mess;

/*===========================================================================*
 *				clock_time				     *
 *===========================================================================*/
PUBLIC time_t clock_time()
{
/* This routine returns the time in seconds since 1.1.1970.  MINIX is an
 * astrophysically naive system that assumes the earth rotates at a constant
 * rate and that such things as leap seconds do not exist.
 */

  register int k;

  clock_mess.m_type = GET_TIME;
  if ( (k = sendrec(clck_tasknr, &clock_mess)) != OK)
  	panic("clock_time err", k);

  return (time_t) clock_mess.NEW_TIME;
}


/*===========================================================================*
 *				fetch_name				     *
 *===========================================================================*/
PUBLIC int fetch_name(path, len, flag)
char *path;			/* pointer to the path in user space */
int len;			/* path length, including 0 byte */
int flag;			/* M3 means path may be in message */
{
/* Go get path and put it in 'user_path'.
 * If 'flag' = M3 and 'len' <= M3_STRING, the path is present in 'message'.
 * If it is not, go copy it from user space.
 */

  register char *rpu, *rpm;
  int r;

  /* Check name length for validity. */
  if (len <= 0) {
	return(EINVAL);
  }

  if (len > PATH_MAX) {
	return(ENAMETOOLONG);
  }

  if (flag == M3 && len <= M3_STRING) {
	/* Just copy the path from the message to 'user_path'. */
	rpu = &user_path[0];
	rpm = fetchname_name;		/* contained in input message */
	do { *rpu++ = *rpm++; } while (--len);
	r = OK;
  } else {
	/* String is not contained in the message.  Get it from user space. */
	r = sys_copy(who, SEG_D, (phys_bytes) path,
		FS_PROC_NR, SEG_D, (phys_bytes) user_path, (phys_bytes) len);
  }
  return(r);
}


/*===========================================================================*
 *				no_sys					     *
 *===========================================================================*/
PUBLIC int no_sys()
{
/* Somebody has used an illegal system call number */

  printf("fs: no_sys(): process %d sends %d\n", who, fs_call);
  if (who >= 0 && fp->fp_pid != PID_SERVER)
	  return(ENOSYS);
  panic(NULL, NO_NUM);
}


/*===========================================================================*
 *				panic					     *
 *===========================================================================*/
PUBLIC void panic(format, num)
char *format;			/* format string */
int num;			/* number to go with format string */
{
/* Something awful has happened.  Panics are caused when an internal
 * inconsistency is detected, e.g., a programming error or illegal value of a
 * defined constant.
 */

  if (panicking) return;	/* do not panic during a sync */

  panicking = TRUE;		/* prevent another panic during the sync */
  if (format)
  {
	  printf("File system panic: %s ", format);
	  if (num != NO_NUM) printf("%d",num); 
	  printf("\n");
  }
  (void) sync_all(GRIMY);	/* flush everything to the disk */
#if (CHIP == INTEL && WORD_SIZE == 4)
  stacktrace();
#endif /* CHIP == INTEL && WORD_SIZE == 4 */
  sys_abort(RBT_PANIC);
}



/*===========================================================================*
 *				swap2					     *
 *===========================================================================*/
PUBLIC unsigned swap2(w)
int w;				/* promotion of 16-bit word to be swapped */
{
/* Swap a 16-bit word between 8086 and 68000 byte order. */

  return( ((w&BYTE) << 8) | ( (w>>8) & BYTE));
}

/*===========================================================================*
 *				swap4					     *
 *===========================================================================*/
PUBLIC long swap4(l)
long l;				/* 32-bit long to be byte swapped */
{
/* Swap a 32-bit long between 8086 and 68000 byte order. */

  return( ((l << 24) & 0xFF000000L)
  	| ((l <<  8) & 0x00FF0000L)
  	| ((l >>  8) & 0x0000FF00L)
  	| ((l >> 24) & 0x000000FFL));
}

/*===========================================================================*
 *				clock_utime				     *
 *===========================================================================*/
PUBLIC int clock_utime(proc, op, ptr)
int proc;
int op;
void *ptr;	/* Actually struct timeval *ptr */
{
	message m;

	m.m1_i1= proc;
	m.m1_i2= op;
	m.m1_p1= ptr;
	return _taskcall(clck_tasknr, SYS_UTIME, &m);
}

#if (CHIP == INTEL && WORD_SIZE == 4)
PUBLIC void stacktrace()
{
	typedef unsigned int reg_t;
	reg_t bp, pc, hbp;

	bp= get_bp();
	while(bp)
	{
		pc= ((reg_t *)bp)[1];
		hbp= ((reg_t *)bp)[0];
		printf("0x%lx ", (unsigned long)pc);
		if (hbp != 0 && hbp <= bp)
		{
			printf("???");
			break;
		}
		bp= hbp;
	}
	printf("\n");
}
#endif /* CHIP == INTEL && WORD_SIZE == 4 */

/*
 * $PchId: utility.c,v 1.6 1996/02/29 23:09:24 philip Exp $
 */
