/* This file contains tty code for the file system. In een ideal world a tty
 * is just another character special device that can be handled be device.c.
 * But in the real world /dev/tty exists and setsid should release the
 * controlling terminal, so it is better to move all tty specific code to
 * one file.
 *
 * The entry points in this file are:
 *	do_fs_setsid:	implement the FS part of setsid.
 *	ntty_open:	perform tty-specific processing for open
 *	ntty_rw:	perform tty-specific processing for read and write
 *	ntty_close:	perform tty-specific processing for close
 */

#include "fs.h"
#include <fcntl.h>
#include <sys/stat.h>
#include <minix/com.h>
#include "file.h"
#include "fproc.h"
#include "inode.h"
#include "param.h"
#include "assert.h"
INIT_ASSERT

/*===========================================================================*
 *				do_setsid				     *
 *===========================================================================*/
PUBLIC int do_setsid()
{
  struct fproc *fp;
  int i, j, status;
  dev_t dev;
  struct od **odpp, *odp;
  struct filp *f, *ctty_filp;
  struct inode *ip;

  assert(fs_call == SETSID);

  /* Only MM may do the SETSID call directly. */
  if (who != MM_PROC_NR) return(ENOSYS);	

  /* Nevertheless, pretend that the call came from the user. */
  fp = &fproc[setsid_proc];

  /* let's get rid of the controlling tty. */
  ctty_filp= NIL_FILP;
  if (fp->fp_tty)
  {
	for(i = 0, odpp= fp->fp_filp; i<OPEN_MAX; i++, odpp++)
	{
		odp= *odpp;
		if (!odp)
			continue;
		f= odp->od_filp;
		ip= f->filp_ino;
		if ((ip->i_mode & I_TYPE) != I_CHAR_SPECIAL)
			continue;
		dev= (dev_t) ip->i_zone[0];
		if (((dev >> MAJOR) & BYTE) != CTTY_MAJOR)
			continue;
		if (!ctty_filp)
		{
			for (j= 0, ctty_filp= filp_table; j<NR_FILPS; j++, 
								ctty_filp++)
			{
				if (ctty_filp->filp_count != 0)
					continue;
				*ctty_filp= *f;
				dup_inode(ctty_filp->filp_ino);
				ctty_filp->filp_int_flags= FIF_CLOSED;
				ctty_filp->filp_count= 1;
				break;
			}
			if (j == NR_FILPS)
				ctty_filp= NIL_FILP;
		}
		if (f->filp_count - 1 == 0 && !(f->filp_int_flags & FIF_CLOSED))
		{
			status= dev_opcl(0, dev, fp-fproc, 0);
			if (status != OK) panic("close failed", NO_NUM);
		}
		if (--f->filp_count == 0) put_inode(ip);
		assert(f->filp_count >= 0);
		if (ctty_filp)
		{
			odp->od_filp= ctty_filp;
			ctty_filp->filp_count++;
		}
		else
		{
			odp->od_flags= 0;
			*odpp= NIL_ODP;
			fp->fp_cloexec &= ~(1L << i);
		}
	}
	if (ctty_filp)
	{
		ctty_filp->filp_count--;
		assert(ctty_filp->filp_count > 0);
	}
	assert((fp->fp_ttyfilp == NIL_FILP && fp->fp_ttylink == 0) ||
		(printf("proc= %d, fp_ttyfilp= 0x%x, fp_ttylink= %d\n",
			setsid_proc, fp->fp_ttyfilp, fp->fp_ttylink), 0));
	fp->fp_tty= 0;
  }
  fp->fp_sesldr= 1;
  return (OK);
}

/*===========================================================================*
 *				exit_sesldr				     *
 *===========================================================================*/
PUBLIC void exit_sesldr()
{
/* A session leader with a controlling terminal died. That terminal is revoked
 * form all processes using it.
 */
  dev_t ctrl_tty;
  struct filp *f;
  struct od **odpp, *odp;
  struct fproc *rfp;
  struct inode *ip;
  int i, j, status, check_tty;
  dev_t dev;

  ctrl_tty= fp->fp_tty;
  assert(ctrl_tty);

  /* First cancel all operations on the underlying tty, and mark those 
   * the tty filedescriptors as closing. Also cancel operations on /dev/tty
   * the process has the right controlling tty and close the underlying filp. */
  for (i= 0, rfp= fproc; i<NR_PROCS; i++, rfp++)
  {
	if (rfp->fp_rootdir == NIL_INODE)
		continue;		/* Process slot is in not in use. */
	check_tty= (rfp->fp_tty == ctrl_tty && rfp->fp_ttylink != 0);
	for(j = 0, odpp= rfp->fp_filp; j<OPEN_MAX; j++, odpp++)
	{			
		odp= *odpp;
		if (!odp)
			continue;
		f= odp->od_filp;
		assert(f);
		if (f->filp_int_flags & FIF_CLOSED)
			continue;
		ip= f->filp_ino;
		if ((ip->i_mode & I_TYPE) != I_CHAR_SPECIAL)
			continue;
		dev= (dev_t) ip->i_zone[0];
		if (dev != ctrl_tty && (!check_tty || 
			((dev >> MAJOR) & BYTE) != CTTY_MAJOR))
		{
			continue;
		}
		cancel_fdops(rfp, j);
		f->filp_int_flags |= FIF_CLOSING;
	}
	if (check_tty)
	{
		assert(rfp->fp_ttyfilp);
		f= rfp->fp_ttyfilp;			/* Free the ttyfilp */
		assert(f->filp_count > 0);
		assert(!(f->filp_int_flags & FIF_CLOSED));
		f->filp_count--;
		if (f->filp_count == 0)
		{
			status= dev_opcl(0, ctrl_tty, FS_PROC_NR, 0);
			if (status != OK) panic("close failed", NO_NUM);
			put_inode(f->filp_ino);
		}
		assert(f->filp_count >= 0);
		rfp->fp_ttyfilp= NIL_FILP;
		rfp->fp_ttylink= 0;
	}
  }

  /* Finally we close all marked references to this tty, and mark them as
   * invalid(closed). */
  for (i= 0, f= filp_table; i<NR_FILPS; i++, f++)
  {
	if (f->filp_count == 0)
		continue;
	if (!(f->filp_int_flags & FIF_CLOSING))
		continue;
	assert(!(f->filp_int_flags & FIF_CLOSED));
	assert ((f->filp_ino->i_mode & I_TYPE) == I_CHAR_SPECIAL);
	dev= f->filp_ino->i_zone[0];
	if (((dev >> MAJOR) & BYTE) != CTTY_MAJOR)
	{
		assert((dev_t)(f->filp_ino->i_zone[0]) == ctrl_tty);
		status= dev_opcl(0, ctrl_tty, FS_PROC_NR, 0);
		if (status != OK) panic("close failed", NO_NUM);
	}
	f->filp_int_flags= (f->filp_int_flags & ~FIF_CLOSING) | FIF_CLOSED;
  }

  /* A consistencty check. Since all references to the tty are closed,
   * we expect that the session leader doesn't have a contrlling terminal
   * anymore.
   */
  if (fp->fp_tty)
	panic("no controlling tty expected", NO_NUM);
}


/*===========================================================================*
 *				ctty_open				     *
 *===========================================================================*/
PUBLIC int ctty_open(open_close, device, dev_task, user_task, flags)
int open_close;
dev_t device;
int dev_task;
int user_task;
int flags;
{
/* This procedure is called from the dmap struct in table.c 
 * to open a controlling tty. */
  
  struct fproc *fp;
  struct filp *f;
  dev_t tty_dev;
  int i;

  fp = &fproc[user_task];
  if (fp->fp_tty == 0)
	return ENXIO;				/* no controlling tty */

  if (fp->fp_ttyfilp == NIL_FILP)
  {
	/* Find a filp for the real tty. */
	tty_dev= fp->fp_tty;
	for (i= 0, f= filp_table; i<NR_FILPS; i++, f++)
	{
		if (f->filp_count == 0)
			continue;
		if ((f->filp_ino->i_mode & I_TYPE) != I_CHAR_SPECIAL)
			continue;
		if ((dev_t)(f->filp_ino->i_zone[0]) != tty_dev)
			continue;
		if (f->filp_int_flags & FIF_CLOSED)
			continue;
		break;
	}
	if (i == NR_FILPS)
		panic("unable to locate tty", NO_NUM);
	fp->fp_ttyfilp= f;
	f->filp_count++;
	assert(fp->fp_ttylink == 0);
  }
  fp->fp_ttylink++;
  
  return OK;
}

/*===========================================================================*
 *				ctty_close				     *
 *===========================================================================*/
PUBLIC int ctty_close(open_close, device, dev_task, user_task, flags)
int open_close;
dev_t device;
int dev_task;
int user_task;
int flags;
{
/* This procedure is called from the dmap struct in table.c when a 
 * controlling tty is closed.
 */

  struct fproc *fp;
  struct filp *f;
  int status;

  fp= &fproc[user_task];
  assert(fp->fp_tty);
  assert(fp->fp_ttylink > 0);
  assert(fp->fp_ttyfilp);

  fp->fp_ttylink--;
  if (fp->fp_ttylink == 0)
  {	/* Let's get rid the ttyfilp. */
	f= fp->fp_ttyfilp;
	fp->fp_ttyfilp= NIL_FILP;
	assert(f->filp_count > 0);
	assert(!(f->filp_int_flags & FIF_CLOSED));
	if (f->filp_count-1 == 0)
	{
		status= dev_opcl(0, fp->fp_tty, user_task, flags);
		if (status != OK) panic("close failed", NO_NUM);
	}
	if (--f->filp_count == 0) put_inode(f->filp_ino);
  }
  return OK;
}

/*===========================================================================*
 *				ctty_rw					     *
 *===========================================================================*/
PUBLIC int ctty_rw(op, device, dev_task, user_task, pos, buf, bytes, ref, flags)
int op;				/* DEV_READ, DEV_WRITE, ... */
dev_t device;			/* major/minor */
int dev_task;			/* which task to call */
int user_task;			/* for whom is the call made */
u64_t pos;			/* position for block special files. */
char *buf;			/* buffer in user space. */
unsigned bytes;			/* number of bytes. */
int ref;			/* extra reference for matching replies. */
int flags;			/* special flags */
{
/* Read or write from the controlling tty. */

  assert(fproc[user_task].fp_tty);

  return dev_rw(op, fproc[user_task].fp_tty, user_task, pos, buf,
	bytes, ref, flags);
}


/*===========================================================================*
 *				ctty_ioctl				     *
 *===========================================================================*/
PUBLIC int ctty_ioctl(device, dev_task, user_task, function, buf, ref, flags)
dev_t device;
int dev_task;
int user_task;
ioreq_t function;
char *buf;
int ref;
int flags;
{
/* Perform an ioctl on the controlling tty. */

  assert(fproc[user_task].fp_tty);

  return dev_ioctl(fproc[user_task].fp_tty, user_task, function, buf, ref, 
									flags);
}


/*===========================================================================*
 *				ctty_cancel				     *
 *===========================================================================*/
PUBLIC int ctty_cancel(operation, device, dev_task, user_task, ref)
int operation;
dev_t device;
int dev_task;
int user_task;
int ref;
{
/* Cancel a read, write or ioctl on the controlling tty. */

  assert(fproc[user_task].fp_tty);

  return dev_cancel(operation, fproc[user_task].fp_tty, user_task, ref);
}


/*===========================================================================*
 *				ctty_msg				     *
 *===========================================================================*/
PUBLIC int ctty_msg(device, dev_task, user_task, m_ptr)
dev_t device;
int dev_task;
int user_task;
message *m_ptr;
{
/* This function is very tricky. It is used to implement the old ioctls
 * that passed many fields in a message instead of passing a pointer
 * to the structure in user space.
 */

  assert(fproc[user_task].fp_tty);

  return dev_msg(fproc[user_task].fp_tty, user_task, m_ptr);
}


/*===========================================================================*
 *				ntty_open				     *
 *===========================================================================*/
PUBLIC int ntty_open(open_close, device, dev_task, user_task, flags)
int open_close;
dev_t device;
int dev_task;
int user_task;
int flags;
{
/* This procedure is called from the dmap struct in table.c on tty opens. */
  
  int r;
  message m;
  struct fproc *fp;

  fp = &fproc[user_task];
  if (!fp->fp_sesldr || (fp->fp_tty != 0)) { /* No session leader. */
	flags |= O_NOCTTY;
  }

  m.m_type	= DEV_OPEN;
  m.NDEV_MINOR	= (device >> MINOR) & BYTE;
  m.NDEV_PROC	= user_task;
  m.NDEV_REF	= 0;
  m.NDEV_MODE	= flags & O_NOCTTY;	/* The only supported flag is O_NOCTTY 
					 */
  /* Call the task. */
  call_ntask(dev_task, &m);
  r= m.REP_STATUS;

  if (r == SUSPEND)
	panic("don't know how to suspend open.", NO_NUM);

  if (r == 1)
  {
	fp->fp_tty = device;	/* set controlling tty */
	r= OK;
  }
  return r;
}


/*===========================================================================*
 *				ntty_close				     *
 *===========================================================================*/
PUBLIC int ntty_close(open_close, device, dev_task, user_task, flags)
int open_close;
dev_t device;
int dev_task;
int user_task;
int flags;
{
/* This procedure is called from the dmap struct in table.c on tty closes.
 */

  int r;
  struct fproc *rfp;
  message m;

  m.m_type	= DEV_CLOSE;
  m.NDEV_MINOR	= (device >> MINOR) & BYTE;
  m.NDEV_PROC	= user_task;
  m.NDEV_REF	= 0;

  /* Call the task. */
  call_ntask(dev_task, &m);
  r= m.REP_STATUS;

  if (r == SUSPEND)
	panic("don't know how to suspend close.", NO_NUM);

  if (r == 1) {	/* Last close. */
	for (rfp = &fproc[0]; rfp < &fproc[NR_PROCS]; rfp++) {
		if (rfp->fp_tty == device) 
		{
			assert(rfp->fp_ttyfilp == NIL_FILP);
			assert(rfp->fp_ttylink == 0);
			rfp->fp_tty = 0;
		}
	}
  }
  return OK;
}


/*
 * $PchId: tty.c,v 1.5 1995/11/28 07:29:49 philip Exp $
 */
