/*
pipe.c

Created:	Feb 4, 1993 by Philip Homburg
*/

#include "fs.h"

#include <signal.h>
#include <minix/com.h>
#include "assert.h"
INIT_ASSERT
#include "file.h"
#include "fproc.h"
#include "inode.h"
#include "param.h"
#include "pipe.h"
#include "super.h"

FORWARD _PROTOTYPE( int pipe_read, (struct od *odp, char *reqbuf,
						unsigned reqsize)	);
FORWARD _PROTOTYPE( int pipe_write, (struct od *odp, char *reqbuf,
						unsigned reqsize)	);
FORWARD _PROTOTYPE( int pipe_copyin, (struct inode *inop, struct pod *podp));

#define VALID_OFFSET(o) ((o) < PIPE_BUF)

/*===========================================================================*
 *				do_pipe					     *
 *===========================================================================*/
PUBLIC int do_pipe()
{
/* Perform the pipe(fil_des) system call. */

  register struct fproc *rfp;
  register struct inode *rip;
  int r;
  struct filp *fil_ptr0, *fil_ptr1;
  struct od *odp0, *odp1;
  int fil_des[2];		/* reply goes here */
  int fs_err= EGENERIC;

  assert(fs_call == PIPE);

  /* Acquire two file descriptors. */
  rfp = fp;
  if ( (r = new_fd(0, R_BIT, &fil_des[0], &odp0)) != OK) return(r);
  fil_ptr0= odp0->od_filp;
  odp0->od_flags |= ODF_INUSE;
  rfp->fp_filp[fil_des[0]] = odp0;
  fil_ptr0->filp_count = 1;
  if ( (r = new_fd(0, W_BIT, &fil_des[1], &odp1)) != OK) {
	odp0->od_flags= 0;
	rfp->fp_filp[fil_des[0]] = NIL_ODP;
	fil_ptr0->filp_count = 0;
	return(r);
  }
  fil_ptr1= odp1->od_filp;
  odp1->od_flags |= ODF_INUSE;
  rfp->fp_filp[fil_des[1]] = odp1;
  fil_ptr1->filp_count = 1;

  /* Make the inode on the pipe device. */
  if ( (rip = alloc_inode(pipe_fs, I_NAMED_PIPE, &fs_err) ) == NIL_INODE) {
	rfp->fp_filp[fil_des[0]] = NIL_ODP;
	odp0->od_flags= 0;
	fil_ptr0->filp_count = 0;
	rfp->fp_filp[fil_des[1]] = NIL_ODP;
	odp1->od_flags= 0;
	fil_ptr1->filp_count = 0;
	return(fs_err);
  }

  if (read_only(rip) != OK) panic("pipe device is read only", NO_NUM);
 
  assert(rip->i_nlinks == 0);
  assert(rip->i_size == 0);
  rip->i_rdfilp= fil_ptr0;
  rip->i_wrfilp= fil_ptr1;
  rip->i_rdhead= NULL;
  rip->i_wrhead= NULL;
  rip->i_update = ATIME | CTIME | MTIME;
  dup_inode(rip);		/* for double usage */
  rw_inode(rip, WRITING);	/* mark inode as allocated */
  fil_ptr0->filp_ino = rip;
  fil_ptr0->filp_flags = O_RDONLY;
  /* The read side maintains the offset of the buffer in the 'pipe file' */
  assert(cmp64u(fil_ptr0->filp_pos, 0) == 0);
  fil_ptr1->filp_ino = rip;
  fil_ptr1->filp_flags = O_WRONLY;
  reply_i1 = fil_des[0];
  reply_i2 = fil_des[1];
  return(OK);
}

/*===========================================================================*
 *				pipe_open				     *
 *===========================================================================*/
PUBLIC int pipe_open(fdnr, filp, rip, bits, oflags)
int fdnr;
struct filp *filp;
struct inode *rip;
Mode_t bits;
int oflags;
{
	struct filp **this_field, **that_field;
	int tmp_fd, i;
	struct fproc *rp;
	unsigned offset;

	assert(bits == R_BIT || bits == W_BIT);

	/* create pointers to the fields in the inode to get some common code
	 * reading and the writing case.
	 */
	if (bits == R_BIT)
	{
		this_field= &rip->i_rdfilp;
		that_field= &rip->i_wrfilp;
	}
	else
	{
		this_field= &rip->i_wrfilp;
		that_field= &rip->i_rdfilp;
	}
	if (*that_field == NULL)
	{
		/* blocking case */
		if (bits == W_BIT && oflags & O_NONBLOCK)
			return ENXIO;	/* common_open will cleanup the mess */
		if (*this_field != NULL)
		{
			/* all readers share a filp and all writers share
			 * another, we have to drop our's.
			 */
			assert(filp->filp_count == 1);
			filp->filp_count= 0;
			put_inode(rip);
			filp= *this_field;
			filp->filp_count++;
			fp->fp_filp[fdnr]->od_filp= filp;
		}
		else
		{
			*this_field= filp;		/* fill in the filp */
		}

		/* non-blocking reads can continue; */
		if (oflags & O_NONBLOCK)
			return OK;

		fd_suspend(EV_POPEN, fdnr);
		return SUSPEND;
	}
	if (*this_field != NULL)
	{
		/* easy case, both a reader and a writer are present already
		 * and since all readers share a filp and all writers share
		 * another, we have to drop our's, and can continue.
		 */
		assert(filp->filp_count == 1);
		filp->filp_count= 0;
		put_inode(rip);
		filp= *this_field;
		filp->filp_count++;
		fp->fp_filp[fdnr]->od_filp= filp;
		return OK;
	}

	*this_field= filp;				/* fill in the filp */

	/* The offset in the buffer is normally stored in the readers filp.
	 * When the filp of the last reader is closed, a pointer to the end of
	 * the buffer is stored in the writers filp (if any). At the first open
	 * of a readers with a writer present we can retrieve that pointer.
	 */
	if (bits == R_BIT)
	{
		offset= cv64u((*that_field)->filp_pos);
		if (offset < rip->i_size)
			offset += PIPE_BUF;
		offset -= rip->i_size;
		assert(VALID_OFFSET(offset));
		filp->filp_pos= cvu64(offset);
	}

	/* Maybe someone is blocked on this fifo. We scan the process table
	 * for EV_POPEN and our inode.
	 */
	for (i= 0, rp = &fproc[0]; i<NR_PROCS; i++,  rp++)
	{
		switch(rp->fp_event)
		{
		case EV_POPEN:
			tmp_fd= rp->fp_state.fp_popen.fp_fd;
			if (rp->fp_filp[tmp_fd]->od_filp->filp_ino != rip)
			{
				/* wrong fifo */
				break;
			}
			assert(rp->fp_filp[tmp_fd]->od_filp == *that_field);
			reply(i, tmp_fd);
			rp->fp_event= EV_NONE;
			break;
		case EV_NONE:
		case EV_LOCK:
		case EV_FWAIT:
		case EV_READ:
		case EV_WRITE:
		case EV_IOCTL:
			break;
		default:
			panic("pipe_open: illegal event", rp->fp_event);
		}
	}
	return OK;
}

/*===========================================================================*
 *				pipe_close				     *
 *===========================================================================*/
PUBLIC void pipe_close(rip, filp)
struct inode *rip;
struct filp *filp;
{
	struct pod *podp;
	int err;
	uoff_t offset;

	if (filp == rip->i_wrfilp)
	{
		/* close write side of the pipe. */

		/* writers must be gone now, remove the filp from the inode */
		assert(rip->i_wrhead == NULL);
		rip->i_wrfilp= NULL;

		/* readers get 0 */
		while(rip->i_rdhead != NULL)
		{
			podp= rip->i_rdhead;
			rip->i_rdhead= podp->pod_rdnext;
			fd_revive(podp->pod_task, podp->pod_fd, ASIO_READ, 0);
		}
		return;
	}
	assert (filp == rip->i_rdfilp);
	/* close read side of the pipe. */

	/* readers must be gone now, remove the filp from the inode */
	assert(rip->i_rdhead == NULL);
	rip->i_rdfilp= NULL;

	/* writers get SIGPIPE/EPIPE */
	while(rip->i_wrhead != NULL)
	{
		podp= rip->i_wrhead;
		rip->i_wrhead= podp->pod_wrnext;
		sys_kill(podp->pod_task, SIGPIPE);
		if (podp->pod_wroffset != 0)
			err= podp->pod_wroffset;
		else
			err= EPIPE;
		fd_revive(podp->pod_task, podp->pod_fd, ASIO_WRITE, err);
	}

	/* if the write side is still open, we store the offset of
	 * buffer in the write filp, for recovery by a future
	 * reader.
	 */
	if (rip->i_wrfilp != NULL)
	{
		offset= cv64u(filp->filp_pos) + rip->i_size;
		if (offset >= PIPE_BUF)
			offset -= PIPE_BUF;
		assert(VALID_OFFSET(offset));
		rip->i_wrfilp->filp_pos= cvu64(offset);
	}
}


/*===========================================================================*
 *				pipe_rdwr				     *
 *===========================================================================*/
PUBLIC int pipe_rdwr(fdnr, rw_flag, reqbuf, reqsize)
int fdnr;
int rw_flag;
char *reqbuf;
unsigned reqsize;
{
	int r;
	struct od *odp;
	unsigned ip_flag;
	int event;
	int fs_err= EGENERIC;

	if (rw_flag == READING)
	{
		ip_flag= ODF_RD_IP;
		event= EV_READ;
	}
	else if (rw_flag == WRITING)
	{
		ip_flag= ODF_WR_IP;
		event= EV_WRITE;
	}
	else 
		panic("pipe_rdwr: strange value in rw_flag", rw_flag);

	odp= get_fd(fdnr, &fs_err);
	assert(odp != NULL);
	if (odp->od_flags & ip_flag)
	{
		return EALREADY;
	}
	if (rw_flag == READING)
		r= pipe_read(odp, reqbuf, reqsize);
	else
		r= pipe_write(odp, reqbuf, reqsize);
	if (r != SUSPEND)
	{
		return r;
	}
	odp->od_pod.pod_task= who;
	odp->od_pod.pod_fd= fdnr;
	odp->od_flags |= ip_flag;
	if (odp->od_flags & ODF_ASYNCH)
		return EINPROGRESS;
	fd_suspend(event, fdnr);
	return SUSPEND;
}

/*===========================================================================*
 *				pipe_read				     *
 *===========================================================================*/
PRIVATE int pipe_read(odp, reqbuf, reqsize)
struct od *odp;
char *reqbuf;
unsigned reqsize;
{
	int r;
	unsigned off;
	unsigned buf_offset;
	unsigned size, offset, wrsize, wroffset;
	struct pod *podp;
	struct inode *inop;
	struct filp *filp;

	assert(odp != NULL);
	filp= odp->od_filp;
	assert(filp != NULL);
	inop= filp->filp_ino;
	assert(inop != NULL);

	/* Do we have a read queue? If so suspend or return EAGAIN depending on
	 * blocking/non-blocking read.
	 */
	if (inop->i_rdhead != NULL)
	{
		if (filp->filp_flags & O_NONBLOCK)
			return EAGAIN;
		podp= inop->i_rdhead;
		while (podp->pod_rdnext != NULL)
			podp= podp->pod_rdnext;
		podp->pod_rdnext= &odp->od_pod;
		odp->od_pod.pod_rdbuf= reqbuf;
		odp->od_pod.pod_rdsize= reqsize;
		odp->od_pod.pod_rdnext= NULL;
		return SUSPEND;
	}

	offset= 0;

	/* Check for data in the buffer. */
	while (inop->i_size != 0 && offset < reqsize)
	{
		size= inop->i_size;
		if (size > reqsize-offset)
			size= reqsize-offset;
		buf_offset= cv64u(filp->filp_pos);
		if (buf_offset + size > PIPE_BUF)
			size= PIPE_BUF-buf_offset;
		off = buf_offset % BLOCK_SIZE;
							/* offset in blk */
		if (off + size > BLOCK_SIZE)
			size= BLOCK_SIZE-off;

		inop->i_size += PIPE_BUF;
		r = rw_chunk(inop, cvu64(buf_offset), off, size, size,
					READING, reqbuf+offset, SEG_D, who);
		inop->i_size -= PIPE_BUF;
		if (r != OK)
		{
			if (offset != 0)
				return offset;
			assert(r == EFAULT || r == EIO);
			return r;
		}

		offset += size;
		inop->i_size -= size;
		buf_offset += size;
		if (buf_offset == PIPE_BUF)
			buf_offset= 0;
		assert(VALID_OFFSET(buf_offset));
		filp->filp_pos= cvu64(buf_offset);
	}
	if (offset == reqsize)
		return reqsize;

	/* At this point, the buffer is empty. We reset the pointer. */
	assert(inop->i_size == 0);
	filp->filp_pos= cvu64(0);

	/* Do we have some writers available for direct user to user copies? */
	while (inop->i_wrhead && offset < reqsize)
	{
		podp= inop->i_wrhead;
		size= reqsize - offset;
		wrsize= podp->pod_wrsize;
		wroffset= podp->pod_wroffset;
		if (size > wrsize-wroffset)
			size= wrsize-wroffset;
		r= sys_copy(podp->pod_task, SEG_D, 
			(phys_bytes) (podp->pod_wrbuf+wroffset), who, SEG_D, 
			(phys_bytes) (reqbuf+offset), (phys_bytes) size);
		if (r != OK)
		{
			/* The reader or the writer supplied a bad buffer. */
			assert(r == EFAULT);
			if (sys_umap(who, SEG_D, (vir_bytes)reqbuf+offset, 
								size) == 0)
			{
				/* The reader messed up */
				if (offset != 0)
					return offset;
				else
					return EFAULT;
			}
			assert(sys_umap(podp->pod_task, SEG_D, 
				(vir_bytes)(podp->pod_wrbuf + wroffset), 
				size) == 0);
			inop->i_wrhead= podp->pod_wrnext;

			if (wroffset == 0)
				r= EFAULT;
			else
				r= wroffset;
			fd_revive(podp->pod_task, podp->pod_fd, ASIO_WRITE, r);
			continue;
		}
		offset += size;
		wroffset += size;
		podp->pod_wroffset= wroffset;
		if (wroffset == podp->pod_wrsize)
		{
			inop->i_wrhead= podp->pod_wrnext;
			fd_revive(podp->pod_task, podp->pod_fd, ASIO_WRITE,
				wroffset);
		}
		if (offset == reqsize && inop->i_wrhead != NULL)
		{
			/* do we have to do a copyin? */
			podp= inop->i_wrhead;
			if (podp->pod_wrsize-podp->pod_wroffset <= PIPE_BUF)
			{
				inop->i_wrhead= podp->pod_wrnext;
				r= pipe_copyin(inop, podp);
				if (r < 0 && podp->pod_wroffset > 0)
					r= podp->pod_wroffset;
				else if (r >= 0)
					r += podp->pod_wroffset;
				fd_revive(podp->pod_task, podp->pod_fd, 
					ASIO_WRITE, r);
			}
		}
	}

	/* Before we suspend, we check for a pipe with only readers, and
	 * for non-blocking I/O. If we got some data already, return
	 * immediately.
	 */
	if (offset != 0)
		return offset;
	if (inop->i_wrfilp == NULL)
		return 0;
	if (filp->filp_flags & O_NONBLOCK)
		return EAGAIN;
	inop->i_rdhead= &odp->od_pod;
	odp->od_pod.pod_rdbuf= reqbuf;
	odp->od_pod.pod_rdsize= reqsize;
	odp->od_pod.pod_rdnext= NULL;
	return SUSPEND;
}

/*===========================================================================*
 *				pipe_write				     *
 *===========================================================================*/
PRIVATE int pipe_write(odp, reqbuf, reqsize)
struct od *odp;
char *reqbuf;
unsigned reqsize;
{
	int r;
	unsigned off;
	unsigned buf_offset;
	unsigned size, offset, buf_size, saved_offset;
	struct pod *podp;
	struct inode *inop;
	struct filp *filp;

	assert(odp != NULL);
	filp= odp->od_filp;
	assert(filp != NULL);
	inop= filp->filp_ino;
	assert(inop != NULL);

	/* Do we have a write queue? If so suspend or return EAGAIN depending on
	 * blocking/non-blocking read.
	 */
	if (inop->i_wrhead != NULL)
	{
		if (filp->filp_flags & O_NONBLOCK)
			return EAGAIN;
		podp= inop->i_wrhead;
		while (podp->pod_wrnext != NULL)
			podp= podp->pod_wrnext;
		podp->pod_wrnext= &odp->od_pod;
		odp->od_pod.pod_wrbuf= reqbuf;
		odp->od_pod.pod_wrsize= reqsize;
		odp->od_pod.pod_wroffset= 0;
		odp->od_pod.pod_wrnext= NULL;
		return SUSPEND;
	}

	offset= 0;

	/* Do we have some readers available for direct user to user copies? */
	while (inop->i_rdhead && offset < reqsize)
	{
		podp= inop->i_rdhead;
		size= reqsize - offset;
		if (size > podp->pod_rdsize)
			size= podp->pod_rdsize;
		r= sys_copy(who, SEG_D, (phys_bytes) (reqbuf+offset),
			podp->pod_task, SEG_D, (phys_bytes) podp->pod_rdbuf,
			(phys_bytes) size);
		if (r != OK)
		{
			/* The reader or the writer supplied a bad buffer. */
			assert(r == EFAULT);
			if (sys_umap(who, SEG_D, (vir_bytes)reqbuf+offset, 
								size) == 0)
			{
				/* The writer messed up */
				if (offset != 0)
					return offset;
				else
					return EFAULT;
			}
			assert(sys_umap(podp->pod_task, SEG_D, 
				(vir_bytes)podp->pod_rdbuf, size) == 0);
			inop->i_rdhead= podp->pod_rdnext;
			fd_revive(podp->pod_task, podp->pod_fd, ASIO_READ,
				EFAULT);
			continue;
		}
		offset += size;
		inop->i_rdhead= podp->pod_rdnext;
		fd_revive(podp->pod_task, podp->pod_fd, ASIO_READ, size);
	}

	if (offset == reqsize)
		return offset;

	assert(inop->i_rdhead == NULL);

	/* Check for a pipe with only writers. Send SIGPIPE is so, and
	 * return EPIPE if no bytes have been transfered so far.
	 */
	if (inop->i_rdfilp == NULL)
	{
		sys_kill(who, SIGPIPE);
		if (offset != 0)
			return offset;
		return(EPIPE);
	}

	/* Check if we can finish this request by storing data in the buffer.
	 * If not, don't do anything
	 */
	if (PIPE_BUF - inop->i_size < reqsize - offset)
	{
		if (filp->filp_flags & O_NONBLOCK)
			return EAGAIN;
		inop->i_wrhead= &odp->od_pod;
		odp->od_pod.pod_wrbuf= reqbuf;
		odp->od_pod.pod_wrsize= reqsize;
		odp->od_pod.pod_wroffset= offset;
		odp->od_pod.pod_wrnext= NULL;
		return SUSPEND;
	}

	buf_size= PIPE_BUF - inop->i_size;
	buf_offset= cv64u(inop->i_rdfilp->filp_pos) + inop->i_size;
	if (buf_offset >= PIPE_BUF)
		buf_offset -= PIPE_BUF;
	assert(VALID_OFFSET(buf_offset));

	saved_offset= offset;
	while (offset < reqsize)
	{
		size= buf_size;
		if (size > reqsize-offset)
			size= reqsize-offset;
		if (buf_offset + size > PIPE_BUF)
			size= PIPE_BUF-buf_offset;
		off = (unsigned int) (buf_offset % BLOCK_SIZE);
							/* offset in blk */
		if (off + size > BLOCK_SIZE)
			size= BLOCK_SIZE-off;

		inop->i_size += PIPE_BUF;
		r = rw_chunk(inop, cvu64(buf_offset), off, size, size,
				WRITING, reqbuf+offset, SEG_D, who);
		inop->i_size -= PIPE_BUF;
		if (r != OK)
		{
			if (saved_offset != 0)
				return saved_offset;
			assert(r == EFAULT || r == EIO || r == ENOSPC);
			return r;
		}

		offset += size;
		buf_size -= size;
		buf_offset += size;
		if (buf_offset == PIPE_BUF)
			buf_offset= 0;
		assert(VALID_OFFSET(buf_offset));
	}
	assert(offset == reqsize);

	assert(VALID_OFFSET(buf_size));
	inop->i_size= PIPE_BUF - buf_size;

	if (buf_offset < inop->i_size)
		buf_offset += PIPE_BUF;
	inop->i_rdfilp->filp_pos= cvu64(buf_offset - inop->i_size);
	assert(VALID_OFFSET(cv64u(inop->i_rdfilp->filp_pos)));
	return reqsize;
}


/*===========================================================================*
 *				pipe_copyin				     *
 *===========================================================================*/
PRIVATE int pipe_copyin(inop, podp)
struct inode *inop;
struct pod *podp;
{
	unsigned buf_size, buf_offset;
	unsigned offset, reqsize;
	unsigned size, off;
	int r;
	char *reqbuf;

	buf_size= PIPE_BUF - inop->i_size;
	buf_offset= cv64u(inop->i_rdfilp->filp_pos) + inop->i_size;
	if (buf_offset >= PIPE_BUF)
		buf_offset -= PIPE_BUF;
	assert(VALID_OFFSET(buf_offset));
	offset= podp->pod_wroffset;
	reqsize= podp->pod_wrsize;
	reqbuf= podp->pod_wrbuf;

	while (offset < reqsize)
	{
		size= buf_size;
		if (size > reqsize-offset)
			size= reqsize-offset;
		if (buf_offset + size > PIPE_BUF)
			size= PIPE_BUF-buf_offset;
		off = (unsigned int) (buf_offset % BLOCK_SIZE);
							/* offset in blk */
		if (off + size > BLOCK_SIZE)
			size= BLOCK_SIZE-off;

		inop->i_size += PIPE_BUF;
		r = rw_chunk(inop, cvu64(buf_offset), off, size, size,
				WRITING, reqbuf+offset, SEG_D, podp->pod_task);
		inop->i_size -= PIPE_BUF;
		if (r != OK)
		{
			assert(r == EFAULT || r == EIO || r == ENOSPC);
			return r;
		}

		offset += size;
		buf_size -= size;
		buf_offset += size;
		if (buf_offset == PIPE_BUF)
			buf_offset= 0;
		assert(VALID_OFFSET(buf_offset));
	}
	assert(offset == reqsize);

	assert(VALID_OFFSET(buf_size));
	inop->i_size= PIPE_BUF - buf_size;

	if (buf_offset < inop->i_size)
		buf_offset += PIPE_BUF;
	inop->i_rdfilp->filp_pos= cvu64(buf_offset - inop->i_size);
	assert(VALID_OFFSET(cv64u(inop->i_rdfilp->filp_pos)));
	return offset - podp->pod_wroffset;
}


/*===========================================================================*
 *				pipe_cancel				     *
 *===========================================================================*/
PUBLIC int pipe_cancel(inop, operation, task, fdnr)
struct inode *inop;
int operation;
int task;
int fdnr;
{
	struct pod *prev, *curr;

	if (operation == ASIO_READ)
	{
		/* cancel a read operation */
		for (prev= NULL, curr= inop->i_rdhead; curr; 
					prev= curr, curr= curr->pod_rdnext)
		{
			if (curr->pod_task == task && curr->pod_fd == fdnr)
				break;
		}
		if (curr == NULL)
			panic("pipe_cancel: unable to find request to cancel", 
				NO_NUM);
		if (prev == NULL)
			inop->i_rdhead= curr->pod_rdnext;
		else
			prev->pod_rdnext= curr->pod_rdnext;
		return EINTR;
	}
	if (operation == ASIO_WRITE)
	{
		/* cancel a write operation */
		for (prev= NULL, curr= inop->i_wrhead; curr; 
					prev= curr, curr= curr->pod_wrnext)
		{
			if (curr->pod_task == task && curr->pod_fd == fdnr)
				break;
		}
		if (curr == NULL)
			panic("pipe_cancel: unable to find request to cancel", 
				NO_NUM);
		if (prev == NULL)
			inop->i_wrhead= curr->pod_wrnext;
		else
			prev->pod_wrnext= curr->pod_wrnext;
		if (curr->pod_wroffset != 0)
			return curr->pod_wroffset;
		return EINTR;
	}
	panic("pipe_cancel: illegal cancel request", operation);
	/*NOTREACHED*/
}

/*
 * $PchId: pipe.c,v 1.6 1996/02/29 23:09:24 philip Exp $
 */
