/*
name.c

Name cache.

Created:	February 1995 by Philip Homburg <philip@cs.vu.nl>
*/

#include "fs.h"
#include <string.h>

#include "assert.h"
INIT_ASSERT
#include "name.h"

_PROTOTYPE( static void unhash_ent, (struct name *np)			);

void flush_names(dev, ino)
dev_t dev;
ino_t ino;
{
	struct name *np;
	int i;

	for (i= 0, np= name_cache; i<NR_NAMES; i++, np++)
	{
		if (np->n_dev == NO_DEV)
			continue;

		/* Note that this will flush a bit too much... */
		if (dev != NO_DEV && np->n_dev != dev && np->n_tdev != dev)
			continue;
		if (ino != 0 && np->n_ino != ino && np->n_tino != ino)
			continue;

		unhash_ent(np);
		np->n_dev= NO_DEV;
	}
}

void flush_name(dev, ino, name)
dev_t dev;
ino_t ino;
char *name;
{
	struct name *np;
	int hash;

	hash= hash_name(dev, ino, name);
	np= lookup_name(hash, dev, ino, name);
	if (np == NULL)
		return;
	unhash_ent(np);
	np->n_dev= NO_DEV;
}

int hash_name(dev, ino, name)
dev_t dev;
ino_t ino;
char *name;
{
	int i;
	size_t l;
	unsigned char *ucp;
	unsigned long hash;

	l= strlen(name);
	if (l+1 > NAME_SIZE)
		return -1;
	hash= 0;
	ucp= (unsigned char *)name;
	for (i= 0; i<l; i++)
		hash ^= (ucp[i] << i);
	hash ^= dev;
	hash ^= ino;
	hash ^= (hash >> 10);
	return hash & (NR_NAMES_HASH-1);
}

struct name *lookup_name(hash, dev, ino, name)
int hash;
dev_t dev;
ino_t ino;
char *name;
{
	struct name *np;
	char *np_str;
	int n;

	if (hash == -1)
		return NULL;			/* Name too long */
	for (np= name_hash[hash]; np != NULL; np= np->n_next)
	{
		if (np->n_dev != dev || np->n_ino != ino)
			continue;
		n= np-name_cache;
		np_str= &name_strings[n * NAME_SIZE];
		if (strcmp(name, np_str) == 0)
		{
			np->n_usage++;
			return np;
		}
	}
	return NULL;
}

void enter_name(hash, dev, ino, name, tdev, tino, ip)
int hash;
Dev_t dev;
Ino_t ino;
char *name;
Dev_t tdev;
Ino_t tino;
struct inode *ip;
{
	static struct name *next_ent= name_cache;

	int i, n;
	struct name *np;
	char *np_str;

#if DEBUG & 0
 { where(); printf("enter_name(0x%x, 0x%x, %d, '%s', 0x%x, %d, 0x%x)\n",
		hash, dev, ino, name, tdev, tino, ip); }
#endif

	if (hash == -1)
		return;	/* Name too long */

	for (i= 0, np= next_ent; i<2*NR_NAMES; i++, np++)
	{
		if (np == &name_cache[NR_NAMES])
			np= name_cache;

		if (np->n_dev == NO_DEV)
			break;

		if (np->n_usage != 0)
		{
			np->n_usage= 0;
			continue;
		}
		unhash_ent(np);
		np->n_dev= NO_DEV;
		break;
	}
	next_ent= np+1;
	np->n_dev= dev;
	np->n_ino= ino;
	np->n_tdev= tdev;
	np->n_tino= tino;
	np->n_ip= ip;
	np->n_usage= 1;
	np->n_hash= hash;

	n= np-name_cache;
	np_str= &name_strings[n * NAME_SIZE];
	strcpy(np_str, name);

	np->n_next= name_hash[hash];
	name_hash[hash]= np;
}

static void unhash_ent(np)
struct name *np;
{
	struct name *prev, *curr;

	for (prev= 0, curr= name_hash[np->n_hash]; curr != np && curr != NULL;
		prev= curr, curr= curr->n_next)
	{
		;	/* Do nothing */
	}
	assert(curr != NULL);
	if (prev == NULL)
		name_hash[np->n_hash]= curr->n_next;
	else
		prev->n_next= curr->n_next;
}

/*
 * $PchId: name.c,v 1.3 1995/11/28 08:32:14 philip Exp $
 */
